UNIT HListBox;    {by Kurt Herzog, 10/16/93                 }
                  {A Horizontal-scrolling ListBox descendant}

{NOTICE:  I make no claims to or for this code- use it at your           }
{         own risk.  I hope that it will save someone the several        }
{         hours I spent learning how to make it work!  If (when!)        }
{         you find any bugs or ways to improve it (surely there          }
{         are some) please send me the changes so we will each           }
{         have learned something.                                        }
{          - -  Kurt Herzog     CompuServe 72122,2023                    }

{         This code was inspired in part by an article                   }
{         in Dr. Dobb's Windows Sourcebook, Fall 1993,                   }
{         by Ted Faison: "Horizontally Scrollable Listboxes".            }


{$X+}     {Extended Syntax}
{------------------------------------------------------------------------}
{-------------------------}     INTERFACE     {--------------------------}
{------------------------------------------------------------------------}

USES
  {.....General purpose Units}
  Strings,               {Null-terminated strings}
  WinTypes,              {Defines Windows API for Pascal}
  WinProcs,              {Defines Func/Proc headers for API}
  {.....ObjectWindows Units}
  OWindows,              {Basic ObjectWindows Unit}
  ODialogs;              {Basic ObjectWindows Unit}

{- - - - - - - - - - -   Define Object Type     - - - - - - - - - - - - -}
TYPE

{THListBox is a TListBox descendant that supports Horizontal scrolling.  }
{This object needs no interventions; just instantiate it in place of the }
{standard TListBox.  Horizontal scroll bar is maintained according to the}
{size of the longest string that is contained in the listbox.            }

{NOTE - - NOTE - - NOTE!  Contrary to the BP7 documentation, the Run-Time}
{library does NOT define virtual methods for TListbox, except for        }
{GetClassName, GetMsgID (Private), and Transfer.                         }

  PHListBox       = ^THListBox;
  THListBox       = Object(TListBox)
    LargestExtent : Integer;   {Largest Text Extent contained in ListBox}
    constructor   Init(AParent: PWindowsObject; AnId: Integer;
                       X, Y, W, H: Integer);
    constructor   InitResource(AParent: PWindowsObject;
                               ResourceID: Integer);
    function      AddString (AString: Pchar): Integer;
    function      InsertString (AString: Pchar; Index: Integer): Integer;
    function      DeleteString (Index: Integer): Integer;
    procedure     ClearList;
    function      Transfer (DataPtr: Pointer; TransferFlag: Word): Word;
                    virtual;
    procedure     ChangeFont (const NewFont: hFont);
    function      MeasureString (Const AItem: Integer): Integer; virtual;
  end;

{------------------------------------------------------------------------}
{-------------------------}  IMPLEMENTATION   {--------------------------}
{------------------------------------------------------------------------}

constructor THListBox.Init(AParent: PWindowsObject; AnId: Integer;
                           X, Y, W, H: Integer);
begin
  Inherited Init(AParent, AnID, X, Y, W, H);
  LargestExtent := 0;
end;

constructor THListBox.InitResource(AParent: PWindowsObject;
                                   ResourceID: Integer);
begin
  Inherited InitResource(AParent, ResourceID);
  LargestExtent := 0;
end;

function THListBox.AddString(AString: Pchar): Integer;
{function returns the index position of the added string}
var
  Item : Integer;

begin
  {Call the ancestral method and then measure the added string}
  Item := Inherited AddString(AString);
  MeasureString(Item);
  AddString := Item;
end;

function THListBox.InsertString(AString: Pchar; Index: Integer): Integer;
{function returns index position of inserted string}
var
  Item : Integer;

begin
  {Call the ancestral method and then measure the inserted string}
  Item := Inherited InsertString(AString, Index);
  MeasureString(Item);
  InsertString := Item;
end;

function THListBox.DeleteString(Index: Integer): Integer;
{function returns the count of strings in the listbox}
var
  Item     : Integer;
  ItemSize : Integer;
  ItemCount: Integer;

begin
  {Measure the item and then call the ancestral method to delete it}
  ItemSize := MeasureString(Index);
  ItemCount := Inherited DeleteString(Index);
  {If item removed was the largest, measure the remaining strings}
  {to find the remaining longest extent.                         }
  if ItemSize >= LargestExtent then
    begin
      {If no items left in list set Horizontal extent to zero  }
      {(removes scroll bar) otherwise read the list to find    }
      {the largest item left & set extent to that size.        }
      {Scrolling the listbox all the way to the left removes   }
      {the scroll bar if the longest string is fully visible.  }
      LargestExtent := 0;
      if ItemCount = 0 then
        SendMessage(HWindow,LB_SetHorizontalExtent,0,0)
      else
        begin
          SendMessage(HWindow,wm_HScroll,sb_Top,0);
          for Item :=  0 to ItemCount -1 do MeasureString(Item);
        end;
    end;
  DeleteString := ItemCount;
end;

procedure THListBox.ClearList;
begin
  {Call the ancestral method to clear the listbox}
  Inherited ClearList;
  {Set size back to zero & remove scroll bar (if any)}
  LargestExtent := 0;
  SendMessage(HWindow,LB_SetHorizontalExtent,0,0);
end;

function THListBox.Transfer (DataPtr: Pointer; TransferFlag: Word): Word;
{function returns the number of bytes transfered}
var
  Item : Integer;

begin
  {Call the ancestral method to Transfer the listbox contents }
  Transfer := Inherited Transfer (DataPtr, TransferFlag);
  {If we are reloading the listbox reset largest extent size        }
  {and then measure all the strings.  This is necessary because     }
  {the AddString Method called by ancestral Transfer is not virtual.}
  if TransferFlag = tf_SetData then
    begin
      LargestExtent := 0;
      for Item :=  0 to GetCount -1 do MeasureString(Item);
    end;
end;

procedure THListBox.ChangeFont (const NewFont: hFont);
var
  Item : Integer;

begin
  {Scroll the list all the way back so the scroll bar will be}
  {deleted if the new font size does not require it.  Then   }
  {change the font and remeasure all the strings.            }
  SendMessage(HWindow,LB_SetHorizontalExtent,0,0);
  SendMessage(HWindow,wm_SetFont,NewFont,LongInt(TRUE));
  LargestExtent := 0;  {remeasure all strings}
  For Item := 0 to GetCount -1 do MeasureString(Item);
end;

function THListBox.MeasureString (Const AItem: Integer): Integer;
{MeasureString returns the width of the string in pixels.}
var
  PItem     : Pchar;
  ItemSize  : Integer;
  LBDC      : HDc;
  OldFont   : hFont;
  LBFont    : hFont;
  TextLength: Integer;

begin
  {Read the specified sting from the listbox and measure it's length.}
  {Reset the listbox horizontal extent if this string is longer.     }
  ItemSize := SendMessage(HWindow,lb_GetTextLen,AItem,0);
  TextLength := LargestExtent;  {Default return value if error}
  GetMem(PItem,ItemSize + 2);   {Leave room for 1 more Char + nul}
  if PItem <> nil then
    begin
      SendMessage(HWindow,lb_GetText,AItem,LongInt(PItem));
      {Add a character to allow space at end of line    }
      {when ListBox is scrolled all the way to the right}
      StrCat(PItem,'X');
      LBDC := GetDC(HWindow);
      LBFont := SendMessage(HWindow,wm_GetFont,0,0);
      if LBFont <> 0 then              {0 if default font in use}
        begin
          OldFont := SelectObject(LBDC,LBFont);
          TextLength := LoWord(GetTextExtent(LBDC,PItem,StrLen(PItem)));
          SelectObject(LBDC,OldFont);  {Restore the original font handle}
        end
      else
        TextLength := LoWord(GetTextExtent(LBDC,PItem,StrLen(PItem)));
      ReleaseDC(HWindow,LBDC);
      if TextLength > LargestExtent then
        begin
          SendMessage(HWindow,LB_SetHorizontalExtent,TextLength,0);
          LargestExtent := TextLength;
        end;
      FreeMem(PItem,ItemSize + 2);
    end;
  MeasureString := TextLength;
end;

END.