(************************************************************************)
(*                                                                      *)
(*  Program ex. to      : "Tips & Tricks in Turbo Pascal", SysTime 1993 *)
(*                                                                      *)
(*  By                  : Martin Israelsen                              *)
(*                                                                      *)
(*  Title               : LZWCOMP.PAS                                   *)
(*                                                                      *)
(*  Description         : LZW compression                               *)
(*                        This example is a "quick" conversion of the   *)
(*                        WRITEGIF/READGIF example.                     *)
(*                                                                      *)
(*                        NOV-93: Overflow bug fixed.                   *)
(*                                                                      *)
(************************************************************************)
(*$I Define.Inc*)
Unit
   LZWComp;

Interface

Uses
  Typ;

Function DeCompresLZW(InName,OutName: PfStr): Integer;
Function CompresLZW(InName,OutName: PfStr): Integer;

Implementation

Uses
   Buffer;

Const
  CMinCodeSize = 8;
  CBufferSize  = 10000;
  CMaxCode     = 4095;
  CTableSize   = 5003;

Type
  TCodeRec     = Record
                    PredCode      : Word;
                    CharCode      : Byte;
                 End;

  PReadLZW     = ^TReadLZW;
  TReadLZW     = Record
                    BitMask,                      (* Bitmask                        *)
                    InitCodeSize,                 (* Original CodeSizeValue         *)

                    ClearCode,                    (* ClearCode, EOFCode, FirstFree  *)
                    EofCode,                      (* FreeCode, MaxCode, Curcode     *)
                    FirstFree,                    (* OldCode, InCode is variables   *)
                    FreeCode,                     (* to the LZW algorithm,          *)
                    MaxCode,
                    CurCode,
                    OldCode,
                    InCode,
                    OutBufferCount: Word;         (* OutBufferCount is the number   *)
                                                  (* of chars in the buffer         *)
                    CurChar       : Byte;         (* New char                       *)

                    OutBuffer     : Array[0..1024] Of Byte;  (* hashing table to quick search *)
                    CodeTable     : Array[0..CTableSize] Of TCodeRec;
                 End;

Function DecompresLZW(InName,OutName: PfStr): Integer;
(* Returncodes:                   *)
(* 0 : Decompression ok           *)
(* -1: Not enough memory          *)
(* -2: Error during fileread      *)
(* -3: Error during decompression *)
Label
  Exit;
Var
  PLZW       : PReadLZW;
  Code,                     (* Code, CodeSize og ReadMask is placed in the stack, *)
  CodeSize,                 (* so that ReadCode can reach them without using      *)
  ReadMask   : Word;        (* the PLZW pointer.                                  *)
  OutBuffer,
  InBuffer   : Array[0..5*1024] Of Byte;
  InUsed,
  OutUsed    : Integer;
  InFile,
  OutFile    : File;

  Function DecodeLZW: Boolean;
  Label
    ErrorExit;
  Var
    BitOffset,W  : Word;        (* BitOffset is bitcodeindex in InBuffer    *)

    Procedure BlockLoad;
    Var
      W,W1,Index,Rest: Word;
    Begin
       Index:=BitOffset Shr 3;

       If (Index>0) And (InUsed=Sizeof(InBuffer)) And
          (Index>(Sizeof(InBuffer))-1000) Then
       Begin
          Rest:=Sizeof(InBuffer)-Index;

          Move(InBuffer[Index],InBuffer[0],Rest);
          BitOffset:=BitOffset And 7;

          BlockRead(InFile,InBuffer[Rest],Sizeof(InBuffer)-Rest,W);

          InUsed:=Rest+W;
       End;
    End;

     Procedure ReadCode;
     Begin
        (* Returns a bitcode of a given length. To make the Pascal       *)
        (* code as quick as possible, its very "compact". Since the code *)
        (* can be up to 12 bits long, we're doing a longint read, and    *)
        (* shift afterwards                                              *)

        Code:=(PLong(@InBuffer[BitOffset Shr 3])^ Shr (BitOffset Mod 8)) And ReadMask;
        Inc(BitOffset,CodeSize);
     End;

     Procedure DoClear;
     Begin
        (* A ClearCode is received, reinit the LZW tree *)

        With PLZW^ Do
        Begin
           CodeSize:=InitCodeSize;
           MaxCode:=4 Shl (CodeSize-2);
           FreeCode:=FirstFree;
           ReadMask:=Not Word($FFF8 Shl (CodeSize-3));
        End;
     End;

     Procedure AddByte(B: Byte);
     Var
       W: Word;
     Begin
        OutBuffer[OutUsed]:=B;
        Inc(OutUsed);

        If OutUsed=Sizeof(OutBuffer) Then
        Begin
           BlockWrite(OutFile,OutBuffer[0],OutUsed,W);
           OutUsed:=0;
        End;

        BlockLoad;
     End;

  Begin
     With PLZW^ Do
     Begin
        BitOffset:=0; DecodeLZW:=False;

        Repeat
           (* Read a bitcode and process it, if it isnt a EofCode *)

           ReadCode;

           If Code<>EofCode Then
           Begin
              If Code=ClearCode Then
              Begin
                 (* If it's a ClearCode, then reinit the LZW-tree    *)
                 (* otherwise put in the extra char                  *)

                 DoClear;
                 ReadCode;

                 CurCode:=Code;
                 OldCode:=Code;
                 CurChar:=Code And BitMask;

                 AddByte(CurChar);
              End Else
              Begin
                 (* It was not a ClearCode and not an EofCode           *)
                 (* So it's gotta be a entrycode                        *)

                 CurCode:=Code;
                 InCode:=Code;

                 If Code>=FreeCode Then
                 Begin
                    (* If the code is larger than Succ(FreeCode), then *)
                    (* there is error in data, otherwise...            *)

                    If Code>Succ(FreeCode) Then Goto ErrorExit;

                    (* it must be an uknown code, process it according *)
                    (* to the problemdesciption i chap. 6              *)

                    CurCode:=OldCode;
                    OutBuffer[OutBufferCount]:=CurChar;
                    Inc(OutBufferCount);
                 End;

                 (* If CodeSize is bigger then Bitmask, it's an Entry-     *)
                 (* code. Follow the tree and output all values until end- *)
                 (* node                                                   *)

                 While CurCode>BitMask Do
                 With CodeTable[CurCode] Do
                 Begin
                    OutBuffer[OutBufferCount]:=CharCode;
                    CurCode:=PredCode;
                    Inc(OutBufferCount);
                 End;

                 (* The last code is a databyte, put it in also            *)

                 CurChar:=CurCode And BitMask;
                 OutBuffer[OutBufferCount]:=CurChar;
                 Inc(OutBufferCount);

                 (* Next, process all the codes via AddByte. Remember       *)
                 (* to do it backwards, since the codes is in reverse order *)

                 For W:=Pred(OutBufferCount) Downto 0 Do AddByte(OutBuffer[W]);
                 OutBufferCount:=0;

                 (* Now, update the LZW tree with the new code            *)

                 With CodeTable[FreeCode] Do
                 Begin
                    PredCode:=OldCode;
                    CharCode:=CurChar;
                 End;

                 OldCode:=InCode;

                 (* .. and calculate the next free position in the LZW-tree *)
                 (* if we pass MaxCode, the Codesize is incremented. If     *)
                 (* codesize is 12, do nothing, since this is the max. size *)

                 Inc(FreeCode);
                 If FreeCode>=MaxCode Then
                 Begin
                    If CodeSize<12 Then
                    Begin
                       Inc(CodeSize);
                       MaxCode:=MaxCode Shl 1;
                       ReadMask:=Not Word($FFF8 Shl (CodeSize-3));
                    End;
                 End;
              End;
           End;
        Until Code=EOFCode;
     End;

     DecodeLZW:=True;

     ErrorExit:

     If OutUsed<>0 Then
     BlockWrite(OutFile,OutBuffer[0],OutUsed,W);
  End;

Begin
   DeCompresLZW:=0;

   PLZW:=Nil;
   New(PLZW);

   With PLZW^ Do
   Begin
      (* Initialize the buffer-system and read the beginning of the file *)

      Assign(InFile,InName); Reset(InFile,1);
      If IoResult<>0 Then
      Begin
         DecompresLZW:=-2;
         Goto Exit;
      End;

      BlockRead(InFile,InBuffer,Sizeof(InBuffer),InUsed);

      Assign(OutFile,OutName); ReWrite(OutFile,1);
      If Ioresult<>0 Then
      Begin
         DecompresLZW:=-3;
         Close(InFile);
      End;

      OutUsed:=0;
      CodeSize:=8;

      (* Calculate the start-up codes    *)

      BitMask:=$FF;
      ClearCode:=(1 Shl CodeSize);

      EofCode:=Succ(ClearCode);
      FirstFree:=Succ(EofCode);
      FreeCode:=FirstFree;

      CodeSize:=Succ(CodeSize);
      InitCodeSize:=CodeSize;
      MaxCode:=4 Shl (CodeSize-2);
      ReadMask:=Not Word($FFF8 Shl (CodeSize-3));

      OutBufferCount:=0;

      If Not DecodeLZW Then DecompresLZW:=-3;

      Close(InFile);
      Close(OutFile);
   End;

   Exit:

   If PLZW<>Nil Then Dispose(PLZW);
End;

Type
  TCodeEntry  = Record
                   PriorCode,
                   CodeId    : Integer;
                   AddedChar : Byte;
                End;

  PWriteLZW   = ^TWriteLZW;
  TWriteLZW   = Record
                    CodeSize,
                    ClearCode,
                    EofCode,
                    MinCode,
                    MaxCode,
                    BitsLeft,
                    FreeCode,
                    PrefixCode,
                    SuffixChar,
                    Hx,Md         : Integer;
                    CodeBuffer    : Array[0..259] Of Byte;
                    CodeTabel     : Array[0..CTableSize] Of TCodeEntry;
                    BitOffset,
                    ByteOffset    : Word;
                    InBuffer      : PBuffer;
                 End;

Function CompresLZW(InName,OutName: PFStr): Integer;
Label
  Break,Exit;
Var
  PWLZW   : PWriteLZW;
  OutFile : File;
  I       : Integer;

  Procedure InitLzwTree;
  Var
    I: Integer;
  Begin
     With PWLZW^ Do
     Begin
        CodeSize:=Succ(CMinCodeSize);
        ClearCode:=1 Shl CMinCodeSize;
        EofCode:=Succ(ClearCode);
        FreeCode:=Succ(EofCode);
        MaxCode:=1 Shl CodeSize;

        For I:=0 To Pred(CTableSize) Do CodeTabel[I].CodeId:=0;
     End;
  End;

  Function GetData: Integer;
  Var
    Resul: Integer;
  Begin
     With PWLZW^ Do
     Begin
        If BufferEof(InBuffer) Then Resul:=-1 Else
        Resul:=BufferGetByte(InBuffer);

        SuffixChar:=Resul;
        GetData:=Resul;
     End;
  End;

  Procedure PutByte(B: Byte);
  Var
    X: Word;
  Begin
     BlockWrite(OutFile,B,1,X);
  End;

  Procedure PutBlock(Var Buffer; Count: Integer);
  Var
    X: Word;
  Begin
     BlockWrite(OutFile,Buffer,Count,X);
  End;

  Procedure WriteCode(Code: Integer);
  Var
    LCode : Longint;
  Begin
     With PWLZW^ Do
     Begin
        ByteOffset:=BitOffset Shr 3; BitsLeft:=BitOffset And 7;

        If ByteOffset>=254 Then
        Begin
           PutBlock(CodeBuffer,ByteOffset);

           CodeBuffer[0]:=CodeBuffer[ByteOffset];
           BitOffset:=BitsLeft; ByteOffset:=0;
        End;

        If BitsLeft>0 Then
        Begin
           LCode:=Longint(Code) Shl BitsLeft;

           CodeBuffer[ByteOffset]:=CodeBuffer[ByteOffset] Or (LCode And $FF);
           PWord(@CodeBuffer[Succ(ByteOffset)])^:=(LCode Shr 8) And $FFFF;
        End Else
        Begin
           PWord(@CodeBuffer[ByteOffset])^:=Code;
        End;

        Inc(BitOffset,CodeSize);
     End;
  End;


Begin
   CompresLZW:=0;

   PWLZW:=Nil;
   New(PWLZW);

   With PWLZW^ Do
   Begin
      If Not BufferInit(InBuffer,CBufferSize,InName) Then
      Begin
         CompresLZW:=-2;
         Goto Exit;
      End;

      Assign(OutFile,OutName); ReWrite(OutFile,1);
      If IoResult<>0 Then
      Begin
         CompresLZW:=-2;
         Goto Exit;
      End;

      BitOffset:=0;
      InitLzwTree;
      WriteCode(ClearCode);

      SuffixChar:=GetData;

      If SuffixChar>=0 Then
      Begin
         PrefixCode:=SuffixChar;

         While GetData>=0 Do
         Begin
            Hx:=(PrefixCode Xor (SuffixChar Shl 5)) Mod CTableSize;
            Md:=1;

            While True Do
            With CodeTabel[Hx] Do
            Begin
               If CodeId=0 Then
               Begin
                  WriteCode(PrefixCode);
                  Md:=FreeCode;

                  If FreeCode<=CMaxCode Then
                  Begin
                     PriorCode:=PrefixCode;
                     AddedChar:=SuffixChar;
                     CodeId:=FreeCode; Inc(FreeCode);
                  End;

                  If Md=MaxCode Then
                  Begin
                     If CodeSize<12 Then
                     Begin
                        Inc(CodeSize); MaxCode:=MaxCode Shl 1;
                     End Else
                     Begin
                        WriteCode(ClearCode); InitLzwTree;
                     End;
                  End;

                  PrefixCode:=SuffixChar; Goto Break;
               End;

               If (PriorCode=PrefixCode) And
                  (AddedChar=SuffixChar) Then
               Begin
                  PrefixCode:=CodeId; Goto Break;
               End;

               Inc(Hx,Md); Inc(Md,2);

               If Hx>=CTableSize Then Dec(Hx,CTableSize);
            End;

            Break:
         End;

         If SuffixChar<>-1 Then Goto Exit;
             WriteCode(PrefixCode);
      End Else
      Begin
         If SuffixChar<>-1 Then Goto Exit;
      End;

      WriteCode(EofCode);

      PutBlock(CodeBuffer[0],(BitOffset+7) Div 8);
      PutByte(0);

      Close(OutFile);
      BufferClose(InBuffer);
   End;

   Exit:

   If PWLZW<>Nil Then Dispose(PWLZW);
End;

End.