(************************************************************************)
(*                                                                      *)
(*  Programex. to.      : "Tips & Tricks in Turbo Pascal", SysTime 1993 *)
(*                                                                      *)
(*  Programmed by       : Martin Israelsen                              *)
(*                                                                      *)
(*  Title               : WRITEGIF.PAS                                  *)
(*                                                                      *)
(*  Chapter             : 6 + 7                                         *)
(*                                                                      *)
(*  Description         : Gif87a compression in Turbo Pascal. Partly    *)
(*                        based on the original CompuServe GIF          *)
(*                        specification.                                *)
(*                                                                      *)
(*                        NOV-93: Overflow bug fixet.                   *)
(*                                                                      *)
(************************************************************************)
(*$I Define.Inc*)

(* Define Trace*) (* Turn on this directive to "trace" the rutines  *)

Unit
  WriteGif;

Interface

Uses
  Typ;

Function  MakeGif(FNavn: PfStr; Width,Height,BackColor,Bpp: Word; PColor: PGPalette): Boolean;
Function  WriteImage(Left,Top,Width,Height,Bpp: Word; Interlace: Boolean; PColor: PGPalette): Boolean;
Procedure FinishGif;
Procedure WriteScreenToGif(Colors: Integer; FName: String);

Implementation

Uses
  Graph;

Const
  CGifImageSeperator = ',';
  CGifTerminator     = ';';

  CTableSize         = 5003;
  CMaxCode           = 4095;

Type
  TCodeEntry  = Record
                   PriorCode : Integer;
                   CodeId    : Integer;
                   AddedChar : Byte;
                End;

  PWriteGif  = ^TWriteGif;
  TWriteGif  = Record
                   CodeSize,
                   ClearCode,
                   EofCode,
                   MinCode,
                   MaxCode,
                   BitsLeft,
                   FreeCode,
                   PrefixCode,
                   SuffixChar,
                   Hx,Md,Gifx,
                   Gify         : Integer;
                   CodeBuffer   : Array[0..259] Of Byte;
                   CodeTabel    : Array[0..CTableSize] Of TCodeEntry;

                   BitOffset,
                   ByteOffset,
                   InterlacePass : Word;
                   MinCodeSize   : Byte;
                   ImageDes      : TGifImageRec;
                   GifFil        : File;
                End;

Var
  PWGif  : PWriteGif;

Procedure InitLzwTree;
Var
  I: Integer;
Begin
   With PWGif^ Do
   Begin
      CodeSize:=Succ(MinCodeSize);
      ClearCode:=1 Shl MinCodeSize;
      EofCode:=Succ(ClearCode);
      FreeCode:=Succ(EofCode);
      MaxCode:=1 Shl CodeSize;

      For I:=0 To CTableSize Do CodeTabel[I].CodeId:=0;
   End;
End;

Function GetGifData: Integer;
Var
  Resul: Integer;
Begin
   With PWGif^ Do
   Begin
      Resul:=GetPixel(Gifx+ImageDes.LeftOffset,Gify+ImageDes.TopOffset);

      (*$IfDef Trace*)
      If Gifx=0 Then PutPixel(Gifx+ImageDes.LeftOffset,Gify+ImageDes.TopOffset,White);
      (*$Endif*)

      Inc(GifX);

      If (GifX=ImageDes.ImageWidth) Then
      Begin
         GifX:=0;

         If InterlacePass<>0 Then
         Begin
            Case Interlacepass Of
               1: Inc(GifY,8);
               2: Inc(GifY,8);
               3: Inc(GifY,4);
               4: Inc(GifY,2);
            End;

            If GifY>=ImageDes.ImageHeight Then
            Begin
               Case InterlacePass Of
                  1: GifY:=4;
                  2: GifY:=2;
                  3: GifY:=1;
                  4: Resul:=-1;
               End;

               Inc(InterlacePass);
            End;
         End Else
         Begin
            If GifY=ImageDes.ImageHeight Then Resul:=-1;
            Inc(GifY);
         End;
      End;

      SuffixChar:=Resul;
      GetGifData:=Resul;
   End;
End;

Procedure PutByte(B: Byte);
Var
  X: Word;
Begin
   BlockWrite(PWGif^.GifFil,B,1,X);
End;

Procedure PutBlock(Var Buffer; Count: Integer);
Var
  X: Word;
Begin
   BlockWrite(PWGIF^.GifFil,Buffer,Count,X);
End;

Procedure PutString(S: String);
Begin
   PutBlock(S[1],Length(S));
End;

Procedure WriteCode(Code: Integer);
Var
  LCode : Longint;
Begin
   With PWGif^ Do
   Begin
      ByteOffset:=BitOffset Shr 3; BitsLeft:=BitOffset And 7;

      If ByteOffset>=254 Then
      Begin
         PutByte(ByteOffset);
         PutBlock(CodeBuffer,ByteOffset);

         CodeBuffer[0]:=CodeBuffer[ByteOffset];
         BitOffset:=BitsLeft; ByteOffset:=0;
      End;

      If BitsLeft>0 Then
      Begin
         LCode:=Longint(Code) Shl BitsLeft;

         CodeBuffer[ByteOffset]:=CodeBuffer[ByteOffset] Or (LCode And $FF);
         PWord(@CodeBuffer[Succ(ByteOffset)])^:=(LCode Shr 8) And $FFFF;
      End Else
      Begin
         PWord(@CodeBuffer[ByteOffset])^:=Code;
      End;

      Inc(BitOffset,CodeSize);
   End;
End;

Function MakeGif(FNavn: PfStr; Width,Height,BackColor,Bpp: Word; PColor: PGPalette): Boolean;
Label
  Exit;
Var
  GifDes : TGifDescrip;
Begin
   MakeGif:=False;

   New(PWGif);

   With PWGif^ Do
   Begin
      Assign(GifFil,FNavn); ReWrite(GifFil,1);
      If IoResult<>0 Then
      Begin
         Dispose(PWGif);
         Goto Exit;
      End;

      PutString('GIF87a');

      With GifDes Do
      Begin
         ScreenWidth:=Width;
         ScreenHeight:=Height;

         GifInfo:=Pred(Bpp);
         If PColor<>Nil Then GifInfo:=GifInfo Or $80;

         BackGround:=0;
         EofMark:=0;
      End;

      PutBlock(GifDes,Sizeof(GifDes));

      If PColor<>Nil Then PutBlock(PColor^,(1 Shl Bpp)*3);
   End;

   MakeGif:=True;

   Exit:
End;

Function WriteImage(Left,Top,Width,Height,Bpp: Word; Interlace: Boolean; PColor: PGPalette): Boolean;
Label
  Exit,Break;
Var
  I        : Integer;
Begin
   WriteImage:=False;

   With PWGif^ Do
   Begin
      PutByte(Ord(CGifImageSeperator));

      With ImageDes Do
      Begin
         LeftOffset:=Left;
         TopOffset:=Top;
         ImageWidth:=Width-Left;
         ImageHeight:=Height-Top;

         (* Calculate mincode size *)

         MinCodeSize:=Bpp;
         ImageInfo:=Pred(MinCodeSize);
         If Interlace Then ImageInfo:=ImageInfo Or $40;
         If PColor<>Nil Then ImageInfo:=ImageInfo Or $80;

         InterlacePass:=Ord(Interlace);

         PutBlock(ImageDes,Sizeof(ImageDes));

         If PColor<>Nil Then PutBlock(PColor^,(1 Shl Bpp)*3);
      End;

      PutByte(MinCodeSize);

      BitOffset:=0;
      InitLzwTree;
      WriteCode(ClearCode);

      GifX:=0; GifY:=0;

      SuffixChar:=GetGifData;

      If SuffixChar>=0 Then
      Begin
         PrefixCode:=SuffixChar;

         While GetGifData>=0 Do
         Begin
            Hx:=(PrefixCode Xor (SuffixChar Shl 5)) Mod CTableSize;
            Md:=1;

            While True Do
            With CodeTabel[Hx] Do
            Begin
               If CodeId=0 Then
               Begin
                  WriteCode(PrefixCode);
                  Md:=FreeCode;

                  If FreeCode<=CMaxCode Then
                  Begin
                     PriorCode:=PrefixCode;
                     AddedChar:=SuffixChar;
                     CodeId:=FreeCode; Inc(FreeCode);
                  End;

                  If Md=MaxCode Then
                  Begin
                     If CodeSize<12 Then
                     Begin
                        Inc(CodeSize); MaxCode:=MaxCode Shl 1;
                     End Else
                     Begin
                        WriteCode(ClearCode); InitLzwTree;
                     End;
                  End;

                  PrefixCode:=SuffixChar; Goto Break;
               End;

               If (PriorCode=PrefixCode) And
                  (AddedChar=SuffixChar) Then
               Begin
                  PrefixCode:=CodeId; Goto Break;
               End;

               Inc(Hx,Md); Inc(Md,2);

               If Hx>CTableSize Then Dec(Hx,CTableSize);
            End;

            Break:
         End;

         If SuffixChar<>-1 Then Goto Exit;

         WriteCode(PrefixCode);
      End Else
      Begin
         If SuffixChar<>-1 Then Goto Exit;
      End;

      WriteCode(EofCode);

      If BitOffset>0 Then
      Begin
         I:=(BitOffset+7) Div 8;
         PutByte(I);
         PutBlock(CodeBuffer[0],I);
      End;

      PutByte(0);

      WriteImage:=True;
   End;

   Exit:
End;

Procedure FinishGif;
Begin
   With PWGif^ Do
   Begin
      PutByte(Ord(CGIFTerminator));
      Close(GifFil);
   End;
End;

Procedure WriteScreenToGif(Colors: Integer; FName: String);
Const
   NormPal = #$00#$00#$00#$00#$00#$AA#$00#$AA#$00+
             #$00#$AA#$AA#$AA#$00#$00#$AA#$00#$AA+
             #$AA#$55#$00#$AA#$AA#$AA#$55#$55#$55+
             #$55#$55#$FF#$55#$FF#$55#$55#$FF#$FF+
             #$FF#$55#$55#$FF#$55#$FF#$FF#$FF#$55+
             #$FF#$FF#$FF;
Var
  Palette : TGPalette;
  S	  : String;
  Bpp	  : Word;
Begin
   If Colors=16 Then
   Begin
      S:=NormPal; Move(S[1],Palette,Length(S)); Bpp:=4;
   End Else
   Begin
      Writeln('To use WriteGif with 256-color pictures');
      Writeln('the palette should be set in this code');

      (* Palette:= 256-Palette *) Bpp:=8;
   End;

   If MakeGif(FName,Succ(GetMaxx),Succ(GetMaxy),0,Bpp,@Palette) Then
   Begin
      If WriteImage(0,0,Succ(GetMaxx),Succ(GetMaxy),Bpp,False,Nil) Then;
      FinishGif;
   End Else
   Begin
      Writeln('Error during MakeGif'^g);
   End;
End;

End.


