unit EGASave;
(* -----------------------------------------------------------
This unit provides all of the routines necessary for saving and restoring EGA
(640x350) 16 color graphics screens to and
from RAM.

Last update: 5/27/88 by John Sieraski
----------------------------------------------------------- *)

{$R-}   {  Turn off range checking  }
{$S-}   {  Turn off stack checking  }

interface

uses
  crt;

const
  BitMapSize = 28000; { Size in bytes of each EGA bit plane    }
  MaxBitMaps = 4;     { Number of Bit planes in video mode $10 }

type
  BitMapBuffer = array[1..BitMapSize] of byte;          { An EGA bit plane  }
  EGABuffer    = array[1..MaxBitMaps] of ^BitMapBuffer; { A full EGA screen }

procedure SaveEgaScreen(Buffer : EGABuffer);
{ Saves an EGA (634x350) 16 color graphics screen into Buffer }

procedure RestoreEgaScreen(Buffer : EGABuffer);
{ Restores an EGA (640x350) 16 color graphics screen image from buffer }

procedure AllocateBuffer(var Buffer : EGABuffer);
{ Allocates a "Buffer" variable on the Heap using GetMem }

procedure DisposeBuffer(var Buffer : EGABuffer);
{ Frees a "Buffer" variable from the Heap using FreeMem }

IMPLEMENTATION

const
  EgaBase    = $A000; { Base address of EGA graphics memory     }

procedure SaveEGAScreen(Buffer : EGABuffer);
var
  BitMap : integer;

   procedure EnableMapRead(Map : byte);
   { Enables reading from one of the EGA's Bit planes 1..4 }
   const
     AddrReg      = $3CE;  { Port address of EGA graphics 1 & 2 address register }
     SetResetReg  = $3CF;  { Port address of EGA Set/Reset register }
     ReadMapReg   = $04;   { Index of EGA Read Map select register }
   begin
     Port[AddrReg] := ReadMapReg;
     Port[SetResetReg] := Pred(map)
   end;

begin
  for BitMap := 1 to MaxBitMaps do
  begin
    EnableMapRead(BitMap);
    Move(ptr(EGABase,0)^, Buffer[BitMap]^, BitMapSize);     {!}
  end
end; { SaveEGAScreen }

function MapsSelected : byte;
{ Returns the number of bit planes enables for writing }
const
  AddrReg     = $3CE; { Port address of EGA graphics 1 & 2 address register }
  SetResetReg = $3CF; { Port address of EGA set/Reset register }
  ReadMapReg  = $04;  { Index of EGA Read Map select register }
var
  BitMap        : integer;
  MemByte       : byte;
  EnabledPlanes : byte;
begin
  EnabledPlanes := 0;
  Port[AddrReg] := ReadMapReg;
  for BitMap := 0 to 3 do
  begin
    Port[SetResetReg] := BitMap;
    MemByte := Mem[EGABase:0000];          { Read a dummy byte from bit plane }
    Mem[EGABase:0000] := not(memByte);     { Write the byte back inverted   }
    if Mem[EGABase:0000] <> MemByte then   { This plane is selected }
    begin
      EnabledPlanes := EnabledPlanes or (1 Shl BitMap);
      Mem[EGABase:0000] := MemByte;         { Reset original byte read }
    end;
  end;
  MapsSelected := EnabledPlanes;
end; { MapsSelected}

procedure RestoreEGAScreen(Buffer : EGABuffer);
const
  SeqAddrReg = $3C4;  { Port address of EGA sequencer address register }
  ResetReg   = $3C5;  { Port address of EGA sequencer reset register }
  MapMaskReg = $02;   { Index of EGA sequencer Map Mask register }

var
  BitMap       : integer;
  MapsEnabled  : byte;

   procedure EnableMapWrite(Map : byte);
   { Enables writing to one of the EGA's bit planes 1...4 }
   begin
     Port[SeqAddrReg] := MapMaskReg;
     Port[ResetReg] := 1 shl Pred(Map)
   end; { EnableMapWrite }

begin
  MapsEnabled := MapsSelected; { Save originally selected write planes }
  for BitMap := 1 to MaxBitMaps do
  begin
    EnableMapWrite(BitMap);
    Move(Buffer[BitMap]^, ptr(EgaBase, 0)^, BitMapSize);
  end;
  Port[ResetReg] := MapsEnabled;  { Restore originally selected write planes }
end; { RestoreEgaScreen }

procedure AllocateBuffer(var Buffer : EGABuffer);
var
  BitMap : integer;
  Ch     : Char;
begin
  for BitMap := 1 to MaxBitMaps do
    if BitMapSize <= MaxAvail then
      GetMem(Buffer[BitMap], BitMapSize)
    else
      begin
        WriteLn('Not enough memory left on the heap.');
        Write('Press any key to halt:');
        Ch := ReadKey;
        Halt;
      end;
end;  { AllocateBuffer }

procedure DisposeBuffer(var Buffer : EGABuffer);
var
  BitMap  :  integer;
begin
  for BitMap := 1 to MaxBitMaps do
    if Buffer[BitMap] <> Nil then
    begin
      FreeMem(Buffer[BitMap], BitMapSize);
      Buffer[BitMap] := Nil;
    end;
end;  { DisposeBuffer }

begin
  DirectVideo := FALSE; {turn off Crt's direct screen writes }
end.