program GetFATsample;

(*

  The File Allocation Table (FAT) is an array used by DOS to keep track of
  which clusters on a disk have been allocated for each file or directory.
  As a program creates a new file or adds to an existing one, DOS allocates
  sectors for that file, writes the data to a given sectors, and keeps track
  of the allocated sectors by recording them in the FAT.

  Depending on the number of clusters on the disk, the FAT consists of an array
  of either 12-bit or 16-bit entries. Disks with more than 4086 clusters have a
  16-bit FAT; those with 4086 or less clusters have a 12-bit FAT.

  This program reads the FAT sectors for the drive specified and then gets
  information about allocated clusters, bad clusters, etc..

*)

 uses crt, DMT;

 var
   LogDrvs          : ListOfDrvs;        { The ListOfDrvs data type is defined in the DMT unit }

   BPBbuffer        : DevParamsStruct;  { The DevParamsStruct data type is predefined in the DMT unit }

   FATbuffer,
   BufferAddr       : pointer;

   BadClusters,
   FreeClusters,
   FileClusters,
   ResClusters,
   Head,
   Track,
   FirstSect,
   StartFATsector,
   SectCounter,
   N               : word;

   DrvLetter       : char;

   DrvsCount,
   Index           : byte;

procedure DecFAT12( BufSeg, BufOfs  : word  );

(* Decodes a FAT of 12-bit entries *)

 var
   Cluster,
   Offset,
   FATentry,
   TotalOfClusts : word;

begin
  (* Determines the number of clusters or FAT entries on disk *)

  TotalOfClusts := Clusters( DrvLetter );

  (* Go through all of the FAT entries *)

  for Cluster := 2 to TotalOfClusts + 1 do
    begin

      { Begin decode 12-bit FAT entries }

      Offset := ( Cluster * 3 ) div 2;
      FATentry := memw[ BufSeg : BufOfs + Offset ];

      if ( ( Cluster mod 2 ) = 0 ) then
        FATentry := ( FATentry and $0FFF )
      else
        FATentry := ( FATentry shr 4 );

      { End decode 12-bit FAT entries }

      case ( FATentry ) of

        (* Possible FAT Entry values *)

               $000 : inc( FreeClusters );

         $FF0..$FF6 : inc( ResClusters );

               $FF7 : inc( BadClusters );

         $FF8..$FFF : inc( FileClusters );

      end;
    end;
end;

procedure DecFAT16( BufSeg, BufOfs : word );

(* Decodes a FAT of 16-bit entries *)

 var
   Cluster,
   Offset,
   FATEntry,
   TotalOfClusts : word;

begin
  Offset := 0;

  (* Determines the number of clusters or FAT entries on disk *)

  TotalOfClusts := Clusters( DrvLetter );

  (* Go through all of the FAT entries *)

  for Cluster := 2 to TotalOfClusts + 1 do
    begin
      Offset := Cluster * 2;

      FATentry := memw[ BufSeg : BufOfs + Offset ];

      case ( FATentry ) of

       (* Possible FAT Entry values *)

               $0000 : inc( FreeClusters );

        $FFF0..$FFF6 : inc( ResClusters );

               $FFF7 : inc( BadClusters );

        $FFF8..$FFFF : inc( FileClusters );

      end;
    end;
end;

begin
  BadClusters := 0;
  FreeClusters := 0;
  FileClusters := 0;
  ResClusters := 0;
  N := 0;

  Color( 7, 0 );
  clrscr;

  GetDrvs( LogDrvs, DrvsCount );   { Call GetDrvs procedure }

  write( #13#10, 'Select Drive ( ');

  for Index := 1 to DrvsCount do
    write( LogDrvs[ Index ], ': ' );

  write( ') [ :]' );

  gotoxy( wherex - 3, wherey );
  DrvLetter := upcase( readkey );
  writeln( DrvLetter );

  GetBPB( DrvLetter, BPBbuffer );  { Call the GetBPB procedure }

  if ( ErrFlag ) then
    begin
      writeln( #7 );
      writeln( ShowError( GetErrCode ) );
    end
  else
    with BPBbuffer do
      begin
        (* Allocates memory for FAT sectors buffer *)

        getmem( FATbuffer, SectsPerFAT * 512 );

        (* Determines the start of the FAT sectors *)

        StartFATsector := HiddenSects + BootSects;

        (* Reads the FAT sectors from disk and place it into a buffer *)

        for SectCounter := StartFATSector to ( StartFATSector + SectsPerFAT - 1 ) do
          begin

            (*
              Translates a logical FAT sector number into a physical sector
              consisting of head, track, and sector numbers.
            *)

             Head      := ( SectCounter div SectsPerTrack ) mod NumOfHeads;
             Track     := SectCounter div ( SectsPerTrack * NumOfHeads );
             FirstSect := SectCounter mod SectsPerTrack;

             BufferAddr := ptr( seg( FATbuffer^ ), ofs( FATbuffer^ ) + n );

             (* Reads a FAT sector from disk and place it into a buffer *)

             ReadSect( DrvLetter, Head, Track, FirstSect, 1, BufferAddr );

             inc( n, BytesPerSect );
          end;

        (* Determines the type of FAT used by the disk *)

        if ( FATtype( DrvLetter ) = 16 ) then
          DecFAT16( seg( FATbuffer^ ), ofs( FATbuffer^ ) )
        else
          DecFAT12( seg( FATbuffer^ ), ofs( FATbuffer^ ) );

        clrscr;
        writeln;
        writeln( 'Disk Drive specified        : ', DrvLetter );
        writeln( 'Number of free clusters     : ', InsComma( FreeClusters ) );
        writeln( 'Number of user files        : ', InsComma( FileClusters ) );
        writeln( 'Number of bad clusters      : ', InsComma( BadClusters ) );
        writeln( 'Number of reserved clusters : ', InsComma( ResClusters ) );
      end;
  GetEnter;
end.


