{*********************************************************}
{*                   CHECKFB.PAS 1.20                    *}
{*      Copyright (c) TurboPower Software 1993,1994.     *}
{*                 All rights reserved.                  *}
{*********************************************************}

{$I-}

program CheckFB;
  {-Check the integrity of a B-Tree Filer fileblock}

{ Notes:
  This program verifies that a B-Tree Filer fileblock's data and
  index files are valid. Valid in this context means the following.

  For the data file:

    - the fields in the system record (record 0) make sense:
         - the 1st free record must be -1, or between 1 and the number
           of records;
         - the number of free records must be not greater than the
           number of records, and both cannot be negative;
         - the record length must be between 21 and 65520;
         - the number of records plus one multiplied by the record
           length must equal the file size as reported by DOS;
         - if the number of free records is non-zero the 1st free
           record number cannot be -1, and vice versa;
         - the number of keys must be zero or greater;
         - the integrity 'save mode' flag must be zero or one (if it
           is 1, the program does NOT report an error).

    - the deleted record chain is complete, and that the number of
      records on this chain agrees with the system record and that
      each link in the chain (a record number) is between 1 and the
      total number of records.

    - if the data file is for a variable record length fileblock then
      the individual section chains for each variable length record
      are also checked to be complete and not corrupt.

  For the index file:

    - the fields in the individual system pages for each index
      make sense:
         - the length of each index page is not more than 65520
         - the root page number is between 1 and the number of
           pages for the index
         - the allow duplicate keys boolean is true or false
         - the fields for free pages are checked in the same manner
           as the data file.

    - the deleted page list for each index is valid (in the same sense
      as the check in the data file).

    - the total size of all the index pages multiplied by the maximum
      number of pages in any index equals the index file size.

    - the key length for each index is an integral value between
      1 and 255.

    - the total number of keys calculated by summing the number of
      keys on each page equals the number of keys expected for each
      index.

  A number of people have asked what the basic format of an B-Tree
  Filer index file is. It is a file of fixed sized key records, each
  key record is divided up into index pages, one for each index. The
  system key record for the index file (record 0) contains system
  pages for each index. For further, more intricate, details check
  out FILER.PAS and the code below.

  This program does NOT check that there is anything 'meaningful' in
  the data portions of the records, as Filer does not 'know' anything
  about the data being held anyway. It should be a trivial matter (if
  somewhat long-winded) to add a routine to this program that checked
  values in individual records.

  This version of the program does not go through and check the
  validity of every single record number and page reference in the
  index file. Nor does it read all the keys on each page and check
  that they are in sequence. I had to draw the line somewhere!

  This program is supplied as-is and is hereby released as freeware.
  There is no official technical support for this program, you use it
  entirely at your own risk.

  Written by Julian M. Bucknall [100116,1572]
}

uses
  {$IFDEF Windows}
  Strings, WinCrt, WinDos;
  {$ELSE}
  Dos;
  {$ENDIF}

const
  VersionStr = '1.20';         {Version number as a string}
  IsFirstSection = true;       {Useful constant}
  DiskReadError = 100;         {Normal IOresult read error}

type
  EnumError = (errDosError, errIOerror, errNotDataFile, errParm,
               errInvRecNo, errBadDelList,
               errBadVarRec, errBadVarRecLink,
               errMemory,
               errNotIndexFile, errIndexKeys, errTotalKeys,
               errInvPageNo);
    {Possible error situations}

  TSystemRecord = record
    {The B-Tree Filer data file system record (record 0)}
    case byte of
      0 : (FirstFree : longint; {Record number of first deleted record}
           NumFree   : longint; {Number of deleted records}
           NumRecs   : longint; {Number of data records excl. system record}
           RecLen    : longint; {Record length}
           NumKeys   : longint; {Number of keys}
           Changed   : byte);   {Data integrity flag}
      1 : (Bytes : array [1..21] of byte);   {...as a block of bytes}
      2 : (Longs : array [0..4] of longint;  {...as an array of longints}
           Chgd  : byte);
  end;

  TIndexSystemPage = record
    {The B-Tree Filer index system record}
    case byte of
      0 : (FirstFree : longint; {Record number of first deleted page}
           NumFree   : longint; {Number of deleted pages}
           NumPages  : longint; {Number of pages in index}
           PageLen   : longint; {Page length}
           RootPage  : longint; {The B-Tree root page number}
           AllowDupK : byte;    {Allow duplicate keys}
           Filler1   : byte;
           NumKeys   : longint);{Number of keys in index}
      1 : (Bytes : array [1..26] of byte);   {...as a block of bytes}
      2 : (Longs : array [0..4] of longint;  {...as an array of longints}
           ADK   : byte;
           Fill2 : byte;
           Long5 : longint);
  end;

  TVarRecLink = record
    {A variable length record link structure}
    SectionSize : word;         {Number of data bytes in this section}
    NextSection : longint;      {Record number of next section}
  end;

  TIndexDefn = record
    {Index definition record for this program}
    IndexPageSize   : word;
    IndexPageOffset : word;
    FirstFree       : longint;
    NumFree         : longint;
    NumPages        : longint;
    NumKeys         : longint;
    Root            : longint;
    KeyLen          : word;
    AllowDupKeys    : boolean;
  end;

  TIndexDefnArray = array [1..100] of TIndexDefn; {dynamically allocated}

  PDeletedPage = ^TDeletedPage;
  TDeletedPage = record
    Next : PDeletedPage;
    Page : longint;
  end;

  TIndexDeletedList = record
    Number : longint;
    First  : PDeletedPage;
  end;

const
  SystemRecordSize = sizeof(TSystemRecord);
  IndexSystemPageSize = sizeof(TIndexSystemPage);
  MaxRecordLength = 65520;

  PageSize     : integer = 62;  {The assumed B-Tree page size}

var
  IOerror      : integer;       {Last IOresult value}
  Verbose      : boolean;       {True if verbose mode on}
  IsVarRecFile : boolean;       {True if a variable length record file}
  SysRecIsRead : boolean;       {True if data system record has been read}
  CheckIndex   : boolean;       {True if index file is to be checked}
  {$IFDEF Windows}
  DirInfo      : TSearchRec;    {Directory info for data/index files}
  DataFileName : array [0..255] of char; {Full data file name}
  IndexFileName  : array [0..255] of char; {Full index file name}
  {$ELSE}
  DirInfo      : SearchRec;     {Directory info for data/index  file}
  DataFileName : string;        {Full data file name}
  IndexFileName: string;        {Full index file name}
  {$ENDIF}
  SystemRecord : TSystemRecord; {Data file's system record}
  FB           : file;          {The data/index file variable}
  IndexDefn    : ^TIndexDefnArray; {The definitions of all indexes}
  IndexDefnSize: word;          {Size of IndexDefn^}
  KeyRecSize   : integer;       {The index file record size}
  NumKeyRecs   : longint;       {The total number of index file records}

  IndexDeletedList : TIndexDeletedList; {The linked list of deleted pages}

procedure DoNothing; inline($90);
  {-A do nothing procedure}

function HexB(B : byte) : string;
  {-Return the hex representation of a byte as a string}
  const
    HexChars : array [0..15] of char = '0123456789abcdef';
  var
    TempSt : string[2];
  begin
    TempSt[0] := #2;
    TempSt[1] := HexChars[(B and $F0) shr 4];
    TempSt[2] := HexChars[(B and $0F)];
    HexB := TempSt;
  end;

procedure HexDumpSystemRecord;
  {-Output the system record in a hex dump style}
  const
    FieldName : array [0..5] of string[11] =
              ('FirstFree',
               'NumFree',
               'Num recs',
               'Rec length',
               'Num indexes',
               'Changed?');
  var
    i, j : integer;
  begin
    writeln('System record in hex:');
    for i := 0 to 4 do
      begin
        write('   ');
        for j := 1 to 4 do
          write(HexB(SystemRecord.Bytes[i*4+j]), ' ');
        writeln('  [', SystemRecord.Longs[i]:11, ']  - ', FieldName[i]);
      end;
    writeln('   ', HexB(SystemRecord.Changed),
            '                           - ', FieldName[5]);
    writeln;
  end;

procedure HexDumpIndexSystemPage(Index : integer; var Rec : TIndexSystemPage);
  {-Output the index system record in a hex dump style}
  const
    FieldName : array [0..7] of string[12] =
              ('FirstFree',
               'NumFree',
               'Num pages',
               'Page size',
               'Root page',
               'AllowDupKeys',
               ' (filler)',
               'Num keys');
  var
    i, j : integer;
  begin
    writeln('Index System record [', Index, '] in hex:');
    for i := 0 to 4 do
      begin
        write('   ');
        for j := 1 to 4 do
          write(HexB(Rec.Bytes[i*4+j]), ' ');
        writeln('  [', Rec.Longs[i]:11, ']  - ', FieldName[i]);
      end;

    writeln('   ', HexB(Rec.AllowDupK),
            '                           - ', FieldName[5]);

    writeln('   ', HexB(Rec.Filler1),
            '                           - ', FieldName[6]);

    write('   ');
    for j := 1 to 4 do
      write(HexB(Rec.Bytes[22+j]), ' ');
    writeln('  [', Rec.Long5:11, ']  - ', FieldName[7]);

    writeln;
  end;

function IOfailed : boolean;
  {-Save value of IOresult, report whether zero or not}
  begin
    IOerror := IOresult;
    IOfailed := (IOerror <> 0);
  end;

procedure CopyrightMessage;
  {-Output a copyright message}
  begin
    writeln('CHECKFB: B-Tree Filer fileblock integrity checker');
    writeln('         Copyright (c) TurboPower Software 1993');
    writeln('         Version ', VersionStr);
    writeln;
  end;

procedure ShowSyntax;
  {-Show the program syntax}
  begin
    writeln;
    writeln('Syntax: CHECKFB [-v] [-q] [-pNN] Fileblock_DAT_Name Fileblock_IX_Name');
    writeln;
    writeln('where  -v    means the file is a variable length record');
    writeln('       -q    means output in quiet (non-verbose) mode');
    writeln('       -pNN  means assume the value of PageSize is NN');
    writeln('             (the default is 64)');
    writeln('       Fileblock_DAT_Name is the name of the fileblock''s');
    writeln('                          data file (including extension)');
    writeln('       Fileblock_IX_Name is the name of the fileblock''s');
    writeln('                         index file (including extension)');
  end;

procedure Abort(Why : EnumError; Info, Info2 : longint);
  {-Abort the program, giving the reason}
  begin
    write('CheckFB aborted: ');
    case Why of
      errParm :
        case integer(Info) of
          0 : writeln('invalid switch');
          1 : writeln('no data file name given');
          2 : writeln('more than two file names given');
        end;{case}
      errDosError :
        writeln('DOS error ', Info);
      errIOError :
        writeln('I/O error ', Info);
      errNotDataFile :
        begin
          case integer(Info) of
            0 : if (Info2 = 0) then
                     writeln('size of data file is less than 21 bytes')
                else writeln('size of index file is less than 21 bytes');
            1 : writeln('system record makes no sense');
            2 : writeln('the file size from the system record is invalid');
            3 : writeln('the first deleted record number is invalid');
            4 : writeln('the number of deleted records > number of records');
            5 : writeln('the first deleted record is the system record');
            6 : writeln('the number of deleted recs > 0; 1st deleted rec is not set');
            7 : writeln('the number of deleted recs = 0; 1st deleted rec is set');
          end;{case}
          if SysRecIsRead then
            HexDumpSystemRecord;
        end;
      errNotIndexFile :
        begin
          case integer(Info) of
            0 : writeln('the system data for index ', Info2, ' makes no sense');
            1 : writeln('index ', Info2, ' is larger than the index file');
            2 : writeln('index file size does not match internal details');
            3 : writeln('index ', Info2, ' has an invalid key length');
            5 : writeln('the first deleted page is the system page for index ', Info2);
            6 : writeln('the number of del pages > 0; 1st del page is not set for index ', Info2);
            7 : writeln('the number of del pages = 0; 1st del page is set for index ', Info2);
          end;{case}
        end;
      errInvRecNo :
        writeln('Invalid record number encountered - ', Info);
      errBadDelList :
        begin
          write('Invalid free record/page list - ');
          case integer(Info) of
            0 : begin
                  writeln('there are not enough elements');
                  writeln('                 in the deleted list');
                end;
            1 : begin
                  writeln('there are more elements in the');
                  writeln('                 deleted list than expected');
                end;
          end;{case}
        end;
      errBadVarRec :
        begin
          writeln('variable length record ', Info, ' has more sections than');
          writeln('                 there are records in the file');
        end;
      errBadVarRecLink :
        begin
          writeln('variable length record ', Info, ' has a bad link on');
          writeln('                 its section number ', Info2);
        end;
      errMemory :
        begin
          writeln('could not allocate ', Info, ' bytes');
        end;
      errIndexKeys :
        begin
          writeln('index ', Info, ' has a page with ', Info2, ' keys');
        end;
      errTotalKeys :
        begin
          writeln('index ', Info, ' does not have the correct number of keys');
        end;
      errInvPageNo :
        writeln('Invalid page number encountered - ', Info);
    end;{case}
    ShowSyntax;
    Halt(1);
  end;

procedure ReadParameters;
  {-Read the parameters from the command line}
  var
    DataFileNameFound : boolean;
    IndexFileNameFound : boolean;
    ec   : integer;
    i    : integer;
    Parm : string[127];
    {$IFDEF Windows}
    ParmZPtr : PChar;
    {$ENDIF}
  begin
    Verbose := true;
    IsVarRecFile := false;
    DataFileNameFound := false;
    IndexFileNameFound := false;
    for i := 1 to ParamCount do
      begin
        Parm := ParamStr(i);
        if (Parm[1] = '-') or (Parm[1] = '/') then
          begin
            if (length(Parm) <> 2) then
              if (length(Parm) > 2) and (Upcase(Parm[2]) = 'P') then
                begin
                  Val(Copy(Parm, 3, 255), PageSize, ec);
                  if (ec <> 0) then
                    Abort(errParm, 0, 0)
                end
              else
                Abort(errParm, 0, 0)
            else
              case Upcase(Parm[2]) of
                'V' : IsVarRecFile := true;
                'Q' : Verbose := false;
              else
                Abort(errParm, 0, 0);
              end{case}
          end
        else
          begin
            if IndexFileNameFound then
              Abort(errParm, 2, 0);
            if DataFileNameFound then
              begin
                IndexFileNameFound := true;
                {$IFDEF Windows}
                ParmZPtr := @IndexFileName;
                ParmZPtr := FileExpand(ParmZPtr, StrPCopy(ParmZPtr, Parm));
                {$ELSE}
                IndexFileName := FExpand(Parm);
                {$ENDIF}
              end
            else
              begin
                DataFileNameFound := true;
                {$IFDEF Windows}
                ParmZPtr := @DataFileName;
                ParmZPtr := FileExpand(ParmZPtr, StrPCopy(ParmZPtr, Parm));
                {$ELSE}
                DataFileName := FExpand(Parm);
                {$ENDIF}
              end;
          end;
      end;
    if not DataFileNameFound then
      Abort(errParm, 1, 0);
    if IndexFileNameFound then
      CheckIndex := true;
  end;

procedure CheckSystemRecord;
  {-Check the system record to see if it makes sense}
  begin
    with SystemRecord do
      begin
        if (FirstFree < -1) or
           (NumFree < 0) or
           (NumRecs < 0) or
           (RecLen < SystemRecordSize) or
           (RecLen > MaxRecordLength) or
           (NumKeys < 0) or
           ((Changed <> 0) and (Changed <> 1)) then
          Abort(errNotDataFile, 1, 0);
        if (DirInfo.Size <> succ(NumRecs)*RecLen) then
          Abort(errNotDataFile, 2, 0);
        if (FirstFree > NumRecs) then
          Abort(errNotDataFile, 3, 0);
        if (NumFree > NumRecs) then
          Abort(errNotDataFile, 4, 0);
        if (FirstFree = 0) then
          Abort(errNotDataFile, 5, 0);
        if (NumFree > 0) and (FirstFree = -1) then
          Abort(errNotDataFile, 6, 0);
        if (NumFree = 0) and (FirstFree <> -1) then
          Abort(errNotDataFile, 7, 0);
      end;
    {all OK, so output the system record}
    writeln('Data file: ', DataFileName);
    writeln;
    HexDumpSystemRecord;
  end;

procedure PrintIndexFileName;
  {Output the index file name}
  begin
    writeln;
    writeln('Index file: ', IndexFileName);
    writeln;
  end;

procedure WriteIndexSummary;
  {Summarise what we've found out about the indexes}
  const
    DupStr : array [boolean] of string[3] = ('No', 'Yes');
  var
    i : integer;
  begin
    writeln('Index file created with PageSize ', PageSize);
    writeln;
    writeln('Summary of indexes:');
    writeln('Index':8, 'Key len':9, 'Dups?':7, 'Total keys':12);
    for i := 1 to SystemRecord.NumKeys do
      with IndexDefn^[i] do
        writeln(i:8, KeyLen:9, DupStr[AllowDupKeys]:7, NumKeys:12);
  end;

procedure CheckIndexSystemPage(Index : integer; var Rec : TIndexSystemPage);
  {-Check the passed index system page to see if it makes sense}
  begin
    with Rec do
      begin
        if (PageLen > MaxRecordLength) or
           (FirstFree > NumPages) or
           (NumFree > NumPages) or
           (RootPage <= 0) or (RootPage > NumPages) or
           ((AllowDupK <> 0) and (AllowDupK <> 1)) then
          Abort(errNotIndexFile, 0, Index);
        if (DirInfo.Size < succ(NumPages)*PageLen) then
          Abort(errNotIndexFile, 2, Index);
        if (FirstFree = 0) then
          Abort(errNotIndexFile, 5, Index);
        if (NumFree > 0) and (FirstFree = -1) then
          Abort(errNotIndexFile, 6, Index);
        if (NumFree = 0) and (FirstFree <> -1) then
          Abort(errNotIndexFile, 7, Index);
      end;

    {all OK, so output the system page}
    HexDumpIndexSystemPage(Index, Rec);

    with IndexDefn^[Index] do
      begin
        IndexPageSize:= Rec.PageLen;
        FirstFree    := Rec.FirstFree;
        NumFree      := Rec.NumFree;
        NumPages     := Rec.NumPages;
        NumKeys      := Rec.NumKeys;
        Root         := Rec.RootPage;
        AllowDupKeys := boolean(Rec.AllowDupK);
      end;
  end;

procedure PatchUpIndexDetails;
  {Having read all the index system pages, calculate remaining data}
  var
    MaxNumPages : longint;
    MaxPageSize : word;
    MaxKeyLenIndex : integer;
    Offset : longint;
    i      : integer;
  begin
    {Note: the index with the longest keys will be that index with
           the biggest page size. As we look for it, calculate the
           index file record length (that reported by BTKeyRecordSize)}
    MaxNumPages := 0;
    MaxPageSize := 0;
    MaxKeyLenIndex := 0;
    Offset := 0;
    for i := 1 to SystemRecord.NumKeys do
      with IndexDefn^[i] do
        begin
          IndexPageOffset := Offset;
          inc(Offset, IndexPageSize);
          if (IndexPageSize > MaxPageSize) then
            begin
              MaxPageSize := IndexPageSize;
              MaxKeyLenIndex := i;
            end;
          if (NumPages > MaxNumPages) then
            MaxNumPages := NumPages;
        end;

    {do some more checks}
    with IndexDefn^[MaxKeyLenIndex] do
      begin
        {check the index file size}
        KeyRecSize := Offset;
        NumKeyRecs := succ(MaxNumPages);
        if (DirInfo.Size <> (KeyRecSize * NumKeyRecs)) then
          Abort(errNotIndexFile, 3, 0);
      end;

    {calculate the key lengths for all indexes}
    for i := 1 to SystemRecord.NumKeys do
      with IndexDefn^[i] do
        begin
          KeyLen := ((IndexPageSize - 6) div PageSize) - 9;
          if (KeyLen <= 0) or (KeyLen > 255) or
             (IndexPageSize <> (((KeyLen+9)*PageSize)+6)) then
            Abort(errNotIndexFile, 2, i);
        end;

    WriteIndexSummary;

  end;

procedure ReadRecordStart(RecNum : longint; var Value : longint);
  {-Read the first 4 bytes of a data record as a longint}
  var
    BytesRead : word;
  begin
    Seek(FB, RecNum*SystemRecord.RecLen);
    if IOfailed then
      Abort(errIOError, IOerror, 0);
    BlockRead(FB, Value, sizeof(longint), BytesRead);
    if (BytesRead <> sizeof(longint)) then
      Abort(errIOError, DiskReadError, 0);
  end;

procedure ReadRecordEnd(RecNum : longint; var Value : TVarRecLink);
  {-Read the link for a variable length record section}
  var
    BytesRead : word;
  begin
    Seek(FB, succ(RecNum)*SystemRecord.RecLen - sizeof(TVarRecLink));
    if IOfailed then
      Abort(errIOError, IOerror, 0);
    BlockRead(FB, Value, sizeof(TVarRecLink), BytesRead);
    if (BytesRead <> sizeof(TVarRecLink)) then
      Abort(errIOError, DiskReadError, 0);
  end;

procedure CheckDataDeletedList;
  {-Read the deleted record list, and check that it is valid}
  var
    NumDeletedRecs,
    RecNum    : longint;
  begin
    {Notes: when a record is deleted, B-Tree Filer adds its record
            number to a deleted record chain linked list. The first
            record number in the chain is at SystemRecord.FirstFree,
            the next number is held in the first four bytes of this
            record, and so on. The end of the list occurs when the
            first 4 bytes are set to $FF (a longint -1).}
    writeln('Checking deleted record list...');
    RecNum := SystemRecord.FirstFree;
    NumDeletedRecs := 0;
    while (RecNum <> -1) and (NumDeletedRecs < SystemRecord.NumFree) do
      begin
        inc(NumDeletedRecs);
        if Verbose then
          writeln('  link ', NumDeletedRecs, ' -> record ', RecNum);
        ReadRecordStart(RecNum, RecNum);
        if not ((RecNum = -1) or
                ((1 <= RecNum) and (RecNum <= SystemRecord.NumRecs))) then
          Abort(errInvRecNo, RecNum, 0);
      end;
    if (NumDeletedRecs <> SystemRecord.NumFree) then
      Abort(errBadDelList, 0, 0);
    if (RecNum <> -1) then
      Abort(errBadDelList, 1, 0);
    writeln('(done)');
  end;

function VerifyLink(var VarRecLink : TVarRecLink; IsFirst : boolean) : boolean;
  {-Verify that the variable length record link is valid}
  var
    OkSoFar : boolean;
    DataLength : word;
  begin
    with VarRecLink, SystemRecord do
      begin
        {calc the max amount of actual data in the section}
        if IsFirst then
             DataLength := RecLen - sizeof(TVarRecLink)
        else DataLength := RecLen - succ(sizeof(TVarRecLink));
        OkSoFar := SectionSize <= DataLength;
        if OkSoFar then
          OkSoFar := (0 <= NextSection) and (NextSection <= NumRecs);
        {if there is another section, this section's data must be
         fully used up}
        if OkSoFar and (NextSection <> 0) then
          OkSoFar := SectionSize = DataLength;
      end;
    VerifyLink := OkSoFar;
  end;

procedure CheckVarRecs;
  {-Go through the file and check variable length record links}
  var
    NumSections  : longint;
    RecNum       : longint;
    DeletedField : longint;
    VarRecLink   : TVarRecLink;
  begin
    {Notes: a variable length record is split into fixed length
            sections. The last 6 bytes of each section hold a
            TVarRecLink record. The second and subsequent sections
            also have the first byte set to $01 (this fools the
            VREORG/VREBUILD routines to assume the record as a whole
            is deleted). The first field of the TVarRecLink record is
            the length of actual data in the section, the second field
            is the record number of the next record in the section
            list (the final section of all has this set to zero).}
    writeln('Checking integrity of variable record links...');
    for RecNum := 1 to SystemRecord.NumRecs do
      begin
        ReadRecordStart(RecNum, DeletedField);
        if (DeletedField = 0) then
          begin
            if Verbose then
              writeln('Record ', RecNum);
            NumSections := 1;
            ReadRecordEnd(RecNum, VarRecLink);
            if not VerifyLink(VarRecLink, IsFirstSection) then
              Abort(errBadVarRecLink, RecNum, NumSections);
            while (VarRecLink.NextSection <> 0) do
              begin
                inc(NumSections);
                if (NumSections > SystemRecord.NumRecs) then
                  Abort(errBadVarRec, 0, 0);
                if Verbose then
                  writeln('  -> ', VarRecLink.NextSection);
                ReadRecordStart(VarRecLink.NextSection, DeletedField);
                if (DeletedField = 0) then
                  Abort(errBadVarRecLink, RecNum, pred(NumSections));
                ReadRecordEnd(VarRecLink.NextSection, VarRecLink);
                if not VerifyLink(VarRecLink, not IsFirstSection) then
                  Abort(errBadVarRecLink, RecNum, NumSections);
              end;
            if Verbose then
              writeln('  -> [end]');
          end;
      end;
    writeln('(done)');
  end;

procedure ReadPageStart(Index   : integer;
                        PageNum : longint;
                        AsAWord : boolean;
                        var Value : longint);
  {-Read the first 2/4 bytes of an index page as a word/longint}
  var
    BytesToRead,
    BytesRead : word;
  begin
    Seek(FB, PageNum*KeyRecSize+IndexDefn^[Index].IndexPageOffset);
    if IOfailed then
      Abort(errIOError, IOerror, 0);
    Value := 0;
    if AsAWord then
         BytesToRead := sizeof(word)
    else BytesToRead := sizeof(longint);
    BlockRead(FB, Value, BytesToRead, BytesRead);
    if (BytesRead <> BytesToRead) then
      Abort(errIOError, DiskReadError, 0);
  end;

procedure AddPageToDeletedList(PageNum : longint);
  {-Adds a page number to our internal index page deleted list}
  var
    Dad, Temp : PDeletedPage;
  begin
    with IndexDeletedList do
      if (Number = 0) then
        begin
          if (MaxAvail <= sizeof(TDeletedPage)) then
            Abort(errMemory, sizeof(TDeletedPage), 0);
          New(First);
          First^.Next := nil;
          First^.Page := PageNum;
          Number := 1;
        end
      else
        begin
          Dad := First;
          Temp := Dad^.Next;
          while Temp <> nil do
            begin
              Dad := Temp;
              Temp := Dad^.Next;
            end;
          if (MaxAvail <= sizeof(TDeletedPage)) then
            Abort(errMemory, sizeof(TDeletedPage), 0);
          New(Temp);
          Dad^.Next := Temp;
          Temp^.Next := nil;
          Temp^.Page := PageNum;
          inc(Number);
        end;
  end;

function PageIsInDeletedList(PageNum : longint) : boolean;
  {-Return true if PageNum is in our internal index page deleted list}
  var
    Temp : PDeletedPage;
  begin
    PageIsInDeletedList := true;
    with IndexDeletedList do
      if (Number <> 0) then
        begin
          Temp := First;
          while Temp <> nil do
            begin
              if (Temp^.Page = PageNum) then Exit;
              Temp := Temp^.Next;
            end;
        end;
    PageIsInDeletedList := false;
  end;

procedure FreeIndexDeletedList;
  {-Free our internal index page deleted list}
  var
    Temp : PDeletedPage;
  begin
    with IndexDeletedList do
      if (Number <> 0) then
        begin
          Temp := First;
          while Temp <> nil do
            begin
              First := Temp^.Next;
              Dispose(Temp);
              Temp := First;
            end;
        end;
    FillChar(IndexDeletedList, sizeof(IndexDeletedList), 0);
  end;

procedure CheckBTreeIndexPages(Index : integer);
  {Read through all the pages for an index and check the number of keys}
  var
    NumKeysOnPage : longint;
    Count   : longint;
    PageNum : longint;
    RootPage: longint;
  begin
    writeln('Checking pages for index ', Index, '...');
    if Verbose then
      writeln('Page':10, 'NumKeys':10);
    Count := 0;
    RootPage := IndexDefn^[Index].Root;
    for PageNum := 1 to pred(NumKeyRecs) do
      if PageIsInDeletedList(PageNum) then
        begin
          if Verbose then
            writeln(PageNum:10, '(deleted)':10);
        end
      else
        begin
          ReadPageStart(Index, PageNum, true, NumKeysOnPage);
          if (NumKeysOnPage > PageSize) then
            Abort(errIndexKeys, Index, NumKeysOnPage)
          else if (NumKeysOnPage > 0) and
                  (NumKeysOnPage < (PageSize shr 1)) and
                  (PageNum <> RootPage) then
            writeln('***WARNING*** ', PageNum, ' has too few keys');
          inc(Count, NumKeysOnPage);
          if Verbose then
            writeln(PageNum:10, NumKeysOnPage:10);
        end;
    if Verbose then
      writeln('Total keys':10, Count:10);
    if (Count <> IndexDefn^[Index].NumKeys) then
      begin
        if not Verbose then
          writeln('Keys found: ', Count);
        writeln('***WARNING*** ',
                'number of keys expected and actually found do not tally');
        {Abort(errTotalKeys, Index, 0); <== an alternative}
      end
    else
      writeln('(done)');
  end;

procedure CheckIndexDeletedList(Index : integer);
  {-Read the deleted page list, and check that it is valid}
  var
    TotalPages,
    NumFreePages,
    NumDeletedPages,
    PageNum    : longint;
  begin
    {Notes: index pages are deleted in the same way as the data file.}
    writeln;
    writeln('---');
    writeln('Checking deleted page list for index ', Index, '...');
    with IndexDefn^[Index] do
      begin
        PageNum := FirstFree;
        NumFreePages := NumFree;
        TotalPages := NumPages;
      end;
    NumDeletedPages := 0;
    while (PageNum <> -1) and (NumDeletedPages < NumFreePages) do
      begin
        inc(NumDeletedPages);
        AddPageToDeletedList(PageNum);
        if Verbose then
          writeln('  link ', NumDeletedPages, ' -> page ', PageNum);
        ReadPageStart(Index, PageNum, false, PageNum);
        if not ((PageNum = -1) or
                ((1 <= PageNum) and (PageNum <= TotalPages))) then
          Abort(errInvPageNo, PageNum, 0);
      end;
    if (NumDeletedPages <> NumFreePages) then
      Abort(errBadDelList, 0, 0);
    if (PageNum <> -1) then
      Abort(errBadDelList, 1, 0);
    writeln('(done)');
  end;

{$IFDEF Windows}
const
  Archive = faArchive;
{$ENDIF}

var
  Index : integer;
  BytesRead : word;
  Offset    : longint;
  IndexSysRec : TIndexSystemPage;

begin
  CopyrightMessage;

  CheckIndex := false;
  SysRecIsRead := false;
  ReadParameters;

  {Data file first}
  FindFirst(DataFileName, Archive, DirInfo);
  if (DosError <> 0) then
    Abort(errDosError, DosError, 0);

  if (DirInfo.Size < SystemRecordSize) then
    Abort(errNotDataFile, 0, 0);

  FileMode := $40; {readonly, deny none}
  Assign(FB, DataFileName);
  Reset(FB, 1);
  if IOfailed then
    Abort(errIOerror, IOError, 0);

  BlockRead(FB, SystemRecord, SystemRecordSize, BytesRead);
  if (BytesRead <> SystemRecordSize) then
    Abort(errIOerror, DiskReadError, 0);
  SysRecIsRead := true;

  CheckSystemRecord;

  CheckDataDeletedList;

  if IsVarRecFile then
    CheckVarRecs;

  Close(FB);
  if IOfailed then
    DoNothing;

  {Index file next}
  if CheckIndex then
    begin

      PrintIndexFileName;

      IndexDefnSize := SystemRecord.NumKeys * sizeof(TIndexDefn);
      if (MaxAvail <= IndexDefnSize) then
        Abort(errMemory, IndexDefnSize, 0);
      GetMem(IndexDefn, IndexDefnSize);
      FillChar(IndexDefn^, IndexDefnSize, 0);

      FindFirst(IndexFileName, Archive, DirInfo);
      if (DosError <> 0) then
        Abort(errDosError, DosError, 0);

      if (DirInfo.Size < SystemRecordSize) then
        Abort(errNotDataFile, 0, 1);

      FileMode := $40; {readonly, deny none}
      Assign(FB, IndexFileName);
      Reset(FB, 1);
      if IOfailed then
        Abort(errIOerror, IOError, 0);

      Offset := 0;
      for Index := 1 to SystemRecord.NumKeys do
        begin
          Seek(FB, Offset);
          if IOfailed then
            Abort(errIOerror, IOError, 0);

          BlockRead(FB, IndexSysRec, IndexSystemPageSize, BytesRead);
          if (BytesRead <> IndexSystemPageSize) then
            Abort(errIOerror, DiskReadError, 0);

          CheckIndexSystemPage(Index, IndexSysRec);

          inc(Offset, IndexSysRec.PageLen);
        end;

      PatchUpIndexDetails;

      FillChar(IndexDeletedList, sizeof(IndexDeletedList), 0);
      for Index := 1 to SystemRecord.NumKeys do
        begin
          CheckIndexDeletedList(Index);
          CheckBTreeIndexPages(Index);
          FreeIndexDeletedList;
        end;

      Close(FB);
      if IOfailed then
        DoNothing;
    end;
end.