unit BP7PBTRV;

interface

uses Dos, WinAPI;

function Btrv( Op: integer;
               var Posn;
               var Data;
               var DataLen: integer;
               var KBuf;
               Key: integer
             ): integer;

implementation

{$R-}         {Range checking off}
{$B+}         {Boolean complete evaluation on}
{$S+}         {Stack checking on}
{$I+}         {I/O checking on}

{                                                                             }
{  Module Name: BP7PBTRV.PAS                                                  }
{                                                                             }
{  Adapted by:  Tony Malins, Barclays Bank plc, UK                            }
{               Any comments/corrections would be appreciated                 }
{               I can be reached on Compuserve 100113,3667                    }
{                                                                             }
{  Description: This is the Btrieve interface for Borland Pascal version 7    }
{               DOS protected mode applications.  This routine sets up the    }
{               parameter block expected by Btrieve, and issues interrupt 7B. }
{                                                                             }
{               This module has been adapted from the base TUR5BTRV.PAS       }
{               supplied by Novell.  BP7PBTRV functions by allocating         }
{               temporary blocks of real memory (ie below 1M), copying the    }
{               callers parameters into these real mode blocks, and then      }
{               calling the DPMI interface to issue Btrieve's real mode       }
{               interrupt.  And of course, any returned data must be copied   }
{               back.  An added complication was the need to issue a STAT     }
{               call to Btrieve to determine the length of the KBuf           }
{               parameter (to avoid protected mode violations copying KBuf    }
{               between real and protected memory).                           }
{                                                                             }
{               BP7PBTRV.PAS should be compiled with the $V- switch so that   }
{               runtime checks will not be performed on the variable          }
{               parameters.                                                   }
{                                                                             }
{  Synopsis:    Status := Btrv (Op, Posn.start, Data.start, DataLen,          }
{                               KBuf.start, Key);                             }
{                    where                                                    }
{                        Op is an integer,                                    }
{                        Posn is a 128 byte array,                            }
{                        Data is an untyped parameter for the data buffer,    }
{                        DataLen is the integer length of the data buffer,    }
{                        KBuf is the untyped parameter for the key buffer,    }
{                    and Key is an integer.                                   }
{                                                                             }
{               Note that Posn, Data & KBuf are passed as untyped var         }
{               parameters.  To avoid protection faults, the caller must      }
{               never pass uninitialised values.  Dummy occurrences should    }
{               be defined thus:                                              }
{                 var                                                         }
{                   DummyPtr: Pointer;                                        }
{                   DummyInt: Integer;                                        }
{                 DummyPtr := @DummyPtr;                                      }
{                 DummyInt := 0;                                              }
{                 Status := btrv( kB_Reset, DummyPtr, ..., DummyInt, ... );   }
{                                                                             }
{  Returns:     Btrieve status code (see Appendix B of the Btrieve Manual).   }
{                                                                             }
{  Note:        The Btrieve manual states that the 2nd, 3rd, and 5th          }
{               parameters be declared as variant records, with an integer    }
{               type as one of the variants (used only for Btrieve calls),    }
{               as is shown in the example below.  This is supported, but     }
{               the restriction is no longer necessary.  In other words, any  }
{               variable can be sent in those spots as long as the variable   }
{               uses the correct amount of memory so Btrieve does not         }
{               overwrite other variables.                                    }
{                                                                             }
{                   var Data = record case boolean of                         }
{                         False: ( Start: integer );                          }
{                         True:  ( Employee_Id: 0..99999;                     }
{                                  Employee_Name: packed array[1..50] of char;}
{                                  Salary: real;                              }
{                                  Date_Of_Hire: Date_Type );                 }
{                         end;                                                }
{                                                                             }
{               There should NEVER be any string variables declared in the    }
{               data or key records, because strings store an extra byte for  }
{               the length, which affects the total size of the record.       }
{                                                                             }
{                                                                             }


function Btrv( Op: integer;
               var Posn;
               var Data;
               var DataLen: integer;
               var KBuf;
               Key: integer
             ): integer;

const

     Var_Id            = $6176;   { id for variable length records - 'va' }
     DPMI_Intr         = $31;
     Btrv_Intr         = $7B;
     Btrv2_Intr        = $2F;
     Btrv_Offset       = $0033;
     Multi_Function    = $AB;
     MaxRecordSize     = 1024;

     { Btrieve operation codes... }
     kB_Open           =  0;
     kB_Close          =  1;
     kB_Insert         =  2;
     kB_Update         =  3;
     kB_Delete         =  4;
     kB_GetEq          =  5;
     kB_GetNext        =  6;
     kB_GetPrevious    =  7;
     kB_GetGT          =  8;
     kB_GetGE          =  9;
     kB_GetLT          = 10;
     kB_GetLE          = 11;
     kB_GetFirst       = 12;
     kB_GetLast        = 13;
     kB_Create         = 14;
     kB_Stat           = 15;
     kB_Extend         = 16;
     kB_SetDir         = 17;
     kB_GetDir         = 18;
     kB_BeginTran      = 19;
     kB_EndTran        = 20;
     kB_AbortTran      = 21;
     kB_GetPos         = 22;
     kB_GetDirect      = 23;
     kB_StepNext       = 24;
     kB_Stop           = 25;
     kB_Version        = 26;
     kB_Unlock         = 27;
     kB_Reset          = 28;
     kB_SetOwner       = 29;
     kB_ClrOwner       = 30;
     kB_CreateIndex    = 31;
     kB_DropIndex      = 32;
     kB_StepFirst      = 33;
     kB_StepLast       = 34;
     kB_StepPrevious   = 35;
     kB_GetNextExt     = 36;
     kB_GetPrevExt     = 37;
     kB_StepNextExt    = 38;
     kB_StepPrevExt    = 39;
     kB_InsertExt      = 40;

     kB_GetKey               =  50;
     kB_WaitLock             = 100;
     kB_NoWaitLock           = 200;
     kB_MultipleWaitLock     = 300;
     kB_MultipleNoWaitLock   = 400;

     { Btrieve status codes... }
     kSuccess          =  0;
     kFileNotOpen      =  3;
     kRecManInactive   = 20;
     kDataBufferLength = 23;

     {  ProcId is used for communicating with the Multi Tasking Version of    }
     {  Btrieve. It contains the process id returned from BMulti and should   }
     {  not be changed once it has been set.                                  }
     ProcessId: integer = 0;       { initialize to no process id }
     Multi: boolean = false;       { set to true if BMulti is loaded }
     VSet: boolean = false;        { set true if we have checked for BMulti   }

type
     SegOfs = record                   { 32 bit address }
        Offset: word;
        Segment: word;
     end;

     tDosAlloc = record                { GlobalDosAlloc variance record }
       case Boolean of
         True:  ( Addresses: LongInt );
         False: ( Selector: word;
                  Segment: word;
                );
     end;

     tBtrv_Parms = record
        Btrv_Buf_Addr:     SegOfs;     { data buffer address        }
        Btrv_Buf_Len:      integer;    { data buffer length         }
        Btrv_Cur_Addr:     SegOfs;     { currency block address     }
        Btrv_FCB_Addr:     SegOfs;     { file control block address }
        Btrv_Function:     integer;    { Btrieve operation          }
        Btrv_Key_Addr:     SegOfs;     { key buffer address         }
        Btrv_Key_Length:   byte;       { key buffer length          }
        Btrv_Key_Number:   shortint;   { key number                 }
        Btrv_Status_Addr:  SegOfs;     { return status address      }
        Btrv_Xface_Id: integer;        { language interface id      }
     end;

     tPosn_Blk = array [1..128] of char;

     tData_Buf = array [1..MaxRecordSize] of char;

     tKey_Buf = array [1..255] of char;

     tKey_Spec = record
       KeyPosn:      integer;
       KeyLen:       integer;
       KeyFlags:     word;
       NoKeys:       longint;
       KeyType:      char;
       NullValue:    char;
       reserved:     array [1..4] of byte;
     end;

     tStatus_Buf = record
       RecordLen:    integer;
       PageSize:     integer;
       NoIndexes:    integer;
       NoRecords:    longint;
       FileFlags:    word;
       reserved:     word;
       UnusedPages:  integer;
       KeySpec:      array [1..10] of tKey_Spec;
     end;


     tDPMI_Regs = record               { DPMI call structure }
       EDI:        longint;
       ESI:        longint;
       EBP:        longint;
       reserved:   longint;
       EBX:        longint;
       EDX:        longint;
       ECX:        longint;
       EAX:        longint;
       flags:      word;
       ES:         word;
       DS:         word;
       FS:         word;
       GS:         word;
       IP:         word;
       CS:         word;
       SP:         word;
       SS:         word;
     end;


var
     DosAlloc:     tDosAlloc;

                                       {Btrieve...}
     pStatus_Code:     ^integer;       {...status code}
     segStatus_Code:   word;
     pBtrv_Parms:      ^tBtrv_Parms;   {...parameter block}
     segBtrv_Parms:    word;
     pPosn_Blk:        ^tPosn_Blk;     {...position block}
     segPosn_Blk:      word;
     pData_Buf:        ^tData_Buf;     {...data buffer}
     segData_Buf:      word;
     pKey_Buf:         ^tKey_Buf;      {...key buffer}
     segKey_Buf:       word;
     KBufLen:          integer;

     RC:               integer;
     Operation:        integer;        { Btrieve op code after masking }

     Regs:         Dos.Registers;  { Registers for software interrupt calls }
     DPMI_Regs:    tDPMI_Regs;     { Registers for DPMI interrupt }

  function BtrvInt( Op: integer;
                    var Posn;
                    var Data;
                    var DataLen: integer;
                    var KBuf;
                    KBufLen: integer;
                    Key: integer
                  ): integer;
  begin

    { Initialise real mode parameters... }

    { ...status code }
    pStatus_Code^ := -1;

    { ...position block }
    if Pointer(Posn) <> @Posn then
      Move( Posn, pPosn_Blk^, SizeOf(tPosn_Blk) )
    else
      FillChar( pPosn_Blk^, SizeOf(tPosn_Blk), #0 );

    { ...data buffer }
    if Pointer(Data) <> @Data then
      Move( Data, pData_Buf^, DataLen )
    else
      FillChar( pData_Buf^, SizeOf(tData_Buf), #0 );

    { ...key buffer }
    if Pointer(KBuf) <> @KBuf then
      Move( KBuf, pKey_Buf^, KBufLen )
    else
      FillChar( pKey_Buf^, SizeOf(tKey_Buf), #0 );

    { Setup Btrieve call/return parameter block }
    with pBtrv_Parms^ do
    begin
      Btrv_Buf_Addr.Segment := segData_Buf;      {...data buffer address}
      Btrv_Buf_Addr.Offset := 0;
      Btrv_Buf_Len := DataLen;
      Btrv_FCB_Addr.Segment := segPosn_Blk;      {...FCB address}
      Btrv_FCB_Addr.Offset := 0;
      Btrv_Cur_Addr.Segment := Btrv_FCB_Addr.Segment; {...currency block seg}
      Btrv_Cur_Addr.Offset := Btrv_FCB_Addr.Offset+38;{...currency block ofs}
      Btrv_Function := OP;                       {...operation code}
      Btrv_Key_Addr.Segment := segKey_Buf;       {...key buffer address}
      Btrv_Key_Addr.Offset := 0;
      Btrv_Key_Length := 255;                    {...assume its large enough}
      Btrv_Key_Number := Key;                    {...key number}
      Btrv_Status_Addr.Segment := segStatus_Code;{...status code address}
      Btrv_Status_Addr.Offset := 0;
      Btrv_Xface_Id := Var_Id;                   {...language id}
    end;

    { Use DPMI interface to issue real mode interrupt to call Btrieve }
    FillChar( DPMI_Regs, SizeOf(tDPMI_Regs), #0 );
    DPMI_Regs.DS := segBtrv_Parms;
    DPMI_Regs.EDX := 0;  {offset for Btr_Parms; always zero}
    Regs.AX := $0300;
    Regs.BL := Btrv_Intr;
    Regs.BH := 0;
    Regs.CX := 0;
    Regs.ES := Seg( DPMI_Regs );
    Regs.DI := Ofs( DPMI_Regs );
    Intr( DPMI_Intr, Regs );

    { Update callers parameters with Btrieve's output... }

    { ...position block }
    if Pointer(Posn) <> @Posn then
      Move( pPosn_Blk^, Posn, SizeOf(tPosn_Blk) );

    { ...data buffer }
    DataLen := pBtrv_Parms^.Btrv_Buf_Len;
    if ( Pointer(Data) <> @Data ) and ( DataLen > 0 ) then
      Move( pData_Buf^, Data, DataLen );

    { ...key buffer }
    if Pointer(KBuf) <> @KBuf then
      Move( pKey_Buf^, KBuf, KBufLen );

    { Return with Btrieve status code }
    BtrvInt := pStatus_Code^;

  end; {of local function BtrvInt}

  function CalcKBufLen: integer;
   var
     KeyBuf:           tKey_Buf;
     StatusBuf:        tStatus_Buf;
     StatusBufLen:     integer;
     x:                integer;
     wKeyNo, wKeyLen, wKeyLength : integer;
  begin

    FillChar( KeyBuf, SizeOf(tKey_Buf), #0 );
    FillChar( StatusBuf, SizeOf(tStatus_Buf), #0 );
    StatusBufLen  := SizeOf( tStatus_Buf );

    BtrvInt( kB_Stat, Posn, StatusBuf, StatusBufLen, KeyBuf, 0, 0 );

    wKeyLength := 0;
    wKeyNo := 0;
    x := 1;

    while (wKeyNo < StatusBuf.NoIndexes) do
    begin

      wKeyLen := StatusBuf.KeySpec[x].KeyLen;

      while (StatusBuf.KeySpec[x].KeyFlags and $10 {segmented key}) > 0 do
      begin
        Inc(x);
        wKeyLen := wKeyLen + StatusBuf.KeySpec[x].KeyLen;
      end;

      if wKeyNo = Key then
        wKeyLength := wKeyLen;

      Inc(x);
      Inc(wKeyNo);
    end;

    CalcKBufLen := wKeyLength;

  end; {of local function CalcKBufLen}

begin

  { Make sure Btrieve is installed }
  { Use DPMI interface to issue real mode interrupt }
  FillChar( DPMI_Regs, SizeOf(tDPMI_Regs), #0 );
  DPMI_Regs.EAX := $3500 + Btrv_Intr;
  Regs.AX := $0300;
  Regs.BL := $21;
  Regs.BH := 0;
  Regs.CX := 0;
  Regs.ES := Seg( DPMI_Regs );
  Regs.DI := Ofs( DPMI_Regs );
  Intr( DPMI_Intr, Regs );
  if (DPMI_Regs.EBX <> Btrv_Offset) then
  begin
    Btrv := kRecManInactive; { return record manager inactive status code }
    Exit;
  end;


  { Allocate and initialise real mode storage for Btrieve... }

  { ...Btrieve call/return parameter block }
  DosAlloc.Addresses := GlobalDosAlloc( SizeOf(tBtrv_Parms) );
  pBtrv_Parms := Ptr( DosAlloc.Selector, 0 );
  segBtrv_Parms := DosAlloc.Segment;

  { ...Status code (return code; type integer) }
  DosAlloc.Addresses := GlobalDosAlloc( SizeOf(Integer) );
  pStatus_Code := Ptr( DosAlloc.Selector, 0 );
  segStatus_Code := DosAlloc.Segment;

  { ...position block }
  DosAlloc.Addresses := GlobalDosAlloc( SizeOf(tPosn_Blk) );
  pPosn_Blk := Ptr( DosAlloc.Selector, 0 );
  segPosn_Blk := DosAlloc.Segment;

  { ...data buffer }
  DosAlloc.Addresses := GlobalDosAlloc( MaxRecordSize );
  pData_Buf := Ptr( DosAlloc.Selector, 0 );
  segData_Buf := DosAlloc.Segment;

  { ...key buffer }
  DosAlloc.Addresses := GlobalDosAlloc( 255 );
  pKey_Buf := Ptr( DosAlloc.Selector,0 );
  segKey_Buf := DosAlloc.Segment;

  { Calculate KBufLen... }

  { ...Operation is masked Op code }
  if (Op-kB_MultipleNoWaitLock) > 0 then
    Operation := Op - kB_MultipleNoWaitLock
  else if (Op-kB_MultipleWaitLock) > 0 then
    Operation := Op - kB_MultipleWaitLock
  else if (Op-kB_NoWaitLock) > 0 then
    Operation := Op - kB_NoWaitLock
  else if (Op-kB_WaitLock) > 0 then
    Operation := Op - kB_WaitLock
  else if (Op-kB_GetKey) > 0 then
    Operation := Op - kB_GetKey
  else
    Operation := Op;

  KBufLen := 0;  { assume zero }
  case Operation of
    kB_Open,
    kB_Create,
    kB_Reset,
    kB_SetOwner:
        if Pointer(KBuf) <> @KBuf then
        while (tKey_Buf(KBuf)[KBufLen] <> #0) and (KBufLen < 255) do
          Inc( KBufLen );
    kB_Close,
    kB_Delete,
    kB_Extend,
    kB_BeginTran,
    kB_EndTran,
    kB_AbortTran,
    kB_GetPos,
    kB_StepNext,
    kB_Stop,
    kB_Version,
    kB_Unlock,
    kB_ClrOwner,
    kB_CreateIndex,
    kB_DropIndex,
    kB_StepFirst,
    kB_StepLast,
    kB_StepPrevious,
    kB_StepNextExt,
    kB_StepPrevExt:
        KBufLen := 0;

    kB_Stat:
        KBufLen := 64;

    kB_GetDir:
        KBufLen := 65;

    else
        KBufLen := CalcKBufLen;

  end {case of Operation};

  Btrv := BtrvInt( Op, Posn, Data, DataLen, KBuf, KBufLen, Key );

  { Dispose of real memory }
  RC := GlobalDosFree( Seg(pStatus_Code^) );
  RC := GlobalDosFree( Seg(pPosn_Blk^) );
  RC := GlobalDosFree( Seg(pData_Buf^) );
  RC := GlobalDosFree( Seg(pKey_Buf^) );
  RC := GlobalDosFree( Seg(pBtrv_Parms^) );  {must be last!}

end;
{$B-}

begin
  { Idea:  Issue GlobalDosAlloc's once only here... }
end.
