program ScreenBlanker;                      { turns off the display }
                                            { and waits for a key   }
{$M 16384,0,0}                              { leave the heap for appl. prgms }

uses
  Dos,Crt;                                  { units we'll use }
const
  PortAddress :array [0..1] of integer = ($3B8,$3D8);
type
  VideoCardType = (Mono,CGA);               { the cards we're looking for }
var
  CrtModeSet :byte absolute $0040:$0065;    { current video mode kept here }
  Regs :Registers;                          { predefined Type in Dos unit }
  Seconds :real;                            { elapsed time since key press }
  OldExit,                                  { addr of Turbo's run-time ExitProc }
  Old_9_Vector,                             { address of old intr 9 vector }
  Old_1C_Vector :pointer;                   { address of old intr $1C vector }
  ScreenOn :boolean;                        { set to false when scr is blanked }
  ErrorCode,
  ClockTicks,                               { BIOS clock ticks 18.2 times/sec }
  Delay :integer;                           { time to wait before blanking scr }
  AddrIndex,                                { 0 = Mono, 1 = CGA }
  DisplayOn,                                { bytes written to I/O port }
  DisplayOff :byte;



procedure RestoreOldVector (IntrNumber :integer; OldVector :pointer);

  { a generic procedure which restores the old interrupt vector entry in the
    interrupt vector table before exiting }

  begin
    SetIntVec (IntrNumber,OldVector);
  end; {procedure}


{$F-}
procedure OnExit; {$F+}                     { custom exit procedures }
  begin
    if ErrorCode <> 0 then
      case ErrorCode of
        1 :begin
             Writeln;
             Writeln ('Unknown video card installed.  Program Aborted.');
             Write ('Please contact the author about this problem.');
             Writeln;
           end;
        2 :begin
             ClrScr;
             Writeln ('Display blanking program successfully installed.  Delay: ',Delay);
             Writeln;
           end;
        else begin                          { abnormal exit }
          RestoreOldVector ($1C,Old_1C_Vector);
          RestoreOldVector (9,Old_9_Vector);
        end; {else}
      end; {case}
      ExitProc := OldExit;
    end; {procedure}


procedure WatchClock;                       { BIOS timer tick inter handler }
interrupt;
  procedure VideoSwitch (PortAddr :integer; DataOut :byte);
    begin
      Port [PortAddr] := DataOut;           { send On/Off byte to I/O port }
    end; {nested procedure}
  begin                                     { main procedure }
    InLine ($FA);                           { disable interrupts }
    ClockTicks := ClockTicks + 1;           { increment the time }
    Seconds := int (ClockTicks/18.2);       { time since last key press }
    if (Seconds >= Delay) and (ScreenOn) then begin
                                            { turn off the display }
      VideoSwitch (PortAddress [AddrIndex],DisplayOff);
      ScreenOn := false;                    { set the flag- screen is off }
    end {if}
    else if (Seconds < Delay) and (not ScreenOn) then begin
                                            { turn screen back on }
      VideoSwitch (PortAddress [AddrIndex], DisplayOn);
      ScreenOn := true;                     { reset the flag }
    end; {else}
    InLine ($FB);                           { re-enable interrupts }
  end; {procedure}


procedure WatchKeyBoard;                    { monitors keyboard via intr 9 }
interrupt;
  begin
    InLine ( $9C/                           { PUSH AF }
             $3E/$FF/$1E/Old_9_Vector       { CALL FAR DS:[OLD_9_VECTOR] }
           );                               { pass keystroke to old intr 9 }
    ClockTicks := 0;                        { reset counter }
    Seconds := 0;
  end; {procedure}


function GetVideoCard :VideoCardType;
  { returns the video controller hardware configuration }
  begin
    Intr ($11,Regs);                        { issue the interrupt }
    case Lo (Regs.ax) AND $30 of
      $30  :GetVideoCard := Mono;
      $20  :GetVideoCard := CGA;            { 80 column text }
      $10  :GetVideoCard := CGA;            { 40 column text }
      else begin                            { video card unknown }
        Writeln;
        Writeln ('Unknown video card installed.  Program Aborted.');
        Write ('Please contact the author about this problem.');
        Writeln;
        Halt;
      end; {else}
    end; {case}
  end; {function}


procedure Initialize;
  begin
    if ParamCount > 0 then                  { get delay time }
      val (ParamStr (1),Delay,ErrorCode)
    else Delay := 5;                        { default delay- 5 min }
    Delay := Delay * 60;                    { convert delay to seconds }
    ClockTicks := 0;                        { init variables }
    Seconds := 0;
    ScreenOn := true;
    OldExit := ExitProc;                    { save old exit procedure address }
    ExitProc := @OnExit;                    { insert custom exit procedure }

    Inline                                  { clear the key buffer }
    ($B4/$06/                               { L1: MOV  AH,6    ;function }
     $B2/$FF/                               {     MOV  DL,0FFH ;subfunction }
     $CD/$21/                               {     INT  21H     ;key in buffer? }
     $75/$F8);                              {     JNZ  L1      ;repeat if yes }

    case GetVideoCard of                    { set byte to send to I/O port }
      Mono :begin                           { Mono card }
              DisplayOff := $21;
              DisplayOn := $29;
              AddrIndex := 0;
            end;
      CGA  :begin                           { CGA }
              DisplayOff := CrtModeSet AND $F7;
              DisplayOn := CrtModeSet;
              AddrIndex := 1;
            end;
    end; {case}

    GetIntVec ($1C,Old_1C_Vector);          { save orig. intr $1C vector }
    SetIntVec ($1C,@WatchClock);            { install the new $1C handler }
    GetIntVec (9,Old_9_Vector);             { save orig intr 9 vector }
    SetIntVec (9,@WatchKeyBoard);           { install new intr 9 handler }
  end; {procedure}



begin                                       { main program }
  Initialize;
  Writeln;
  Writeln ('Display blanking program successfully installed.  Delay: ',
            Delay div 60,' minute(s).');
  Writeln;
  Keep (2);
  RestoreOldVector ($1C,Old_1C_Vector);
  RestoreOldVector (9,Old_9_Vector);
end. {program}
