{
  BIOSFIX - a program to update the BIOS data area and equipment flag to
  reflect the number and base addresses of standard serial ports Com1 through
  Com4

  Usage: BIOSFIX [option]
    Q       quiet mode - suppress messages
    ?       show help

  Normally, the BIOS data area that contains the addresses of all known
  standard UARTs, [$0040:0000], is filled in at bootstrap time. Unfortunately,
  not all BIOSs check for UARTs at the Com3 and Com4 addresses. Some
  communications programs check the BIOS data area for UART addresses and will
  refuse to use a serial port unless its address is listed in the BIOS data
  area. In fact, you may choose to use such a check in your programs before
  opening a requested serial port (as a very safe method of operation to
  guarantee that no other hardware is operating at a standard UART address).

  BIOSFIX helps out in this area by checking for UARTs at all standard
  addresses and updating the BIOS data area and the equipment flags. You would
  typically run BIOSFIX as part of your AUTOEXEC.BAT -- preferably before
  installing any network drivers or any other hardware drivers (in case the
  addresses of the related hardware conflicts with standard UART addresses).

  Requires Async Professional to compile.
  Released to the public domain.

  Written by Terry Hughes, TurboPower Software

  Version 1.0 - 5/21/91
    initial release
}

{$R-,S-}
program BiosFix;
  {-Adds standard serial ports, Com1..Com4, to the BIOS data area}
uses
  ApMisc,
  ApPort,
  ApUart;

const
  {Search range}
  FirstCom = Com1;
  LastCom = Com4;

  {To show update messages}
  ShowMsg : Boolean = True;

var
  Com : ComNameType;
  BaseAddr : Word;
  BiosIndex : Byte;
  BiosData : array[1..4] of Word absolute $40:$0;
  EquipFlag : Word absolute $40:$10;

  function HexW(W : Word) : string;
    {-Return hex string for word}
  const
    Digits : array[0..$F] of Char = '0123456789ABCDEF';
  begin
    HexW[0] := #4;
    HexW[1] := Digits[hi(W) shr 4];
    HexW[2] := Digits[hi(W) and $F];
    HexW[3] := Digits[lo(W) shr 4];
    HexW[4] := Digits[lo(W) and $F];
  end;

procedure ShowHelp;
  {-Show usage}
begin
  WriteLn('Usage: BIOSFIX [option]');
  WriteLn('  Q     quiet mode - suppress messages');
  WriteLn('  ?     show help');
  Halt;
end;

procedure ParseCommandLine;
  {-Get args}
var
  S : String;
begin
  if ParamCount <> 0 then begin
    S := ParamStr(1);
    case upcase(S[1]) of
      'Q' : ShowMsg := False;
      else  ShowHelp;
    end;
  end;
end;

procedure UpdateEquip;
  {-Update the equipment flag to reflect the number of serial ports found}
begin
  Dec(BiosIndex);
  EquipFlag := (EquipFlag and $F1FF) or (BiosIndex shl 9);
  if ShowMsg then
    WriteLn('Updated equipment flag to show ', BiosIndex, ' serial ports');
end;

begin
  ParseCommandLine;

  BiosIndex := 1;
  for Com := FirstCom to LastCom do begin
    {Get the base address for this serial port}
    if ((Com = Com3) or (Com = Com4)) and IsPS2 then
      BaseAddr := DefPS2Addr[Com]
    else
      BaseAddr := DefBaseAddr[Com];

    {If a UART is found, update the BIOS data area}
    if UartTest1(BaseAddr) and UartTest2(BaseAddr) then begin
      if ShowMsg then
        WriteLn('BIOS updated for ', ComNameString(Com), ' at ', HexW(BaseAddr));
      BiosData[BiosIndex] := BaseAddr;
      Inc(BiosIndex);

      {Prevent overflowing the BIOS data area}
      if BiosIndex > 4 then begin
        UpdateEquip;
        Halt;
      end;
    end;
  end;

  {Update the equipment flags}
  UpdateEquip;
end.
