{ Turbo Pascal Beautifier Version 0.5c }
{ by Toupao Chieng, 1-1-91 12:00 pm    }

program TurboPascalBeautifier(Input, Output);
(*
 |-------------------------------------------------------------------------|
 |                                                                         |
 |                    Turbo Pascal Beautifier Version 0.5c                 |
 |                                   by                                    |
 |                              Toupao Chieng                              |
 |                             January 1, 1991                             |
 |                                                                         |
 |-------------------------------------------------------------------------|

 NOTES: BEAUTIFY, based on original code (PB.PAS) by a company in Texas,
        which I don't know the name.

        TechnoJock's Turbo Tool Kit was used to help build this program.
        You must have it's TPU files in order to recompile the program.

        BEAUTIFY is distributed as public domain.

 WARNNINGS: There are minor bugs. This program assumes that there are enough
            spaces for its temp files. If there's not enough free space, the
            program gives a runtime error, and halts.

            You can't assign work drives (maybe in the future), so make sure
            you have enough space on the current drive before formatting.

            EXAMPLE: if you where formatting a file with 20000 bytes, you
                     must have at least 21000-25000 bytes free, depending
                     on how you choose to format it. With the -z+ option
                     it doubles, because it needs two temp files (for now).

            Other than that, the program works great!

 OTHERS: Other bug reports are welcomed to me...
*)

{$R-}

uses 
    Crt, Dos, TPFast, TPMisc, TPStr, TPWin; { Turbo Tool Kit Units. The
    units and procedures/functions are renamed for my own convenience }

const
    HexDigits    = ['0'..'9', 'A'..'f', 'a'..'f'];
    MaxRwLen     = 32;
    OrdMinChar   = 32;
    OrdMaxChar   = 126;
    MaxBuiltins  = 750;
    MaxInLen     = 255;

type
    Natural      = 0..MaxInt;
    InRange      = 0..MaxInLen;
    OutRange     = 0..256;
    RwStringType = String[MaxRwLen];

    ErrorType = (
        LongLine, NoEndComm, NotQuote, LongWord, NotDo, NotOf, NotEnd, NotThen,
        NotBegin, NotUntil, NotSemicolon, NotColon, NotParen, NoEof, NotInterface,
        NotImplementation
);
    CharType = (
        Illegal, Special, ChaPostrophe, ChLeftParen, ChLeftBracket, ChRightParen,
        ChPeriod, Digit, ChColon, ChComma, ChSemicolon, ChRightBracket, ChLessThan,
        ChGreaterThan, Letter, ChLeftBrace, ChRightBrace, ChEqual, ChHash,
        ChDollar, ChCaret
);
    ResWord = (
        RwIf, RwDo, RwOf, RwTo, RwIn, RwOr, RwEnd, RwFor, RwVar, RwDiv, RwMod,
        RwSet, RwAnd, RwNot, RwNil, RwShl, RwShr, RwXor, RwThen, RwElse, RwWith,
        RwGoto, RwCase, RwType, RwFile, RwUnit, RwUses, RwBegin, RwUntil, RwWhile,
        RwArray, RwConst, RwLabel, RwRepeat, RwRecord, RwDownto, RwPacked, RwInline,
        RwString, RwObject, RwProgram, RwForward, RwVirtual, RwFunction, RwAbsolute,
        RwExternal, RwProcedure, RwInterface, RwInterrupt, RwDestructor,
        RwConstructor, RwImplementation, RwX
);
    FirstClass = (
        NewClause, Continue, AlComm, ContAlComm, UnComm, ContUnComm, StmtLabel,
        BackIndent
);
    SymbolType = (
        SemiColon, SyBegin, SyEnd, SyIf, SyDo, SyOf, SyThen, SyElse, SyGoto, SyCase,
        SyUntil, SyRepeat, SyRecord, ForWhileWith, ProgProcFunc, Declarator, SyVar,
        OtherWord, OtherSym, LeftParen, RightParen, Period, LeftBracket, RightBracket,
        SySubrange, IntConst, Colon, Comma, Ident, Comment, Equal, SyEOf, CommentEnd,
        ProcDirect, Hash, Dollar, Caret, SyUnit, SyInterface, SyImplementation,
        SyInline, SyObject, SyVirtual
);
    WordType = record
        WhenFirst      : FirstClass;
        PuncFollows    : Boolean;
        BlankLnCount   : Natural;
        Spaces         : Integer;
        Base           : - 9..MaxInLen;
        Size           : InRange
    end;

    InsertType = SemiColon..SyEnd;
    SymbolSet  = Set of SymbolType;

var
    D: DirStr;
    N: NameStr;
    E: ExtStr;

    H, M, S, Hund, HB, MB, SB, HundB, Errors, Lines, TotalLn: Word;

    InFile, OutFile, TempF: Text;
    InF, OutF, FileName: String;

    X, Y, X1, Y1: Byte;

    Tab_Width, InitMargin, MaxOutLen, CommThresh, AlCommBase, Indent,
    ContIndent, EndSpaces, CommIndent, Num_Builtins, Loop, FileCount,
    Params, Nfiles: Integer;

    Use_Builtins, Rw_Case, Nl_On_End, Proc_Indent, Sep_ParamList, Indent_of,
    Wrap_Struct, Snuggle_Ifs, Sunggle_Dos, Param_List, Last_Was_Space, Is_Rw,
    Is_Builtin, EndComment, In_Directive, LnPending, NoFiles, CommandBuiltins,
    DefaultPB, DefaultBuiltins, StripSpaces: Boolean;

    InLin: record
        EndOfFile  : Boolean;
        Ch         : Char;
        Index      : InRange;
        Len        : Natural;
        Buf        : Array [- 8..256] of Char
    end;

    OutLine: record
        BlankLns   : Natural;
        Len        : OutRange;
        Buf        : Array [1..256] of Char
    end;

    Cur_Word: WordType;
    Margin: OutRange;
    Symbol, Prev_Sym: SymbolType;
    HearderSyms, StrucSyms, StmtBeginSyms, StmtEndSyms,
    StopSyms, RecendSyms, DataWords, NoLeadSpace: SymbolSet;
    NewWord: Array [InsertType] of WordType;
    InString: packed Array [1..9] of Char;
    FirstRw: Array [1..MaxRwLen] of ResWord;
    RwWord: Array [RwIf..RwImplementation] of RwStringType;
    RwSy: Array [RwIf..RwImplementation] of SymbolType;
    Builtins: Array [1..MaxBuiltins] of RwStringType;
    CharClass: Array [Char] of CharType;
    SymbolClass: Array [CharType] of SymbolType;

function ValidCommand(Command: Char): Boolean;
begin
    ValidCommand := UpCase(Command) in ['A','B','C','E','F','I','L','M',
                                        'N','O','P','R','S','T','W','Z'];
end { ValidCommand };

procedure Beep;
begin
    Sound(800); Delay(50); NoSound;
end { Beep };

function StriCmp(var S1, S2: RwStringType): Integer;
var
    Caps1, Caps2: String;
    I: Integer;
begin
    Caps1 := S1;
    for I := 1 to Length(Caps1) do Caps1[I] := UpCase(Caps1[I]);
    Caps2 := S2;
    for I := 1 to Length(Caps2) do Caps2[I] := UpCase(Caps2[I]);
    if Caps1 < Caps2 then StriCmp := - 1
    else if Caps1 = Caps2 then StriCmp := 0
    else StriCmp := 1;
end { StriCmp };

procedure MakeBuiltins(Name: String);
var
    Start, I, J: Integer;
    DataFile: Text;
    Out_Form, Str: String;
    Temp: RwStringType;
begin
    Str := Name;
    if not Use_Builtins then Exit;
    if Pos('.', Name) = 0 then Name := Name + '.PB';
    if not Exist(Name) then Name := FSearch(Name, GetEnv('PATH'));
    if Name = '' then begin
        Beep; WriteLn('Sorry, I can''t find the identifier file: ', Upper(Str),'  <--- I Quit.');
        Halt(1);  { Important, don't just exit, terminate the program }
    end;
    Assign(DataFile, Name);
    Reset(DataFile);
    Start := Num_Builtins;
    while not EOF(DataFile) and (Num_Builtins < MaxBuiltins) do begin
        ReadLn(DataFile, Out_Form);
        Inc(Num_Builtins);
        Builtins[Num_Builtins] := Out_Form;
    end;
    if Num_Builtins >= MaxBuiltins then begin
        Beep; WriteLn('Builtin array too small!');
        Halt(1);
    end;
    Close(DataFile);
    if Name <> 'BUILTINS.PB' then
        for I := Start + 1 to Num_Builtins do begin
            J := I;
            Temp := Builtins[I];
            while StriCmp(Builtins[J - 1], Temp) > 0 do begin
                Builtins[J] := Builtins[J - 1];
                Dec(J);
            end;
            Builtins[J] := Temp;
        end;
end { MakeBuiltins };

procedure StrucConsts;
var
    I: OrdMinChar..OrdMaxChar;
    Ch: Char;

procedure BuildInsert(Symbol: InsertType; InClass: FirstClass;
                      InPuncFollows: Boolean; InSpaces, InBase: Integer;
                      InSize: InRange);
begin
    with NewWord[Symbol] do begin
        WhenFirst := InClass;
        PuncFollows := InPuncFollows;
        BlankLnCount := 0;
        Spaces := InSpaces;
        Base := InBase;
        Size := InSize
    end;
end { BuildInsert };

procedure BuildRw(Rw: ResWord; SymWord: String; Symbol: SymbolType);
var
    I: Integer;
begin
    RwWord[Rw][0] := Chr(Length(SymWord));
    for I := 1 to Length(SymWord) do RwWord[Rw][I] := SymWord[I];
    RwSy[Rw] := Symbol;
end { BuildRw };

begin
    HearderSyms := [ProgProcFunc, Declarator, SyVar, SyBegin, SyEof, SyInterface, SyImplementation];
    StrucSyms := [SyCase, SyRepeat, SyIf, ForWhileWith];
    StmtBeginSyms := StrucSyms + [SyBegin, Ident, SyGoto, SyInline];
    StmtEndSyms := [SemiColon, SyEnd, SyUntil, SyElse, SyEof];
    StopSyms := HearderSyms + StrucSyms + StmtEndSyms + [SyGoto];
    RecendSyms := [RightParen, SyEnd, SyEof, ProgProcFunc];
    DataWords := [OtherWord, IntConst, Ident, SyEnd];
    NoLeadSpace := [SemiColon, RightParen, Period, Comment, Caret, RightBracket, SySubrange, Colon, Comma];
    BuildInsert(SemiColon, Continue, False, 0, - 9, 1);
    BuildInsert(SyBegin, Continue, False, 1, - 8, 5);
    if Nl_On_End then BuildInsert(SyEnd, BackIndent, True, EndSpaces, - 3, 3)
    else BuildInsert(SyEnd, NewClause, True, EndSpaces, - 3, 3);
    InString := ';beginend';
    FirstRw[1] := RwIf;
    FirstRw[2] := RwIf;
    BuildRw(RwIf, 'IF', SyIf);
    BuildRw(RwDo, 'DO', SyDo);
    BuildRw(RwOf, 'OF', SyOf);
    BuildRw(RwTo, 'TO', OtherSym);
    BuildRw(RwIn, 'IN', OtherSym);
    BuildRw(RwOr, 'OR', OtherSym);
    FirstRw[3] := RwEnd;
    BuildRw(RwEnd, 'END', SyEnd);
    BuildRw(RwFor, 'FOR', ForWhileWith);
    BuildRw(RwVar, 'VAR', SyVar);
    BuildRw(RwDiv, 'DIV', OtherSym);
    BuildRw(RwMod, 'MOD', OtherSym);
    BuildRw(RwSet, 'SET', OtherSym);
    BuildRw(RwAnd, 'AND', OtherSym);
    BuildRw(RwNot, 'NOT', OtherSym);
    BuildRw(RwShl, 'SHL', OtherSym);
    BuildRw(RwShr, 'SHR', OtherSym);
    BuildRw(RwXor, 'XOR', OtherSym);
    BuildRw(RwNil, 'NIL', OtherWord);
    FirstRw[4] := RwThen;
    BuildRw(RwThen, 'THEN', SyThen);
    BuildRw(RwElse, 'ELSE', SyElse);
    BuildRw(RwWith, 'WITH', ForWhileWith);
    BuildRw(RwGoto, 'GOTO', SyGoto);
    BuildRw(RwCase, 'CASE', SyCase);
    BuildRw(RwType, 'TYPE', Declarator);
    BuildRw(RwUnit, 'UNIT', SyUnit);
    BuildRw(RwUses, 'USES', Declarator);
    BuildRw(RwFile, 'FILE', OtherSym);
    FirstRw[5] := RwBegin;
    BuildRw(RwBegin, 'BEGIN', SyBegin);
    BuildRw(RwUntil, 'UNTIL', SyUntil);
    BuildRw(RwWhile, 'WHILE', ForWhileWith);
    BuildRw(RwArray, 'ARRAY', OtherSym);
    BuildRw(RwConst, 'CONST', Declarator);
    BuildRw(RwLabel, 'LABEL', Declarator);
    FirstRw[6] := RwRepeat;
    BuildRw(RwRepeat, 'REPEAT', SyRepeat);
    BuildRw(RwRecord, 'RECORD', SyRecord);
    BuildRw(RwDownto, 'DOWNTO', OtherSym);
    BuildRw(RwPacked, 'PACKED', OtherSym);
    BuildRw(RwInline, 'INLINE', SyInline);
    BuildRw(RwString, 'STRING', OtherSym);
    BuildRw(RwObject, 'OBJECT', SyObject);
    FirstRw[7] := RwProgram;
    BuildRw(RwProgram, 'PROGRAM', ProgProcFunc);
    BuildRw(RwForward, 'FORWARD', ProcDirect);
    BuildRw(RwVirtual, 'VIRTUAL', SyVirtual);
    FirstRw[8] := RwFunction;
    BuildRw(RwFunction, 'FUNCTION', ProgProcFunc);
    BuildRw(RwAbsolute, 'ABSOLUTE', OtherSym);
    BuildRw(RwExternal, 'EXTERNAL', ProcDirect);
    FirstRw[9] := RwProcedure;
    BuildRw(RwProcedure, 'PROCEDURE', ProgProcFunc);
    BuildRw(RwInterface, 'INTERFACE', SyInterface);
    BuildRw(RwInterrupt, 'INTERRUPT', ProcDirect);
    FirstRw[10] := RwDestructor;
    BuildRw(RwDestructor, 'DESTRUCTOR', ProgProcFunc);
    FirstRw[11] := RwConstructor;
    BuildRw(RwConstructor, 'CONSTRUCTOR', ProgProcFunc);
    FirstRw[12] := RwImplementation;
    FirstRw[13] := RwImplementation;
    FirstRw[14] := RwImplementation;
    BuildRw(RwImplementation, 'IMPLEMENTATION', SyImplementation);
    FirstRw[15] := RwX;
    for I := OrdMinChar to OrdMaxChar do CharClass[Chr(I)] := Illegal;
    for Ch := 'a' to 'z' do begin
        CharClass[Ch] := Letter;
        CharClass[UpCase(Ch)] := Letter
    end;
    for Ch := '0' to '9' do CharClass[Ch] := Digit;
    CharClass['#'] := ChHash;
    CharClass['_'] := Letter;
    CharClass[' '] := Special;
    CharClass['$'] := ChDollar;
    CharClass[''''] := ChaPostrophe;
    CharClass['('] := ChLeftParen;
    CharClass[')'] := ChRightParen;
    CharClass['['] := ChLeftBracket;
    CharClass[']'] := ChRightBracket;
    CharClass['*'] := Special;
    CharClass['+'] := Special;
    CharClass['-'] := Special;
    CharClass['.'] := ChPeriod;
    CharClass['/'] := Special;
    CharClass[':'] := ChColon;
    CharClass[','] := ChComma;
    CharClass[';'] := ChSemicolon;
    CharClass['<'] := ChLessThan;
    CharClass['='] := ChEqual;
    CharClass['>'] := ChGreaterThan;
    CharClass['@'] := Special;
    CharClass['^'] := ChCaret;
    CharClass['{'] := ChLeftBrace;
    CharClass['}'] := ChRightBrace;
    SymbolClass[Illegal] := OtherSym;
    SymbolClass[Special] := OtherSym;
    SymbolClass[ChCaret] := Caret;
    SymbolClass[ChDollar] := Dollar;
    SymbolClass[ChHash] := Hash;
    SymbolClass[ChaPostrophe] := OtherWord;
    SymbolClass[ChLeftParen] := LeftParen;
    SymbolClass[ChRightParen] := RightParen;
    SymbolClass[ChLeftBracket] := LeftBracket;
    SymbolClass[ChRightBracket] := RightBracket;
    SymbolClass[ChPeriod] := Period;
    SymbolClass[Digit] := IntConst;
    SymbolClass[ChColon] := Colon;
    SymbolClass[ChComma] := Comma;
    SymbolClass[ChSemicolon] := SemiColon;
    SymbolClass[ChLessThan] := OtherSym;
    SymbolClass[ChGreaterThan] := OtherSym;
    SymbolClass[Letter] := Ident;
    SymbolClass[ChLeftBrace] := Comment;
    SymbolClass[ChRightBrace] := CommentEnd;
    SymbolClass[ChEqual] := Equal;
end { StrucConsts };

procedure WriteLine;
var
    I: OutRange;
begin
    with OutLine do begin
        while BlankLns > 0 do begin
            WriteLn(OutFile);
            BlankLns := BlankLns - 1
        end;
        if Len > 0 then begin
            for I := 1 to Len do Write(OutFile, Buf[I]);
            WriteLn(OutFile);
            Len := 0
        end
    end;
end { WriteLine };

procedure WriteError(Error: ErrorType);
var
    I, IX: InRange;
begin
    Beep;
    WriteAT(X+7, Y, LightRed + Blink, Black, ' ERROR !!! ');
    WriteLine;
    Write(OutFile, ' (*  !!! ERROR, ');
    case Error of
        LongLine: Write(OutFile, 'shorter line');
        NoEndComm: Write(OutFile, 'end of Comment');
        NotQuote: Write(OutFile, 'final "''" on line');
        LongWord: Write(OutFile, 'shorter word');
        NotDo: Write(OutFile, '"do"');
        NotOf: Write(OutFile, '"of"');
        NotEnd: Write(OutFile, '"end"');
        NotThen: Write(OutFile, '"then"');
        NotBegin: Write(OutFile, '"begin"');
        NotUntil: Write(OutFile, '"until"');
        NotSemicolon: Write(OutFile, '";"');
        NotColon: Write(OutFile, '":"');
        NotParen: Write(OutFile, '")"');
        NoEof: Write(OutFile, 'end of file');
        NotInterface: Write(OutFile, 'interface');
        NotImplementation: Write(OutFile, 'implementation');
    end;
    Write(OutFile, ' expected');
    if Error >= LongWord then begin
        Write(OutFile, ', not "');
        with InLin, Cur_Word do begin
            if Size > MaxRwLen then IX := MaxRwLen
            else IX := Size;
            for I := 1 to IX do Write(OutFile, Buf[Base + I])
        end;
        Write(OutFile, '"')
    end;
    if Error = NoEof then begin
        Beep;
        WriteAT(X+7, Y, LightRed + Blink, Black, ' ERROR !!! ');
        Write(OutFile, ', FORMATTING STOPS');
    end;
    WriteLn(OutFile, ' !!!  *)');
    Inc(Errors);
end { WriteError };

procedure ReadLine;
var
    C: Char;
    NonBlank: Boolean;
begin
    with InLin do begin
        Len := 0;
        if EOF(InFile) then EndOfFile := True
        else begin
            while not EOLN(InFile) do begin
                Read(InFile, C);
                if (C < ' ') then begin
                    if (C = #9) and (not EndComment) then
                        while (Len + 1) Mod Tab_Width > 0 do begin
                            Len := Len + 1;
                            if Len < MaxInLen then Buf[Len] := ' ';
                        end;
                    if not ((C = #12) and (Len > 0)) then C := ' ';
                end;
                Len := Len + 1;
                if Len < MaxInLen then Buf[Len] := C
            end;
            ReadLn(InFile);
            if Len >= MaxInLen then begin
                WriteError(LongLine);
                Len := MaxInLen - 1
            end;
            NonBlank := False;
            repeat
                if Len = 0 then NonBlank := True
                else if Buf[Len] <> ' ' then NonBlank := True
                else Len := Len - 1
            until NonBlank
        end;
        Len := Len + 1;
        Buf[Len] := ' ';
        Index := 0
    end;
    Inc(TotalLn); Inc(Lines);
    WriteAT(X+1, Y, LightGray, Black, '(' + IntToStr(Lines) + ')');
end { ReadLine };

procedure GetChar;
begin
    with InLin do begin
        Index := Index + 1;
        Ch := Buf[Index]
    end;
end { GetChar };

function NextChar: Char;
begin
    with InLin do NextChar := Buf[Index + 1];
end { NextChar };

procedure StartWord(StartClass: FirstClass);
var
    First: Boolean;
begin
    First := False;
    with InLin, Cur_Word do begin
        WhenFirst := StartClass;
        BlankLnCount := 0;
        while (Index >= Len) and not EndOfFile do begin
            if Len = 1 then BlankLnCount := BlankLnCount + 1;
            if StartClass = ContUnComm then WriteLine
            else First := EndComment;
            ReadLine;
            GetChar;
            if Ch = Chr(12) then begin
                WriteLine;
                WriteLn(OutFile, Chr(12));
                BlankLnCount := 0;
                GetChar
            end;
        end;
        Spaces := 0;
        if not EndOfFile then begin
            while Ch = ' ' do begin
                GetChar;
                if not EndComment then Spaces := Spaces + 1;
            end
        end;
        if First then Spaces := 1;
        Base := Index - 1
    end;
end { StartWord };

procedure FinishWord;
begin
    with InLin, Cur_Word do begin
        PuncFollows := (Symbol in DataWords) and (Ch <> ' ');
        Size := Index - Base - 1
    end;
end { FinishWord };

procedure CopyWord(NewLine: Boolean; Cur_Word: WordType);
var
    I: Integer;
begin
    with Cur_Word, OutLine do begin
        I := MaxOutLen - Len - Spaces - Size;
        if NewLine or (I < 0) or ((I = 0) and PuncFollows) or ((Symbol = SyEnd) and Nl_On_End) then WriteLine;
        if Len = 0 then begin
            BlankLns := BlankLnCount;
            case WhenFirst of
                NewClause: Spaces := Margin;
                Continue: Spaces := Margin + ContIndent;
                AlComm: Spaces := AlCommBase;
                ContAlComm: Spaces := AlCommBase + CommIndent;
                UnComm:;
                ContUnComm:;
                StmtLabel: Spaces := InitMargin;
                BackIndent: Spaces := Margin - Indent
            end;
            if Spaces + Size > MaxOutLen then begin
                Spaces := MaxOutLen - Size;
                if Spaces < 0 then begin
                    WriteError(LongWord);
                    Size := MaxOutLen;
                    Spaces := 0
                end
            end
        end else if EndComment then begin
            if (Prev_Sym = Equal) or (Sep_ParamList and (Symbol = LeftParen) and (Prev_Sym in [SyInline, Ident])) or not (
                    Symbol = SyEnd) and not (Symbol in [Comment, CommentEnd]) and not Last_Was_Space and not (Symbol in
                    NoLeadSpace) and not (Prev_Sym in [Period, Caret]) and not ((Symbol in [LeftParen, LeftBracket]) and (
                    Prev_Sym in [SyInline, Ident, RightBracket])) and not ((Symbol = IntConst) and (Prev_Sym = Hash)) and not
                    (Prev_Sym in [LeftParen, LeftBracket, SySubrange]) then
                Spaces := 1
        end;
        for I := 1 to Spaces do begin
            Len := Len + 1;
            Buf[Len] := ' '
        end;
        for I := 1 to Size do begin
            Len := Len + 1;
            Buf[Len] := InLin.Buf[Base + I]
        end;
        Last_Was_Space := Buf[Len] = ' ';
    end;
    Prev_Sym := Symbol;
end { CopyWord };

procedure DoComment;

procedure CopyComment(commclass: FirstClass; commbase: InRange);
begin
    EndComment := False;
    with Cur_Word do begin
        WhenFirst := commclass;
        Spaces := commbase - OutLine.Len;
        CopyWord((Spaces < 0) or (BlankLnCount > 0), Cur_Word)
    end;
    commclass := Succ(commclass);
    Symbol := OtherSym;
    if InLin.Ch = '$' then begin
        StartWord(commclass);
        GetChar;
        FinishWord;
        In_Directive := True;
        CopyWord(False, Cur_Word);
    end;
    with InLin do begin
        repeat
            StartWord(commclass);
            EndComment := EndOfFile;
            if EndComment then WriteError(NoEndComm)
            else begin
                repeat
                    if Ch = '*' then begin
                        GetChar;
                        if Ch = ')' then begin
                            Dec(Index);
                            FinishWord;
                            CopyWord(False, Cur_Word);
                            StartWord(commclass);
                            GetChar;
                            EndComment := True;
                            In_Directive := False;
                            Symbol := CommentEnd;
                            GetChar
                        end
                    end else if Ch = '}' then begin
                        FinishWord;
                        CopyWord(False, Cur_Word);
                        StartWord(commclass);
                        EndComment := True;
                        In_Directive := False;
                        Symbol := CommentEnd;
                        GetChar
                    end else GetChar
                until (Ch = ' ') or EndComment
            end;
            FinishWord;
            CopyWord(False, Cur_Word);
        until EndComment
    end;
end { CopyComment };

begin
    if Cur_Word.Base < CommThresh then CopyComment(UnComm, Cur_Word.Base)
    else CopyComment(AlComm, AlCommBase)
end { DoComment };

procedure CopySymbol(Symbol: SymbolType; Cur_Word: WordType);
begin
    if Symbol = Comment then begin
        DoComment;
        LnPending := True
    end else if Symbol = SemiColon then begin
        CopyWord(False, Cur_Word);
        LnPending := not Param_List;
    end else begin
        CopyWord(LnPending, Cur_Word);
        LnPending := False
    end;
end { CopySymbol };

procedure Insert(NewSymbol: InsertType);
var
    Old_Sym: SymbolType;
begin
    Old_Sym := Symbol;
    Symbol := NewSymbol;
    CopySymbol(NewSymbol, NewWord[NewSymbol]);
    Symbol := Old_Sym;
end { Insert };

procedure GetSymbol;

procedure FindSymbol;
var
    ChClass: CharType;

procedure CheckResWord;
var
    Rw, RwBeyond: ResWord;
    All_Caps, SymWord: RwStringType;
    I, B, U, L: Integer;
begin
    with Cur_Word, InLin do begin
        Size := Index - Base - 1;
        if Size < MaxRwLen then begin
            SymWord[0] := Chr(Size);
            for I := 1 to Size do SymWord[I] := UpCase(Buf[Base + I]);
            Rw := FirstRw[Size];
            RwBeyond := FirstRw[Size + 1];
            Symbol := SemiColon;
            repeat
                if Rw >= RwBeyond then Symbol := Ident
                else if SymWord = RwWord[Rw] then Symbol := RwSy[Rw]
                else Rw := Succ(Rw)
            until Symbol <> SemiColon;
            Is_Rw := Symbol <> Ident;
            if Is_Rw then for I := 1 to Size do begin
                if not Rw_Case then SymWord[I] := Chr(Ord(SymWord[I]) + 32);
                Buf[Base + I] := SymWord[I];
            end;
            if Use_Builtins and not Is_Rw then begin
                L := 1;
                U := Num_Builtins;
                Is_Builtin := False;
                while (L <= U) and not Is_Builtin do begin
                    B := (L + U) DIV 2;
                    All_Caps := Builtins[B];
                    for I := 1 to Length(All_Caps) do All_Caps[I] := UpCase(All_Caps[I]);
                    if SymWord < All_Caps then U := B - 1
                    else if SymWord > All_Caps then L := B + 1
                    else Is_Builtin := True;
                end;
                if Is_Builtin then for I := 1 to Size do Buf[Base + I] := Builtins[B][I];
            end;
            if Symbol in [SyEnd, SyUntil] then begin
                if Spaces < EndSpaces then Spaces := EndSpaces;
                if Nl_On_End then WhenFirst := BackIndent
                else WhenFirst := NewClause;
            end
        end
    end;
end { CheckResWord };

procedure GetName;
begin
    while (CharClass[InLin.Ch] in [Letter, Digit]) or
          (CharClass[InLin.Ch] = ChPeriod) and
          (CharClass[NextChar] in [Letter, Digit]) do GetChar;
    CheckResWord;
end { GetName };

procedure GetNumber;
begin
    with InLin do begin
        while CharClass[Ch] = Digit do GetChar;
        if Ch = '.' then begin
            if CharClass[NextChar] = Digit then begin
                Symbol := OtherWord;
                GetChar;
                while CharClass[Ch] = Digit do GetChar
            end
        end;
        if UpCase(Ch) = 'E' then begin
            Symbol := OtherWord;
            GetChar;
            if (Ch = '+') or (Ch = '-') then GetChar;
            while CharClass[Ch] = Digit do GetChar
        end
    end;
end { GetNumber };

procedure GetHex;
begin
    with InLin do while (Ch in HexDigits) do GetChar;
end { GetHex };

procedure GetStringLiteral;
var
    EndString: Boolean;
begin
    with InLin do begin
        EndString := False;
        repeat
            if Ch = '''' then begin
                GetChar;
                if Ch = '''' then GetChar
                else EndString := True
            end else if Index >= Len then begin
                WriteError(NotQuote);
                Symbol := SyEof;
                EndString := True
            end else GetChar
        until EndString
    end;
end { GetStringLiteral };

begin
    StartWord(Continue);
    with InLin do begin
        if EndOfFile then Symbol := SyEof
        else begin
            ChClass := CharClass[Ch];
            Symbol := SymbolClass[ChClass];
            GetChar;
            case ChClass of
                ChSemicolon, ChRightParen, ChLeftBrace, Special, Illegal:;
                Letter: GetName;
                Digit: GetNumber;
                ChDollar: GetHex;
                ChaPostrophe: GetStringLiteral;
                ChComma: Symbol := Comma;
                ChColon: begin
                    if Ch = '=' then begin
                        Symbol := OtherSym;
                        GetChar
                    end
                end;
                ChLessThan: begin
                    if (Ch = '=') or (Ch = '>') then GetChar
                end;
                ChGreaterThan: begin
                    if Ch = '=' then GetChar
                end;
                ChLeftParen: begin
                    if Ch = '*' then begin
                        Symbol := Comment;
                        GetChar
                    end
                end;
                ChLeftBracket: Symbol := LeftBracket;
                ChRightBracket: Symbol := RightBracket;
                ChEqual: Symbol := Equal;
                ChPeriod: begin
                    if Ch = '.' then begin
                        Symbol := SySubrange;
                        GetChar
                    end
                end
            end
        end
    end;
    FinishWord;
end { FindSymbol };

begin
    repeat
        CopySymbol(Symbol, Cur_Word);
        FindSymbol;
        if LnPending and (Symbol = SyVirtual) then LnPending := False;
    until Symbol <> Comment;
end { GetSymbol };

procedure StartClause;
begin
    if (Symbol in [SyEnd, SyUntil]) and Nl_On_End then Cur_Word.WhenFirst := BackIndent
    else Cur_Word.WhenFirst := NewClause;
    LnPending := True;
end { StartClause };

procedure PassSemicolons;
begin
    while Symbol = SemiColon do begin
        GetSymbol;
        StartClause
    end;
end { PassSemicolons };

procedure StartPart(Need_Blank: Boolean);
begin
    with Cur_Word do begin
        if (BlankLnCount = 0) and Need_Blank then BlankLnCount := 1
    end;
    StartClause;
end { StartPart };

procedure StartBody(NeedIndent: Boolean);
begin
    PassSemicolons;
    if NeedIndent then Margin := Margin + Indent;
    StartClause;
end { StartBody };

procedure FinishBody(NeedIndent: Boolean);
begin
    if NeedIndent then Margin := Margin - Indent;
end { FinishBody };

procedure PassPhrase(FinalSymbols: SymbolSet);
var
    EndSyms: SymbolSet;
begin
    if Symbol <> SyEof then begin
        EndSyms := StopSyms + FinalSymbols;
        repeat
            GetSymbol
        until Symbol in EndSyms
    end;
end { PassPhrase };

procedure Expect(ExpectedSym: SymbolType; Error: ErrorType; Syms: SymbolSet);
begin
    if Symbol = ExpectedSym then GetSymbol
    else begin
        WriteError(Error);
        while not (Symbol in [ExpectedSym] + Syms) do GetSymbol;
        if Symbol = ExpectedSym then GetSymbol
    end;
end { Expect };

procedure DoLabel;
var
    NextFirst: FirstClass;
begin
    with Cur_Word do begin
        NextFirst := WhenFirst;
        WhenFirst := StmtLabel;
        LnPending := True;
        GetSymbol;
        Expect(Colon, NotColon, StopSyms);
        WhenFirst := NextFirst;
        LnPending := True
    end;
end { DoLabel };

procedure Block(NeedBody: Boolean);

procedure Heading;

procedure MatchParens;
begin
    GetSymbol;
    while not (Symbol in RecendSyms) do begin
        if Symbol = LeftParen then MatchParens
        else GetSymbol
    end;
    Expect(RightParen, NotParen, StopSyms + RecendSyms);
end { MatchParens };

begin
    GetSymbol;
    PassPhrase([LeftParen]);
    if Symbol = LeftParen then begin
        Param_List := True;
        MatchParens;
        Param_List := False;
    end;
    if Symbol = Colon then PassPhrase([SemiColon]);
    Expect(SemiColon, NotSemicolon, StopSyms);
end { Heading };

procedure Statement; Forward;

procedure StmtList;
begin
    repeat
        Statement;
        PassSemicolons
    until Symbol in StmtEndSyms;
end { StmtList };

procedure CompoundStmt(StmtPart: Boolean);
begin
    GetSymbol;
    StartBody(True);
    StmtList;
    Expect(SyEnd, NotEnd, StmtEndSyms);
    FinishBody(True);
end { CompoundStmt };

procedure Statement;

procedure CheckCompound;
begin
    if Symbol = IntConst then DoLabel;
    if (Symbol in StrucSyms) and Wrap_Struct then begin
        Insert(SyBegin);
        StartBody(True);
        Statement;
        Insert(SyEnd);
        FinishBody(True);
    end else Statement;
end { CheckCompound };

procedure IfStmt;
var
    New_Body: Boolean;
begin
    PassPhrase([SyThen]);
    Expect(SyThen, NotThen, StopSyms);
    New_Body := not (Snuggle_Ifs or ((Symbol in StrucSyms) and Wrap_Struct) or (Symbol = SyBegin));
    if New_Body then StartBody(True);
    if Symbol <> SyBegin then CheckCompound
    else Statement;
    if New_Body then FinishBody(True);
    if Symbol = SyElse then begin
        if (Prev_Sym = SyEnd) and not Nl_On_End then begin
            FinishBody(True);
            StartClause;
        end else if (Prev_Sym <> SyEnd) then StartClause;
        GetSymbol;
        if Symbol = SyIf then IfStmt
        else begin
            New_Body := not (Snuggle_Ifs or ((Symbol in StrucSyms) and Wrap_Struct) or (Symbol = SyBegin));
            if New_Body then StartBody(True);
            if Symbol <> SyBegin then CheckCompound
            else Statement;
            if New_Body then FinishBody(True);
        end;
    end;
end { IfStmt };

procedure RepeatStmt;
begin
    GetSymbol;
    StartBody(True);
    StmtList;
    StartClause;
    Expect(SyUntil, NotUntil, StmtEndSyms);
    FinishBody(True);
    PassPhrase([SemiColon]);
end { RepeatStmt };

procedure FwwStmt;
var
    New_Body: Boolean;
begin
    PassPhrase([SyDo]);
    Expect(SyDo, NotDo, StopSyms);
    New_Body := not (Sunggle_Dos or ((Symbol in StrucSyms) and Wrap_Struct) or (Symbol = SyBegin));
    if New_Body then StartBody(True);
    if Symbol <> SyBegin then CheckCompound
    else Statement;
    if New_Body then FinishBody(True);
end { FwwStmt };

procedure CaseStmt;
begin
    PassPhrase([SyOf]);
    Expect(SyOf, NotOf, StopSyms);
    StartBody(True);
    repeat
        if Symbol = SyElse then GetSymbol
        else begin
            PassPhrase([Colon]);
            Expect(Colon, NotColon, StopSyms);
        end;
        if Indent_of then StartBody(True);
        if Symbol <> SyBegin then CheckCompound
        else Statement;
        if Indent_of then FinishBody(True);
        PassSemicolons
    until Symbol in StopSyms - [SyElse];
    Expect(SyEnd, NotEnd, StmtEndSyms);
    FinishBody(True);
end { CaseStmt };

begin
    if Symbol = IntConst then DoLabel;
    if Symbol in StmtBeginSyms then begin
        case Symbol of
            SyBegin: CompoundStmt(False);
            SyCase: CaseStmt;
            SyIf: IfStmt;
            SyRepeat: RepeatStmt;
            ForWhileWith: FwwStmt;
            SyInline, Ident, SyGoto: PassPhrase([SemiColon])
        end
    end;
    if not (Symbol in StmtEndSyms) then begin
        WriteError(NotSemicolon);
        PassPhrase([SemiColon])
    end;
end { Statement };

procedure PassFields(ForVariant: Boolean); Forward;

procedure DoRecord;
begin
    GetSymbol;
    StartBody(True);
    PassFields(False);
    Expect(SyEnd, NotEnd, RecendSyms);
    FinishBody(True);
end { DoRecord };

procedure DoObject;
begin
    GetSymbol;
    if Symbol = LeftParen then begin
        PassPhrase([RightParen]);
        GetSymbol;
    end;
    StartBody(True);
    PassFields(False);
    Block(False);
    Expect(SyEnd, NotEnd, RecendSyms);
    FinishBody(True);
end { DoObject };

procedure DoVariant;
begin
    PassPhrase([SyOf]);
    Expect(SyOf, NotOf, StopSyms);
    StartBody(True);
    PassFields(True);
    FinishBody(True);
end { DoVariant };

procedure DoParens(ForVariant: Boolean);
begin
    GetSymbol;
    if ForVariant then StartBody(True);
    PassFields(False);
    LnPending := False;
    Expect(RightParen, NotParen, RecendSyms);
    if ForVariant then FinishBody(True);
end { DoParens };

procedure PassFields;
begin
    while not (Symbol in RecendSyms) do begin
        if Symbol = SemiColon then PassSemicolons
        else if Symbol = SyRecord then DoRecord
        else if Symbol = SyCase then DoVariant
        else if Symbol = LeftParen then DoParens(ForVariant)
        else GetSymbol
    end;
end { PassFields };

procedure Do_Decl;
begin
    StartPart(True);
    GetSymbol;
    StartBody(True);
    repeat
        PassPhrase([SyObject, SyRecord]);
        if Symbol = SyRecord then DoRecord;
        if Symbol = SyObject then DoObject;
        if Symbol = SemiColon then PassSemicolons
    until Symbol in HearderSyms;
    FinishBody(True)
end { Do_Decl };

begin
    while Symbol in [Declarator, SyVar] do Do_Decl;
    while Symbol in [ProgProcFunc, SyVar] do begin
        if Symbol = SyVar then Do_Decl
        else begin
            StartPart(NeedBody);
            Heading;
            if Symbol = SyVirtual then begin
                PassPhrase([SemiColon]);
                PassSemicolons;
            end;
            if NeedBody then begin
                StartBody(Proc_Indent);
                if Symbol in HearderSyms then Block(True)
                else if Symbol in [ProcDirect, SyInline] then begin
                    StartPart(False);
                    PassPhrase([SemiColon]);
                    PassSemicolons
                end else WriteError(NotBegin);
                FinishBody(Proc_Indent);
            end;
        end;
    end;
    if Symbol = SyUnit then begin
        StartPart(True);
        Heading;
        StartBody(False);
        if Symbol <> SyInterface then WriteError(NotInterface)
        else begin
            StartPart(True);
            GetSymbol;
        end;
        Block(False);
        if Symbol <> SyImplementation then WriteError(NotImplementation)
        else begin
            StartPart(True);
            GetSymbol;
        end;
        Block(True);
        if Symbol <> SyEof then PassPhrase([Period]);
        GetSymbol;
    end;
    if Symbol = SyBegin then begin
        StartPart(True);
        CompoundStmt(True);
        if Symbol in [SySubrange, Period] then Symbol := SemiColon;
        PassSemicolons
    end;
end { Block };

procedure CopyRem;
begin
    WriteError(NoEof);
    with InLin do begin
        repeat
            CopyWord(False, Cur_Word);
            StartWord(ContUnComm);
            if not EndOfFile then begin
                repeat
                    GetChar
                until Ch = ' '
            end;
            FinishWord;
        until EndOfFile
    end;
end { CopyRem };

procedure Initialize;
var
    I: 1..9;
begin
    with InLin do begin
        for I := 1 to 9 do Buf[I - 9] := InString[I];
        EndOfFile := False;
        Ch := ' ';
        Index := 0;
        Len := 0
    end;
    with OutLine do begin
        BlankLns := 0;
        Len := 0
    end;
    with Cur_Word do begin
        WhenFirst := ContUnComm;
        PuncFollows := False;
        BlankLnCount := 0;
        Spaces := 0;
        Base := 0;
        Size := 0
    end;
    Margin := InitMargin;
    LnPending := False;
    Symbol := OtherSym;
    Prev_Sym := OtherSym;
    EndComment := True;
    In_Directive := False;
    Param_List := False;
    Errors := 0;
    Lines := 0;
    GetTime(HB, MB, SB, HundB);
    FindCursor(X, Y, X1, Y1);
end { Initialize };

procedure ProcessOption(Str: String);
var
    Temp, VCode: Integer;
begin
    if not ValidCommand(Str[1]) then begin
        Beep; WriteLn('Sorry, ',Str,' is not an option. I''m skipping it.');
        Exit;
    end;
    case UpCase(Str[1]) of
        'A': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then AlCommBase := Temp;
        end;
        'B': Use_Builtins := Str[2] = '+';
        'C': case UpCase(Str[2]) of
            'C': begin
                Val(Copy(Str, 3, 7), Temp, VCode);
                if VCode = 0 then CommIndent := Temp;
            end;
            'I': begin
                Val(Copy(Str, 3, 7), Temp, VCode);
                if VCode = 0 then ContIndent := Temp;
            end;
            'T': begin
                Val(Copy(Str, 3, 7), Temp, VCode);
                if VCode = 0 then CommThresh := Temp;
            end;
        end;
        'E': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then EndSpaces := Temp;
        end;
        'F': Sep_ParamList := Str[2] = '+';
        'I': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then Indent := Temp;
        end;
        'L': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then MaxOutLen := Temp;
        end;
        'M': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then InitMargin := Temp;
        end;
        'N': Nl_On_End := Str[2] = '+';
        'O': Indent_of := Str[2] = '+';
        'P': Proc_Indent := Str[2] = '+';
        'R': Rw_Case := Str[2] = '+';
        'S': case UpCase(Str[2]) of
            'D': Sunggle_Dos := Str[3] = '+';
            'I': Snuggle_Ifs := Str[3] = '+';
            '+': begin
                Sunggle_Dos := True;
                Snuggle_Ifs := True;
            end;
            '-': begin
                Sunggle_Dos := False;
                Snuggle_Ifs := False;
            end;
        end;
        'T': begin
            Val(Copy(Str, 2, 7), Temp, VCode);
            if VCode = 0 then Tab_Width := Temp;
        end;
        'W': Wrap_Struct := Str[2] = '+';
        'Z': StripSpaces := Str[2] = '+';
    end
end { ProcessOption };

procedure ProcessResponseFile(Name: String);
var
    IFile: Text;
    Str: String;
begin
    Str := Name;
    if Pos('.', Name) = 0 then Name := Name + '.PB';
    if not Exist(Name) then Name := FSearch(Name, GetEnv('PATH'));
    if not Exist(Name) then begin
        Beep; WriteLn('Sorry, I can''t find the response file: ',Upper(Str),'  <--- Skipped.');
        Exit;
    end;
    Assign(IFile, Name);
    Reset(IFile);
    while not EOF(IFile) do begin
        ReadLn(IFile, Str);
        if Str[1] = '@' then ProcessResponseFile(Copy(Str, 2, 79))
        else if Str[1] = '#' then begin
            CommandBuiltins := True;
            MakeBuiltins(Copy(Str, 2, 79))
        end else ProcessOption(Copy(Str, 2, 79));
    end;
    Close(IFile);
end { ProcessResponseFile };

procedure SetSwitches;
var
    Str: String;
    I: Integer;
    Def_Path: PathStr;
begin
    TotalLn := 0;
    Params := 0;
    Tab_Width := 5;
    Num_Builtins := 0;
    InitMargin := 0;
    MaxOutLen := 125;
    CommThresh := 6;
    AlCommBase := 50;
    Indent := 2;
    ContIndent := 8;
    EndSpaces := 2;
    CommIndent := 2;
    Use_Builtins := True;
    NoFiles := False;
    Rw_Case := False;
    Nl_On_End := True;
    Proc_Indent := False;
    Sep_ParamList := False;
    Indent_of := False;
    DefaultPB := False;
    DefaultBuiltins := False;
    CommandBuiltins := False;
    Wrap_Struct := False;
    Sunggle_Dos := False;
    Snuggle_Ifs := False;
    StripSpaces := False;
    Def_Path := FSearch('DEFAULT.PB', GetEnv('PATH'));
    if (Def_Path <> '') or Exist('DEFAULT.PB') then begin
        DefaultPB := True;
        ProcessResponseFile(Def_Path);
    end;
    if ParamCount >= 1 then begin
        for I := 1 to ParamCount do begin
            Str := ParamStr(I);
            if Str[1] in ['-', '/'] then begin
                if (Pos('.',Str) = 0) or (not Exist(Str)) then begin
                    Inc(Params);
                    ProcessOption(Copy(Str, 2, 79))
                end;
            end else if (Str[1] = '@') and (not Exist(Str)) then begin
                Inc(Params);
                ProcessResponseFile(Copy(Str, 2, 79))
            end else if (Str[1] = '#') and (Use_Builtins) then begin
                Inc(Params);
                CommandBuiltins := True;
                MakeBuiltins(Copy(Str, 2, 79));
            end;
        end;
    end;
    if Params = ParamCount then NoFiles := True;
    if (Use_Builtins) and (not CommandBuiltins) and (not NoFiles) then begin
        DefaultBuiltins := True;
        MakeBuiltins('BUILTINS.PB');
    end;
end { SetSwitches };

procedure Summary;
begin
    GetTime(H, M, S, Hund);
    if Hund < HundB then begin
        Hund := Hund + 100;
        S := S - 1;
        if S < SB then S := S + 60;
    end else if S < SB then S := S + 60;
    S := S - SB;
    Hund := Hund - HundB;
    WriteLn(' ', Lines, ' lines, ', S, '.', Hund, ' seconds, ', Errors, ' error(s).');
end { Summary };

procedure StripProcFunc;
var
    Loop: Integer;
    Temp, Buff: String;
begin
    Assign(InFile, 'TEMP.$$$');
    Assign(OutFile, 'TEMP.TMP');
    Reset(InFile);
    ReWrite(OutFile);
    while not EOF(InFile) do begin
        ReadLn(InFile, Buff);
        if (Pos('PROCEDURE', Upper(Buff))<>0) or (Pos('FUNCTION', Upper(Buff))<>0) or
           (Pos('CONSTRUCTOR', Upper(Buff))<>0) or (Pos('DESTRUCTOR', Upper(Buff))<>0) then begin
            WriteLn(OutFile, Buff);
            repeat
                ReadLn(InFile, Buff);
                Temp := Buff;
                if Temp = '' then begin
                    ReadLn(InFile, Buff);
                    if (Pos('PROCEDURE',Upper(Buff))<>0) or (Pos('FUNCTION',Upper(Buff))<>0) then
                        WriteLn(OutFile, Temp);
                        WriteLn(OutFile, Buff);
                    end else WriteLn(OutFile, Temp);
            until (Pos('BEGIN', Upper(Buff))<>0) or (Pos('IMPLEMENTATION', Upper(Buff))<>0);
        end else WriteLn(OutFile, Buff);
    end;
    Close(InFile);
    Close(OutFile);
    Reset(InFile);
    Reset(OutFile);
    Erase(InFile);
    Assign(InFile, InF);
    Reset(InFile);
    FSplit(InF,D,N,E);
    if Exist(D + N + '.BAK') then begin
        Assign(TempF, D + N + '.BAK');
        Erase(TempF);
    end;
    ReName(InFile, D + N + '.BAK');
    ReName(OutFile, InF);
    Close(InFile);
    Close(OutFile);
end { StripProcFunc };

procedure DoFormatting(S: String);
var
    Ch: Char;
begin
    InF := S;
    OutF := 'TEMP.$$$';
    Assign(InFile, InF);
    Reset(InFile);
    Assign(OutFile, OutF);
    ReWrite(OutFile);
    Initialize;
    GetSymbol;
    Block(True);
    if not InLin.EndOfFile then CopyRem;
    WriteLine;
    Close(InFile);
    Close(OutFile);
    if not StripSpaces then begin
        Reset(InFile);
        Reset(OutFile);
        FSplit(S,D,N,E);
        if Exist(D + N + '.BAK') then begin
            Assign(TempF, D + N + '.BAK');
            Erase(TempF);
        end;
        ReName(InFile, D + N + '.BAK');
        ReName(OutFile, InF);
        Close(InFile);
        Close(OutFile);
    end else StripProcFunc;
    Summary;
end { DoFormatting };

procedure ProcessFile(S: String);
var
    OK: Boolean;
    Ch: Char;

procedure Approve(Name: String);
begin  { Perhaps the user may still want the old unformatted file }
    OK := True; Beep;
    Write('WARNNING: backup file exists for "',Upper(S),'," proceed with format (y/n) ');
    if Not Yes then OK := False;
    WriteLn;
end { Approve };

begin
    FSplit(S, D, N, E);
    if Exist(D + N + '.BAK') then begin
        Approve(S);
        if Not OK then Exit;
    end;
    Write('Formatting ---> ', Upper(S));
    DoFormatting(S);
end { ProcessFile };

procedure Usage;
begin
    Write('Pascal Beautifier Version 0.5c                 by Toupao Chieng 1/1/1991 12:00pm');
    Write('usage: BEAUTIFY [{/|-}options...] [#identifier_file] [@response_file] [files...]');
    WriteLn;
    WriteLn('[options]');
    Write('/an    comment alignment column. n=50       /b+|-  use builtin indentifiers. b+ ');
    Write('/ccn   indent continued comments. n=2       /cin   indent continued lines. n=8  ');
    Write('/ctn   comment threshing column. n=6        /en    indent space before END. n=2 ');
    Write('/f+|-  space in proc/func & param list. f-  /in    indent level space. n=2      ');
    Write('/ln    maximum length of all lines. n=125   /mn    spaces of left margin. n=0   ');
    Write('/n+|-  new line of output after END. n+     /o+|-  new line after CASE const. o-');
    Write('/p+|-  indenting of proc or func. p-        /r+|-  upcase reserved words. r-    ');
    Write('/s+|-  follow THEN, ELSE or DO. s-          /sd+|- follow DO only. sd-          ');
    Write('/si+|- follow THEN, ELSE only. si-          /tn    tab width. n=5               ');
    Write('/w+|-  insert BEGIN-END pair around         /z+|-  strip spaces between proc or ');
    Write('       structured statements. w-                   func and its body. z-        ');
    WriteLn;
    Write('See BEAUTIFY.DOC for more details on options, identifier, and response files.   ');
    Halt(1);
end { Usage };

procedure Message;
begin
    Beep; WriteLn;
    WriteLn('Sorry, you must specify the file(s) to format.');
    WriteLn;
    WriteLn('example: BEAUTIFY myfile.pas bgidemo.pas tcalc.pas');
    WriteLn('example: BEAUTIFY *.pas');
    WriteLn('example: BEAUTIFY *.*');
    WriteLn;
    WriteLn('You can run BEAUTIFY without any parameters to get a listing of options.');
    Halt(1);
end { Message };

begin
    if ParamCount = 0 then Usage;
    WriteLn('BEAUTIFY Version 0.5c 1991 - Toupao Chieng');
    SetSwitches;
    if NoFiles then Message;
    StrucConsts;
    LoadFiles(Nfiles);
    FP := FileListHead;
    if FP = nil then begin
        Beep; WriteLn('Sorry, no matching files found.');
        Halt(1);
    end else WriteLn;
    WriteLn(Nfiles, ' file(s) total to format.'); WriteLn;
    if DefaultBuiltins then WriteLn('Using default builtins in BUILTINS.PB.');
    if DefaultPB then WriteLn('Using default options in DEFAULT.PB.');
    WriteLn;
    repeat
        ProcessFile(FP^.Name);
        FP := FP^.Next;
    until FP = nil;
    WriteLn;
    WriteLn('Total lines processed = ', TotalLn);
end { Pascal Beautifier }.