* Program: Xcatalog.prg
* Dialect: FoxPro 2.5, either DOS or Windows
*  Author: Tom Rettig
*        :
*  Notice: This utility program is part of Xcatalog, an Xbase-oriented
*        :    "system catalog" or "data dictionary" placed in the
*        :    public domain on October 5, 1992.
*        : This program may be freely used, modified, and distributed
*        :    in source-code or compiled form by anyone without any
*        :    payment, registration, or limitation.
*        : This program is provided "as is" without warranty of any kind,
*        :    expressed or implied.  IN NO EVENT SHALL ITS AUTHORS,
*        :    CONTRIBUTORS, OR DISTRIBUTORS BE LIABLE FOR ANY COMMERCIAL,
*        :    SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.
*        :
*  Syntax: DO Xcatalog WITH <C method> [,<C TableName>]
*      or: Xcatalog(<C method> [,<C TableName>])
*        :
*  Return: <expL> true if event completed successfully,
*        :        otherwise false.
*        :
* Methods: Valid event methods are:
*        : "Create" = Create Xcatalog tables from SysCat definition
*        : "Table"  = Update SysTab, SysCol, SysInd from existing tables
*        : "Import" = Import into Xcatalog through SysMap definition
*        : "Export" = Export from Xcatalog through SysMap definition
*        : "Update" = Update Xcatalog from existing Xcatalog objects
*        : "Modify" = Modify Xcatalog objects from Xcatalog definitions
*        :
* Options: Pass the optional table name to limit the event
*        :    to a single table.
*        :
*   Notes:
*       1: All methods require EXCLUSIVE use of tables.
*       2: There is very little error trapping.
*        :
*Routines: Xcreate()         Create Xcatalog structures
*        : Ximport()         Import from external catalog
*        : Xtable()          Update SysTab, SysCol, SysInd from tables
*        : Ximport()         Import from external catalog
*        : Xdelete()         Relational delete from SysKey definitions
*        : ColUpdate()       Create SysCol from table
*        : IndUpdate()       Create SysInd from table
*        : KeyUpdate()       Create SysKey from SysCat
*        : IndCreate()       Create indexes from SysKey
*        : NotImplemented()  Not implemented yet message
*        : ErrorHandler()    Error message
*
PARAMETERS pcMethod, pcTableName
PRIVATE llSetTalkOn, llCompleted, lcMethod

* Global compiler directives.
#define cXCATVERS    "1.05"
#define cXCATDATE    "July 16, 1993"
#define cDOMAINSEP   " = "
#define cCOMMENTSEP  REPLICATE("&",2)
#define cSYSUSERID   "SYS"
#define cAUTOUSERID  "AUTO"
#define cCRLF        CHR(13)+CHR(10)

* Set up.
IF SET("TALK")=="ON"
   SET TALK OFF
   llSetTalkOn = .T.
ELSE
   llSetTalkOn = .F.
ENDIF
IF PARAMETERS()<2  && set to expected data type if not passed
   pcTableName = ""
   IF PARAMETERS()<1
      pcMethod = ""
   ENDIF
ENDIF
llCompleted = .F.

* List of event methods.
lcMethod = UPPER(m.pcMethod)
DO CASE
CASE m.lcMethod=="CREATE"
   llCompleted = Xcreate(m.pcTableName)
CASE m.lcMethod=="TABLE"
   llCompleted = Xtable(m.pcTableName)
CASE m.lcMethod=="IMPORT"
   llCompleted = Ximport()
CASE m.lcMethod=="EXPORT"
   llCompleted = NotImplemented(m.lcMethod)
CASE m.lcMethod=="UPDATE"
   llCompleted = NotImplemented(m.lcMethod)
CASE m.lcMethod=="MODIFY"
   llCompleted = NotImplemented(m.lcMethod)
OTHERWISE  && display the syntax
   IF EMPTY(WONTOP())
      ?   &&123456789 123456789 123465789 123456798 123456789 1
      ? "*********************************************************"
      ? "***"+PADC("Xcatalog.prg, version "+cXCATVERS+", "+cXCATDATE,;
                   51)+"***"
      ? "***   FoxPro tools for the Xcatalog data dictionary   ***"
      ? "*********************************************************"
      ? "Notice: Xcatalog is in the public domain"
      ? "Syntax:"
      ? "   DO Xcatalog WITH <C Method> [, <C TableName>]"
      ? " or"
      ? "   Xcatalog(<C Method> [, <C TableName>])"
      ? "Methods:"
      ? "   Create  = Create Xcatalog tables from SysCat definition"
      ? "   Table   = Update SysTab, SysCol, SysInd from existing tables"
      ? "   Import  = Import into Xcatalog through SysMap definition"
      ? "   Export* = Export from Xcatalog through SysMap definition"
      ? "   Update* = Update Xcatalog from existing Xcatalog objects"
      ? "   Modify* = Modify Xcatalog objects from Xcatalog definitions"
      ? "         * = not implemented yet"
      ? "Options:"
      ? "   Pass <TableName> to limit event to a single table"
      ? "Return:"
      ? "   <expL> true if completed successfully, otherwise false"
   ELSE
      WAIT WINDOW NOWAIT;
         "Syntax: DO Xcatalog WITH <C Method> [, <C TableName>]"
   ENDIF
ENDCASE  && list of methods

IF m.llCompleted
   WAIT WINDOW NOWAIT "*** Xcatalog "+PROPER(m.pcMethod)+;
                      " Complete ***"
ENDIF
IF m.llSetTalkOn
   SET TALK ON
ENDIF
RETURN m.llCompleted
*** Xcatalog.prg ********************************************


FUNCTION Xcreate
*  Syntax: DO Xcreate
*  Return: <expL> true if event completed successfully,
*        :        otherwise false.
*        :
*        : "Create" = Create Xcatalog tables from SysCat definition
*        :
* Options: Pass the optional table name to limit the event
*        :    to a single table.
*
PARAMETERS pcTableName
PRIVATE lcTextFile, lnTable, lnDomain, laCatalog, laCreate,;
        llSetSafetyOn, lnLastEventNumber, llSetExclusiveOn,;
        llWasCreated, lcMemoLine, lnMemLines, lnMemoWidth, lnSeconds

CLOSE ALL  && because this event requires many work areas

lcTextFile = "Xcatalog.txt"
llSetExactOn = SET("EXACT")=="ON"
SET EXACT OFF
llSetSafetyOn = SET("SAFETY")=="ON"
SET SAFETY ON
llSetExclusiveOn = SET("EXCLUSIVE")=="ON"
SET EXCLUSIVE ON

IF EMPTY(pcTableName)
   IF FILE(m.lcTextFile)
      * Start a fresh text output file if creating all tables.
      ERASE (m.lcTextFile)
   ENDIF

   * Get unique list of tables defined in SysCat excluding
   * SysCat itself.  Second column is set true if table is
   * created.
   SELECT DISTINCT UPPER(SysCat.TableName), .F.;
      FROM SysCat;
      INTO ARRAY laCatalog;
      WHERE NOT (SysCat.SysOrder<2 OR EMPTY(SysCat.TableName) OR;
                 UPPER(SysCat.SysStatus)="E");
      ORDER BY SysCat.TableName
ELSE
   DECLARE laCatalog[1, 2]
   laCatalog[1,1] = UPPER(ALLTRIM(m.pcTableName))
ENDIF

* Generate each table and write structure to the text file.
FOR lnTable = 1 TO ALEN(laCatalog, 1)
   SELECT SysCat.ColumnName, SysCat.ColumnType,;
          SysCat.ColumnLen,  SysCat.ColumnDec;
      FROM SysCat;
      INTO ARRAY laCreate;
      WHERE UPPER(SysCat.TableName)=laCatalog[m.lnTable, 1] AND;
            NOT UPPER(SysCat.SysStatus)="E";
      ORDER BY SysCat.SysOrder
   IF _TALLY==0
      WAIT WINDOW "Error: "+laCatalog[m.lnTable, 1]+;
                  "not defined in SysCat ..."
      LOOP
   ENDIF
   IF TRIM(laCatalog[m.lnTable, 1])=="SYSMAP" AND FILE("SysMap.dbf")
      LOOP  && never overwrite SysMap if it exists
   ENDIF
   WAIT WINDOW NOWAIT "Creating "+TRIM(laCatalog[m.lnTable, 1])+" table"
   CREATE TABLE (TRIM(laCatalog[m.lnTable, 1]));
      FROM ARRAY laCreate
   IF USED(TRIM(laCatalog[m.lnTable, 1]))
      * Write created table's structure to text file.
      LIST STRUCTURE IN (TRIM(laCatalog[m.lnTable, 1]));
                     TO FILE (m.lcTextFile) ADDITIVE
      * Set created table's second array element to true.
      laCatalog[m.lnTable, 2] = .T.
   ENDIF
ENDFOR
CLEAR TYPEAHEAD  && from CREATE overwrite warnings, if any

* Update the SysTab file if it exists with Xcatalog table entries.
IF FILE("SysTab.dbf")
   WAIT WINDOW NOWAIT "Updating SysTab table"
   IF NOT USED("SysTab")
      USE SysTab IN SELECT(1)
   ENDIF
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         SELECT TRIM(laCatalog[m.lnTable, 1])
         DO TabUpdate WITH TRIM(laCatalog[m.lnTable, 1]), cSYSUSERID,;
                           "_"+RIGHT(TRIM(laCatalog[m.lnTable, 1]),3), "S"
      ENDIF
   ENDFOR
ENDIF

* Update the SysCol file if it exists with Xcatalog table entries.
IF FILE("SysCol.dbf")
   WAIT WINDOW NOWAIT "Updating SysCol table"
   IF NOT USED("SysCol")
      USE SysCol IN SELECT(1)
   ENDIF
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         SELECT TRIM(laCatalog[m.lnTable, 1])
         DO ColUpdate WITH TRIM(laCatalog[m.lnTable, 1]), cSYSUSERID
      ENDIF
   ENDFOR
ENDIF

* Update the SysKey file if it exists with Xcatalog table entries,
* and create indexes based on all keys in SysKey.
IF FILE("SysKey.dbf")
   WAIT WINDOW NOWAIT "Updating SysKey table"
   IF NOT USED("SysKey")
      USE SysKey IN SELECT(1)
   ENDIF
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         SELECT TRIM(laCatalog[m.lnTable, 1])
         DO KeyUpdate WITH TRIM(laCatalog[m.lnTable, 1]), cSYSUSERID
         DO IndCreate WITH TRIM(laCatalog[m.lnTable, 1])
      ENDIF
   ENDFOR
ENDIF

* Update the SysInd file if it exists with indexes created above.
IF FILE("SysInd.dbf")
   WAIT WINDOW NOWAIT "Updating SysInd table"
   IF NOT USED("SysInd")
      USE SysInd IN SELECT(1)
   ENDIF
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         SELECT TRIM(laCatalog[m.lnTable, 1])
         DO IndUpdate WITH TRIM(laCatalog[m.lnTable, 1]),;
                           cSYSUSERID, "CDX"
      ENDIF
   ENDFOR
ENDIF

* Update the SysDom file if it exists, and if SysCat.Detail
* has a domain definition.
IF FILE("SysDom.dbf")
   WAIT WINDOW NOWAIT "Updating SysDom table"
   lnMemoWidth = SET("MEMOWIDTH")
   SET MEMOWIDTH TO 80
   SELECT SysCat
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         * Scan only records with domain definitions.
         SCAN FOR UPPER(SysCat.TableName)=laCatalog[m.lnTable, 1] AND;
                  (NOT UPPER(SysCat.SysStatus)="E") AND;
                  ATCLINE("BeginDomain", SysCat.Detail)>0
            _MLINE = ATCLINE("BeginDomain", SysCat.Detail)
            IF _MLINE>1  && make sure it starts a line
               _MLINE = ATCLINE(cCRLF+"BeginDomain",;
                                SysCat.Detail)
               IF _MLINE==0
                  LOOP
               ENDIF
            ENDIF
            * Next line starts the domain list.
            _MLINE = _MLINE + AT(cCRLF,;
                                 SUBSTR(SysCat.Detail, _MLINE))
            lnMemLines = MEMLINES(SysCat.Detail)
            FOR lnDomain = 1 TO m.lnMemLines
               lcMemoLine = MLINE(SysCat.Detail, 1, _MLINE)
               IF EMPTY(m.lcMemoLine) OR;
                     UPPER(LEFT(m.lcMemoLine,9))="ENDDOMAIN"
                  EXIT
               ELSE
                  IF cCOMMENTSEP$m.lcMemoLine
                     lcMemoLine = LEFT(m.lcMemoLine,;
                                       AT(cCOMMENTSEP,m.lcMemoLine)-1)
                  ENDIF
                  INSERT INTO SysDom;
                        (TableName, ColumnName,;
                         DataValue, PromptName,;
                         OrderBy);
                     VALUES;
                        (UPPER(SysCat.TableName),;
                         UPPER(SysCat.ColumnName),;
                         IIF(cDOMAINSEP$m.lcMemoLine,;
                             LEFT(ALLTRIM(m.lcMemoLine),;
                                  AT(cDOMAINSEP,m.lcMemoLine)),;
                             ALLTRIM(m.lcMemoLine)),;
                         IIF(cDOMAINSEP$m.lcMemoLine,;
                             SUBSTR(ALLTRIM(m.lcMemoLine),;
                                    AT(cDOMAINSEP,m.lcMemoLine)+3),;
                             ""),;
                         m.lnDomain)
               ENDIF
            ENDFOR
         ENDSCAN
      ENDIF
   ENDFOR
   SET MEMOWIDTH TO m.lnMemoWidth
ENDIF

* Update the SysUsa file if it exists.
IF FILE("SysUsa.dbf")
   WAIT WINDOW NOWAIT "Updating SysUsa table"
   * Get last usage event number.
   SELECT MAX(SysUsa.UsageEvent);
      FROM SysUsa;
      INTO ARRAY laLastEvent
   IF _TALLY==0  && no records in SysUsa
      DECLARE laLastEvent[1]
      laLastEvent[1] = 0
   ENDIF
   lnSeconds = INT(SECONDS())  && for time consistency in loop
   FOR lnTable = 1 TO ALEN(laCatalog, 1)
      * Second dimension element is true if table was created above.
      IF laCatalog[m.lnTable, 2]
         INSERT INTO SysUsa;
               (UsageEvent,   TableName,   ObjectName,;
                UserId,       UsageType,   CreateDate,;
                CreateTime,   Detail);
            VALUES;
               (laLastEvent[1]+m.lnTable,;
                   TRIM(laCatalog[m.lnTable, 1]),;
                      "SYSCAT",;
                cSYSUSERID,   "C",         DATE(),;
                m.lnSeconds,  "Xcatalog.prg create")
      ENDIF
   ENDFOR
ENDIF

* Clean up.
CLOSE ALL
IF m.llSetExactOn
   SET EXACT ON
ENDIF
IF NOT m.llSetSafetyOn
   SET SAFETY OFF
ENDIF
IF NOT m.llSetExclusiveOn
   SET EXCLUSIVE OFF
ENDIF

* View the text file.
IF FILE(m.lcTextFile)
   WAIT WINDOW NOWAIT "Viewing created structures"
   CLEAR TYPEAHEAD  && from overwrite warnings, if any
   MODIFY COMMAND (m.lcTextFile)
ENDIF

RETURN .T.
*** Xcreate() ***********************************************


FUNCTION Xtable
*  Syntax: DO Xtable
*  Return: <expL> true if event completed successfully,
*        :        otherwise false.
*        :
*        : "Table" = Update SysTab, SysCol, SysInd from existing tables
*        :
* Options: Pass the optional table name to limit the event
*        :    to a single table, otherwise does the entire
*        :    current directory.
*
PARAMETERS pcTableName
PRIVATE laTables, lnTable, laCatalog, lnCatalog,;
        llSetExactOn, llSetExclusiveOn, llWasOpen

IF NOT FILE("SysTab.DBF")
   DO ErrorHandler WITH "SysTab.DBF not found", LINENO()
   RETURN .F.  && early exit
ENDIF

WAIT WINDOW NOWAIT "Updating SysTab table"

CLOSE ALL  && because this event requires many work areas

llSetExactOn = SET("EXACT")=="ON"
SET EXACT OFF
llSetExclusiveOn = SET("EXCLUSIVE")=="ON"
SET EXCLUSIVE ON

* Get list of tables in current directory, or use the one passed.
IF EMPTY(m.pcTableName)
   =ADIR(laTables, "*.DBF")
ELSE
   DECLARE laTables[1,1]
   laTables[1,1] = UPPER(ALLTRIM(m.pcTableName))
ENDIF

* Get unique list of tables defined in SysCat including SysCat itself.
SELECT DISTINCT UPPER(SysCat.TableName);
   FROM SysCat;
   INTO ARRAY laCatalog;
   WHERE NOT (SysCat.SysOrder<1 OR EMPTY(SysCat.TableName) OR;
              UPPER(SysCat.SysStatus)="E");
   ORDER BY SysCat.TableName
IF USED("SysCat")
   USE IN SysCat
ENDIF

IF NOT USED("SysTab")
   USE SysTab IN SELECT(1)
ENDIF
IF (NOT USED("SysCol")) AND FILE("SysCol.DBF")
   USE SysCol IN SELECT(1)
ENDIF
IF (NOT USED("SysInd")) AND FILE("SysInd.DBF")
   USE SysInd IN SELECT(1)
ENDIF
FOR lnTable = 1 TO ALEN(laTables, 1)
   SET EXACT ON  && for ASCAN()
   IF EMPTY(m.pcTableName) AND;
         ASCAN(laCatalog, PADR(STRTRAN(laTables[m.lnTable,1],".DBF"),10))>0
      SET EXACT OFF
      LOOP  && skip over Xcatalog tables picked up by ADIR()
   ENDIF
   SET EXACT OFF
   IF USED(TRIM(laTables[m.lnTable, 1]))
      llWasOpen = .T.
      SELECT (TRIM(laTables[m.lnTable, 1]))
   ELSE
      llWasOpen = .F.
      SELECT 0
      USE (TRIM(laTables[m.lnTable, 1]))
   ENDIF
   DO TabUpdate WITH TRIM(laTables[m.lnTable, 1]), cSYSUSERID
   IF USED("SysCol")
      DO ColUpdate WITH TRIM(laTables[m.lnTable, 1]), cSYSUSERID
   ENDIF
   IF USED("SysInd")
      DO IndUpdate WITH TRIM(laTables[m.lnTable, 1]), cSYSUSERID
   ENDIF
   IF NOT m.llWasOpen
      USE IN (TRIM(laTables[m.lnTable, 1]))
   ENDIF
ENDFOR

* Clean up.
CLOSE ALL
IF NOT m.llSetExclusiveOn
   SET EXCLUSIVE OFF
ENDIF
IF m.llSetExactOn
   SET EXACT ON
ENDIF
RETURN .T.
*** Xtable() ************************************************


FUNCTION Ximport
*  Syntax: DO Ximport
*  Return: <expL> true if event completed successfully,
*        :        otherwise false.
*        :
*        : "Import" = Import into Xcatalog through SysMap definition
*        :
* Options: Pass the optional table name to limit the event
*        :    to a single table.
*
PRIVATE laCreate, laMapTables, laMapXcatalog, laMapXImport,;
        laMapXCreate, laMapData, laFields, laXPrimary,;
        lcSearch, lcFileSpec, lcAliasName, lcMapAliasName,;
        llSetExactOn, llSetExclusiveOn, llSetFullPathOn,;
        lnTable, lnCatalog, lnData, lnTags, lnCDX, llChar,;
        lnSeconds
           
WAIT WINDOW NOWAIT "Selecting tables"

CLOSE ALL  && because this event requires many work areas
llSetExactOn = SET("EXACT")=="ON"
SET EXACT OFF
llSetExclusiveOn = SET("EXCLUSIVE")=="ON"
SET EXCLUSIVE ON
llSetFullPathOn = SET("FULLPATH")=="ON"
SET FULLPATH ON

* Get unique list of all Xcatalog tables defined in SysMap.
* Third column is set true if table is created.
SELECT DISTINCT UPPER(SysMap.SysTable), SysMap.SysPath, .F.;
   FROM SysMap;
   INTO ARRAY laMapXcatalog;
   WHERE NOT EMPTY(SysMap.SysTable);
   ORDER BY SysMap.SysTable

* Get unique list of all mapped tables defined in SysMap.
SELECT DISTINCT UPPER(SysMap.MapTable), SysMap.MapPath;
   FROM SysMap;
   INTO ARRAY laMapTables;
   WHERE NOT EMPTY(SysMap.MapTable);
   ORDER BY SysMap.MapTable

* Open all Xcatalog's files needed by SysMap.
WAIT WINDOW NOWAIT "Opening catalog"
FOR lnCatalog = 1 TO ALEN(laMapXcatalog, 1)
   * Get full file spec including path (defaults to current directory),
   * and open or create the Xcatalog table.
   lcFileSpec = IIF(EMPTY(laMapXcatalog[m.lnCatalog, 2]),;
                    FULLPATH(""),;
                    TRIM(EVALUATE(TRIM(laMapXcatalog[m.lnCatalog, 2])))+;
                       IIF(RIGHT(TRIM(EVALUATE(;
                                 TRIM(laMapXcatalog[m.lnCatalog, 2]))),;
                                 1)=="\",;
                           "", "\"))+;
                TRIM(laMapXcatalog[m.lnCatalog, 1])+".DBF"
   IF FILE(m.lcFileSpec)
      lcAliasName = TRIM(laMapXcatalog[m.lnCatalog, 1])
      IF NOT USED(lcAliasName)  && for SysMap
         USE (m.lcFileSpec) IN SELECT(1) ALIAS (m.lcAliasName)
      ENDIF
   ELSE  && create automatically if it doesn't exist
      SELECT SysCat.ColumnName, SysCat.ColumnType,;
             SysCat.ColumnLen,  SysCat.ColumnDec;
         FROM SysCat;
         INTO ARRAY laCreate;
         WHERE UPPER(SysCat.TableName)=laMapXcatalog[m.lnCatalog, 1] AND;
               NOT UPPER(SysCat.SysStatus)="E";
         ORDER BY SysCat.SysOrder
      IF _TALLY==0
         DO ErrorHandler WITH TRIM(laMapXcatalog[m.lnCatalog, 1])+;
                              " not defined in SysCat", LINENO()
      ELSE
         CREATE TABLE (m.lcFileSpec) FROM ARRAY laCreate
         laMapXcatalog[m.lnCatalog, 3] = .T.
      ENDIF
   ENDIF
ENDFOR

* Collect mapped data by scanning SysMap once for
* every row in every mapped table outside Xcatalog.
FOR lnTable = 1 TO ALEN(laMapTables, 1)
   IF laMapTables[m.lnTable, 1]="<CREATE>"
      * Creating from a literal file name, not an external dictionary.
      SELECT SysMap
      LOCATE FOR UPPER(SysMap.MapTable)=laMapTables[m.lnTable, 1]
      * This next loop handles all of SysMap's <create> literal actions.
      DO WHILE FOUND("SysMap")
         * Get full file spec including path (defaults to current directory).
         lcAliasName = TRIM(EVALUATE(TRIM(SysMap.PromptName)))
         lcFileSpec = IIF(EMPTY(SysMap.MapPath),;
                          FULLPATH(""),;
                          TRIM(EVALUATE(TRIM(SysMap.MapPath)))+;
                             IIF(RIGHT(TRIM(EVALUATE(TRIM(SysMap.MapPath))),;
                                       1)=="\",;
                                 "", "\"))+;
                          m.lcAliasName+IIF("."$m.lcAliasName,"",".DBF")
         IF "."$m.lcAliasName
            lcAliasName = LEFT(m.lcAliasName, AT(".",m.lcAliasName)-1)
         ENDIF  

         * Open the mapped table, which is always closed at the
         * end of this DO WHILE loop structure.
         IF USED(m.lcAliasName)
            SELECT (m.lcAliasName)
         ELSE
            IF NOT FILE(m.lcFileSpec)
               DO ErrorHandler WITH m.lcFileSpec+" file not found",;
                                    LINENO()
               CONTINUE
               LOOP  && to start of DO WHILE
            ENDIF
            SELECT 0
            USE (m.lcFileSpec) ALIAS (m.lcAliasName)
         ENDIF

         DO CASE
         CASE UPPER(TRIM(SysMap.SysTable))=="SYSTAB"
            DO TabUpdate WITH m.lcFileSpec, SysMap.UserId
         CASE UPPER(TRIM(SysMap.SysTable))=="SYSCOL"
            DO ColUpdate WITH m.lcFileSpec, SysMap.UserId
         CASE UPPER(TRIM(SysMap.SysTable))=="SYSIND"
            DO IndUpdate WITH m.lcFileSpec, SysMap.UserId,;
                              IIF(EMPTY(SysMap.ImportExp), "",;
                                  TRIM(EVALUATE(TRIM(SysMap.ImportExp))))
         ENDCASE

         SELECT SysMap
         USE IN m.m.lcAliasName
         CONTINUE
      ENDDO  && while found "<CREATE>"
      LOOP   && back to FOR (next table)
   ENDIF

   * Get unique list of Xcatalog tables needed by this
   * mapped table's <import> actions only.  Columns must
   * match precompiler array element definitions.
   SELECT DISTINCT UPPER(SysMap.SysTable), SysMap.MapTable,;
                   SysMap.MapColumn, SysMap.PromptName;
      FROM SysMap;
      INTO ARRAY laMapXImport;
      ORDER BY SysMap.SysTable;
      WHERE UPPER(SysMap.MapTable)=laMapTables[m.lnTable, 1] AND;
            UPPER(SysMap.MapType)="I" AND;
            UPPER(SysMap.SysColumn)="<IMPORT>"

   * Get unique list of Xcatalog tables needed by this
   * mapped table's <create> actions only.  Columns must
   * match precompiler array element definitions.
   SELECT DISTINCT UPPER(SysMap.SysTable), SysMap.MapTable,;
                   SysMap.MapColumn, SysMap.PromptName;
      FROM SysMap;
      INTO ARRAY laMapXCreate;
      ORDER BY SysMap.SysTable;
      WHERE UPPER(SysMap.MapTable)=laMapTables[m.lnTable, 1] AND;
            UPPER(SysMap.MapType)="C" AND;
            UPPER(SysMap.SysColumn)="<CREATE>" AND;
            NOT EMPTY(SysMap.SysTable)

   * Use the literal table name (not expression) for foreign catalog,
   * and select or open the foreign catalog.
   lcMapAliasName = TRIM(laMapTables[m.lnTable, 1])
   lcFileSpec = IIF(EMPTY(laMapTables[m.lnTable, 2]) OR;
                    EMPTY(EVALUATE(TRIM(laMapTables[m.lnTable, 2]))),;
                    FULLPATH(""),;
                    TRIM(EVALUATE(TRIM(laMapTables[m.lnTable, 2])))+;
                       IIF(RIGHT(TRIM(;
                          EVALUATE(TRIM(laMapTables[m.lnTable, 2]))),;
                                 1)=="\",;
                           "", "\"))+;
                    m.lcMapAliasName+IIF("."$m.lcMapAliasName,"",".DBF")
   IF "."$m.lcMapAliasName
      lcMapAliasName = LEFT(m.lcMapAliasName, AT(".",m.lcMapAliasName)-1)
   ENDIF  
   IF USED(m.lcMapAliasName)
      SELECT (m.lcMapAliasName)
   ELSE
      IF NOT FILE(m.lcFileSpec)
         DO ErrorHandler WITH m.lcFileSpec+" file not found", LINENO()
         LOOP  && to start of FOR (next mapped table in laMapTables[])
      ENDIF
      SELECT 0
      USE (m.lcFileSpec) ALIAS (m.lcMapAliasName)
   ENDIF

   SCAN  && mapped table
      * One event for each Xcatalog table which is mapped to
      * this map table as an <import> event only.  We don't
      * need to open any tables since definitions are
      * imported from data in the foreign catalog.
      FOR lnCatalog = 1 TO ALEN(laMapXImport, 1)

* Local compiler directives.
 * laMapXImport[] and laMapXCreate[] array constants,
 * two dimensional, dynamic.  Order must match SQL SELECT statements.
 * These are columns only.
#define cXSYSTABLE   1
#define cXMAPTABLE   2
#define cXMAPCOLUMN  3
#define cXPROMPTNAME 4
 * laMapData[] array constants, two dimensional, dynamic.
 * Order must match SQL SELECT statements.  These are columns only.
#define cDMAPCOLUMN  1
#define cDSYSTABLE   2
#define cDSYSCOLUMN  3
#define cDIMPORTEXP  4


         * File name is found in the evaluated fields MapTable.PromptName.
         * Short name is found in the evaluated fields MapTable.MapColumn.
         lcAliasName = TRIM(EVALUATE(IIF("."$laMapXImport[m.lnCatalog,;
                                                          cXMAPTABLE],;
                          LEFT(laMapXImport[m.lnCatalog, cXMAPTABLE],;
                               AT(".",laMapXImport[m.lnCatalog,;
                                                     cXMAPTABLE])-1),;
                          TRIM(laMapXImport[m.lnCatalog, cXMAPTABLE]))+;
                       "."+laMapXImport[m.lnCatalog, cXPROMPTNAME]))
         IF "."$m.lcAliasName
            lcAliasName = LEFT(m.lcAliasName, AT(".",m.lcAliasName)-1)
         ENDIF
         IF m.lcAliasName==m.lcMapAliasName
            * The foreign catalog is defined in itself, so skip.
            EXIT
         ENDIF

         * Get the whole mapped data set for this Xcatalog table.
         SELECT SysMap.MapColumn, SysMap.SysTable,;
                SysMap.SysColumn, SysMap.ImportExp;
            FROM SysMap;
            INTO ARRAY laMapData;
            WHERE (UPPER(SysMap.MapTable) = laMapTables[m.lnTable, 1] OR;
                   UPPER(SysMap.MapTable) = lcMapAliasName) AND;
                  UPPER(SysMap.SysTable)  = laMapXImport[m.lnCatalog,;
                                                         cXSYSTABLE] AND;
                  UPPER(SysMap.MapType)="I" AND NOT SysMap.SysColumn="<"

         * Check to see if this entry already exists.  First get the
         * primary key for this Xcatalog table from SysCat.
         SELECT UPPER(SysCat.ColumnName);
            FROM SysCat;
            INTO ARRAY laXPrimary;
            WHERE UPPER(SysCat.TableName)=laMapXImport[m.lnCatalog,;
                                                       cXSYSTABLE] AND;
                  (UPPER(SysCat.KeyType)="P" OR VAL(SysCat.KeyType)>0)
         * Next, look at the entire mapped data set, but only
         * take the primary key parts to search on.
         SET EXACT ON  && for ASCAN()
         lcSearch = ""
         FOR lnData = 1 TO ALEN(laMapData, 1)
            * Add to search string if this is a primary key.
            IF ASCAN(laXPrimary, UPPER(laMapData[m.lnData,cDSYSCOLUMN]))>0
               llChar = IIF(TYPE(TRIM(m.lcMapAliasName)+"."+;
                             TRIM(laMapData[m.lnData,cDMAPCOLUMN]))=="C",;
                            .T., .F.)
               lcSearch = m.lcSearch +;
                          IIF(LEN(m.lcSearch)>0, " AND ", "")+;
                          TRIM(laMapData[m.lnData, cDSYSTABLE])+"."+;
                          TRIM(laMapData[m.lnData, cDSYSCOLUMN])+"="+;
                          IIF(EMPTY(laMapData[m.lnData, cDIMPORTEXP]),;
                              IIF(m.llChar, "TRIM(", "")+;
                                 TRIM(m.lcMapAliasName)+"."+;
                                 TRIM(laMapData[m.lnData, cDMAPCOLUMN])+;
                                 IIF(m.llChar, ")", ""),;
                              TRIM(laMapData[m.lnData, cDIMPORTEXP]))

             ENDIF
         ENDFOR  && mapped data set
         SET EXACT OFF

         IF EMPTY(m.lcSearch)
            DO ErrorHandler WITH "primary key not in SysCat", LINENO()
            LOOP  && to top of FOR (next Xcatalog table for this entry)
         ENDIF

         * Search Xcatalog table for this entry, and add
         * a row to it if not found.
         SELECT (TRIM(laMapData[1, cDSYSTABLE]))
         LOCATE FOR &lcSearch  && macro alert
         IF NOT FOUND(TRIM(laMapData[1, cDSYSTABLE]))
            APPEND BLANK
         ENDIF
   
         * Add the mapped data to the Xcatalog table.
         FOR lnData = 1 TO ALEN(laMapData, 1)
            WAIT WINDOW NOWAIT;
               PADR(PROPER(m.lcAliasName),                   10)+"."+;
               PADR(PROPER(laMapData[m.lnData, cDMAPCOLUMN]),10)+" --> "+;
               PADR(       laMapData[m.lnData, cDSYSTABLE] , 10)+"."+;
               PADR(PROPER(laMapData[m.lnData, cDSYSCOLUMN]),10)
            REPLACE (TRIM(laMapData[m.lnData, cDSYSTABLE])+"."+;
                     TRIM(laMapData[m.lnData, cDSYSCOLUMN]));
               WITH IIF(EMPTY(laMapData[m.lnData, cDIMPORTEXP]),;
                        EVALUATE(m.lcMapAliasName+"."+;
                                 TRIM(laMapData[m.lnData, cDMAPCOLUMN])),;
                        EVALUATE(TRIM(laMapData[m.lnData, cDIMPORTEXP])))
         ENDFOR  && mapped data set

         * Update the UserId and timestamp if the
         * appropriate columns exist.
         =AFIELDS(laFields)
         SET EXACT ON  && for ASCAN()
         lnSeconds = INT(SECONDS())  && for time consistency
         IF ASCAN(laFields, "USERID")>0
            REPLACE UserId WITH UPPER(SysMap.UserId)
         ENDIF
         IF NOT FOUND(TRIM(laMapData[1, cDSYSTABLE]))
            IF ASCAN(laFields, "CREATEDATE")>0
               REPLACE CreateDate WITH DATE()
            ENDIF
            IF ASCAN(laFields, "CREATETIME")>0
               REPLACE CreateTime WITH m.lnSeconds
            ENDIF
         ENDIF
         IF ASCAN(laFields, "REFDATE")>0
            REPLACE RefDate WITH DATE()
         ENDIF
         IF ASCAN(laFields, "REFTIME")>0
            REPLACE RefTime WITH m.lnSeconds
         ENDIF
         SET EXACT OFF
         RELEASE laFields  && because AFIELDS() is additive
      ENDFOR  && each Xcatalog table

      * One event for each Xcatalog table's <create>
      * action which is mapped to this map table.  We need
      * to open tables here since definitions are "imported"
      * from tables instead of data in a foreign catalog.
      FOR lnCatalog = 1 TO ALEN(laMapXCreate, 1)
         * The mapped data set for this event is all in
         * one record, so use it direct from the SysMap table.
         SELECT SysMap
         LOCATE FOR UPPER(SysMap.SysTable) = laMapXCreate[m.lnCatalog,;
                                                        cXSYSTABLE] AND;
                    UPPER(SysMap.MapTable) = m.lcMapAliasName AND;
                    UPPER(SysMap.MapType)  = "C" AND;
                    UPPER(SysMap.SysColumn)= "<CREATE>"
         IF NOT FOUND("SysMap")
            DO ErrorHandler WITH "SysMap search failed", LINENO()
            LOOP  && to start of FOR (next <create> action)
         ENDIF

         * File name is found in the evaluated fields MapTable.PromptName.
         * Short name is found in the evaluated fields MapTable.MapColumn.
         lcAliasName = TRIM(EVALUATE(IIF("."$laMapXCreate[m.lnCatalog,;
                                                          cXMAPTABLE],;
                           LEFT(laMapXCreate[m.lnCatalog, cXMAPTABLE],;
                                AT(".",laMapXCreate[m.lnCatalog,;
                                                    cXMAPTABLE])-1),;
                           TRIM(laMapXCreate[m.lnCatalog, cXMAPTABLE]))+;
                        "."+laMapXCreate[m.lnCatalog, cXPROMPTNAME]))
         lcFileSpec = IIF(EMPTY(laMapTables[m.lnTable, 2]),;
                          FULLPATH(""),;
                          TRIM(EVALUATE(TRIM(laMapTables[m.lnTable,2])))+;
                             IIF(RIGHT(TRIM(EVALUATE(;
                                       TRIM(laMapTables[m.lnTable, 2]))),;
                                       1)=="\",;
                                 "", "\"))+;
                          m.lcAliasName+IIF("."$m.lcAliasName,"",".DBF")
         IF "."$m.lcAliasName
            lcAliasName = LEFT(m.lcAliasName, AT(".",m.lcAliasName)-1)
         ENDIF  
         IF m.lcAliasName==m.lcMapAliasName
            * The foreign catalog is defined in itself, so skip.
            EXIT
         ENDIF

         * Open the mapped table, which is always closed at the
         * end of the next DO CASE structure unless it's also
         * the mapped catalog containing a reference to itself.
         IF USED(m.lcAliasName)
            SELECT (m.lcAliasName)
         ELSE
            IF NOT FILE(m.lcFileSpec)
               DO ErrorHandler WITH m.lcFileSpec+" file not found",;
                                    LINENO()
            ENDIF
            SELECT 0
            USE (m.lcFileSpec) ALIAS (m.lcAliasName)
         ENDIF

         * Create action depends on the mapped Xcatalog table.
         DO CASE
            CASE UPPER(TRIM(SysMap.SysTable))=="SYSTAB"
               DO TabUpdate WITH m.lcFileSpec, SysMap.UserId,;
                   IIF(EMPTY(laMapXCreate[m.lnCatalog, cXMAPCOLUMN]), "",;
                       TRIM(EVALUATE(m.lcMapAliasName+"."+;
                           TRIM(laMapXCreate[m.lnCatalog, cXMAPCOLUMN]))))
            CASE UPPER(TRIM(SysMap.SysTable))=="SYSCOL"
               DO ColUpdate WITH m.lcFileSpec, SysMap.UserId
            CASE UPPER(TRIM(SysMap.SysTable))=="SYSIND"
               DO IndUpdate WITH m.lcFileSpec, SysMap.UserId,;
                              IIF(EMPTY(SysMap.ImportExp), "",;
                                  TRIM(EVALUATE(TRIM(SysMap.ImportExp))))
         ENDCASE  && <create> table, column, or index

         * Close the mapped table unless it's the foreign catalog
         * or an Xcatalog table.
         IF NOT m.lcAliasName==m.lcMapAliasName  && foreign catalog
            SET EXACT ON  && for ASCAN()
            IF ASCAN(laMapXcatalog,;
                     PADR(m.lcAliasName, FSIZE("SysTable","SysMap")))==0
               USE
            ENDIF
            SET EXACT OFF
         ENDIF
      ENDFOR  && each Xcatalog table's <create>

      SELECT (TRIM(laMapTables[m.lnTable, 1]))
   ENDSCAN  && each mapped table

   * Close the mapped table, unless it's also an Xcatalog table.
   SET EXACT ON  && for ASCAN()
   IF ASCAN(laMapXcatalog,;
            PADR(m.lcMapAliasName, FSIZE("SysTable","SysMap")))==0
      USE IN (m.lcMapAliasName)
   ENDIF
   SET EXACT OFF
ENDFOR  && unique list of mapped tables

* Update the SysUsa file if it exists.
IF FILE("SysUsa.dbf")
   WAIT WINDOW NOWAIT "Updating Xcatalog usage"

   * Update the usage table with one event for
   * each Xcatalog table's mapped <import> event.
   FOR lnCatalog = 1 TO ALEN(laMapXcatalog, 1)
      * Third dimension element is true if table was created above.
      IF laMapXcatalog[m.lnCatalog, 3]
         * Get last usage event number.
         SELECT MAX(SysUsa.UsageEvent);
            FROM SysUsa;
            INTO ARRAY laLastEvent
         IF _TALLY==0  && no records in SysUsa
            DECLARE laLastEvent[1]
            laLastEvent[1] = 0
         ENDIF
      ENDIF

      * Get the whole mapped data set for this Xcatalog table.
      SELECT DISTINCT UPPER(SysMap.MapName);
         FROM SysMap;
         INTO ARRAY laMapData;
         WHERE UPPER(SysMap.SysTable)=laMapXCatalog[m.lnCatalog, 1] AND;
               UPPER(SysMap.MapType)="I" AND;
               UPPER(SysMap.SysColumn)="<IMPORT>"
      IF _TALLY>0
         * Get last usage event number.
         SELECT MAX(SysUsa.UsageEvent);
            FROM SysUsa;
            INTO ARRAY laLastEvent
         IF _TALLY==0  && no records in SysUsa
            DECLARE laLastEvent[1]
            laLastEvent[1] = 0
         ENDIF
         FOR lnTable = 1 TO ALEN(laMapData, 1)
            INSERT INTO SysUsa;
                 (UsageEvent,   TableName,    ObjectName,;
                  UserId,       UsageType,    CreateDate,;
                  CreateTime,   Detail);
               VALUES;
                 (laLastEvent[1]+m.lnTable,;
                     TRIM(laMapXcatalog[m.lnCatalog, 1]),;
                        "SYSMAP",;
                  UPPER(SysMap.UserId), "I",  DATE(),;
                  m.lnSeconds,;
                  "Map "+UPPER(TRIM(SysMap.MapName))+" <import>")
         ENDFOR
      ENDIF
   ENDFOR

   * Get last usage event number again.
   SELECT MAX(SysUsa.UsageEvent);
      FROM SysUsa;
      INTO ARRAY laLastEvent
   IF _TALLY==0  && still no records in SysUsa
      DECLARE laLastEvent[1]
      laLastEvent[1] = 0
   ENDIF

   * Update the usage table with one event for
   * each Xcatalog table's mapped <create> event.
   FOR lnCatalog = 1 TO ALEN(laMapXCreate, 1)
      SELECT SysMap
      LOCATE FOR UPPER(SysMap.SysTable) = laMapXCreate[m.lnCatalog,;
                                                       cXSYSTABLE] AND;
                 UPPER(SysMap.MapType)  = "C" AND;
                 UPPER(SysMap.SysColumn)= "<CREATE>"
      IF FOUND("SysMap")
         INSERT INTO SysUsa;
              (UsageEvent,   TableName,    ObjectName,;
               UserId,       UsageType,    CreateDate,;
               CreateTime,   Detail);
            VALUES;
              (laLastEvent[1]+m.lnCatalog,;
                  TRIM(laMapXCreate[m.lnCatalog, cXSYSTABLE]),;
                     "SYSMAP",;
               UPPER(SysMap.UserId), "I",  DATE(),;
               m.lnSeconds,;
                  "Map "+UPPER(TRIM(SysMap.MapName))+" <create>")
      ENDIF
   ENDFOR
ENDIF

* Clean up.
CLOSE ALL
IF m.llSetExactOn
   SET EXACT ON
ENDIF
IF NOT m.llSetExclusiveOn
   SET EXCLUSIVE OFF
ENDIF
IF NOT m.llSetFullPathOn
   SET FULLPATH OFF
ENDIF

RETURN .T.
*** Ximport() ***********************************************


FUNCTION TabUpdate
*  Syntax: DO TabUpdate WITH <C file name>
*        :                   [,<C user's id name>
*        :                    [,<C table's short name>]]
*        :                     [,<C table type>]]]
*  Action: Update SysTab entry from an existing table.
*  Return: Nothing
*   Notes: Called by Xcreate and Ximport.
*        : Expects table open.
*        : Changes select area to that of passed file name.
*
PARAMETERS pcFileName, pcUserId, pcShortName, pcTableType
PRIVATE lcFileName, lcAliasName, laNameCalc, lnSeconds, lcTableName
lcFileName = IIF("\"$m.pcFileName,;
                 SUBSTR(m.pcFileName, RAT("\",m.pcFileName)+1),;
                 TRIM(m.pcFileName))  && trim path
IF NOT "."$m.lcFileName
   lcFileName = m.lcFileName+".DBF"   && default file extension
ENDIF
lcAliasName = UPPER(LEFT(m.lcFileName, AT(".",m.lcFileName)-1))
lcTableName = PADR(m.lcAliasName, FSIZE("TableName","SysTab"))

WAIT WINDOW NOWAIT;
   PADR(PROPER(m.lcAliasName),10)+"."+;
   PADR("<create>",           10)+" --> "+;
   PADR("SysTab",             10)+"."+;
   PADR("FileName",           10)

SELECT SysTab
LOCATE FOR SysTab.TableName = m.lcTableName
lnSeconds = INT(SECONDS())  && for time consistency
IF FOUND("SysTab")  && any passed short name and type are ignored
   REPLACE SysTab.FileName   WITH m.lcFileName,;
           SysTab.UserId     WITH IIF(EMPTY(m.pcUserId), cAUTOUSERID,;
                                      TRIM(m.pcUserId)),;
           SysTab.RefDate    WITH DATE(),;
           SysTab.RefTime    WITH m.lnSeconds
ELSE
   * Create unique short name if none is passed.
   DECLARE laNameCalc[1]
   IF EMPTY(m.pcShortName)
      * Create a unique short name for the table.
      laNameCalc[1] = 0
      SELECT INT(MAX(VAL(TableId))) FROM SysTab INTO ARRAY laNameCalc
      laNameCalc[1] = PADL(laNameCalc[1]+1, 5, "0")
   ELSE
      laNameCalc[1] = m.pcShortName
   ENDIF
   INSERT INTO SysTab;
         (TableId,      TableType,    FileName,;
          TableName,    UserId,       CreateDate,;
          CreateTime,   RefDate,      RefTime);
      VALUES;
         (laNameCalc[1],;
             IIF(EMPTY(m.pcTableType), "?", UPPER(TRIM(m.pcTableType))),;
                 m.lcFileName,;
          m.lcAliasName,;
             IIF(EMPTY(m.pcUserId), cAUTOUSERID, TRIM(m.pcUserId)),;
                DATE(),;
          m.lnSeconds,    DATE(),       m.lnSeconds)
ENDIF

SELECT (m.lcAliasName)
RETURN
*** TabUpdate() *********************************************


FUNCTION ColUpdate
*  Syntax: DO ColUpdate WITH <C file name>
*        :                   [,<C user id name>]
*  Action: Update SysCol entry from an existing table.
*  Return: Nothing
*   Notes: Called by Xcreate and Ximport.
*        : Expects table open.
*        : Changes select area to that of passed file name.
*
PARAMETERS pcFileName, pcUserId
PRIVATE lcFileName, lcAliasName, laFields, laTableCount, laDelete,;
        lnSeconds, lcTableName
lcFileName = IIF("\"$m.pcFileName,;
                    SUBSTR(m.pcFileName, RAT("\",m.pcFileName)+1),;
                    TRIM(m.pcFileName))  && trim path
IF NOT "."$m.lcFileName
   lcFileName = m.lcFileName+".DBF"   && default file extension
ENDIF
lcAliasName = UPPER(LEFT(m.lcFileName, AT(".",m.lcFileName)-1))
lcTableName = PADR(m.lcAliasName, FSIZE("TableName","SysTab"))

* Get TableName from SysTab for this alias name, or
* create a SysTab entry if one doesn't exist.
SELECT COUNT(*) FROM SysTab INTO ARRAY laTableCount;
   WHERE SysTab.TableName = m.lcTableName
IF _TALLY==0
   DO TabUpdate WITH lcFileName, pcUserId
   SELECT COUNT(*) FROM SysTab INTO ARRAY laTableCount;
      WHERE SysTab.TableName = m.lcTableName
   IF _TALLY==0  && should never happen
      DO ErrorHandler WITH "table creation failed", LINENO()
      RETURN
   ENDIF
ENDIF

=AFIELDS(laFields)
SELECT SysCol
lnSeconds = INT(SECONDS())  && for time compare at end of loop
FOR lnFields = 1 TO ALEN(laFields, 1)
   WAIT WINDOW NOWAIT;
      PADR(PROPER(m.lcAliasName),          10)+"."+;
      PADR(PROPER(laFields[m.lnFields, 1]),10)+" --> "+;
      PADR("SysCol",                       10)+"."+;
      PADR("ColumnName",                   10)
   LOCATE FOR SysCol.TableName = m.lcTableName AND;
              SysCol.ColumnName = laFields[m.lnFields, 1]
   IF FOUND("SysCol")
      REPLACE SysCol.ColumnType WITH laFields[m.lnFields, 2],;
              SysCol.ColumnLen  WITH laFields[m.lnFields, 3],;
              SysCol.ColumnDec  WITH laFields[m.lnFields, 4],;
              SysCol.ColumnPos  WITH m.lnFields,;
              SysCol.UserId     WITH IIF(EMPTY(m.pcUserId), cAUTOUSERID,;
                                         TRIM(m.pcUserId)),;
              SysCol.RefDate    WITH DATE(),;
              SysCol.RefTime    WITH m.lnSeconds
   ELSE
      INSERT INTO SysCol;
            (TableName,   ColumnName,  ColumnType,;
             ColumnLen,   ColumnDec,   ColumnPos,;
             UserId,      CreateDate,  CreateTime,;
             RefDate,     RefTime);
         VALUES;
            (m.lcAliasName,;
                laFields[m.lnFields, 1],;
                   laFields[m.lnFields, 2],;
             laFields[m.lnFields, 3],;
                laFields[m.lnFields, 4],;
                   m.lnFields,;
             IIF(EMPTY(m.pcUserId), cAUTOUSERID, TRIM(m.pcUserId)),;
                DATE(),   m.lnSeconds,;
             DATE(),      m.lnSeconds)
   ENDIF
ENDFOR  && each column in the table
SELECT (m.lcAliasName)

* Remove SysCol entries no longer valid for this table.
SELECT "SYSCOL    ", UPPER(SysCol.TableName+SysCol.ColumnName), "";
   FROM SysCol;
   INTO ARRAY laDelete;
   WHERE SysCol.TableName=m.lcTableName AND;
         NOT DTOS(SysCol.RefDate)+STR(SysCol.RefTime,5)=;
             DTOS(DATE())+STR(m.lnSeconds,5)
IF _TALLY>0
   IF NOT Xdelete(@laDelete)
      DO ErrorHandler WITH "deleting in SysCol", LINENO()
   ENDIF
ENDIF

RETURN
*** ColUpdate() *********************************************


FUNCTION KeyUpdate
*  Syntax: DO KeyUpdate WITH <C file name>
*        :                   [,<C user id name>]
*  Action: Update SysKey entries from SysCat definition.
*  Return: Nothing
*   Notes: Called by Xcreate.
*        : Expects SysCat and SysKey open.
*
PARAMETERS pcFileName, pcUserId
PRIVATE lcFileName, lcAliasName, laFields, laKeyExp,;
        lnKeyExp, lcKeyExp, laKeys, lnKeys,;
        lcTableName, lnPointer, lnSeconds
lcFileName = IIF("\"$m.pcFileName,;
                    SUBSTR(m.pcFileName, RAT("\",m.pcFileName)+1),;
                    TRIM(m.pcFileName))  && trim path
IF NOT "."$m.lcFileName
   lcFileName = m.lcFileName+".DBF"   && default file extension
ENDIF
lcAliasName = UPPER(LEFT(m.lcFileName, AT(".",m.lcFileName)-1))
lcTableName = PADR(m.lcAliasName, FSIZE("TableName","SysCat"))

* Composite primary key array.
DECLARE laKeyExp[1, 5]
laKeyExp = ""

SELECT SysCat
lnSeconds = INT(SECONDS())  && for time consistency
SCAN FOR UPPER(SysCat.TableName)=m.lcTableName AND;
         NOT EMPTY(SysCat.KeyType)
   WAIT WINDOW NOWAIT;
      PADR(PROPER(SysCat.TableName),  10)+"."+;
      PADR(PROPER(SysCat.ColumnName), 10)+" --> "+;
      PADR("SysKey",                  10)+"."+;
      PADR("KeyType",                 10)

   * Parse SysCat.KeyType into an array where each element is
   * one part of KeyType's comma-delimited list.

   * Count the number of commas in the list.  Keys are one more.
   lnKeys = 1
   DO WHILE AT(",", SysCat.KeyType, m.lnKeys)>0
      lnKeys = m.lnKeys+1
   ENDDO
   DECLARE laKeys[m.lnKeys,5]
   * Get the key types, segments, and foreign key references into the
   * array.  Array columns are: 1=Type, 2=Segment, 3=Foreign table,
   * 4=Foreign column, 5=Foreign key rule.
   lnPointer = 1
   FOR lnKeys = 1 TO ALEN(laKeys,1)
      * Get the key type and segment.
      IF AT(",", SysCat.KeyType, m.lnKeys)>1
         laKeys[m.lnKeys,1] =;
            UPPER(TRIM(SUBSTR(SysCat.KeyType, m.lnPointer,;
                       AT(",", SysCat.KeyType, m.lnKeys)-m.lnPointer)))
         lnPointer = AT(",", SysCat.KeyType, m.lnKeys)+1
      ELSE
         laKeys[m.lnKeys,1] =;
            UPPER(TRIM(SUBSTR(SysCat.KeyType, m.lnPointer)))
      ENDIF

      laKeys[m.lnKeys,2] = IIF(VAL(RIGHT(laKeys[m.lnKeys,1],1))==0,;
                               1, VAL(RIGHT(laKeys[m.lnKeys,1],1)))

      * Get the foreign key reference from a list in SysCat.Detail
      * if KeyType contains a foreign key.
      IF LEFT(laKeys[m.lnKeys,1], 1)=="F"
         IF NOT laKeys[m.lnKeys,1]+cDOMAINSEP $ SysCat.Detail
            DO ErrorHandler WITH "foreign key detail not found", LINENO()
            LOOP
         ENDIF

         * Get this reference from the key list in SysCat.Detail.
         _MLINE = 0
         laKeys[m.lnKeys,3] = MLINE(SysCat.Detail, 1, _MLINE)
         DO WHILE NOT (EMPTY(laKeys[m.lnKeys,3]) OR laKeys[m.lnKeys,1]==;
                  UPPER(LEFT(laKeys[m.lnKeys,3], LEN(laKeys[m.lnKeys,1])))))
            laKeys[m.lnKeys,3] = MLINE(SysCat.Detail, 1, _MLINE)
         ENDDO

         laKeys[m.lnKeys,4] =;
            UPPER(TRIM(SUBSTR(laKeys[m.lnKeys,3],;
                              AT(".", laKeys[m.lnKeys,3])+1)))
         IF cCOMMENTSEP$laKeys[m.lnKeys,4]
            laKeys[m.lnKeys,4] =;
               LEFT(laKeys[m.lnKeys,4],;
                    AT(cCOMMENTSEP,laKeys[m.lnKeys,4])-1)
         ENDIF
         laKeys[m.lnKeys,5] = SUBSTR(laKeys[m.lnKeys,4],;
                                     AT("(",laKeys[m.lnKeys,4])+1, 1)
         laKeys[m.lnKeys,4] =;
            PADR(LEFT(laKeys[m.lnKeys,4],;
                      AT("(",laKeys[m.lnKeys,4])-1),;
                 FSIZE("ForColName","SysKey"))
         laKeys[m.lnKeys,3] =;
            UPPER(TRIM(SUBSTR(laKeys[m.lnKeys,3],;
                              AT(cDOMAINSEP, laKeys[m.lnKeys,3])+;
                              LEN(cDOMAINSEP))))
         laKeys[m.lnKeys,3] =;
            PADR(LEFT(laKeys[m.lnKeys,3],;
                      AT(".", laKeys[m.lnKeys,3])-1),;
                 FSIZE("ForTabName","SysKey"))

         * Build the key expression array with one row for
         * each part of a foreign key.
         IF NOT EMPTY(laKeyExp[1, 1])
            DIMENSION laKeyExp[ALEN(laKeyExp,1)+1, 5]
         ENDIF
         laKeyExp[ALEN(laKeyExp,1), 1] = laKeys[m.lnKeys,1]+;
                               STR(laKeys[m.lnKeys,2],2) && type+segment
         laKeyExp[ALEN(laKeyExp,1), 2] = UPPER(TRIM(SysCat.ColumnName))
         laKeyExp[ALEN(laKeyExp,1), 3] = UPPER(SysCat.ColumnType)
         laKeyExp[ALEN(laKeyExp,1), 4] = TRIM(laKeys[m.lnKeys,3])
         laKeyExp[ALEN(laKeyExp,1), 5] = TRIM(laKeys[m.lnKeys,4])
      ELSE  && not a foreign key
         STORE "" TO laKeys[m.lnKeys,3], laKeys[m.lnKeys,4],;
                     laKeys[m.lnKeys,5]

         * Build the key expression array with one row for
         * each part of a primary or alternate key.
         IF NOT EMPTY(laKeyExp[1, 1])
            DIMENSION laKeyExp[ALEN(laKeyExp,1)+1, 5]
         ENDIF
         laKeyExp[ALEN(laKeyExp,1), 1] =;
            laKeys[m.lnKeys,1]+STR(laKeys[m.lnKeys,2],2) && type+segment
         laKeyExp[ALEN(laKeyExp,1), 2] = UPPER(TRIM(SysCat.ColumnName))
         laKeyExp[ALEN(laKeyExp,1), 3] = UPPER(SysCat.ColumnType)
      ENDIF
      laKeys[m.lnKeys,1] = LEFT(laKeys[m.lnKeys,1], 1)

      SELECT SysKey   && six-part primary key
      LOCATE FOR TableName  = m.lcTableName  AND;
                 ColumnName = UPPER(SysCat.ColumnName) AND;
                 KeyType    = laKeys[m.lnKeys,1] AND;
                 KeySegment = laKeys[m.lnKeys,2] AND;
                 ForTabName = laKeys[m.lnKeys,3] AND;
                 ForColName = laKeys[m.lnKeys,4]
      IF NOT FOUND("SysKey")
         INSERT INTO SysKey;
               (TableName,   ColumnName,  ForTabName,;
                ForColName,  KeyType,     KeySegment,;
                KeyRule,     UserId,      CreateDate,;
                CreateTime,  RefDate,     RefTime);
            VALUES;          
               (UPPER(SysCat.TableName),;
                   UPPER(SysCat.ColumnName),;
                      laKeys[m.lnKeys,3],;
                laKeys[m.lnKeys,4],;
                   laKeys[m.lnKeys,1],    laKeys[m.lnKeys,2],;
                laKeys[m.lnKeys,5],;
                   IIF(EMPTY(m.pcUserId),cAUTOUSERID,TRIM(m.pcUserId)),;
                      DATE(),;
                m.lnSeconds, DATE(),      m.lnSeconds)
      ENDIF
      SELECT SysCat
   ENDFOR  && each key expression in laKeyExp
ENDSCAN  && each designated key defined in SysCat for one table

* Put the entire key expression into segment 1's SysKey.Detail.
=ASORT(laKeyExp, 1)  && ascending on KeyType+KeySegment
FOR lnKeyExp = 1 TO ALEN(laKeyExp,1)
   lcKeyExp = ""
   lnOffset = m.lnKeyExp
   * One loop here for all segments of each different key.
   DO WHILE .T.  && bottom test
      DO CASE
         CASE ALEN(laKeyExp,1)==1 OR;
              laKeyExp[m.lnOffset,3]=="C"
            lcKeyExp = m.lcKeyExp+;
                       m.lcAliasName+"."+laKeyExp[m.lnOffset,2]+"+"
         CASE laKeyExp[m.lnOffset,3]$"NF"
            lcKeyExp = m.lcKeyExp + "STR("+;
                       m.lcAliasName+"."+laKeyExp[m.lnOffset,2]+")+"
         CASE laKeyExp[m.lnOffset,3]=="D"
            lcKeyExp = m.lcKeyExp + "DTOS("+;
                       m.lcAliasName+"."+laKeyExp[m.lnOffset,2]+")+"
         CASE laKeyExp[m.lnOffset,3]=="L"
            lcKeyExp = m.lcKeyExp + "IIF("+;
                       m.lcAliasName+"."+laKeyExp[m.lnOffset,2]+;
                                  ", '.T.', '.F.')+"
      ENDCASE
      * Test the DO loop condition.
      IF m.lnOffset==ALEN(laKeyExp,1) OR;
            VAL(RIGHT(laKeyExp[m.lnOffset+1,1],1))==1
         EXIT
      ELSE  && increment loop counter if staying in
         lnOffset = m.lnOffset+1
      ENDIF
   ENDDO
   lcKeyExp = LEFT(m.lcKeyExp, LEN(m.lcKeyExp)-1) && remove extra plus sign

   SELECT SysKey
   IF LEFT(laKeyExp[m.lnKeyExp,1],1)=="F"  && foreign
      LOCATE FOR SysKey.TableName=m.lcTableName AND;
                 SysKey.ColumnName=;
                   PADR(laKeyExp[m.lnKeyExp,2],;
                        FSIZE("ColumnName","SysKey")) AND;
                 SysKey.ForTabName=;
                   PADR(laKeyExp[m.lnKeyExp,4],;
                        FSIZE("ForTabName","SysKey")) AND;
                 SysKey.ForColName=;
                   PADR(laKeyExp[m.lnKeyExp,5],;
                        FSIZE("ForColName","SysKey")) AND;
                 SysKey.KeyType=LEFT(laKeyExp[m.lnKeyExp,1],1) AND;
                 SysKey.KeySegment=1
   ELSE  && primary, alternate
      LOCATE FOR SysKey.TableName=m.lcTableName AND;
                 SysKey.KeyType=LEFT(laKeyExp[m.lnKeyExp,1],1) AND;
                 SysKey.KeySegment=1
   ENDIF
   IF FOUND("SysKey")
      REPLACE SysKey.Detail WITH m.lcKeyExp
   ENDIF

   * Increment FOR's counter by offset from DO loop.
   lnKeyExp = m.lnOffset
ENDFOR

SELECT (m.lcAliasName)
RETURN
*** KeyUpdate() *********************************************


FUNCTION IndUpdate
*  Syntax: DO IndUpdate WITH <C file name>
*        :                   [,<C user id name>
*        :                    [,<C import expression>]]
*  Action: Update SysInd entry from an existing index.
*  Return: Nothing
*   Notes: Called by Xcreate, Ximport, and XTable.  Expects table open.
*        : Changes select area to that of passed file name.
*
PARAMETERS pcFileName, pcUserId, pcImportExp
PRIVATE lcFileName, lcAliasName, laTableCount, laMapData, laDelete,;
        lnSeconds, lcTableName

* Local compiler directives.
* laMapData[] array constants.
#define cMAXINDDATA  5
#define cTAGNAME     1
#define cFILENAME    2
#define cINDKEYEXP   3
#define cINDFOREXP   4
#define cDIRECTION   5

lcFileName = IIF("\"$m.pcFileName,;
                    SUBSTR(m.pcFileName, RAT("\",m.pcFileName)+1),;
                    TRIM(m.pcFileName))  && trim path
IF NOT "."$m.lcFileName
   lcFileName = m.lcFileName+".DBF"   && default file extension
ENDIF
lcAliasName = UPPER(LEFT(m.lcFileName, AT(".",m.lcFileName)-1))
lcTableName = PADR(m.lcAliasName, FSIZE("TableName","SysTab"))

* Get TableName from SysTab for this alias name, or
* create a SysTab entry if one doesn't exist.
SELECT COUNT(*) FROM SysTab INTO ARRAY laTableCount;
   WHERE SysTab.TableName = m.lcTableName
IF _TALLY==0
   DO TabUpdate WITH lcFileName
   SELECT COUNT(*) FROM SysTab INTO ARRAY laTableCount;
      WHERE SysTab.TableName = m.lcTableName
   IF _TALLY==0  && should never happen
      DO ErrorHandler WITH "table creation failed", LINENO()
      RETURN
   ENDIF
ENDIF

IF EMPTY(pcImportExp)  && default to a compact-structural index
   pcImportExp = "CDX"
ENDIF

DO CASE
CASE UPPER(RIGHT(m.pcImportExp, 3))=="IDX"
   * Set the individual index file to the table.
   SET INDEX TO (IIF("."$m.pcImportExp, m.pcImportExp, m.lcAliasName))
   WAIT WINDOW NOWAIT;
      PADR(PROPER(m.lcAliasName),10)+"."+;
      PADR(PROPER(NDX(1)),       10)+" --> "+;
      PADR("SysInd",             10)+"."+;
      PADR("TagName",            10)
               
   * Get the current tag info in an array.
   DIMENSION laMapData[cMAXINDDATA]
   laMapData[cTAGNAME]   = NDX(1)
   laMapData[cFILENAME]  = SUBSTR(NDX(1), RAT("\", NDX(1))+1)
   laMapData[cINDKEYEXP] = KEY(1)
   laMapData[cINDFOREXP] = SYS(2021, SELECT())
   laMapData[cDIRECTION] = IIF(" DESCENDING"$SET("ORDER"), "D", "A")

   * Check to see if an entry already exists.
   SELECT SysInd
   LOCATE FOR SysInd.TableName = m.lcTableName AND;
              SysInd.IndexName = m.lcTableName AND;
              SysInd.TagName   = laMapData[cTAGNAME]
   * Update existing record, or add a new one.
   lnSeconds = INT(SECONDS())  && for time consistency
   IF FOUND("SysInd")
      REPLACE SysInd.FileName   WITH laMapData[cFILENAME],;
              SysInd.IndexType  WITH "T",;
              SysInd.ExpKey     WITH laMapData[cINDKEYEXP],;
              SysInd.ExpFor     WITH laMapData[cINDFOREXP],;
              SysInd.OrderDir   WITH laMapData[cDIRECTION],;
              SysInd.UserId     WITH IIF(EMPTY(m.pcUserId), cAUTOUSERID,;
                                         TRIM(m.pcUserId)),;
              SysInd.RefDate    WITH DATE(),;
              SysInd.RefTime    WITH m.lnSeconds
   ELSE
      INSERT INTO SysInd;
            (IndexName,     TableName,     TagName,;
             FileName,      IndexType,     ExpKey,;
             ExpFor,        OrderDir,      UserId,;
             CreateDate,    CreateTime,    RefDate,;
             RefTime);
         VALUES;
            (m.lcAliasName, m.lcAliasName, laMapData[cTAGNAME],;
             laMapData[cFILENAME], "T",    laMapData[cINDKEYEXP],;
             laMapData[cINDFOREXP],;
                laMapData[cDIRECTION],;
                   IIF(EMPTY(m.pcUserId),cAUTOUSERID,TRIM(m.pcUserId)),;
             DATE(),       m.lnSeconds,    DATE(),;
             m.lnSeconds)
   ENDIF
   SELECT (m.lcAliasName)

CASE UPPER(RIGHT(m.pcImportExp, 3))=="CDX"
   * Reset the compound index file if it's not the same
   * name as the open table.
   IF "."$m.pcImportExp AND NOT;
         UPPER(LEFT(m.pcImportExp,;
               AT(".", m.pcImportExp)-1))=;
         UPPER(m.pcImportExp))
      SET INDEX TO (TRIM(m.pcImportExp))
   ENDIF

   * Insert one record for each CDX and each tag in the CDX.
   lnCDX = 1

   IF EMPTY(CDX(m.lnCDX))
      * Remove any SysInd entries for a compound structural index.
      SELECT "SYSIND    ",;
             UPPER(SysInd.TableName+SysInd.IndexName+SysInd.TagName), "";
         FROM SysInd;
         INTO ARRAY laDelete;
         WHERE SysInd.FileName=m.lcAliasName+".CDX" AND;
               (SysInd.IndexType="C" OR SysInd.IndexType="T")
      IF _TALLY>0
         IF NOT Xdelete(@laDelete)
            DO ErrorHandler WITH "deleting in SysInd", LINENO()
         ENDIF
      ENDIF
   ENDIF

   DO WHILE NOT EMPTY(CDX(m.lnCDX))
      * Check to see if an entry already exists for this CDX.
      SELECT SysInd
      LOCATE FOR SysInd.TableName = m.lcTableName AND;
                 SysInd.IndexName = m.lcTableName AND;
                 SysInd.IndexType = "C"
      lnSeconds = INT(SECONDS())  && for time consistency in each tag
      * Add a new record if none exists (no update existing).
      IF NOT FOUND("SysInd")
         INSERT INTO SysInd;
               (IndexName,     TableName,     FileName,;
                IndexType,     UserId,        CreateDate,;
                CreateTime,    RefDate,       RefTime);
            VALUES;
               (m.lcAliasName, m.lcAliasName, m.lcAliasName+".CDX",;
                "C",;
                   IIF(EMPTY(m.pcUserId),cAUTOUSERID,TRIM(m.pcUserId)),;
                      DATE(),;
                m.lnSeconds,     DATE(),        m.lnSeconds)
      ENDIF
      SELECT (m.lcAliasName)

      lnTags = 1
      DO WHILE NOT EMPTY(TAG(CDX(m.lnCDX), m.lnTags))
         WAIT WINDOW NOWAIT;
            PADR(PROPER(m.lcAliasName),              10)+"."+;
            PADR(PROPER(TAG(CDX(m.lnCDX), m.lnTags)),10)+;
            " --> "+;
            PADR("SysInd",                           10)+"."+;
            PADR("TagName",                          10)

         * Get the current tag info in an array.
         SET ORDER TO (TAG(CDX(m.lnCDX), m.lnTags))
         DIMENSION laMapData[cMAXINDDATA]
         laMapData[cTAGNAME]   = TAG(CDX(m.lnCDX), m.lnTags)
         laMapData[cFILENAME]  = SUBSTR(CDX(m.lnCDX),;
                                        RAT("\", CDX(m.lnCDX))+1)
         laMapData[cINDKEYEXP]    = KEY(CDX(m.lnCDX), m.lnTags)
         laMapData[cINDFOREXP]    = SYS(2021, SELECT())
         laMapData[cDIRECTION] = IIF(" DESCENDING"$SET("ORDER"), "D", "A")

         * Check to see if an entry already exists for this tag.
         SELECT SysInd
         LOCATE FOR SysInd.TableName = m.lcTableName AND;
                    SysInd.IndexName = m.lcTableName AND;
                    SysInd.TagName   = laMapData[cTAGNAME] AND;
                    SysInd.IndexType = "T"
         * Update existing record, or add a new one.
         IF FOUND("SysInd")
            REPLACE SysInd.FileName   WITH laMapData[cFILENAME],;
                    SysInd.ExpKey     WITH laMapData[cINDKEYEXP],;
                    SysInd.ExpFor     WITH laMapData[cINDFOREXP],;
                    SysInd.OrderDir   WITH laMapData[cDIRECTION],;
                    SysInd.UserId     WITH IIF(EMPTY(m.pcUserId),;
                                               cAUTOUSERID,;
                                               TRIM(m.pcUserId)),;
                    SysInd.RefDate    WITH DATE(),;
                    SysInd.RefTime    WITH m.lnSeconds
         ELSE
            INSERT INTO SysInd;
                  (IndexName,    TableName,    TagName,;
                   FileName,     IndexType,    ExpKey,;
                   ExpFor,       OrderDir,     UserId,;
                   CreateDate,   CreateTime,   RefDate,;
                   RefTime);
               VALUES;
                  (m.lcAliasName,;
                      m.lcAliasName,;
                         laMapData[cTAGNAME],;
                   laMapData[cFILENAME], "T",  laMapData[cINDKEYEXP],;
                   laMapData[cINDFOREXP], laMapData[cDIRECTION],;
                         IIF(EMPTY(m.pcUserId),;
                             cAUTOUSERID, TRIM(m.pcUserId)),;
                   DATE(),       m.lnSeconds,   DATE(),;
                   m.lnSeconds)
         ENDIF
         SELECT (m.lcAliasName)
         lnTags = m.lnTags+1
      ENDDO  && tags in one compound index file

      * Remove SysInd tag entries no longer valid for this compound index.
      SELECT "SYSIND    ",;
             UPPER(SysInd.TableName+SysInd.IndexName+SysInd.TagName), "";
         FROM SysInd;
         INTO ARRAY laDelete;
         WHERE SysInd.FileName=m.lcAliasName+".CDX" AND;
               SysInd.IndexType="T" AND;
               NOT DTOS(SysInd.RefDate)+STR(SysInd.RefTime,5)=;
                   DTOS(DATE())+STR(m.lnSeconds,5)
      IF _TALLY>0
         IF NOT Xdelete(@laDelete)
            DO ErrorHandler WITH "deleting in SysInd", LINENO()
         ENDIF
      ENDIF
      lnCDX = m.lnCDX+1
   ENDDO  && compound index files

OTHERWISE
   DO ErrorHandler WITH "invalid import expression", LINENO()
ENDCASE
RETURN
*** IndUpdate() *********************************************


FUNCTION IndCreate
*  Syntax: DO IndCreate WITH <C file name>
*  Action: Create an index file from SysKey definitions.
*  Return: Nothing
*   Notes: Called by Xcreate.  Expects table and SysKey open.
*        : Changes select area to that of passed file name.
*
PARAMETERS pcFileName
PRIVATE lcFileName, lcAliasName, laTableCount, laKeys, lnKeys,;
        lcTableName

* Precompiler directives.
#define cPRIKEYTAGNAME "PrimaryKey"

lcFileName = IIF("\"$m.pcFileName,;
                    SUBSTR(m.pcFileName, RAT("\",m.pcFileName)+1),;
                    TRIM(m.pcFileName))  && trim path
IF NOT "."$m.lcFileName
   lcFileName = m.lcFileName+".DBF"   && default file extension
ENDIF
lcAliasName = UPPER(LEFT(m.lcFileName, AT(".",m.lcFileName)-1))
lcTableName = PADR(m.lcAliasName, FSIZE("TableName","SysTab"))

SELECT (m.lcAliasName)

* Get TableName from SysTab for this alias name, or
* return an error if one doesn't exist.
SELECT COUNT(*) FROM SysTab INTO ARRAY laTableCount;
   WHERE SysTab.TableName = m.lcTableName
IF _TALLY==0
   DO ErrorHandler WITH "table not in SysTab", LINENO()
   RETURN  && early exit
ENDIF

* Get all individual key entries (by column) from SysKey for this table.
SELECT DISTINCT SysKey.ColumnName;
  FROM SysKey;
  INTO ARRAY laKeys;
  WHERE SysKey.TableName = m.lcTableName;
  ORDER BY SysKey.ColumnName

IF _TALLY==0
   DO ErrorHandler WITH m.lcAliasName+" not in SysKey", LINENO()
   RETURN  && early exit
ENDIF

* Create one index tag (ascending) for every column this table
* has in laKeys[].
FOR lnKeys = 1 TO ALEN(laKeys,1)
   WAIT WINDOW NOWAIT;
      PADR(PROPER(m.lcAliasName),      10)+"."+;
      PADR(PROPER(laKeys[m.lnKeys,1]), 10)+" --> "+;
      PADR(PROPER(m.lcAliasName),      10)+"."+;
      PADR("CDX",                      10)
   INDEX ON &laKeys[m.lnKeys,1];
        TAG (TRIM(laKeys[m.lnKeys,1]))  && macro alert
ENDFOR

* Get primary key entries for a compound primary key index tag.
SELECT SysKey.Detail;
  FROM SysKey;
  INTO ARRAY laKeys;
  WHERE SysKey.TableName = m.lcTableName AND;
        SysKey.TableName IN (SELECT DISTINCT SysKey.TableName;
                               FROM SysKey;
                              WHERE SysKey.KeySegment>1) AND;
        SysKey.KeyType="P" AND SysKey.KeySegment=1 AND;
        NOT EMPTY(SysKey.Detail)
IF _TALLY>0
   * STRTRAN() doesn't always work in SQL SELECT.
   laKeys[1] = STRTRAN(laKeys[1], m.lcAliasName+".")
   * Create one index tag (character, ascending) for primary key.
   INDEX ON &laKeys[1] TAG (cPRIKEYTAGNAME)   && macro alert
ENDIF

CLEAR TYPEAHEAD  && from INDEX overwrite warnings, if any
SET ORDER TO
RETURN
*** IndCreate() *********************************************


FUNCTION Xdelete
*  Syntax: Xdelete(<A array of primary key values>)
*  Action: Create an index file from SysKey definitions.
*  Return: <expL> false if any deletions failed, otherwise true.
*   Notes: The passed array is a two dimension array
*        :    with three columns, passed by reference.
*        :    All character expressions must be uppercase.
*        : Column 1 contains the tablename, padded to
*        :    SysTab.TableName width.  Column 2 contains the
*        :    primary key expression to delete.  All rows of
*        :    Column 1 must contain the same table name.
*        : This routine sets the third column to <expC>
*        :    representing the results of the delete action taken.
*        :    Possible action results are D=deleted, R=restricted,
*        :    L=lock failed, S=search failed, E=error.
*        : May open a lot of files and leave them open.
*        : Called from TabUpdate, ColUpdate, IndUpdate (so far)
*        : No subroutine calls!
*
PARAMETERS paDelete
EXTERNAL ARRAY paDelete
PRIVATE llRet, lnDelete, laKeys, lnKeys, lnWasSelect,;
        llSetExactOn, laDelete, lnDelete, laCascade, laKeyExp,;
        laTables, lnTables

* paDelete[] array constants, two dimensional, dynamic.  Columns only.
#define cMAXDELCOL  3
#define cDELTABLE   1
#define cDELKEYEXP  2
#define cDELRESULT  3

* laKeys[] array constants, two dimensional, dynamic.
* Order must match SQL SELECT statements.  These are columns only.
#define cMAXLAKEYS  8
#define cKEYTABLE   1
#define cKEYEXP     2
#define cKEYRULE    3
#define cFORTABLE   4
#define cFORKEYEXP  5
#define cBAKPTR     6
#define cFWDPTR     7
#define cDELETEFLAG 8

* Get primary key expression from SysKey.  Only the data was
* passed, so we need to know what expression made up the data.
llRet = USED("SysKey")
SELECT SysKey.Detail;
  FROM SysKey;
  INTO ARRAY laKeyExp;
  WHERE SysKey.TableName = paDelete[1,cDELTABLE] AND;
        SysKey.KeyType="P" AND SysKey.KeySegment=1
IF _TALLY==0
   DO ErrorHandler WITH "primary key not in SysKey", LINENO()
   IF NOT m.llRet
      USE IN SysKey
   ENDIF
   paDelete[1,cDELRESULT] = "E"
   RETURN .F.  && early exit
ENDIF

lnWasSelect = SELECT()
llSetExactOn = SET("EXACT")=="ON"
SET EXACT ON  && for ASCAN(), scope, and all comparisons (not SQL SELECT)

IF USED(TRIM(paDelete[1,cDELTABLE]))
   SELECT (TRIM(paDelete[1,cDELTABLE]))
   SET ORDER TO 0  && for fastest Rushmore searching
ELSE
   SELECT 0
   USE (TRIM(paDelete[1,cDELTABLE]))
ENDIF

llRet = .T.  && until set false by one failed delete action

* Build a list of all foreign keys to this table and
* all foreign keys to those foreign keys, cascading.
* No data checking here; we're just getting the list.
SELECT DISTINCT SysKey.ForTabName, "", SysKey.KeyRule,;
                SysKey.TableName, PADR(SysKey.Detail,250), 0, 0, .F.;
   FROM SysKey;
   INTO ARRAY laKeys;
   WHERE SysKey.ForTabName=paDelete[1,cDELTABLE] AND;
         SysKey.KeyType="F" AND SysKey.KeySegment=1 AND;
         NOT SysKey.KeyRule="I"
IF _TALLY==0
   DECLARE laKeys[1,cMAXLAKEYS]
   laKeys = ""
ELSE
   * Trim the foreign key expression which was padded to make
   * an expression so that DISTINCT will work on memo fields.
   FOR lnKeys = 1 TO ALEN(laKeys,1)
      laKeys[m.lnKeys,cFORKEYEXP] =;
         TRIM(f_aKeys[m.lnKeys,cFORKEYEXP])
   ENDFOR
   lnKeys = 1   && laKeys[] is dynamic, DO reevaluates every time
   DO WHILE m.lnKeys<=ALEN(laKeys,1)
      IF laKeys[m.lnKeys,cKEYRULE]=="C"
         SELECT DISTINCT SysKey.ForTabName, "", SysKey.KeyRule,;
                         SysKey.TableName, PADR(SysKey.Detail,250);
            FROM SysKey;
            INTO ARRAY laCascade;
            WHERE SysKey.ForTabName=laKeys[m.lnKeys,cFORTABLE] AND;
                  SysKey.KeyType="F" AND SysKey.KeySegment=1 AND;
              NOT SysKey.KeyRule="I"
         IF _TALLY>0
            FOR lnCascade = 1 TO ALEN(laCascade,1)
               * Add the cascading table unless it's already
               * present and this one is not restricted.
               llAdd = .T.
               FOR lnKeys2 = 1 TO ALEN(laKeys,1)
                  IF laKeys[m.lnKeys2,cFORTABLE]==;
                        laCascade[m.lnCascade,cFORTABLE] AND;
                        laKeys[m.lnKeys2,cKEYRULE]$"CR" AND;
                        NOT laCascade[m.lnCascade,cKEYRULE]=="R"
                     llAdd = .F.
                     EXIT
                  ENDIF
               ENDFOR
               IF m.llAdd  && add this table to the foreign key array
                  * Get primary key expression of foreign key table.
                  SELECT SysKey.Detail;
                    FROM SysKey;
                    INTO ARRAY laPriKeyExp;
                    WHERE SysKey.TableName=laCascade[m.lnCascade,cKEYTABLE];
                      AND SysKey.KeyType="P" AND SysKey.KeySegment=1
                  DIMENSION laKeys[ALEN(laKeys,1)+1,cMAXLAKEYS]
                  laKeys[ALEN(laKeys,1),cKEYTABLE] =;
                     laCascade[m.lnCascade,cKEYTABLE]
                  laKeys[ALEN(laKeys,1),cKEYEXP] = laPriKeyExp[1]
                  laKeys[ALEN(laKeys,1),cFORTABLE] =;
                     laCascade[m.lnCascade,cFORTABLE]
                  laKeys[ALEN(laKeys,1),cFORKEYEXP] =;
                     TRIM(laCascade[m.lnCascade,cFORKEYEXP])
                  laKeys[ALEN(laKeys,1),cKEYRULE] =;
                     laCascade[m.lnCascade,cKEYRULE]
                  laKeys[ALEN(laKeys,1),cBAKPTR] = m.lnKeys
                  laKeys[ALEN(laKeys,1),cFWDPTR] = 0
                  laKeys[m.lnKeys,cFWDPTR] = ALEN(laKeys,1)
               ENDIF  && add to the foreign key array
            ENDFOR  && cascading tables
         ENDIF  && cascading tables
      ENDIF  && marked to cascade
      lnKeys = m.lnKeys+1
   ENDDO  && cascading foreign key list (dynamic)
ENDIF  && any foreign keys

* Loop through passed array.
SELECT (TRIM(paDelete[1,cDELTABLE]))
FOR lnDelete = 1 TO ALEN(paDelete,1)
   * Search for primary key data.  Macro alert.
   LOCATE FOR &laKeyExp[1] = paDelete[m.lnDelete,cDELKEYEXP]
   IF NOT FOUND()
      paDelete[m.lnDelete,cDELRESULT] = "S"
      LOOP
   ENDIF

   * Check for restricted data.
   IF ASCAN(laKeys, "R")>0
      * Traverse laKeys from bottom to top for restricted key rules.
      llRestrict = .F.
      FOR lnKeys = ALEN(laKeys,1) TO 1 STEP -1
         IF NOT laKeys[m.lnKeys,cKEYRULE]=="R"
            LOOP
         ENDIF
         * Traverse laKeys back to top using the backward pointer
         * to build the search lists.
         STORE "" TO lcTableList, lcJoinCondition
         lnKeys2 = m.lnKeys
         DO WHILE m.lnKeys2>0
            lcTableList = m.lcTableList+;
                          TRIM(laKeys[m.lnKeys2,cFORTABLE])+;
                          IIF(laKeys[m.lnKeys2,cBAKPTR]==0,;
                              ","+TRIM(laKeys[m.lnKeys2,cKEYTABLE]),;
                              ",")
            lcJoinCondition = m.lcJoinCondition+;
                              IIF(laKeys[m.lnKeys2,cBAKPTR]==0,;
                                  laKeys[m.lnKeys2,cFORKEYEXP]+"="+;
                                     laKeyExp[1],;
                                  laKeys[m.lnKeys2,cFORKEYEXP]+"="+;
                                     laKeys[m.lnKeys2,cKEYEXP]+" AND ")
            lnKeys2 = laKeys[m.lnKeys2,cBAKPTR]
         ENDDO  && cascade chain
         * Search for restricted data in the chain.
         SELECT COUNT(*);
            FROM &lcTableList;
            INTO ARRAY laRestrict;
            WHERE &lcJoinCondition  && double macro alert
         IF _TALLY>0 AND laRestrict[1]>0
            llRestrict = .T.
            EXIT
         ENDIF
      ENDFOR  && next restricted keyrule
      IF m.llRestrict
         paDelete[m.lnDelete,cDELRESULT] = "R"
         llRet = .F.
         LOOP  && next primary key value in passed array
      ENDIF
   ENDIF  && restricted data

   * Delete action: "C"=delete record, "N"=nullify foreign key columns.
   * Traverse laKeys from top to bottom for delete action.
   FOR lnKeys = 1 TO ALEN(laKeys,1)
      DO CASE
         CASE NOT laKeys[m.lnKeys,cKEYTABLE]==;
                  paDelete[m.lnDelete,cDELTABLE]
            * No foreign keys, or the rest of laKeys[] cascade
            * from foreign keys.
            EXIT
         CASE NOT laKeys[m.lnKeys,cKEYRULE]$"CN"
            LOOP  && restrictions were already processed
      ENDCASE

      * Open the files to locate the data.
      IF NOT USED(TRIM(laKeys[m.lnKeys,cKEYTABLE]))
         USE (TRIM(laKeys[m.lnKeys,cKEYTABLE])) IN SELECT(1)
      ENDIF
      IF NOT USED(TRIM(laKeys[m.lnKeys,cFORTABLE]))
         USE (TRIM(laKeys[m.lnKeys,cFORTABLE])) IN SELECT(1)
      ENDIF

      * Search and mark for deletion.
      lcJoinCondition = laKeys[m.lnKeys,cFORKEYEXP]+"="+laKeyExp[1]
      SELECT (TRIM(laKeys[m.lnKeys,cFORTABLE]))
      LOCATE FOR &lcJoinCondition  && macro alert
      DO WHILE FOUND()
         IF laKeys[m.lnKeys,cKEYRULE]=="C"
            DELETE
            laKeys[m.lnKeys,cDELETEFLAG] = .T.  && for packing later

            * Traverse laKeys[] forward for cascading entries.
            lnKeys2 = laKeys[m.lnKeys,cFWDPTR]
            DO WHILE m.lnKeys2>0
               * Open the files to locate the data.
               IF NOT USED(TRIM(laKeys[m.lnKeys2,cKEYTABLE]))
                  USE (TRIM(laKeys[m.lnKeys2,cKEYTABLE])) IN SELECT(1)
               ENDIF
               IF NOT USED(TRIM(laKeys[m.lnKeys2,cFORTABLE]))
                  USE (TRIM(laKeys[m.lnKeys2,cFORTABLE])) IN SELECT(1)
               ENDIF

               * Search.
               lcJoinCondition = laKeys[m.lnKeys2,cFORKEYEXP]+"="+;
                                 laKeys[m.lnKeys2,cKEYEXP]
               SELECT (TRIM(laKeys[m.lnKeys2,cFORTABLE]))
               LOCATE FOR &lcJoinCondition  && macro alert
               DO WHILE FOUND()
                  IF laKeys[m.lnKeys2,cKEYRULE]=="C"
                     DELETE
                     laKeys[m.lnKeys2,cDELETEFLAG] = .T.
                  ELSE  && nullify found data
                     SELECT SysCol.ColumnName, SysCol.ColumnType,;
                            SysCol.Detail;
                       FROM SysCol, SysKey;
                       INTO ARRAY laCascade;
                       WHERE SysCol.TableName=SysKey.TableName AND;
                             SysCol.ColumnName=SysKey.ColumnName AND;
                             SysKey.TableName=;
                                laKeys[m.lnKeys2,cFORTABLE] AND;
                             SysKey.ForTabName=;
                                laKeys[m.lnKeys2,cKEYTABLE] AND;
                             SysKey.KeyType="F"
                     FOR lnCascade = 1 TO ALEN(laCascade,1)
                        REPLACE (laCascade[m.lnCascade,1]) WITH;
                           IIF(NOT EMPTY(laCascade[m.lnCascade,3]),;
                               EVALUATE(TRIM(laCascade[m.lnCascade,3])),;
                               IIF(laCascade[m.lnCascade,2]=="C", "",;
                                IIF(laCascade[m.lnCascade,2]$"NF", 0,;
                                 IIF(laCascade[m.lnCascade,2]=="D", {},;
                                  IIF(laCascade[m.lnCascade,2]=="L", .F.,;
                                      "null error")))))
                     ENDFOR
                  ENDIF
                  CONTINUE
               ENDDO  && data search
               lnKeys2 = laKeys[m.lnKeys2,cFWDPTR]
            ENDDO  && cascade chain
            SELECT (TRIM(laKeys[m.lnKeys,cFORTABLE]))
         ELSE  && nullify found data
            SELECT SysCol.ColumnName, SysCol.ColumnType, SysCol.Detail;
              FROM SysCol, SysKey;
              INTO ARRAY laCascade;
              WHERE SysCol.TableName=SysKey.TableName AND;
                    SysCol.ColumnName=SysKey.ColumnName AND;
                    SysKey.TableName=laKeys[m.lnKeys,cFORTABLE] AND;
                    SysKey.ForTabName=laKeys[m.lnKeys,cKEYTABLE] AND;
                    SysKey.KeyType="F"
            FOR lnCascade = 1 TO ALEN(laCascade,1)
               REPLACE (laCascade[m.lnCascade,1]);
                  WITH IIF(NOT EMPTY(laCascade[m.lnCascade,3]),;
                           EVALUATE(TRIM(laCascade[m.lnCascade,3])),;
                           IIF(laCascade[m.lnCascade,2]=="C", "",;
                            IIF(laCascade[m.lnCascade,2]$"NF", 0,;
                             IIF(laCascade[m.lnCascade,2]=="D", {},;
                              IIF(laCascade[m.lnCascade,2]=="L", .F.,;
                                  "null error")))))
            ENDFOR
         ENDIF  && cascade or nullify found data
         CONTINUE
      ENDDO  && foreign key data found
   ENDFOR  && cascading deletes

   * After all foreign deletes are handled, delete
   * the table row that was passed to us.
   SELECT (TRIM(paDelete[m.lnDelete,cDELTABLE]))
   DELETE
   paDelete[m.lnDelete,cDELRESULT] = "D"
ENDFOR  && rows of the passed array

* Remove cascading deleted records from the foreign table(s).
IF ASCAN(laKeys,.T.)>0
   DIMENSION laCascade[1]
   laCascade = ""
   FOR lnKeys = 1 TO ALEN(laKeys,1)
       IF laKeys[m.lnKeys,cDELETEFLAG] AND;
             ASCAN(laCascade,laKeys[m.lnKeys,cFORTABLE])==0
          SELECT (TRIM(laKeys[m.lnKeys,cFORTABLE]))
          PACK  && remove deleted records from the foreign table

          * Save table we packed in, so we don't repeat.
          laCascade[ALEN(laCascade,1)] = laKeys[m.lnKeys,cFORTABLE]
          DIMENSION laCascade[ALEN(laCascade,1)+1]
       ENDIF
   ENDFOR
ENDIF
* Remove deleted records from the passed table.
IF ASCAN(paDelete,"D")>0
   SELECT (TRIM(paDelete[1,cDELTABLE]))
   PACK
ENDIF

SELECT (m.lnWasSelect)
IF NOT m.llSetExactOn
   SET EXACT OFF
ENDIF
RETURN m.llRet
*** Xdelete() ***********************************************


FUNCTION NotImplemented
*  Syntax: NotImplemented(<C event name>)
*  Action: Display not implemented message
*  Return: <expL> false
*        :
PARAMETERS pcEventName
WAIT WINDOW NOWAIT PROPER(m.pcEventName)+;
                   " is not implemented yet"
RETURN .F.
*** NotImplemented() ****************************************


FUNCTION ErrorHandler
*  Syntax: DO ErrorHandler WITH <C error message>
*  Action: Display error message
*  Return: nothing
*        :
PARAMETERS pcErrorMessage, pnLineNo
PRIVATE lcResponse
WAIT WINDOW "Line "+LTRIM(STR(m.pnLineNo))+;
            " Error "+m.pcErrorMessage+;
            "...[S=SUSPEND]";
         TO lcResponse
IF m.lcResponse$"sS"
   SUSPEND
ENDIF
RETURN
*** ErrorHandler() ******************************************

*** EOF Xcatalog.prg ****************************************
