/*
**	tyield.c: source code for TYIELD.PLB Foxpro LCK library
**
**		RBWooster, February 1993
*/

#include "C:\FOXLCK\H\stdlib.h"
#include "C:\FOXLCK\H\math.h"	/* for pow(), floor() */
#include "C:\FOXLCK\H\pro_ext.h"
#include "datesubs.h"

#define BASEDATE 2415019L

#include "tsysubs.h"

#define TOLERANCE 1E-8

/************************************************************
*
*						FOXPRO LCK STUFF
*
************************************************************/

void FAR bd2p(	ParamBlk FAR *parm )
/* Treasury bill discount rate to price */
{
	double dval, dmat, disc, price;
	unsigned val, mat;

	dval  = parm->p[0].val.ev_real;
	dmat  = parm->p[1].val.ev_real;
	disc = parm->p[2].val.ev_real;
	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);

	price = billpx(val, mat, disc);
	_RetFloat( price, 20, 8);
}

void FAR bd2y(	ParamBlk FAR *parm )
/* Treasury bill discount to bond equivalent yield */
{
	double dval, dmat, disc, yield;
	unsigned val, mat;

	dval  = parm->p[0].val.ev_real;
	dmat  = parm->p[1].val.ev_real;
	disc  = parm->p[2].val.ev_real;
	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);

	yield = billbey(val, mat, disc);
	_RetFloat( yield, 20, 8);
}

void FAR ny2p(	ParamBlk FAR *parm )
/* Treasury note/bond yield to price */
{
	double dval, dmat, disu, coupon, yield, price;
	unsigned val, mat, isu;

	dval  = parm->p[0].val.ev_real;
	dmat  = parm->p[1].val.ev_real;
	disu  = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	yield  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	price = notey2p(val, mat, isu, coupon, yield);
	_RetFloat( price, 20, 8);
}

void FAR np2y(	ParamBlk FAR *parm )
/* Treasury note/bond price to yield */
{
	double dval, dmat, disu, coupon, price, yield;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	price  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	yield = notep2y(val, mat, isu, coupon, price);
	_RetFloat( yield, 20, 8);
}

void FAR zy2p(	ParamBlk FAR *parm )
/* Treasury strip yield to price */
{
	double dval, dmat, yield, price;
	unsigned val, mat;

	dval  = parm->p[0].val.ev_real;
	dmat  = parm->p[1].val.ev_real;
	yield = parm->p[2].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);

	price = zeroy2p(val, mat, yield);
	_RetFloat( price, 20, 8);
}

void FAR zp2y(	ParamBlk FAR *parm )
/* Treasury strip price to yield */
{
	double dval, dmat, price, yield;
	unsigned val, mat;

	dval  = parm->p[0].val.ev_real;
	dmat  = parm->p[1].val.ev_real;
	price = parm->p[2].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);

	yield = zerop2y(val, mat, price);
	_RetFloat( yield, 20, 8);
}

void FAR ntai(	ParamBlk FAR *parm )
/* accrued interest on a Tsy note */
{
	double dval, dmat, disu, coupon, ai;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	ai = accdInt(val, mat, isu, coupon);
	_RetFloat( ai, 20, 8);
}

void FAR mdur(	ParamBlk FAR *parm )
/* Modified duration */
{
	double dval, dmat, disu, coupon, yield, duration;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	yield  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	duration = modDur(val, mat, isu, coupon, yield);
	_RetFloat( duration, 20, 8);
}

void FAR cvxy(	ParamBlk FAR *parm )
/* Convexity */
{
	double dval, dmat, disu, coupon, yield, cnvxty;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	yield  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	cnvxty = convexity(val, mat, isu, coupon, yield);
	_RetFloat( cnvxty, 20, 8);
}

void FAR yv32(	ParamBlk FAR *parm )
/* Convexity */
{
	double dval, dmat, disu, coupon, price, retval;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	price  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	retval = yval32nd(val, mat, isu, coupon, price);
	_RetFloat( retval, 20, 8);
}

void FAR dv01(	ParamBlk FAR *parm )
/* Convexity */
{
	double dval, dmat, disu, coupon, yield, retval;
	unsigned val, mat, isu;

	dval   = parm->p[0].val.ev_real;
	dmat   = parm->p[1].val.ev_real;
	disu   = parm->p[2].val.ev_real;
	coupon = parm->p[3].val.ev_real;
	yield  = parm->p[4].val.ev_real;

	val = (unsigned)((long)floor(dval)-BASEDATE);
	mat = (unsigned)((long)floor(dmat)-BASEDATE);
	isu = (unsigned)((long)floor(disu)-BASEDATE);

	retval = pval01(val, mat, isu, coupon, yield);
	_RetFloat( retval, 20, 8);
}

FoxInfo tlibInfo[] = {
	{ "BD2P",		bd2p,	3,	"D,D,N" },
	{ "BD2Y",		bd2y,	3,	"D,D,N" },
	{ "NY2P",		ny2p,	5,	"D,D,D,N,N" },
	{ "NP2Y",		np2y,	5,	"D,D,D,N,N" },
	{ "ZY2P",		zy2p,	3,	"D,D,N"},
	{ "ZP2Y",		zp2y,   3,  "D,D,N"},
	{ "NTAI",		ntai,	4,	"D,D,D,N"},
	{ "MDUR",		mdur,	5,  "D,D,D,N,N"},
	{ "CVXY",		cvxy,	5,  "D,D,D,N,N"},
	{ "YV32",		yv32,	5,	"D,D,D,N,N"},
	{ "DV01",		dv01,	5,	"D,D,D,N,N"}
};

FoxTable _FoxTable = {
	(FoxTable FAR *) 0,
	sizeof(tlibInfo)/sizeof(FoxInfo),
	tlibInfo
};
/*****************end of FOXPRO LCK STUFF**************************/

/* datesubs.c
*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++
**					DATE FUNCTIONS
++++++++++++++++++++++++++++++++++++++++++++++++++*/

/*--------------------------------------------------------------
MDY2JUL() - return 0 on error, else return julian
	NB: this isn't called anywhere. I just put it in
		for completeness.
---------------------------------------------------------------*/
unsigned mdy2jul(int m, int d, int y)
{
	long int lj;

	/* check m,d */
	if(m==0 || m>12)	return 0;
	else if(d == 0)		return 0;
	else switch(m){
	case 1:
	case 3:
	case 5:
	case 7:
	case 8:
	case 10:
	case 12:
		if(d>31) return 0;
		else break;
	case 4:
	case 6:
	case 9:
	case 11:
		if(d>30) return 0;
		else break;
	case 2:
		if(y%4==0){
			if(d>29) return 0;
			else break;
		}else{
			if(d>28) return 0;
			else break;
		}
	}
	if(y<50){
		y += 2000;
	} else if (y<1900){
		y += 1900;
	}
	/* now calculate julian
	*/
	lj = jday(d,m,y);
	return (unsigned int) lj;

}

/*-------------------------------------------------
ADDMOS(): add m months to j, preserve monthend
--------------------------------------------------*/
unsigned addmos(unsigned j, int m)
{
	int mm, dd, yy, mx, yx;
	long int lj;

	jd(j, &dd, &mm, &yy);
	mx = yy*12 + mm + m;

	yx = mx/12;
	mx = mx%12;
	if(mx==0){
		mx = 12; yx--;
	}

	if(ismoend(mm, dd, yy)) switch(mx){
		case 1:
		case 3:
		case 5:
		case 7:
		case 8:
		case 10:
		case 12:
			dd = 31; break;
		case 4:
		case 6:
		case 9:
		case 11:
			dd = 30; break;
		case 2:
			if(yx%4==0)	dd = 29;
			else dd = 28;
			break;
	}

	lj = jday(dd,mx,yx);
	return (unsigned int)lj;

}

int ismoend(int m, int d, int y)
/* return 1 if m/d/y is a monthend, otherwise return 0
*/
{
	switch(m){
		case 1:
		case 3:
		case 5:
		case 7:
		case 8:
		case 10:
		case 12:
			return (d == 31? 1 : 0);
		case 4:
		case 6:
		case 9:
		case 11:
			return (d == 30? 1 : 0);
		case 2:
			if(y%4==0)
				return (d == 29? 1 : 0);
			else
				return (d == 28? 1 : 0);
	}
	return 0;
}

/*	jday() and jdate() are the same algorithms that Fox
	uses for julian number calculations.  They are published
	in Collected Algorithms from ACM, Volume I.
	Note, however that these return/accept a daycount from
	12/30/1899 as an unsigned integer rather than a long or a
	double.  These shorter numbers, in addition to taking up
	a lot less space, are also the same day numbers that LOTUS 1-2-3,
	Quattro, and their ilk use.
*/
unsigned int jday( int d, int m, int y)
/* find the julian number from day, month, year
	using ACM algorithm 199

*/
{
	int c, ya;
	long int  j;

	if (m>2)
		m -= 3;
	else {
		m += 9; y--;
	}
	c = y / 100;
	ya = y - 100 * c;

	j = (146097L * (long)c)/4
		+ (1461L * (long)ya)/4
		+ (153L * m + 2)/5
		+ (long)d
		+ 1721119L;

	return (unsigned int)(j-BASEDATE);
}

void jd(unsigned int j, int* d, int* m, int* y)
/* extract day, month, year from a julian number
	using ACM algorithm 199
*/
{
	long int lj, yy, dd, mm;

	lj = j + BASEDATE;

	lj -= 1721119L;
	yy = (4*lj-1)/146097L;
	lj = 4*lj-1-146097L*yy;
	dd = lj/4;
	lj = (4L*dd+3)/1461L;
	dd = 4L*dd+3-(1461L*lj);
	dd = (dd+4)/4;
	mm = (5L * dd - 3)/153L;
	dd = 5L*dd - 3 - 153L * mm;
	dd = (dd+5)/5;
	yy = 100 * yy + lj;
	if (mm < 10L)
		mm += 3;
	else {
		mm -= 9; yy++;
	}
	*m = (int)mm;
	*d = (int)dd;
	*y = (int)yy;

}
/*	TSYSUBS.C
*/
/*	INTERNAL FUNCTION PROTOTYPES
*/
double SIA1(double af, double c, double p);
double SIA2(double af, double c, double y);
double SIA4(double af, double c, double y, int n);
double SIA4A(double af, double c, double p, int n);
double SIA10(double dsce, double dfce, double af, double c,
		double y, int n);
double SIA10A(double dsce, double dfce, double af, double c,
		double p, int n);
double SIA14(double p, double af);
double SIA15(double y, double af);
double SIA16(double p, double af, int n);
double SIA17(double p, double af, int n);
int ncp(unsigned matdt, unsigned valdt);
double dpdy4(double af, double c, double y, int n);
double dpdy10(double dsce, double dfce, double c, double y, double n);

/*=======================================================
**
**				STANDARD PRICE <> YIELD ROUTINES
**
**=======================================================*/
/*-----------------------------------------------------------
ACCDINT() return accrued interest on Tsy note/bond
-----------------------------------------------------------*/
double accdInt(unsigned val, unsigned mat, unsigned idt, double cpn)
{
	unsigned lacd, nacd;
	double af;

	if(idt>val) return 0.0; /* when-issued */
	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if(idt > lacd){ /* short first coupon */
		af   = (double)(val - idt)/(double)(nacd-lacd);
	} else
		af = (double)(val-lacd)/(double)(nacd-lacd);
	return af * cpn/2.0;
}

/*----------------------------------------------------------------
MODDUR() return modified duration on a note/bond
----------------------------------------------------------------*/
double modDur(unsigned val, unsigned mat, unsigned idt, double cpn, double yld)
{
	unsigned lacd, nacd;
	double c, af, caf, m, y, yy, x;
	int n;

	if(cpn == 0.0){	/* Tsy bill or strip: duration = maturity */
		return (double)(mat-val)/365.25;
	}
	if(idt>val) val = idt; /* when-issued */
	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if(idt > lacd){ /* short first coupon */
		af   = (double)(val - idt)/(double)(nacd-lacd);
	} else
		af = (double)(val-lacd)/(double)(mat-lacd);
	caf = 1.0-af;
	n = ncp(mat, val);
	m = (double)(n-1)+caf;
	y = yld/200.0;
	yy = 1.0 + y;
	c = cpn/200.0;

	x = (1.0 + caf*y)*pow(yy,n);
	x -= yy;
	x -= m * y;
	x *= c;

	x /= (y*y*pow(yy,m));
	x += m/pow(yy,m);
	x /= 2*(notey2p(val, mat, idt, cpn, yld)/100.0+c*af);
	x /= yy;
	return x;
}

/*-----------------------------------------------------
CONVEXITY(): return convexity of Tsy issue
------------------------------------------------------*/
double convexity(unsigned val, unsigned mat, unsigned idt, double cpn, double yld)
{
	double p1, p2, p3;

	if( cpn == 0.0 )	return 0.0; /* bill */

	p1 = notey2p(val, mat, idt, cpn, yld);
	p2 = notey2p(val, mat, idt, cpn, yld+1.0);
	p3 = notey2p(val, mat, idt, cpn, yld-1.0);

	return(p2+p3-2.0*p1);
}

/*---------------------------------------------------------------------
YVAL32ND(): return yield value of a price change of 1/32 in bp's
---------------------------------------------------------------------*/
double yval32nd(unsigned val, unsigned mat, unsigned idt, double cpn, double px)
{
	if( cpn == 0.0){ /* bill px is discount rate */
		double p1, p2, y1, y2, r1, r2;

		p1 = p2 = billpx(val, mat, px);
		p1 -= 0.015625;
		p2 += 0.015625;
		r1 = (100.0 - p1) * 360.0/(double)(mat - val);
		r2 = (100.0 - p2) * 360.0/(double)(mat - val);
		y1 = billbey(val, mat, r1);
		y2 = billbey(val, mat, r2);
		return fabs(100.0*(y1-y2));

	} else {
		double y1, y2;
		y1 = notep2y(val, mat, idt, cpn, px-0.015625);
		y2 = notep2y(val, mat, idt, cpn, px+0.015625);
		return fabs(100.0*(y1-y2));
	}
}

/*-----------------------------------------------------------------------
PVAL01(): return value of a 01 yield change in $/MM
-----------------------------------------------------------------------*/
double pval01(unsigned val, unsigned mat, unsigned idt, double cpn, double yld)
{
	if( cpn == 0.0 ){ /* bill */
		double p1, p2;

		p1 = billy2dpx(val, mat, yld-0.005);
		p2 = billy2dpx(val, mat, yld+0.005);

		return (p1-p2)*1e4;

	} else {
		double p1, p2;

		p1 = notey2p(val, mat, idt, cpn, yld-0.005);
		p2 = notey2p(val, mat, idt, cpn, yld+0.005);

		return (p1-p2)*1e4;
	}
}

/*---------------------------------------------------------------
BILLY2DPX(): return dollar price, given bey
---------------------------------------------------------------*/
double billy2dpx( unsigned val, unsigned mat, double yld)
{
	unsigned dtemp;
	double px;

	dtemp = addmos(val,6);
	if( dtemp >= mat){
		double dr;

		dr = 36000.0*yld/(36500.0 + yld * (double)(mat-val));
		px = 100.0 - dr * (double)(mat - val)/360.0;
	} else {
		double high, low, dr, yy;

		high = 36000.0*yld/(36500.0 + yld * (mat-val));
		high += 1.0;
		low  = high - 2.0;
		do {
			dr = (high+low)/2.0;
			yy = billbey(val, mat, dr);
			if( yy > yld) high = dr;
			else low = dr;
		} while(fabs(yy - yld)>0.0004);
		px = 100.0 - dr * (double)(mat - val)/360.0;
	}
	return px;

}
/*-------------------------------------------
BILLPX() return bill dollar price
-------------------------------------------*/
double billpx(unsigned val, unsigned mat, double disc)
{
	double px;

	if( val>mat ) return 0.0;
	else if( val==mat) return 1.0;
	else {
		px = 1.0 - (mat-val)*disc/36000.0;
	}
	return px*100.0;
}

/*----------------------------------------------------
BILLBEY()	return bond equivalent yields
----------------------------------------------------*/
double billbey(unsigned val, unsigned mat, double disc)
{
	unsigned j6;

	j6 = addmos(val,6);
	if( mat <= j6){	/* use short form */
		return 365.0*disc/(360.0-(mat-val)*disc/100.0);
	} else { /* more than 6 months to maturity--use quadratic */
		unsigned ydays;
		double a, b, c, p, temp;

		ydays = addmos(val,12) - val;
		p = billpx(val, mat, disc)/100.0;
		a = (mat-val)/(2.0*ydays) - 0.25;
		b = (mat-val)/(double)ydays;
		c = (p-1.0)/p;
		temp = b*b - 4.0*a*c;
		temp = sqrt(temp);
		temp -= b;
		temp /= 2.0*a;
		return temp*100.0;

	}
}

/*------------------------------------------------
NOTEY2P() note yield to price
------------------------------------------------*/
double notey2p(unsigned val, unsigned mat, unsigned idate,
			double coupon, double yield)
{
	unsigned lacd, nacd;
	int n;
	double af, c, y, p;

	if(idate>val) val = idate;
	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if( nacd == mat){	/* note in last period; use SIA 2 */
		af = (double)(val-lacd)/(double)(mat-lacd);
		c  = coupon/200.0;
		y  = yield/200.0;
		p  = SIA2(af, c, y);

	} else if(idate > lacd){ /* short first coupon; use SIA 10 */
		double dsce, dfce;

		dsce = (double)(nacd-val)/(double)(nacd-lacd);
		dfce = (double)(nacd-idate)/(double)(nacd-lacd);
		af   = (double)(val - idate)/(double)(nacd-lacd);
		c = coupon/200.0;
		y = yield/200.0;
		n = ncp(mat, val);
		p = SIA10(dsce, dfce, af, c, y, n);

	} else {	/* use SIA 4 */
		af = (double)(val-lacd)/(double)(nacd-lacd);
		c  = coupon/200.0;
		y  = yield/200.0;
		n  = ncp(mat, val);
		p = SIA4(af, c, y, n);
	}

	return p*100.0;
}

/*-----------------------------------------------------
NOTEP2Y() note price to yield
-----------------------------------------------------*/
double notep2y(unsigned val, unsigned mat, unsigned idate,
			double coupon, double price)
{
	unsigned lacd, nacd;
	int n;
	double af, c, y, p;

	if(idate>val) val = idate;
	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if( nacd == mat){	/* note in last period; use SIA 2 */
		af = (double)(val-lacd)/(double)(mat-lacd);
		c  = coupon/200.0;
		p  = price/100.0;
		y  = SIA1(af, c, p);

	} else if(idate > lacd){ /* short first coupon; use SIA 10 */
		double dsce, dfce;

		dsce = (double)(nacd-val)/(double)(nacd-lacd);
		dfce = (double)(nacd-idate)/(double)(nacd-lacd);
		af   = (double)(val - idate)/(double)(nacd-lacd);
		c = coupon/200.0;
		p = price/100.0;
		n = ncp(mat, val);
		y = SIA10A(dsce, dfce, af, c, p, n);

	} else {	/* use SIA 4 */
		af = (double)(val-lacd)/(double)(nacd-lacd);
		c  = coupon/200.0;
		p  = price/100.0;
		n  = ncp(mat, val);
		y  = SIA4A(af, c, p, n);
	}

	return y*100.0;
}


/*------------------------------------------------------
ZEROY2P() Yield to price for strips
------------------------------------------------------*/
double zeroy2p(unsigned val, unsigned mat, double yield)
{
	unsigned lacd, nacd;
	int n;
	double af, y, p;

	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if( nacd == mat){	/* strip in last period; use SIA 15 */
		af = (double)(val-lacd)/(double)(mat-lacd);
		y  = yield/200.0;
		p  = SIA15(y, af);

	} else {	/* use SIA 17 */
		af = (double)(val-lacd)/(double)(nacd-lacd);
		y  = yield/200.0;
		n  = ncp(mat, val);
		p = SIA17(y, af, n);
	}

	return p*100.0;

}

/*------------------------------------------------------
ZEROP2Y() price to yield for strips
------------------------------------------------------*/
double zerop2y(unsigned val, unsigned mat, double price)
{
	unsigned lacd, nacd;
	int n;
	double af, y, p;

	lacd = mat;
	while(lacd>val)	lacd = addmos(lacd,-6);
	nacd = addmos(lacd,6);
	if( nacd == mat){	/* strip in last period; use SIA 14 */
		af = (double)(val-lacd)/(double)(mat-lacd);
		p  = price/100.0;
		y  = SIA14(p, af);

	} else {	/* use SIA 16 */
		af = (double)(val-lacd)/(double)(nacd-lacd);
		p  = price/100.0;
		n  = ncp(mat, val);
		y = SIA16(p, af, n);
	}

	return y*100.0;

}

/*===================================================
	INTERNAL FUNCTIONS, called by above functions
===================================================*/
int ncp(unsigned matdt, unsigned valdt)
/* number of coupon payments from valdt to matdt */
{
	int n=0;
	unsigned td;

	td = valdt;
	while( td<matdt){
		n++;	td = addmos(td, 6);
	}
	return n;
}

double SIA1(double af, double c, double p)
/*	SIA FORMULA 1 */
{
	double x;
	x = ((1.0+c)-(p+af*c))/(p+af*c);
	x *= 2.0/(1.0-af);
	return x;
}


double SIA2(double af, double c, double y)
/*	SIA FORMULA 2 */
{
	return (1.0+c)/(1.0+(1.0-af)*y) - af*c;
}

double SIA10(double dsce, double dfce, double af, double c,
		double y, int n)
/*	SIA FORMULA 10 */
{
	double cy, y1, x;

	cy = c/y;
	y1 = 1.0 + y;

	x  = cy + ((1.0-cy)/pow(y1,n)) - (c/y1);
	x *= pow(y1,(1.0-dsce));
	x += (c*dfce)/(pow(y1,dsce));
	x -= c*af;

	return x;
}

double SIA10A(double dsce, double dfce, double af, double c,
		double p, int n)
/*	SIA FORMULA 10  price to yield version, using Newton's method */
{
	double z, px;

	z = c/p;
	do {
		px = SIA10(dsce, dfce, af, c, z, n);
		z += (p-px)/dpdy10(dsce, dfce, c, z, n);

	} while( fabs(px-p)>TOLERANCE);

	return z*2.0;
}


double dpdy10(double dsce, double dfce, double c, double y, double n)
/* first derivative of SIA 10 */
{
	double cy, y1, y1n, y1dsce, y1dsce1, w, x, z;

	cy = c/y;
	y1 = 1.0+y;
	y1n = pow(y1,n);
	y1dsce = pow(y1, dsce);
	y1dsce1 = pow(y1, (1.0-dsce));

	w = -c*(dfce/y1dsce)*(dsce/y1);
	x = -((1.0-cy)/y1n)*(n/y1);
	x -= c/(y*y);
	x += c/(y*y*y1n);
	x += c/(y1*y1);
	x *= y1dsce1;
	z = cy + (1.0-cy)/y1n - c/y1;
	z *= y1dsce1;
	z *= (1.0-dsce)/y1;

	x = w+x+z;

	return x;
}

double SIA4(double af, double c, double y, int n)
/* SIA formula 4: Bond yield to price, normal coupons */
{
	double cy, y1, x;

	cy = c/y;
	y1 = 1.0+y;

	x  = cy+(1-cy)/pow(y1,n);
	x *= pow(y1,af);
	x -= c*af;
	return x;
}

double SIA4A(double af, double c, double p, int n)
/* Price-to-yield version of SIA 4 */
{
	double z, px;

	z = c/p;
	do {
		px = SIA4(af, c, z, n);
		z += (p-px)/dpdy4(af, c, z, n);

	} while( fabs(px-p)>TOLERANCE);

	return z*2.0;
}

double dpdy4(double af, double c, double y, int n)
/* first derivative of SIA 4 with respect to yield */
{
	double cy, y1, y1af, y1n, x, z;

	cy = c/y;
	y1 = 1.0+y;
	y1af = pow(y1, af);
	y1n  = pow(y1, n);

	x = -((1.0-cy)/y1n) * (n/y1);
	x += c/(y*y*y1n);
	x -= c/(y*y);
	x *= y1af;

	z = ((1.0-cy)/y1n)+cy;
	z *= y1af;
	z *= af/y1;

	x += z;

	return x;
}

double SIA14(double p, double af)
/* SIA formula 14: Zero coupon p->y, in last six months */
{
	double y;

	y = (1.0-p)/p;
	y *= 2.0/(1.0-af);
	return y;
}

double SIA15(double y, double af)
/* SIA formula 15: Zero coupon y->p, in last six months */
{
	double p;

	p = 1.0/(1.0+(1.0-af)*y);
	return p;
}

double SIA16(double p, double af, int n)
/* SIA formula 16: Zero coupon p->y */
{
	double y;

	y = pow(p,-(1.0/((double)n-af)));
	y -= 1.0;
	y *= 2.0;
	return y;
}

double SIA17(double y, double af, int n)
/* SIA formula 17: Zero coupon y->p */
{
	double p;

	p = 1.0/(pow(1.0+y,(double)n-af));
	return p;
}
