/*****
 *
 * TBR32.PRG
 *
 * A "multi-line" TBrowse in which the memo field
 * is shown in up to four lines
 * Editing is allowed
 * 
 */

#include "inkey.ch"
#include "setcurs.ch"
#include "set.ch"

#include "tbrowse.ch"
#include "samples.ch"

#define        WHAT_IT_DOES           "ESC - Quit"    +;
                                      "          "    +;
                                      "ENTER - Edit"

FUNCTION Tbr32()
   LOCAL oBrow, oCol, nKey
   // This variable holds the line number 
   // Lets start as line number 1
   //
   LOCAL nLine := 1

   SET(_SET_PATH, WHERE_TO_FIND_FILES)

   DBUSEAREA( NEW, "DbfNtx", "Memotest" )
   DBSETINDEX( "Memotest" )

   // Save
   PushSets()
   PushScr()

   SET(_SET_SCOREBOARD, .F.)
   SET(_SET_CONFIRM, .T.)
   SETBLINK(.F.)
   SETCURSOR(SC_NONE)
   SETCOLOR(BGND_CLR)
   SCROLL()
   @ MAXROW(), 0 SAY PADC( WHAT_IT_DOES, MAXCOL() + 1) COLOR MSG_ROW_CLR

   // STEP 1
   oBrow := TBROWSEDB( 3, 7, MAXROW() - 3, 72 )
   oBrow:colorSpec := CLR_SPEC
   oBrow:colSep    := COLSEP
   oBrow:headSep   := HEADSEP

   // Custom Skippers
   //
   // nLine is passed by reference
   // since we need to know which
   // data should be retrieved.
   //
   // It means field contents or "" or even spaces
   // for the first and the third columns depending
   // if nLine is 1 or != 1.
   // 
   oBrow:skipBlock     := {|nSkip| SkipDB(nSkip, @nLine)}
   oBrow:goTopBlock    := {|| GoTopDB(@nLine)}
   oBrow:goBottomBlock := {|| GoBottomDB(@nLine)}

   // STEP 2
   oCol := TBColumnNew( HEAD_CITY, {|| IF(nLine == 1,;
                          memotest->charfld, "")})
   
   // Get-set block should be attached to cargo
   // Better way to attach things to cargo is thru a
   // dictionary but just for the sample...
   //
   oCol:cargo := {|xParm| IF(xParm == NIL, memotest->charfld,;
                             memotest->charfld := xParm)}
   oBrow:addColumn( oCol )

   // As you can see, the "trick" is the line counter
   // which will point out for the line you want to 
   // show on the screen. 
   oCol := TBColumnNew( HEAD_MEMO, {|| RetrieveWhat(nLine)} )
   oCol:width   := WIDTH
   // Get-set block should be attached to cargo
   oCol:cargo := {|xParm| IF(xParm == NIL, memotest->memofld,;
                             memotest->memofld := xParm)}
   oBrow:addColumn( oCol )

   // Browse it!
   WHILE .T.
      // Step 3
      ForceStable( oBrow )
      IF (oBrow:hitTop .OR. oBrow:hitBottom)
         // Make some noise!
         TONE(1000, 2)

      ENDIF

      // Step 4
      nKey := INKEY(0)         

      // Process key
      IF !TBMoveCursor( nKey, oBrow )
         IF ( nKey == K_ESC )
            EXIT

         ELSEIF ( nKey == K_ENTER )
            // Editing
            // Is it a memo?
            IF ( oBrow:colPos == MEMOCOLUMN )
               IF EditMemo(oBrow, nLine)
                  oBrow:refreshAll()

               ELSE
                  oBrow:invalidate()

               ENDIF

            ELSE
               DoGet(oBrow, nLine)

            ENDIF

         ENDIF
   
      ENDIF

   END

   PopSets()
   PopScr()

   RETURN (NIL)

/*****
 *
 * Bottom of file
 *
 */

STATIC FUNCTION GoBottomDB( nLine )
   // You are receiving a reference
   DBGOBOTTOM()
   nLine := LAST_LINE
   RETURN (NIL)

/*****
 *
 * Top of File
 *
 */

STATIC FUNCTION GoTopDB( nLine )
   // You are receiving a reference
   DBGOTOP()
   // Since you are pointing to the first record
   // your current line should be 1
   nLine := 1
   RETURN (NIL)

/*****
 *
 * Skip records
 *
 */

STATIC FUNCTION SkipDB( nRequest, nLine )
   // nLine is a reference
   LOCAL nActually := 0

   IF nRequest == 0
      DBSKIP(0)

   ELSEIF nRequest > 0 .AND. !EOF()
      WHILE nActually < nRequest
         IF nLine < LAST_LINE
            // This will print up to LAST_LINE of text
            // Some of them (or even all) might be empty
            ++nLine

         ELSE
            // Go to the next record
            DBSKIP(+1)
            nLine := 1

         ENDIF
         IF EOF()
            DBSKIP(-1)
            nLine := LAST_LINE
            EXIT

         ENDIF
         nActually++

      END

   ELSEIF nRequest < 0
      WHILE nActually > nRequest
         // Go to previous line
         IF nLine > 1
            --nLine

         ELSE
            DBSKIP(-1)
            IF !BOF()
               nLine := LAST_LINE

            ELSE
               // You need this. Believe me!
               nLine := 1
               GOTO RECNO()
               EXIT

            ENDIF

         ENDIF
         nActually--

      END

   ENDIF
   RETURN (nActually)

/*****
 *
 * Which line should be retrieved?
 *
 */

STATIC FUNCTION RetrieveWhat(nLine)
   LOCAL cReturn := ""
   LOCAL cStr

   // Strip all hard returns, soft returns and line feeds                
   cStr := STRTRAN(memotest->memofld, CHR(141), "")
   cStr := STRTRAN(cStr, CHR(10), "")
   cStr := STRTRAN(cStr, CHR(13), "")

   IF nLine == 1
      cReturn := SUBSTR(cStr, 1, WIDTH)

   ELSEIF nLine == 2
      cReturn := SUBSTR(cStr, WIDTH + 1, WIDTH)

   ELSEIF nLine == 3
      cReturn := SUBSTR(cStr, WIDTH * 2 + 1, WIDTH)

   ELSEIF nLine == 4
      cReturn := SUBSTR(cStr, WIDTH * 3 + 1, WIDTH)

   ENDIF
   RETURN (cReturn)

/*****
 *
 * Edits the memo field
 *
 */

STATIC FUNCTION EditMemo(oBrow, nLine)
   LOCAL nBot, nRig, nTop, nLef, oCol
   LOCAL cStr1, cStr0, bBlock
   LOCAL cClr, nCur, lRet := .F.

   IF nLine == 1
      oCol := oBrow:getColumn(oBrow:colPos)
      nTop   := ROW()
      nLef   := COL()

      IF (oBrow:rowCount == oBrow:rowPos)
         nBot := nTop

      ELSEIF ( (oBrow:rowPos + LAST_LINE - 1 ) > oBrow:rowCount)
         nBot := nTop + (oBrow:rowCount - oBrow:rowPos)

      ELSE
         nBot := nTop + LAST_LINE - 1

      ENDIF

      nRig   := nLef + WIDTH

      cClr  := SETCOLOR("I")
      nCur  := SETCURSOR(SC_NORMAL)
      cStr0 := EVAL(oCol:cargo)

      // Edit it
      cStr1 := MEMOEDIT(cStr0, nTop, nLef, nBot, nRig, .T.)

      // Reset
      SETCOLOR(cClr)
      SETCURSOR(SC_NONE)

      IF !(cStr0 == cStr1)
         // Replace field
         EVAL(oCol:cargo, cStr1)
         lRet := .T.

      ENDIF

   ENDIF

   RETURN (lRet)

/*****
 *
 * @...GET
 *
 */

STATIC FUNCTION DoGet(oBrow, nLine)
   LOCAL nCursSave, xOldKey, xNewKey
   LOCAL oCol, oGet, nKey

   IF nLine == 1
      PushSets()

      // Cursors are for GETs, so:
      SETCURSOR(SC_NORMAL)

      // make sure browse is stable
      ForceStable(oBrow)

      oCol := oBrow:getColumn( oBrow:colPos )

      // create a corresponding GET and READ it
      // Pay attention to oCol:cargo!!!!!!!!!
      oGet := GetNew(ROW(), COL(), oCol:cargo,;
                  oCol:heading,, "W+/BG,W+/B")

      // Get old key value or NIL
      xOldKey := IF( EMPTY(INDEXKEY()), NIL, &(INDEXKEY()) )
      READMODAL( {oGet} )
      SETCURSOR(SC_NONE)
      xNewKey := IF( EMPTY(INDEXKEY()), NIL, &(INDEXKEY()) )

      // If key was changed...
      IF xOldKey != xNewKey
         // Refresh
         oBrow:refreshAll()
         ForceStable( oBrow )

         // Make sure we are still in the right record
         // after stabilizing
         WHILE &(INDEXKEY()) > xNewKey .AND. !oBrow:hitTop()
            oBrow:up()
            ForceStable( oBrow )

         END

      ELSE
         oBrow:refreshCurrent()
         ForceStable( oBrow )

      ENDIF

      // check exit key
      nKey := LASTKEY()
      IF ( nKey == K_UP .OR. nKey == K_DOWN .OR. ;
         nKey == K_PGUP .OR. nKey == K_PGDN )
         KEYBOARD CHR( nKey )

      ENDIF

      PopSets()

   ENDIF

   RETURN (NIL)

// EOF - TBR32.PRG //
