*------------------------------------------------------------------------------
*-- Program     : MARIAN.PRG
*-- Programmer..: Jay Parsons (Jparsons)
*-- Date........: 05/05/1992
*-- Notes.......: Manages addition and deletion of routines to Ken Mayer's
*--             : procedure and function library, LIB15.ZIP on the BORBBS.
*-- Written for.: dBASE IV, Version 1.5. ( uses low-level file handling )
*-- Rev. History: None
*-- Calls       : No functions or procedures external to this file
*-- Called by...: Any, but see Side Effects
*-- Uses........: Setmouse.bin, by Bowen Moursund
*--             : Exactime.bin, by Jay Parsons
*-- Usage.......: DO Marian
*-- Parameters..: None
*-- Side effects: In addition to changing the user's library file(s),
*--             : clears all dBASE variables and work areas.
*-------------------------------------------------------------------------------

*--             General organization of the program file

*--     Header and description (this part)

*--     Global setup code and semi-public variables

*--     Data, including file structures

*--     Main flow

*--     Menu and popup definitions

*--     Principal program action modules
*--             - Menu and popup actions
*--             - File handling

*--     Subroutines, procedures and functions
*--             - Special-purpose routines
*--             - Library-type routines

*--     Minor startup code and exit code

*--     Sign-on banner and help screens

*----------------------------------------------------------------------------
*--                     Narrative description

*--     This is a stand-alone program in dBASE IV, Version 1.5 minimum.
*--     It cannot usefully be made part of a larger application unless
*--     changed to remove the CLEAR ALL, CLOSE ALL and similar statements
*--     in its setup code.

*--                     Files used

*--     This program works with fifteen files, although the actual
*--     number of files may be larger as the user selects new ones from
*--     some of the categories.

*--     .MEM  (1)
*--     One .mem file, LIB15MV.MEM.  This is not essential, it simply
*--     holds default filenames for the catalog, description, report and
*--     target files (see below), the selected overwriting mode, colors
*--     and whether or not the program has been run before.  If this .MEM file
*--     does not exist, it will be created.  Changing its identity requires
*--     changing code in the data section.

*--     .PRG  (2)
*--     Two text files containing dBASE IV procedures and functions.
*--     The program considers one to be the SOURCE, one of the new
*--     program files received with a library update or made by the user,
*--     and the other to be the TARGET, the library file being updated.
*--     It is ESSENTIAL that these files follow the dUFLP structure:

*--     Each function or procedure ("routine") must be headed by its type
*--     and name at the left of a line, as with any compilable .prg.

*--     Each must have a header using the *-- commenting notation used
*--     throughout this file following the "FUNCTION" or "PROCEDURE" line.

*--     One of the header lines should start "*-- Date" and have the date
*--     of last revision of the routine, in MM/DD/YY format, beginning
*--     one space to the right of the first colon in the line.  If dates
*--     are not included in this format, the program will not "freshen"
*--     routines of the same name and will not add new routines if the
*--     overwriting mode has been set to "If_Older".

*--     If the routine calls any other routines, the header should contain
*--     a line starting"*-- Calls".  This and all following lines while
*--     they start with the "*--" and six spaces will be considered to
*--     hold, starting one space to the right of the first colon in the line,
*--     the name of a called routine, followed by a left parenthesis if
*--     a function or a space if a procedure, and after the word "in" in
*--     its first appearance later in the line, the name of the program
*--     file in which the called routine resides.  If this format is
*--     followed, if the program file holding the calling routine can be
*--     found by this program and if that file follows the same conventions,
*--     the called routine will be added to the target file along with
*--     the calling routine.  Note, however, that if the called routine
*--     is old and the target file already contains a routine of the
*--     same name and type and same or later date, it will not be added if
*--     the "If_Older" overwriting mode is set, or during a freshen.

*--     .DBF ( 5 ) and .MDX  ( 3 )
*--     The program requires a .dbf known as the CATALOG file containing
*--     a list of program files to use as source and target files.  The
*--     program allows changes of or to the catalog file, but not use without
*--     one.  See "Acat" in the data section for its structure.  It gives the
*--     the name and date of each indexing .dbf created for a .prg file; its
*--     .mdx has three keys, one for all file names, one for source files and
*--     one for target files.  As furnished, the structure of this file and
*--     the window used to choose from it are not wide enough to include
*--     paths with the file names, but you can change those if you wish to
*--     allow paths.

*--     The program builds a .dbf ( if one is not found ) to INDEX each Source
*--     and Target .prg file. See "Aindex" in the data section for a list
*--     of the structure of these files.  Each is created with a production
*--     .mdx. and holds information on the names, types, locations, sizes
*--     and dates of the routines in the .prg file.

*--     The program builds, uses and eventually destroys a .dbf of CALLEES,
*--     simply a stack of called routines awaiting transfer with their
*--     parents.  See "Acallees" for its structure.  It builds a second
*--     temporary .dbf of the same structure to hold the names of the
*--     indexing .dbfs created during the run, to allow the user to decide
*--     what to do with them at the end.

*--     .TXT  ( 2 )
*--     The program will print out the file README.TXT, Ken's file containing
*--     the names and descriptions of all routines in his library, or other
*--     text file.

*--     If ALTERNATE is set on, a list of routines transferred and of problems
*--     will be sent to a text file, by default REPORT.TXT.  These will
*--     normally be printed only to the screen.

*--     .BIN ( 2 )
*--     If available, the program uses SETMOUSE.BIN to allow the user to
*--     turn the mouse ( if any ) on and off.  It also uses EXACTIME.BIN,
*--     if available, to time a delay loop to within 1/18 seconds.

*--                     Program Operation

*--     Most of the program simply manages selection and creation of the
*--     various indexing files.  It uses the low-level file functions of
*--     Version 1.5 to read, parse and write the .prg files, relying on
*--     their structure being as stated above.

*--     The indexing files will be built with the same names as their
*--     .prg files if no name conflict exists, otherwise with temporary names.
*--     The "Never" overwriting mode may be used in order to have
*--     this program build the indexes without affecting the .prg files.

*--     At the end of the run, the user is given the option to save,
*--     rename or destroy the indexing files built during the run.

*--   ===================================================================

*--                     Global Setup Code

*----------------------------------------------------------------------------

  *--   Executable code begins here
  CLEAR ALL
  gn_oldsize = iif( set("DISPLAY") = "EGA43",43,25)
  gc_bell   =SET("BELL")
  gc_carry  =SET("CARRY")
  gc_century=SET("CENTURY")
  gc_clock  =SET("CLOCK")
  gc_color  =SET("ATTRIBUTE")
  gc_confirm=SET("CONFIRM")
  gc_console=set("CONSOLE")
  gc_cursor =SET("CURSOR")
  gc_delete =SET("DELETED")
  gc_deli   =SET("DELIMITERS")
  gc_dir    =set("DIRECTORY")
  gc_display=SET("DISPLAY")
  gc_escape =set("ESCAPE")
  gc_exact  =set("EXACT")
  gc_safety =SET("SAFETY")
  gc_status =SET("STATUS")
  gc_score  =SET("SCOREBOARD")
  gc_talk   =SET("TALK")
  gc_title  =set("TITLE")

  *--   Close all, do SET commands
  DO Setenv

  *-- Initialize App Gen global variables
  gl_color= ISCOLOR() .AND. .NOT. "MONO" $ SET("DISPLAY")
  gn_error=0          && 0 if no error, otherwise an error occurred
  gn_send=0           && return value from popup of position menus
  gc_alt = 'OFF'      && set("ALTERNATE")
  gc_dev='CON'        && Device to use for printing - See Proc. PrintSet
  gl_leave=.f.        && leave the application
  gc_quit=' '         && memvar for return to caller

*--                     Data, statuses and file structures
*-------------------------------------------------------------------------
*--                     Miscellaneous data

*--     Define the key parts of the .prg structure used

Remark          = "*--"                 && start of dFULP comment line
RCalls          = " CALLS"              && dFULP for what's called
RDate           = "DATE"                && dFULP for routine date
Blanks          = space( 6 )            && succeeding "Calls" lines
NoCalls         = "NONE"
Fromflag        = " IN "

*--     Miscellaneous program values

Delaysecs       = 0.2                   && delay loop time in seconds
Showlines       = iif( set("DISPLAY") = "EGA43", 41, 23 )  && screen lines

*-- Load or default file names, etc.
Memofile        = "LIB15MV.MEM"         && standard mem file for defaults

IF file( Memofile )
  RESTORE FROM ( Memofile ) ADDITIVE
ELSE
  Def_cat       = "LIB15CAT"            && catalog file
  Def_crash     = "Ask_User"            && overwrite mode
  Def_desc      = "README.TXT"          && description-text file
  Def_first     = .T.                   && first time run
  Def_rept      = "REPORT.TXT"          && alternate file
  Def_targ      = "LIBRARY.PRG"         && target file
  IF gl_color
    Def_attrs   = "W+/B,RG+/GB,N/N "+chr(38)+chr(38)+" W+/N,W/B,RG+/GB,B/W,N/GB"
  ELSE
    Def_attrs   = "W+/N,N/W,N/N "+chr(38)+chr(38)+" W/N,W/N,W+/N,N/W,N/W"
  ENDIF
ENDIF

Catfile         = Def_cat
Crash           = Def_crash
Descfile        = Def_desc
Reptfile        = Def_rept
Target          = Def_targ
cAttrs          = Def_attrs

*--     Structure of temporary and index files

DECLARE Acallees[3,5]                 && structure for Callees file
  Acallees[1,1] = "ProcName"          && also used inappropriately for
  Acallees[1,2] = "C"                 && Made file of index .dbfs made
  Acallees[1,3] = 12
  Acallees[1,4] =  0
  Acallees[1,5] = "N"

  Acallees[2,1] = "FromFile"
  Acallees[2,2] = "C"
  Acallees[2,3] =  12
  Acallees[2,4] =  0
  Acallees[2,5] = "N"

  Acallees[3,1] = "ProcType"
  Acallees[3,2] = "C"
  Acallees[3,3] =  1
  Acallees[3,4] =  0
  Acallees[3,5] = "N"

*--
DECLARE Aindex[5,5]                   && structure for Index .dbf files
  Aindex[1,1] = "Proctype"
  Aindex[1,2] = "C"
  Aindex[1,3] =  9
  Aindex[1,4] =  0
  Aindex[1,5] = "N"

  Aindex[2,1] = "Procname"
  Aindex[2,2] = "C"
  Aindex[2,3] =  11
  Aindex[2,4] =  0
  Aindex[2,5] = "N"                   && we want index FOR recno()>2 only

  Aindex[3,1] = "Line"
  Aindex[3,2] = "N"
  Aindex[3,3] =  5
  Aindex[3,4] =  0
  Aindex[3,5] = "N"

  Aindex[4,1] = "Bytes"
  Aindex[4,2] = "N"
  Aindex[4,3] =  10
  Aindex[4,4] =  0
  Aindex[4,5] = "N"

  Aindex[5,1] = "Date"
  Aindex[5,2] = "D"
  Aindex[5,3] =  8
  Aindex[5,4] =  0
  Aindex[5,5] = "N"

*--
DECLARE Acat[4,5]                 && structure for catalog file
  Acat[1,1] = "Procfile"
  Acat[1,2] = "C"
  Acat[1,3] = 12
  Acat[1,4] =  0
  Acat[1,5] = "Y"

  Acat[2,1] = "Indxfile"
  Acat[2,2] = "C"
  Acat[2,3] =  8
  Acat[2,4] =  0
  Acat[2,5] = "N"

  Acat[3,1] = "Date"
  Acat[3,2] = "D"
  Acat[3,3] =  8
  Acat[3,4] =  0
  Acat[3,5] = "N"

  Acat[4,1] = "Used"
  Acat[4,2] = "C"
  Acat[4,3] =  1
  Acat[4,4] =  0
  Acat[4,5] = "N"

*--------------------------------------------------------------------------

*                       Main processing

*--------------------------------------------------------------------------

*-- Define menus and colors
DO ReColor WITH cAttrs
*-- Prevents clearing of menus from commands:
*-- SET STATUS and SET SCOREBOARD
DEFINE WINDOW FullScr FROM 0,0 TO 24,79 NONE
ACTIVATE WINDOW FullScr

*-- Entertain them while they are waiting
DO Showsign

DO MPDEF                                        && Menu Process DEFinition

*-- Initialize file handles and statuses
STORE 0 TO nTarg, nSrce                          && file handles for .PRG files
STORE .F. TO lSndx, lTndx, lCalled, lMade        && status flags for .DBF files
STORE "" TO Sindex, Tindex, Osrce, Otarg, Ocat   && Aliases, names
STORE space( 12 ) TO Fname                       && global unknowns
Source = "None"                                  && No source file yet
STORe .T. TO lDesc, lType                        && description file, library

IF .NOT. file( Catfile + ".DBF" )                && try to open catalog file
  lCat = .F.
ELSE
  SELECT select()
  USE ( Catfile ) ALIAS Catfile
  ?? "."
  IF .NOT. file( Catfile + ".MDX" )
    INDEX ON Procfile TAG Procfile
    ?? "."
    INDEX ON Procfile TAG Sources FOR Used = "S"
    ?? "."
    INDEX ON Procfile TAG Targets FOR Used = "T"
  ENDIF
  lCat = .T.
ENDIF
?? "."
IF .NOT. file( Target )                         && and target file
  nTarg = 0
  Target = "None"
ELSE
  nTarg = fopen( Target,"rw" )
  ?? "."
ENDIF

*-- Create a temporary file for stacking names of called functions
Trash = rand( -1 )
SELECT select()
Callees = Tempname("DBF")
?? "."
Strufile = MakeStru()
?? "."
DO Makedbf WITH Callees, Strufile, "Acallees"
?? "."
USE
?? "."
USE ( Callees ) ALIAS Callees NOSAVE
?? "."
lCallees = .T.

*-- and a like one for names of index .dbfs created
Made = Tempname("DBF")
?? "."
COPY STRU TO ( Made )
?? "."
SELECT select()
USE ( Made ) ALIAS Made NOSAVE
?? "."

*-- Toss out the ill-equipped
IF val( substr( version(), 9, 5 ) ) < 1.5
  DO Pause WITH "Sorry - This program requires dBASE IV Version 1.5 or higher."
  DO Resetenv
  RETURN
ENDIF

*-- Help the first-timers
IF Def_first
  ON ESCAPE DO Quickout
  DO Showtext WITH "",-1
  ON ESCAPE
  Def_first =.F.
  SAVE ALL LIKE Def_* TO ( Memofile )
ENDIF

*-- Activate mouse
Mousebin = .T.
STORE .F. TO Ismouse, Mouse
Gotmouse = 8
ON ERROR Mousebin = .F.
LOAD SetMouse
ON ERROR
IF Mousebin
  CLEAR
  IF call( "SetMouse", .T. )
    Store .T. TO Mouse, Ismouse
    Gotmouse = 9
    DO Showtext WITH "", -2
  ELSE
    DO Pause WITH "Be a sport - get a mouse!"
  ENDIF
ENDIF
DO OptDef

*-- Load timer
Timerbin = .T.
ON ERROR Timerbin = .F.
LOAD Exactime
ON ERROR
 
*-- Plan for help and alternate output
ON KEY LABEL F1 DO Help
SET ALTERNATE TO ( Reptfile )

*--                     Main loop
*--------------------------------------------------------------
*-- Execute main menu

DO WHILE .NOT. gl_leave
  CLEAR
  DO Paint
  DO Showstatus WITH .T.

  ACTIVATE MENU MAIN

  CLEAR
  gl_leave = NodShake( " ;   Do you want to leave this application?   ", ;
                          13, 18, 2, 44, .T. )
ENDDO
*---------------------------------------------------------------

DO Shutdown
DO Resetenv

RETURN
*-- EoP: MARIAN
*--              End of principal module
*-----------------------------------------------------------------

*--                Menu and Popup Definitions

*-----------------------------------------------------------------

PROCEDURE MPDEF
*-- Description..: Defines menus in the system for C:MARIAN, except
*-- OPTIONS which was moved out to wait for decision on mouse support

  *-- Pause message box
  DEFINE WINDOW Pause FROM 15,00 TO 19,79 DOUBLE
  *-- Help screens
  DEFINE WINDOW Help FROM 3,5 TO 18,74 DOUBLE COLOR N/BG,GR+/BG,BG/N
  *-- Progress reports
  DEFINE WINDOW Counting FROM 10,10 TO 15,69 DOUBLE
  @ 24,00
  @ 24,00 SAY "Loading..."

  SET BORDER TO DOUBLE

  *-- Bar: Main
  DEFINE MENU MAIN MESSAGE 'Position with: '+CHR(27)+CHR(26)+' - <Enter> to select choice, <F1> for help'
  DEFINE PAD PAD_1 OF MAIN PROMPT "Files" AT 1,1
    ON PAD PAD_1 OF MAIN ACTIVATE POPUP FFILES
  DEFINE PAD PAD_2 OF MAIN PROMPT "Updates" AT 1,16
    ON PAD PAD_2 OF MAIN ACTIVATE POPUP UPDATES
  DEFINE PAD PAD_3 OF MAIN PROMPT "Reports" AT 1,31
    ON PAD PAD_3 OF MAIN ACTIVATE POPUP REPORTS
  DEFINE PAD PAD_4 OF MAIN PROMPT "Options" AT 1,46
    ON PAD PAD_4 OF MAIN ACTIVATE POPUP OPTIONS
  ?? "."

  *-- Popup: Files
  DEFINE POPUP FFILES FROM 2,0 TO 9,36 ;
  MESSAGE 'Position: '+CHR(27)+CHR(26)+CHR(25)+CHR(24)+'  Select: '+CHR(17)+CHR(196)+CHR(217) + '  <F1> for help'
    DEFINE BAR 1 OF FFILES PROMPT "Select a catalog of files used"
    DEFINE BAR 2 OF FFILES PROMPT "Select a source file of routines"
    DEFINE BAR 3 OF FFILES PROMPT "Select a library target file"
    DEFINE BAR 4 OF FFILES PROMPT "Select a program target file"
    DEFINE BAR 5 OF FFILES PROMPT "Select an description text file"
    DEFINE BAR 6 OF FFILES PROMPT "Quit"
  ON SELECTION POPUP FFILES DO Fileacts
  ?? "."

  *-- Popup: Updates
  DEFINE POPUP UPDATES FROM 2,15 TO 9,50 ;
  MESSAGE 'Position: '+CHR(27)+CHR(26)+CHR(25)+CHR(24)+'  Select: '+CHR(17)+CHR(196)+CHR(217) + '  <F1> for help'
    DEFINE BAR 1 OF UPDATES PROMPT "Add the source file to target file"
    DEFINE BAR 2 OF UPDATES PROMPT "Add single routine to target file"
    DEFINE BAR 3 OF UPDATES PROMPT "Freshen target file from all files"
    DEFINE BAR 4 OF UPDATES PROMPT "Delete routine from target file"
    DEFINE BAR 5 OF UPDATES PROMPT "Delete a file from the catalog"
    DEFINE BAR 6 OF UPDATES PROMPT "Split the target file into two"
  ON SELECTION POPUP UPDATES DO Upacts
  ?? "."

  *-- Popup: Reports
  DEFINE POPUP REPORTS FROM 2,30 TO 9,66 ;
  MESSAGE 'Position: '+CHR(27)+CHR(26)+CHR(25)+CHR(24)+'  Select: '+CHR(17)+CHR(196)+CHR(217) + '  <F1> for help'
    DEFINE BAR 1 OF REPORTS PROMPT "List of changed/added routines"
    DEFINE BAR 2 OF REPORTS PROMPT "Print the whole description file"
    DEFINE BAR 3 OF REPORTS PROMPT "Text of a routine in source file"
    DEFINE BAR 4 OF REPORTS PROMPT "Text of a routine in target file"
    DEFINE BAR 5 OF REPORTS PROMPT "Print index .dbf for source file"
    DEFINE BAR 6 OF REPORTS PROMPT "Print index .dbf for target file"
    ON SELECTION POPUP REPORTS DO Repacts
  ?? "."

  *-- Popup: Catfiles
  DEFINE POPUP Catfiles FROM 6,30 TO 20,43 PROMPT FIELD ProcFile
  ON SELECTION POPUP Catfiles DO Pickfile
  ?? "."

  *-- Popup: Routines
  DEFINE POPUP Routines FROM 6,30 TO 20,42 PROMPT Field ProcName
  ON SELECTION POPUP Routines DO PickRoute
  ?? "."

RETURN
*-- EOP: MPDEF.PRG

PROCEDURE OptDef
  *-- Popup: Options
    DEFINE POPUP OPTIONS FROM 2,45 TO Gotmouse,72 ;
    MESSAGE 'Position: '+CHR(27)+CHR(26)+CHR(25)+CHR(24)+'  Select: '+CHR(17)+CHR(196)+CHR(217) + '  <F1> for help'
    DEFINE BAR 1 OF OPTIONS PROMPT "Change printer setting"
    DEFINE BAR 2 OF OPTIONS PROMPT "Change Alternate file"
    DEFINE BAR 3 OF OPTIONS PROMPT "Set Alternate On/Off"
    DEFINE BAR 4 OF OPTIONS PROMPT "Change overwrite mode"
    DEFINE BAR 5 OF OPTIONS PROMPT "Save settings as defaults"
    IF Ismouse
      DEFINE BAR 6 OF OPTIONS PROMPT "Toggle mouse On/Off"
    ENDIF
    ON SELECTION POPUP OPTIONS DO Opacts
RETURN
*-- EoP: OptDef

*------------------------------------------------------------------

*--                 Routines acting on Popups

*------------------------------------------------------------------

PROCEDURE Fileacts
* FFILES: POPUP Menu actions
*------------------------------------------------------------------
  Abort = .F.
  DO CASE

    CASE bar() = 1            && select catalog file
      Ocat = Catfile
      DO WHILE .T.
        Catfile = Getfname( Catfile, "catalog of files" )
        IF "" = Catfile .OR. upper( left( Catfile+"   ", 4 ) ) = "NONE"
          Catfile = "None"
          IF NodShake( "No file selected. Try again?", 13,18,2,44,.F.)
            Catfile = ""
            LOOP
          ELSE
            Abort = .T.
            EXIT
          ENDIF
        ENDIF
        IF Catfile # Ocat .AND. lCat
          SELECT Catfile
          USE
          lCat = .F.
        ENDIF
        IF .NOT. lCat
          IF file( Catfile + ".DBF" )
            SELECT select()
            USE ( Catfile ) ALIAS Catfile
            lCat = .T.
            EXIT
          ELSE
            IF NodShake( "File "+ Catfile + " not found.  Create it?",13,18,2,44,.T.)
              DO Makedbf WITH Catfile, Strufile, "ACat"
              USE
              USE ( Catfile ) ALIAS Catfile
              lCat = .T.
              EXIT
            ELSE
              IF NodShake( "Try again?",13,18,2,44,.F.)
                Catfile = ""
                LOOP
              ELSE
                Catfile = "None"
                Abort = .T.
                EXIT
              ENDIF
            ENDIF
          ENDIF
        ELSE
          SELECT Catfile
          EXIT
        ENDIF
      ENDDO
      IF .NOT. Abort .AND. lCat .AND. tagcount() < 3
        INDEX ON Procfile TAG Sources FOR Used = "S"
        INDEX ON Procfile TAG Targets FOR Used = "T"
      ENDIF

    CASE bar() = 2            && select source file
      IF .NOT. lCat
        DO PAUSE WITH "We need a catalog file first.  Please select one"
        Abort = .T.
      ENDIF
      IF .NOT. Abort
        Osrce = Source
        Addfile = .F.
        DO WHILE .T.
          SELECT Catfile
          SET ORDER TO Sources
          ACTIVATE POPUP Catfiles
          IF bar() = 0
            Source = "None"
            IF NodShake( "No file selected from catalog.  Add one to it?",13,18,2,46,.T.)
              Source = Getfname( space( 12 ),"source file to add" )
              Addfile = .T.
            ELSE
              IF NodShake( "No file selected. Try again?", 13,18,2,44,.F.)
                Source = ""
                SET ORDER TO ProcFile
                LOOP
              ELSE
                Abort = .T.
                EXIT
              ENDIF
            ENDIF
          ELSE
            Source = Procfile
          ENDIF
          IF Source # "None"
            IF Source = Osrce .AND. nSrce > 0
              EXIT
            ELSE
              IF nSrce > 0
                Trash = fclose( nSrce )
                nSrce = 0
              ENDIF
            ENDIF
            IF .NOT. file( Source )
              IF NodShake( "File "+ Source +" cannot be found.  Try again?",13,18,2,44,.F.)
                Source = ""
                LOOP
              ELSE
                Source = "None"
                Abort = .T.
                EXIT
              ENDIF
            ELSE
              IF Used = "T" .AND. .NOT. Addfile
                IF .NOT. NodShake( "  File "+ Source + " was last used as a target." + ;
                    "  OK to use it as a source?", 13,18,2,44,.T.)
                  Source = ""
                  SET ORDER TO Sources
                  LOOP
                ENDIF
              ENDIF
            ENDIF
            nSrce = fopen( Source, "r" )
            IF nSrce = 0
              IF Nodshake( "Cannot open file "+ Source+".  Try again?",13,18,2,44,.F.)
                Source = ""
                LOOP
              ELSE
                Source = "None"
                Abort = .T.
                EXIT
              ENDIF
            ENDIF
          ENDIF
          IF Source # "None" .AND. ( Addfile .OR. Used = "T" )
            SET ORDER TO ProcFile
            IF .NOT. seek( Source )
              APPEND BLANK
              REPLACE Procfile WITH Source,Date WITH date(),Used WITH "S"
            ELSE
              REPLACE Used WITH "S"
            ENDIF
          ENDIF
          EXIT
        ENDDO
      ENDIF

    CASE bar() = 3            && select library target file
      IF .NOT. lCat
        DO PAUSE WITH "We need a catalog file first.  Please select one"
        Abort = .T.
      ENDIF
      IF .NOT. Abort
        Otarg = Target
        Addfile = .F.
        DO WHILE .T.
          SELECT Catfile
          SET ORDER TO Targets
          ACTIVATE POPUP Catfiles
          IF bar() = 0
            Target = "None"
            IF NodShake( "No file selected from catalog.  Add one to it?",13,18,2,46,.T.)
              Target = Getfname( space( 12 ),"target file to add" )
              Addfile = .T.
            ELSE
              IF NodShake( "No file selected. Try again?", 13,18,2,44,.F.)
                Target = ""
                SET ORDER TO Procfile
                LOOP
              ELSE
                Abort = .T.
                EXIT
              ENDIF
            ENDIF
          ELSE
            Target = Procfile
          ENDIF
          Maketarg = .F.
          IF Target # "None"
            IF Target = Otarg .AND. Ntarg > 0
              EXIT
            ELSE
              IF nTarg > 0
                Trash = fclose( nTarg )
                nTarg = 0
              ENDIF
            ENDIF
            IF .NOT. file( Target )
              IF NodShake( "File "+ Target +" cannot be found.  Create it?",13,18,2,46,.F.)
                Maketarg = .T.
              ELSE
                IF NodShake( "File "+ Target +" cannot be found.  Try again?",13,18,2,46,.F.)
                  Target = ""
                  LOOP
                ELSE
                  Target = "None"
                  Abort = .T.
                  EXIT
                ENDIF
              ENDIF
            ELSE
              IF Used = "S" .AND. .NOT. Addfile
                IF .NOT. NodShake( "File "+ Target + " was last used as a source." + ;
                    "  OK to use it as a target?", 13,18,2,45,.T.)
                  Target = ""
                  SET ORDER TO Sources
                  LOOP
                ENDIF
              ENDIF
            ENDIF
          ENDIF
          IF Target # "None"
            IF Maketarg
              nTarg = fcreate( Target, "rw" )
            ELSE
              nTarg = fopen( Target, "rw" )
            ENDIF
            IF nTarg = 0
              IF Nodshake( "Cannot open file "+ Target+".  Try again?",13,18,2,44,.F.)
                Target = ""
                LOOP
              ELSE
                Target = "None"
                Abort = .T.
                EXIT
              ENDIF
            ENDIF
          ENDIF
          IF Target # "None" .AND. ( Addfile .OR. Used = "S" )
            SET ORDER TO Procfile
            IF .NOT. seek( Target )
              APPEND BLANK
              REPLACE Procfile WITH Target,Date WITH date(),Used WITH "T"
            ELSE
              REPLACE Used WITH "T"
            ENDIF
          ENDIF
          Ltype = .T.
          EXIT
        ENDDO
      ENDIF

    CASE bar() = 4                           && select program target file
      Otarg = Target
      DO WHILE .T.
        Target = Getfname( Target, "program target file" )
        IF "" # Target .AND. upper( left( Target +"   ", 4 ) ) # "NONE"
          IF .NOT. "." $ Target
            Target = Target + ".PRG"
          ENDIF
          IF Target = Otarg .OR. file( Target )
            EXIT
          ENDIF
        ENDIF
        IF NodShake("File "+Target+" cannot be found.  Try again?",13,18,2,44,.F.)
          Target = space( 12 )
          LOOP
        ELSE
          Target = "None"
          Abort = .T.
          EXIT
        ENDIF
      ENDDO
      IF .NOT. Abort .AND. Target # Otarg
        IF nTarg > 0
          Trash = fclose( nTarg )
          nTarg = 0
        ENDIF
        nTarg = fopen( Target, "rw" )
        Ltype = .F.
      ENDIF

    CASE bar() = 5                           && choose description file
      DO WHILE .T.
        Descfile = Getfname( Descfile, "description file" )
        IF "" # Descfile .AND. upper( left( Descfile+"   ", 4 ) ) # "NONE"
          IF .NOT. "." $ Descfile
            Descfile = Descfile + ".TXT"
          ENDIF
          IF file( Descfile )
            lDesc = .T.
            EXIT
          ENDIF
        ENDIF
        IF NodShake("File "+Descfile+" cannot be found.  Try again?",13,18,2,44,.F.)
          Descfile = space( 12 )
          LOOP
        ELSE
          Descfile = "None"
          lDesc = .F.
          EXIT
        ENDIF
      ENDDO
      Abort = .NOT. lDesc

    CASE bar() = 6            && quit
      gc_quit = 'Q'
      DEACTIVATE MENU

  ENDCASE
  SET MESSAGE TO
  DO Showstatus WITH Abort

RETURN
*-- EoP: Fileacts

PROCEDURE Upacts
* UPDATES: POPUP Menu Actions.
*---------------------------------------------------------------------
  Abort = .F.
  DO CASE

    CASE bar() = 1            && add an entire source file to target
      IF nSrce * nTarg = 0
        IF nSrce = 0
          DO Pause WITH "You must select a source file first"
          IF nTarg = 0
            ?? chr( 7 )
          ENDIF
        ENDIF
        IF nTarg = 0
          DO Pause WITH "You must select a target file first"
        ENDIF
        Abort = .T.
      ELSE
        IF Ltype
          DO Tdex
        ENDIF
        DO Sdex
        DO Addit WITH "*"
      ENDIF
      gn_error = 0

    CASE bar() = 2            && add a single routine to target
      IF nSrce * nTarg = 0
        IF nSrce = 0
          DO Pause WITH "You must select a source file first"
          IF nTarg = 0
            ?? chr( 7 )
          ENDIF
        ENDIF
        IF nTarg = 0
          DO Pause WITH "You must select a target file first"
        ENDIF
        Abort = .T.
      ELSE
        IF Ltype
          DO Tdex
        ENDIF
        DO Sdex
        ACTIVATE POPUP Routines
        IF "" # ProcName
          DO Addit WITH upper( left( ProcType, 1 ) + ProcName )
          gn_error = 0
        ELSE
          DO Pause WITH "No routine selected.  Returning."
        ENDIF
      ENDIF

    CASE bar() = 3                              && freshen all
      IF .not. lCat
        DO Pause WITH "We need a catalog file - Please select one"
        Abort = .T.
      ENDIF
      IF .NOT. Abort .AND. .NOT. Ltype
        DO Pause WITH "You must select a library target file first"
        Abort = .T.
      ENDIF
      IF .NOT. Abort .AND. nTarg = 0
        DO Pause WITH "You must select a target file first"
        Abort = .T.
      ENDIF
      IF .NOT. Abort
        SELECT Catfile
        SET ORDER TO Sources
        Ocrash = Crash
        IF Crash = "Always"
          Crash = "If_Older"
        ENDIF
        SCAN
          IF nSrce > 0
            Trash = fclose( nSrce )
            nSrce = 0
          ENDIF
          Source = Filename
          nSrce = fopen( Source,"r" )
          IF nSrce > 0
            DO Sdex
            DO Addit WITH "*"
          ENDIF
        ENDSCAN
        Crash = Ocrash
        gn_error = 0
      ENDIF

    CASE bar() = 4                              && delete a routine
      Abort = .F.
      IF .NOT. Ltype .OR. Target = "None"
        DO Pause WITH "You must select a library target file first"
        Abort = .T.
      ENDIF
      IF .NOT. Abort .AND. nTarg = 0
        nTarg = fopen( Target,"rw" )
        IF nTarg = 0
          DO PAUSE WITH "No target file to delete routines from!"
          Abort = .T.
        ENDIF
      ENDIF
      IF nTarg > 0 .AND. .NOT. Abort
        DO Tdex
        ACTIVATE POPUP Routines
        IF "" = ProcName
          DO Pause WITH "No routine selected.  Returning."
        ELSE
          DO Delone WITH upper( left( ProcType, 1 ) + ProcName )
        ENDIF
      ENDIF

    CASE bar() = 5                              && delete file from catalog
      IF .not. lCat
        DO Pause WITH "We need a catalog file - Please select one"
        Abort = .T.
      ENDIF
      IF .NOT. Abort
        SELECT Catfile
        SET ORDER TO ProcFile
        DO WHILE .T.
          ACTIVATE POPUP Catfiles
          IF bar() = 0
            IF NodShake( "No file selected. Try again?", 13,18,2,44,.F.)
              LOOP
            ELSE
              Abort = .T.
              EXIT
            ENDIF
          ELSE
            IF NodShake( "Delete file "+ ProcFile + " from catalog?",13,18,2,44,.T.)
              DELETE NEXT 1
              PACK
            ENDIF
            IF .NOT. NodShake( "  Delete another?  ", 13,18,2,44,.T.)
              EXIT
            ENDIF
          ENDIF
        ENDDO
      ENDIF

    CASE Bar() = 6                              && split target file
      Abort = .F.
      IF .not. lCat
        DO Pause WITH "We need a catalog file - Please select one"
        Abort = .T.
      ENDIF
      IF .NOT. Abort .AND. (.NOT. Ltype .OR. nTarg = 0 )
        DO Pause WITH "You must select a library target file first"
        Abort = .T.
      ENDIF
      IF .NOT. Abort
        DO WHILE .T.
          Bpoint = space( 34 )
          ACTIVATE WINDOW Pause
          @ 0,0 SAY "Break point is before designated:"
          @ 1,0 GET Bpoint PICTURE "@M Routine <procedure files only>, Line number, Number of bytes" ;
          MESSAGE "Press SPACE to view choices and "+CHR(17)+CHR(196)+CHR(217)+" to Select"
          READ
          DEACTIVATE WINDOW Pause
          Bpoint = left( Bpoint,1 )
          Break = 0
          DO CASE
            CASE Bpoint = "R"
              DO Tdex
              ACTIVATE POPUP Routines
              IF Bar() = 0
                IF NodShake( "No routine selected. Try again?", 13,18,2,44,.F.)
                  LOOP
                ELSE
                  Abort = .T.
                  EXIT
                ENDIF
              ELSE
                Break = Bytes
                Bpoint = "N"
                ACTIVATE WINDOW PAUSE
                @ 0,0 SAY "Breaking at "+ ProcType + ProcName
              ENDIF
            CASE Bpoint = "L"
              ACTIVATE WINDOW PAUSE
              @ 0,0 SAY "Enter the number of the line to break at: " GET Break Range 0,
            CASE Bpoint = "N"
              ACTIVATE WINDOW PAUSE
              @ 0,0 SAY "Enter the number of the byte to break at: " GET Break Range 0, fsize( nTarg )
            OTHERWISE
              LOOP
          ENDCASE
          EXIT
        ENDDO
      ENDIF
      IF .NOT. Abort
        cF1 = Target
        nAt = at( ".", cF1 )
        IF nAt > 0 .AND. nAt < 9
          cF1 = left( Target, nAt - 1 ) + "1" + right( Target, 4 )
          cF2 = left( Target, nAt - 1 ) + "2" + right( Target, 4 )
        ELSE
          STORE "" TO cF1,cF2
        ENDIF
        cF1 = left( cF1 + space(30), 30 )
        cF2 = left( cF2 + space(30), 30 )
        DO WHILE .T.
          @ 1,0 SAY "Enter name of the new first part:  " GET cF1
          @ 2,0 SAY "Enter name of the new second part: " GET cF2
          READ
          cF1 = upper( ltrim( trim( cF1 ) ) )
          cF2 = upper( ltrim( trim( cF2 ) ) )
          IF cF1 = cF2
            IF NodShake ("Names of parts must be different.  Try again?",13,18,2,44,.F.)
              LOOP
            ELSE
              Abort = .T.
              EXIT
            ENDIF
          ENDIF
          New1 = .T.
          IF cF1 # Target .AND. file( cF1 )
            IF NodShake( cF1 + " exists - unable to write to it.  Try Again?", 13,18,2,44,.T.)
              LOOP
            ELSE
              Abort = .T.
              EXIT
            ENDIF
          ENDIF
          IF cF2 = Target
            Writefile = TempName( ".PRG" )
            EXIT
          ENDIF
          Writefile = cF2
          IF .NOT. file( Writefile )
            EXIT
          ENDIF
          nWrite = fopen( Writefile, "w" )
          IF nWrite > 0
            EXIT
          ENDIF
          IF NodShake("Unable to open "+ cF2 + ".  Try Again?", 13,18,2,44,.T.)
            LOOP
          ELSE
            Abort = .T.
            EXIT
          ENDIF
        ENDDO
        DEACTIVATE WINDOW Pause
      ENDIF
      IF .NOT. Abort
        IF file( Writefile )
          IF NodShake( "Overwrite "+ Writefile +"? <No to append>",13,18,2,44,.T.)
            Trash = fseek( nWrite, 0, 0 )
          ELSE
            Trash = fseek( nWrite, 0, 2 )
          ENDIF
        ELSE
          nWrite = fcreate( Writefile, "w" )
        ENDIF
        * seek byte or line of break
        IF Bpoint = "N"
          Trash = fseek( nTarg, Break, 0 )
        ELSE
          Gotlines = 0
          Trash = fseek( nTarg, 0, 0 )
          DO WHILE Gotlines < Break
            Trash = fgets( nTarg )
            Gotlines = Gotlines + 1
          ENDDO
        ENDIF
        * mark spot as new end of file
        Newend = fseek( nTarg, 0 , 1 )
        * copy the rest of the file
        DO WHILE .NOT. feof( nTarg )
          cLine = fgets( nTarg )
          Trash = fputs( nWrite, cLine )
        ENDDO
        * close one, truncate and close the other
        Trash = fclose( nWrite )
        Trash = fseek( nTarg, Newend, 0 )
        Trash = fwrite( nTarg, "", 0 )
        Trash = fclose( nTarg )
        * and rename if needed
        IF Writefile = cF2
          IF Target # cF1
            RENAME ( Target ) TO ( cF1 )
          ENDIF
        ELSE
          RENAME ( Target ) TO ( cF1 )
          RENAME ( Writefile ) TO ( Target )
        ENDIF
        IF select( "Tindex" ) > 0
          SELECT Tindex
          USE
        ENDIF
        ERASE ( Tindex ) + ".DBF"
        ERASE ( Tindex ) + ".MDX"
        SELECT Catfile
        SEEK Target
        REPLACE ProcFile WITH cF1, Indxfile WITH " ", Date WITH {}
        APPEND BLANK
        REPLACE ProcFile WITH cF2, Indxfile WITH " ", Date WITH {}
        nTarg = 0
        Target = "None"
        Otarg = ""
      ENDIF

  ENDCASE
  SET MESSAGE TO
  DO Showstatus WITH Abort

RETURN
*-- EoP: Upacts

PROCEDURE Repacts
* REPORTS: POPUP Menu Actions.
*----------------------------------------------------------------
  Abort = .F.
  DO CASE

    CASE bar() = 1              &&  changed/added routines
      IF nSrce * nTarg = 0
        IF nSrce = 0
          DO Pause WITH "You must select a source file first"
          IF nTarg = 0
            ?? chr( 7 )
          ENDIF
        ENDIF
        IF nTarg = 0 .OR. .NOT. Ltype
          DO Pause WITH "You must select a library target file first"
        ENDIF
        Abort = .T.
      ELSE
        DO Tdex
        DO Sdex
        SET CURSOR OFF
        ACTIVATE WINDOW Counting
        STORE 0 TO nAdded, nChanged
        @ 3,5 SAY nAdded PICTURE "######"
        @ 3,12 SAY " routines added,"
        @ 3,29 SAY nChanged PICTURE "######"
        @ 3,36 SAY " changed."
        SCAN
          IF seek( upper( left( ProcType, 1 ) + ProcName ), "Tindex" )
            IF Date <= Tindex->Date
              LOOP
            ELSE
              @ 0,0 CLEAR TO 0,57
              @ 0,1 SAY trim( ProcType  ) +" " + trim( ProcName ) +" has been changed"
              nChanged = nChanged + 1
              @ 3,29 SAY nChanged PICTURE "######"
              IF gc_alt = "ON"
                SET CONSOLE OFF
                ? trim( ProcType  ) +" " + trim( ProcName ) +" has been changed"
                SET CONSOLE ON
              ENDIF
            ENDIF
          ELSE
            @ 0,0 CLEAR TO 0,57
            @ 0,1 SAY trim( ProcType  ) +" " + trim( ProcName ) +" has been added"
            nAdded = nAdded + 1
            @ 3,5 SAY nAdded PICTURE "######"
            IF gc_alt = "ON"
              SET CONSOLE OFF
              ? trim( ProcType  ) +" " + trim( ProcName ) +" has been added"
              SET CONSOLE ON
            ENDIF
          ENDIF
        ENDSCAN
        DEACTIVATE WINDOW Counting
        IF gc_alt = "ON"
          SET CONSOLE OFF
          ? nAdded PICTURE "######" AT 5,
          ?? " routines added, ",
          ?? nChanged PICTURE "######",
          ?? " changed."
          SET CONSOLE ON
        ENDIF
        SET CURSOR ON
      ENDIF

    CASE bar() = 2              &&  print description file
      if .NOT. lDesc
        DO PAUSE WITH "No description file selected."
        Abort = .T.
      ELSE
        IF gc_dev = "PRN"
          TYPE ( Descfile ) TO PRINTER
        ELSE
          nHandle = fopen( Descfile, "r" )
          IF nHandle = 0
            DO PAUSE WITH "Cannot open " + Descfile
          ELSE
            SAVE SCREEN TO Sscr
            SET CLOCK OFF
            CLEAR
            Outlines = 1
            DO WHILE .NOT. feof( nHandle )
              cLine = fgets( nHandle )
              ? cLine
              Outlines = Outlines + 1
              IF mod( Outlines, Showlines ) = 0
                WAIT
              ENDIF
            ENDDO
            IF mod( Outlines, Showlines ) > 0
              DO WHILE mod( Outlines, Showlines ) > 0
                ?
                Outlines = Outlines + 1
              ENDDO
              WAIT
            ENDIF
            RESTORE SCREEN FROM Sscr
            SET CLOCK ON
            Trash = fclose( nHandle )
          ENDIF
        ENDIF
      ENDIF

    CASE Bar() = 3 .OR. Bar() = 4             && source/target text
      Abort = .F.
      IF Bar() = 3
        IF nSrce = 0
          DO Pause WITH "You must select a source file first"
          Abort = .T.
        ELSE
          nHandle = nSrce
          cFname  = Source
          DO Sdex
        ENDIF
      ELSE
        IF nTarg = 0 .OR. .NOT. Ltype
          DO Pause WITH "You must select a library target file first"
          Abort = .T.
        ELSE
          nHandle = nTarg
          cFname  = Target
          DO Tdex
        ENDIF
      ENDIF
      IF .NOT. Abort
        DO WHILE .T.
          ACTIVATE POPUP Routines
          IF Bar() = 0
            IF NodShake( "No routine selected. Try again?", 13,18,2,44,.F.)
              LOOP
            ELSE
              Abort = .T.
              EXIT
            ENDIF
          ELSE
            EXIT
          ENDIF
        ENDDO
      ENDIF
      IF .NOT. Abort
        Trash = fseek( nHandle, Bytes, 0 )
        Rlines = Line
        Atrec = recno()
        IF recno() < reccount()
          GO Atrec + 1
          Rlines = Line - Rlines
          cCond = "Rlines > 0"
        ELSE
          cCond = ".not. feof( nHandle )"
        ENDIF
        IF gc_dev = "CON"
          SAVE SCREEN TO Sscr
          SET CLOCK OFF
          CLEAR
        ENDIF
        Outlines = 1
        DO WHILE &cCond
          cLine = fgets( nHandle )
          ? cLine
          Rlines = Rlines - 1
          Outlines = Outlines + 1
          IF gc_dev = "CON" .AND. mod( Outlines, Showlines ) = 0
            WAIT
          ENDIF
        ENDDO
        IF gc_dev = "CON"
          IF mod( Outlines, Showlines ) > 0
            DO WHILE mod( Outlines, Showlines ) > 0
              ?
              Outlines = Outlines + 1
            ENDDO
            WAIT
          ENDIF
          RESTORE SCREEN FROM Sscr
          SET CLOCK ON
        ENDIF
      ENDIF

    CASE Bar() = 5 .OR. Bar() = 6                  && print index .dbf
      Abort = .F.
      IF Bar() = 5
        IF nSrce = 0
          DO Pause WITH "You must select a source file first"
          Abort = .T.
        ELSE
          DO Sdex
        ENDIF
      ELSE
        IF nTarg = 0 .OR. .NOT. Ltype
          DO Pause WITH "You must select a library target file first"
          Abort = .T.
        ELSE
          DO Tdex
        ENDIF
      ENDIF
      IF .NOT. Abort
        IF NodShake( "Print in alphabetic order? <No for natural>",13,18,2,44,.F.)
          SET ORDER TO Prname
          GO TOP
        ELSE
          SET ORDER TO
          GO 3
        ENDIF
        IF gc_dev = "PRN"
          LIST REST TO PRINT
        ELSE
          SET CLOCK OFF
          SAVE SCREEN TO Sscr
          CLEAR
          DISP REST
          WAIT
          RESTORE SCREEN FROM Sscr
          SET CLOCK ON
        ENDIF
      ENDIF

  ENDCASE
  SET MESSAGE TO
  DO Showstatus WITH Abort

RETURN
*-- EOP: Repacts - Menu REPORTS

PROCEDURE Opacts
*-- OPTIONS: POPUP Menu Actions.
*---------------------------------------------------------------------
  DO CASE

    CASE bar() = 1              && printer options
      DO Printset

    CASE bar() = 2              && change alternate
      Altfile = Getfname( Reptfile,"Alternate File")
      IF Altfile # Reptfile
        CLOSE ALTERNATE
        SET ALTERNATE TO ( Altfile )
        Reptfile = Altfile
      ENDIF

    CASE bar() = 3              && toggle alternate
      IF NodShake( "  ALTERNATE is "+gc_alt+". Set it "+ iif( gc_alt ="ON","OFF?  ","ON?  "),13,18,2,44,.F.)
        gc_alt = iif( gc_alt ="ON","OFF","ON")
        SET ALTERNATE &gc_alt
      ENDIF

    CASE bar() = 4              && overwrite mode
      DEFINE WINDOW Getcrash FROM 8,15 TO 10,64 DOUBLE
      ACTIVATE WINDOW Getcrash
      @ 0,0 SAY "Overwrite routine of same name: " GET Crash ;
        PICTURE "@M Always, Never, If_Older, Ask_User";
        MESSAGE "Press SPACE to view choices and "+CHR(17)+CHR(196)+CHR(217)+" to Select"
        READ
      DEACTIVATE WINDOW Getcrash
      RELEASE WINDOW Getcrash

    CASE bar() = 5             && save status as defaults
      Def_attrs = cAttrs
      Def_cat   = Catfile
      Def_crash = Crash
      Def_desc  = Descfile
      Def_first = .F.
      Def_rept  = Reptfile
      Def_targ  = Target
      SAVE ALL LIKE Def_* TO ( Memofile )

    CASE bar() = 6              && toggle mouse on or off
      IF NodShake( "Mouse is "+iif( Mouse, "ON","OFF")+".  Turn it " ;
        +iif(Mouse,"OFF?","ON?"), 13,18,2,44,.F.)
        IF Mouse
          Trash = call("SetMouse", .F. )
        ELSE
          Trash = call("SetMouse", .T. )
          @ 12,40 SAY " "
        ENDIF
        Mouse = .NOT. Mouse
      ENDIF

  ENDCASE
  SET MESSAGE TO
  DO Showstatus

RETURN
*-- EOP: Opacts - Menu OPTIONS

*--                     Minor Popups
*--------------------------------------------------------------------
PROCEDURE Pickfile
  IF lastkey() # 27
    GO TOP
    SKIP bar() - 1
  ENDIF
  DEACTIVATE POPUP
RETURN
*--

PROCEDURE Pickroute
  IF lastkey() # 27
    GO TOP
    SKIP bar() - 1
  ENDIF
  DEACTIVATE POPUP
RETURN
*--------------------------------------------------------------------

*--          Principal Routines other than Menus and Popups

*--------------------------------------------------------------------
*--                     Major file handling
*--------------------------------------------------------------------

PROCEDURE Addit
*-- Add from source file to target file
PARAMETERS Pwhich
SET CURSOR OFF
ACTIVATE WINDOW Counting
nCopied = 0
DO Copy WITH Pwhich
* and then pick up any called files found
IF reccount( "Callees" ) > 0
  Osource = Source
  SELECT Callees
  GO TOP
  SCAN
    Which = upper( left( Proctype, 1 ) + ProcName )
    IF Fromfile = " " .OR. ( Pwhich = "*" .AND. Fromfile = Source )
      LOOP
    ENDIF
    Orec = recno()
    IF FromFile # Source
      DEACTIVATE WINDOW Counting
      if nSrce > 0
        Trash = fclose( nSrce )
      endif
      Source = FromFile
      IF file( Source )
        nSrce = fopen( Source,"r" )
        DO sDex
      ELSE
        nSrce = 0
      ENDIF
      ACTIVATE WINDOW Counting
    ENDIF
    IF nSrce > 0
      DO Copy WITH Which
      SELECT Callees
    ENDIF
    GO Orec
  ENDSCAN
  ZAP
  IF Osource # Source
    Source = Osource
    if nSrce > 0
      Trash = fclose( nSrce )
    endif
    nSrce = fopen( Source,"r" )
  ENDIF
ENDIF
DEACTIVATE WINDOW Counting
SET CURSOR ON
SELECT Sindex
RETURN
*-- EoP: Addit

PROCEDURE Copy
*--  Copy routines between .prg files
PARAMETERS Which
@ 3,30 SAY ncopied PICTURE "######"
@ 3,37 SAY "routine(s) added"
SELECT Sindex
IF Which = "*"
  SET ORDER TO
  GO 3
  SCAN REST FOR .NOT. deleted()
    cResult = ""
    PrName = upper( left( ProcType, 1 ) + ProcName )
    nOk = CopyOne( PrName )
    @ 0,0 CLEAR TO 0,57
    @ 0,1 SAY trim( ProcType  ) +" " + trim( ProcName ) +" " + cResult
    nCopied = nCopied + nOk
    @ 3,30 SAY ncopied PICTURE "######"
    IF gc_alt = "ON"
      SET CONSOLE OFF
      ? trim( ProcType  ) +" " + trim( ProcName ) +" " + cResult
      SET CONSOLE ON
    ENDIF
  ENDSCAN
ELSE
  cResult = ""
  SET ORDER TO Prname
  SEEK Which
  nOk = CopyOne( Which )
  @ 0,0 CLEAR TO 0,57
  @ 0,1 SAY trim( ProcType  ) +" " + trim( ProcName ) +" " + cResult
  Ncopied = Ncopied + nOk
  @ 3,30 SAY ncopied PICTURE "######"
  IF gc_alt = "ON"
    SET CONSOLE OFF
    ?  trim( ProcType  ) +" " + trim( ProcName ) +" " + cResult
    SET CONSOLE ON
  ENDIF
ENDIF
RETURN
*-- Eop: Copy

FUNCTION CopyOne
*-- Copy one routine
PARAMETERS Prname
IF Crash = "Never"
  cResult = "Not copied - Overwrite is Never"
  RETURN 0
ENDIF
IF Ltype
  SELECT Tindex
  SET ORDER TO Prname
  IF seek( Prname )
    DO CASE
      CASE Crash = "Ask_User"
        IF .not. NodShake( " There's already a " + ProcName ;
          + iif( ProcType = "FUNCTION", " function", " procedure" ) + ;
          iif( Tindex->Date # {}, " dated "+ dtoc( Tindex->Date ),"" ) + ;
          ". Overwrite it? ", 13,10,2,60,.T.)
          cResult = "Not added by user choice"
          RETURN 0
        ENDIF
      CASE Crash = "If_Older"
        if Tindex->Date = {} .OR. Date < Tindex->Date
          cResult = "Not added - older than target"
          RETURN 0
        ENDIF
    ENDCASE
    DO Delone WITH Prname
  ENDIF
ENDIF
cResult = " - added"
RETURN Addone( )
*-- EoP: Copyone

PROCEDURE Delone
*-- Delete one routine from the target file
PARAMETERS Prname
SELECT Tindex
SET ORDER TO Prname
SEEK Prname
Atrec = recno()
Atbytes = Bytes
nLines = Line
SET ORDER TO
IF Atrec = reccount()
  nBytes = fsize( Target ) - Atbytes
  Atend = .T.
ELSE
  GO Atrec + 1
  nLines = Line - nLines
  nBytes = Bytes - Atbytes
  REPLACE Line WITH Line - nLines, Bytes WITH Bytes - nBytes REST
  GO Atrec
  Atend = .F.
ENDIF
DELETE NEXT 1
PACK

* If this is not at the end of the file, copy the rest
* of the file down into its place
* The easy way to do this is to open the file again for reading and maintain
* separate read and write pointers for the two "files".
* However, this does not work.

Writeptr = Atbytes
IF .NOT. Atend
  Trash = fseek( nTarg, Atbytes + nBytes, 0 )
  DO WHILE .NOT. feof( nTarg )
    cLine = fread( nTarg, 254 )              && read some
    Readptr = fseek( nTarg, 0, 1 )           && save read pointer for next
    Trash = fseek( nTarg, Writeptr, 0 )      && move file pointer for write
    Trash = fwrite( nTarg, cLine )           && write some
    Writeptr = fseek( nTarg, 0, 1 )          && save write pointer for next
    Trash = fseek( nTarg, Readptr, 0 )       && move file pointer for read
  ENDDO
ENDIF
* and truncate the file at its new end
Trash = fseek( nTarg, Writeptr, 0 )
Trash = fwrite( nTarg, "", 0 )
Trash = fclose( nTarg )
nTarg = fopen( Target,"rw" )

RETURN
*-- EoP: Delone

FUNCTION Addone
*-- Add one routine from the source file, pointed to by Sindex
SELECT Sindex
Atrec = recno()
IF Atrec = 0
  RETURN 0
ENDIF
Startread = Bytes
IF Atrec = reccount()
  NextSfn = fsize( Source )
ELSE
  GO Atrec + 1
  NextSfn = Bytes
  GO Atrec
ENDIF
nBytes = NextSfn - Startread
IF Ltype
  SELECT Tindex
  IF reccount() > 2
    GO reccount()
    nLines = Line
    Trash = fseek( nTarg, Bytes, 0 )
    DO WHILE .NOT. feof( nTarg )
      cLine = fgets( nTarg )
      nLines = nLines + 1
    ENDDO
  ELSE
    nLines = 1
  ENDIF
ENDIF
Writeptr = fseek( nTarg, 0, 2 )
lCalls =.F.
SELECT Callees
Trash = fseek( nSrce, Startread, 0 )
DO WHILE nBytes > 0
  cLine = fgets( nSrce )
  nBytes = nBytes - fputs( nTarg, cLine )
  cLine = upper( trim( cLine ) )
  cL2 = left( ltrim( cLine ), 9 )
  IF lCalls
    IF left( cL2, 3 ) #  Remark .OR. .NOT. right( cL2, 6 ) = Blanks
      lCalls = .F.
    ENDIF
  ENDIF
  IF cL2 = Remark + Rcalls
    lCalls = .T.
  ENDIF
  IF lCalls
    cL2 = substr( cLine, 19 )
    Callee = ParseWord( cL2 )
    IF .NOT. Callee = Nocalls
      STORE " " TO Ptype, Frfile
      IF "(" $ Callee
        Callee = left( Callee, at( "(", Callee ) - 1 )
        Ptype = "F"
      ELSE
        Ptype = left( StripWord( CL2 ), 1 )
        IF Ptype = "("
          Ptype = "F"
        ENDIF
      ENDIF
      IF Fromflag $ cL2
        cL2 = substr( cL2, at( Fromflag, cL2 ) + 4)
        Frfile = ParseWord( CL2 )
      ENDIF
      IF Ptype $ "FP" .AND. Frfile # " "
        APPEND Blank
        REPLACE ProcName WITH Callee, FromFile WITH Frfile, ProcType WITH Ptype
      ENDIF
    ENDIF
  ENDIF
ENDDO
IF Ltype
  SELECT Tindex
  APPEND BLANK
  REPLACE ProcType WITH Sindex->ProcType, ProcName WITH Sindex->ProcName,;
     Date WITH Sindex->Date, Line WITH nLines, Bytes WITH Writeptr
ENDIF
SELECT Sindex
RETURN 1
*-- EoF: Addone

*---------------------------------------------------------------------
*--                Heart of the Matter - indexing .prg files
*---------------------------------------------------------------------

PROCEDURE Sdex
*-- be sure source file is indexed, or index it
IF Source # Osrce
  IF select( "Sindex" ) > 0
    SELECT Sindex
    USE
  ENDIF
  lSndx = .F.
ENDIF
IF .NOT. lSndx
  SELECT Catfile
  SET ORDER TO Sources
  SEEK Source
  IDate = Date
  Sindex = Indxfile
  lSndxOpen = .F.
  IF " " # Sindex .AND. file ( Sindex +".DBF" )
    lSndxOpen = .T.
    SELECT select()
    USE ( Sindex ) ALIAS Sindex
    IF .NOT. file( Sindex + ".MDX" )
      INDEX ON upper( left( ProcType, 1 )+ Procname ) TAG Prname FOR recno() > 2
    ENDIF
    SET ORDER TO
    GO TOP
    IF Idate >= fdate( Source ) .AND. fldcount() = 5
      IF field( 1 ) = "PROCTYPE" .AND. field( 2 ) = "PROCNAME" .AND. field( 3 ) ;
        = "LINE" .AND. field( 4 ) = "BYTES" .AND. field( 5 ) = "DATE" .AND. ;
        ProcType = "*File:" .AND. ProcName = Source
        lSndx = .T.
      ENDIF
    ENDIF
  ENDIF
ENDIF
* if not indexed, index it
IF .NOT. lSndx
  IF .NOT. lSndxopen
    SELECT select()
    Sindex = Source
    IF "." $ Sindex
      Sindex = left( Sindex, at( ".", Sindex ) - 1 )
    ENDIF
    IF file ( Sindex + ".MDX" )
      Sindex = tempname("DBF")
    ENDIF
    DO Makedbf WITH Sindex, Strufile, "Aindex"
    USE
    USE ( Sindex ) ALIAS Sindex
    APPEND BLANK
    REPLACE ProcType WITH "*File:", ProcName WITH Source, Date WITH date()
    APPEND BLANK
    REPLACE ProcType WITH Catfile
    INDEX ON upper( left( ProcType, 1 ) + ProcName ) TAG Prname FOR recno() > 2
    lsndxopen = .T.
  ENDIF
  DO Indexfile WITH nSrce
  SELECT Made
  APPEND BLANK
  REPLACE ProcName WITH ( Sindex ), FromFile WITH Source
  lSndx = .T.
  * and note new index date in the file of filenames
  SELECT Catfile
  REPLACE Indxfile WITH Sindex, Date WITH date()
ENDIF
Osrce = Source
SELECT Sindex
SET ORDER TO Prname
RETURN
*-- EoP: Sdex

PROCEDURE Tdex
*-- be sure target file is indexed, or index it
IF Otarg # Target
  IF select( "Tindex" ) > 0
    SELECT Tindex
    USE
  ENDIF
  lTndx = .F.
ENDIF
IF .NOT. lTndx
  SELECT Catfile
  SET ORDER TO Targets
  IF .NOT. seek( Target )
    DO Pause WITH "Target file not in catalog -- please add it"
    gn_error = 2
    RETURN
  ENDIF
  IDate = Date
  Tindex = Indxfile
  lTndxOpen = .F.
  IF file( Tindex +".DBF" )
    lTndxOpen = .T.
    SELECT select()
    USE ( Tindex ) ALIAS Tindex
    IF .NOT. file( Tindex + ".MDX" )
      INDEX ON upper( left( ProcType, 1 ) + ProcName ) TAG Prname FOR recno() > 2
    ENDIF
    SET ORDER TO
    GO TOP
    IF Idate >= fdate( Target ) .AND. fldcount() = 5
      IF field( 1 ) = "PROCTYPE" .AND. field( 2 ) = "PROCNAME" .AND. field( 3 ) ;
        = "LINE" .AND. field( 4 ) = "BYTES" .AND. field( 5 ) = "DATE" .AND. ;
        ProcType = "*File:" .AND. ProcName = Target
        lTndx = .T.
      ENDIF
    ENDIF
  ENDIF
ENDIF
* if not indexed, index it
IF .NOT. lTndx
  IF .NOT. lTndxopen
    SELECT select()
    Tindex = Target
    IF "." $ Tindex
      Tindex = left( Tindex, at( ".", Tindex ) - 1 )
    ENDIF
    IF file ( Tindex + ".MDX" )
      Tindex = tempname("DBF")
    ENDIF
    DO Makedbf WITH Tindex, Strufile, "Aindex"
    USE
    USE ( Tindex ) ALIAS Tindex
    APPEND BLANK
    REPLACE ProcType WITH "*File:", ProcName WITH Target, Date WITH date()
    APPEND BLANK
    REPLACE Proctype WITH Catfile
    INDEX ON upper( left( ProcType, 1 ) + ProcName ) TAG Prname FOR recno() > 2
    lTndxopen = .T.
  ENDIF
  DO Indexfile WITH nTarg
  SELECT Made
  APPEND BLANK
  REPLACE ProcName WITH ( Tindex ), FromFile WITH Target
  lTndx = .T.
  * and note new index date in the file of filenames
  SELECT Catfile
  REPLACE Indxfile WITH Tindex, Date WITH date()
ENDIF
Otarg = Target
SELECT Tindex
SET ORDER TO Prname
RETURN
*-- EoP: Tdex

PROCEDURE Indexfile
*-- index a file using its handle to the .dbf in current work area
  parameters nHandle
  SET ORDER TO
  DELETE ALL FOR recno() > 2
  PACK
  STORE 0 TO nAtline, nIndexed, nTopline
  SET CURSOR OFF
  ACTIVATE WINDOW Counting
  @ 1,5 SAY "Building indexing .dbf"
  Trash = fseek( nHandle, 0, 0 )
  do while .not. feof( nHandle )
    STORE "" TO cType, cName
    dDate = {}
    STORE 0 TO nLines, nBytes, nTopline
    DO Getfn
    IF cType $ "PF" .AND. .NOT. "" = cName
      APPEND BLANK
      REPLACE Proctype WITH iif( cType = "P", "PROCEDURE","FUNCTION"), ;
        Procname WITH cName, Date WITH dDate, Bytes WITH nBytes, ;
        Line With nTopline + nAtline
      nAtline = nAtline + nLines
      nIndexed = nIndexed + 1
      @ 0,0 CLEAR TO 2,30
      @ 0,0 SAY "Indexing "+ trim( Proctype )+ " " + cName
      @ 3,10 SAY nIndexed PICTURE "######"
      @ 3,17 SAY "routines indexed"
      IF gc_alt = "ON"
        SET CONSOLE OFF
          ? "Indexing "+ trim( Proctype )+ " " + cName
          ? space(5), nIndexed PICTURE "######","routines indexed"
          ?
        SET CONSOLE ON
      ENDIF
    ENDIF
  ENDDO
  DEACTIVATE WINDOW Counting
  GO TOP
  REPLACE Date WITH date()
  SET CURSOR ON
RETURN
*-- EoP: Indexfile

*-------------------------------------------------------------------
*--                     Other special-purpose routines
*-------------------------------------------------------------------

PROCEDURE Getfn
*-- find type, date, name and place of next routine in a file
DO WHILE .not. feof( nHandle )
  nBytes = fseek( nHandle, 0 , 1 )
  cLine = trim( ltrim( fgets( nHandle ) ) )
  nLines = nLines + 1
  if upper( left( cLine, 4 ) ) $ "PROC FUNC"
    nTopLine = nLines
    exit
  endif
enddo
cType = upper( left( cLine, 1 ) )
cName = upper( ParseWord( StripWord ( cLine ) ) )
do while .not. feof( nHandle )
  cLine = trim( ltrim( fgets( nHandle ) ) )
  nLines = nLines + 1
  if left( cLine, 3 ) # Remark
    exit
  else
    if upper( ltrim( substr( cLine, 5, 4 ) ) ) = Rdate
      cLine = ltrim( substr( cLine, at( ":", Cline) + 1 ) )
      dDate = ctod( ParseWord( cLine ) )
      exit
    endif
  endif
ENDDO
RETURN
*-- EoP: Getfn

FUNCTION Getfname
*-- get a file or other name from user
  PARAMETERS Name, Msg
  Fname = left( Name + space(30), 30 )
  ACTIVATE WINDOW PAUSE
  @ 0,3 SAY "Enter the name of the "+Msg
  @ 1,3 GET Fname PICTURE replicate( "X", 30 )
  READ
  DEACTIVATE WINDOW PAUSE
RETURN upper( ltrim( trim( Fname ) ) )
*-- EoF: Getfname

FUNCTION Testrout
*--  Precede name of a routine with what it is
PARAMETERS Name
PRIVATE PrName
PrName = trim( ltrim( upper( Name ) ) )
IF "(" $ PrName
  PrName = "F" + PrName
ELSE
  IF Nodshake( " Is " + PrName + " a function? ",13,18,2,44,.F.)
    PrName = "F" + PrName
  ELSE
    PrName = "P" + PrName
  ENDIF
ENDIF
RETURN Prname
*-- EoF: Getrout

*---------------------------------------------------------------------

*--            Library-type routines from App Gen

*---------------------------------------------------------------------
PROCEDURE Pause
PARAMETER pc_msg
*---------------------------------------------------------------------
* Procedure to display a message or errors in a window
* Parameters : pc_msg = message line
*---------------------------------------------------------------------
  PRIVATE lc_msg

  IF TYPE("lc_message")="U"
    gn_error=ERROR()
  ENDIF
  lc_msg = pc_msg
  lc_option='0'

  ACTIVATE WINDOW Pause
  IF gn_error > 0
    IF TYPE("lc_message")="U"
      @ 0,1 SAY [An error has occurred !! - Error message: ]+MESSAGE()
    ELSE
      @ 0,1 SAY [Error # ]+lc_message
    ENDIF
  ENDIF

  @ 1,1 SAY lc_msg
  WAIT " Press any key to continue..."
  DEACTIVATE WINDOW Pause

RETURN
*-- EOP: Pause WITH pc_msg

PROCEDURE PrintSet
*---------------------------------------------------------------------
* Procedure for toggling printer output
*---------------------------------------------------------------------
  PRIVATE lc_window

  gc_dev='CON'
  lc_choice=' '
  gn_pkey=0
  gn_send=3

  DEFINE WINDOW printemp FROM 08,25 TO 17,56

  DEFINE POPUP SavePrin FROM 10,40
    DEFINE BAR 1 OF SavePrin PROMPT " Send output to ..." SKIP
    DEFINE BAR 2 OF SavePrin PROMPT REPLICATE(CHR(196),24) SKIP
    DEFINE BAR 3 OF SavePrin PROMPT " CON:   Console" MESSAGE "Send output to Screen"
    DEFINE BAR 4 OF SavePrin PROMPT " LPT1:  Parallel port 1 " MESSAGE "Send output to LPT1:"
    DEFINE BAR 5 OF SavePrin PROMPT " LPT2:  Parallel port 2" MESSAGE "Send output to LPT2:"
    DEFINE BAR 6 OF SavePrin PROMPT " COM1:  Serial port 1" MESSAGE "Send output to COM1:"
  ON SELECTION POPUP SavePrin DEACTIVATE POPUP

  lc_window = WINDOW()                  && Capture current window name
  IF .NOT. ISBLANK( lc_window )         && If window was active
    ACTIVATE SCREEN                     && Activate screen for correct popup posit
  ENDIF

  ACTIVATE POPUP SavePrin
  gn_send = BAR()
  RELEASE POPUP SavePrin

  IF .NOT. ISBLANK( lc_window )         && If window was active before
    ACTIVATE WINDOW &lc_window.         && Reactivate it
  ENDIF

  IF gn_send <> 0                       && If user made a popup selection

    IF gn_send <> 3                   && Output not to the screen
      gc_dev = 'PRN'
      ON ERROR DO prntrtry
      DO CASE
        CASE gn_send = 4
          SET PRINTER TO LPT1
        CASE gn_send = 5
          SET PRINTER TO LPT2
        CASE gn_send = 6
          SET PRINTER TO COM1
      ENDCASE
      SET PRINT ON
      ON ERROR
    ENDIF

  ELSE
    gn_pkey = 27                        && Signal escape pressed to caller
  ENDIF

  RELEASE WINDOW printemp

RETURN
*-- EOP: PrintSet

PROCEDURE PrntRtry
*---------------------------------------------------------------------
* On error routine for handling printer errors.
*---------------------------------------------------------------------
  PRIVATE ll_escape
  ll_escape = SET("ESCAPE") = "ON"

  IF .NOT. PRINTSTATUS()                && If printer not ready

    IF ll_escape                        && If Escape on, set it off
      SET ESCAPE OFF
    ENDIF

    ACTIVATE WINDOW printemp
    @ 1,0 SAY "Please ready your printer or"
    @ 2,0 SAY "     press ESC to cancel"

    *-- Loop until printer is ready or user presses escape
    gn_pkey = 0
    DO WHILE ( .NOT. PRINTSTATUS()) .AND. gn_pkey <> 27
      gn_pkey = INKEY()
    ENDDO

    DEACTIVATE WINDOW printemp
    IF ll_escape
      SET ESCAPE ON
    ENDIF

    IF gn_pkey <> 27                    && If user wants to retry
      RETRY                             && Retry the print command
    ENDIF

  ENDIF

RETURN
*-- EOP: PrntRtry

FUNCTION NodShake
PARAMETERS pc_mssg, pn_up, pn_left, pn_height, pn_max, pl_dflt_no
*---------------------------------------------------------------------------
* NAME
*   NodShake
*
* DESCRIPTION
*   Accepts a YES/NO response from user
*
* SYNOPSIS
*   ? NodShake (pc_mssg, pn_up, pn_left, pn_height, pn_max, pl_dflt_no)
*
* PARAMETERS
*   pc_mssg:    dialog box message
*   pn_up:      upper corrdinate of dialog box
*   pn_left:    left coordinate of dialog box
*   pn_height:  height of dialog box
*   pn_max:     maximum width of a line in message
*   pl_dflt_no: flag indicating if default pad highlighted should be "NO"
*   	
* EXAMPLE
*    pl_set = NodShake( pc_vermssg, 13, 25, 2, 28, .T. )
*---------------------------------------------------------------------------

  PRIVATE ll_ans, ll_wrapset, ln_pspset

  ll_wrapset = _wrap
  ln_pspset = _pspacing
  _wrap = .F.
  _pspacing = 1

  DEFINE WINDOW NodShake DOUBLE ;
     FROM pn_up, pn_left TO pn_up + pn_height + 4, pn_left + pn_max + 1

  DEFINE MENU NodShake
  DEFINE PAD Yes OF NodShake PROMPT "Yes" ;
     AT pn_height + 1, (pn_max - 12) / 2;
     MESSAGE "Select option and press ENTER, or press first letter" + ;
             " of desired option"

  ON SELECTION PAD Yes OF NodShake DEACTIVATE MENU
  DEFINE PAD No OF NodShake PROMPT "No" ;
     AT pn_height + 1, (pn_max - 12) / 2 + 10 ;
     MESSAGE "Select option and press ENTER, or press first letter" + ;
             " of desired option"

  ON SELECTION PAD No OF NodShake DEACTIVATE MENU
  ACTIVATE WINDOW NodShake
  CLEAR
  ?
  @ 0, 0
  ?? pc_mssg FUNCTION ";"

  ON KEY LABEL Y KEYBOARD "{Alt-Y}{13}"
  ON KEY LABEL N KEYBOARD "{Alt-N}{13}"

  IF pl_dflt_no
    KEYBOARD "{Alt-N}"
  ENDIF

  ON KEY LABEL RIGHTARROW
  ON KEY LABEL LEFTARROW

  ACTIVATE MENU NodShake

  ON KEY LABEL Y
  ON KEY LABEL N

  IF PAD() = "YES"
    ll_ans = .T.
  ELSE
    ll_ans = .F.
  ENDIF

  RELEASE WINDOW NodShake
  RELEASE MENU NodShake
  _wrap = ll_wrapset
  _pspacing = ln_pspset

RETURN ll_ans
*-- EOF: NodShake( pc_mssg, pn_up, pn_left, pn_height, pn_max, pl_dflt_no )

*----------------------------------------------------------------------

*--                Utility Functions and Procedures

*----------------------------------------------------------------------

FUNCTION ParseWord
*----------------------------------------------------------------------
*-- Programmer..: Jay Parsons (Jparsons).
*-- Date........: 04/26/1992
*-- Notes.......: returns the first word of a string
*-- Written for.: dBASE IV, 1.1, 1.5
*-- Rev. History: None
*-- Calls       : None
*-- Called by...: Any
*-- Usage.......: ? ParseWord(<cString>)
*-- Example.....: Command = ParseWord( cProgramline )
*-- Parameters..: cString - character string to be stripped.
*-- Returns     : that portion, trimmed on both ends, of the passed string
*--               that includes the characters up to the first interior space.
*------------------------------------------------------------------------------
parameters string
PRIVATE cW
cW = trim( ltrim( string ) )
RETURN iif( " " $ cW, rtrim( left( cW, at( " ", cW ) - 1 ) ), cW )
*-- EoP: ParseWord

FUNCTION StripWord
*----------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (Jparsons).
*-- Date........: 04/26/1992
*-- Notes.......: discards first word of a string
*-- Written for.: dBASE IV, 1.1, 1.5
*-- Rev. History: None
*-- Calls       : None
*-- Called by...: Any
*-- Usage.......: ? StripWord(<cString>)
*-- Example.....: Lastname = StripWord( "Carrie Nation" )
*-- Parameters..: cString - character string to be stripped.
*-- Returns     : string trimmed of trailing spaces, and trimmed on the
*--               left to remove leading spaces and, if the passed string
*--               contained interior spaces, also to remove everything before
*--               the first nonspace character after the first interior space.
*-------------------------------------------------------------------------------
parameters string
PRIVATE cW
cW = trim( ltrim( string ) )
RETURN iif( " " $ cW, ltrim( substr( cW, at( " ", cW ) + 1 ) ), cW )
*-- EoF: StripWord()

PROCEDURE ReColor
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (Jparsons)
*-- Date........: 04/23/1992
*-- Notes.......: Restores colors to those held in a string of the form
*--               returned by set("ATTRIBUTE").
*-- Written for.: dBASE IV, Versions 1.0 - 1.5.
*-- Rev. History: None
*-- Calls       : None
*-- Called by...: Any
*-- Usage.......: DO ReColor WITH <cColors>
*-- Example.....: DO Recolor WITH OldColors
*-- Parameters..: cColors, a string in the form returned by set("ATTRIBUTE").
*-- Side effects: Changes the screen colors.
*-------------------------------------------------------------------------------
  parameters cColors
  private cThis, cNext, nAt, cLeft, nX, cAreas
  cAreas = "   NORMHIGHBORDMESSTITLBOX INFOFIEL"
  cLeft = cColors + ", "
  nX = 0
  do while nX < 8
    nX = nX + 1
    cThis = substr( cAreas, 4 * nX, 4 )
    if nX = 3
      nAt = at( "&", cLeft )
      cNext = left( cLeft, nAt - 2 )
      cLeft = substr( cLeft, nAt + 3 )
      SET COLOR TO , , &cNext
    else
      nAt = at( ",", cLeft )
      cNext = left( cLeft, nAt - 1 )
      cLeft = substr( cLeft, nAt + 1 )
      SET COLOR OF &cThis TO &cNext
    endif
  enddo

RETURN
*-- EoP: ReColor

FUNCTION Tempname
*-------------------------------------------------------------------------------
*-- Programmer..: Martin Leon (HMAN)  Former Sysop, ATBBS
*-- Date........: 04/26/92
*-- Notes.......: Obtain a name for a temporary file of a given extension
*--               that does not conflict with existing files.
*-- Rev. History: Originally part of Makestru(), 6-12-1991
*-- Written for.: dBASE IV
*-- Rev. History: 04/26/92, made a separate function - Jay Parsons
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: TempName( cExt )
*-- Example.....: Sortfile = TempName( "DBF" )
*-- Returns.....: Name not already in use
*-- Parameters..: cExt = Extension to be given file ( without the "." )
*-------------------------------------------------------------------------------

 parameters cExt
 DO WHILE .T.
    Fname = "TMP" + ltrim( str( rand() * 100000, 5 ) )
    IF .NOT. FILE( Fname + "." + cExt ) .AND. ( upper( cExt ) # "DBF" .OR.;
        .NOT. ( file( Fname + ".MDX" ) .OR. file ( Fname + ".DBT" ) ) )
      EXIT
    ENDIF
  ENDDO
RETURN Fname
*-- EoF: Tempname

PROCEDURE Makedbf
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (Jparsons).
*-- Date........: 04/26/1992
*-- Notes.......: Makes an empty dBASE .dbf file
*-- Written for.: dBASE IV, 1.1, 1.5
*-- Rev. History: None
*-- Calls       : Tempname()          function in MARIAN.PRG
*-- Called by...: Any
*-- Usage.......: DO MakeDbf WITH <cFilename>, <cStrufile>, <cArray>
*-- Example.....: DO MakeDbf WITH Customers, cCustfields, cCuststuff
*-- Parameters..: cFilename - filename ( without extension ) of the .dbf to be
*--               created.
*--               cStrufile - name ( without extension ) of a STRUC EXTE .dbf
*--               cArray - name of the array holding field information for the
*--               .dbf.  The array must be dimensioned [ F, 5 ] where F is the
*--               number of fields.  Each row must hold data for one field:
*--                     [ F, 1 ]  field name, character
*--                     [ F, 2 ]  field type, character from set "CDFLMN"
*--                     [ F, 3 ]  field length, numeric.  If field type is
*--                                 D, L, or M, will be ignored
*--                     [ F, 4 ]  field decimals, numeric. optional if 0.
*--                     [ F, 5 ]  field is mdx tag, char $ "YN", optional if N
*-------------------------------------------------------------------------------
  parameters cFname, cSname, aAname
  private nX,cF1,cF2,cF3,cF4,cF5,cStrufile,cFtype
  cF1 = aAname + "[nX,1]"
  cF2 = aAname + "[nX,2]"
  cF3 = aAname + "[nX,3]"
  cF4 = aAname + "[nX,4]"
  cF5 = aAname + "[nX,5]"
  select select()
  use ( cSname ) ALIAS cSname
  zap
  nX = 1
  do while type( cF1 ) # "U"
    cFtype = &cF2
    append blank
    replace Field_name with &cF1, Field_type with cFtype
    do case
      case cFtype = "D"
        replace Field_len with 8
      case cFtype = "M"
        replace Field_len with 10
      case cFtype = "L"
        replace Field_len with 1
      otherwise
        replace Field_len with &cF3
    endcase
    if type( cF4 ) = "N" .and. cFtype $ "FN"
      replace Field_dec with &cF4
    else
      replace Field_dec with 0
    endif
    if type( cF5 ) # "U" .and. cFtype $ "CDFN" .and. &cF5 = "Y"
      replace Field_idx with "Y"
    else
      replace Field_idx with "N"
    endif
    nX = nX + 1
  enddo
  use
  create ( cFname + ".DBF" ) from ( cSname )
RETURN
*-- EoP: Makedbf

FUNCTION Makestru
*-------------------------------------------------------------------------------
*-- Programmer..: Martin Leon (Hman), formerly sysop of A-T BBS
*--             : Revised by Jay Parsons, (Jparsons).
*-- Date........: 04/24/1992
*-- Notes.......: Makes an empty dBASE STRUCTURE EXTENDED file and returns
*--             : its root name
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: Original function published 6-12-1991.
*--             : Changed to take no parameter, return filename, 4-7-1992.
*--             : Code added to preserve catalog status and name, 4-10-1992.
*--             : Use of Tempname() added 4-24-92.
*-- Calls       : Tempname()          function in MARIAN.PRG
*-- Called by...: Any
*-- Usage.......: Makestru()
*-- Example.....: Tempfile = Makestru()
*-- Returns.....: Name of file created
*-- Parameters..: None
*-------------------------------------------------------------------------------
private cCatfile, cCatstat, cTitle, cAlias, cConsole, cStruname, cNewcat

* Preserve work area and environment
cCatfile = catalog()
cCatstat = set("CATALOG")
cTitle = set("TITLE")
set title off                   && Otherwise we get ugly question box
cAlias = Alias()
select select()

* Create needed files
cStruname = Tempname( "DBF" )
cNewcat =   Tempname( "CAT" ) +".CAT"

* Create .dbf by the SET CATALOG command, copy structure and kill it
set catalog to ( cNewcat)
set catalog to
use ( cNewcat  ) nosave
copy to ( cStruname ) structure extended

* remove the records relating to the catalog from the structure file
use ( cStruname) exclusive
zap
use

if "" # cAlias
  select ( cAlias )
endif
set title &cTitle
if "" # cCatfile
  set catalog to ( cCatfile )
else
  set catalog to
endif
set catalog to &cCatstat
RETURN cStruname
*-- Eof: Makestru()

FUNCTION ExactDelay
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 05/09/92
*-- Notes.......: Delay Loop.  Returns .T. after lapse of given number of 
*--               seconds.   Accurate only to nearest 1/18 second.
*--               <some published ways to do this do not work, because they 
*--               ignore the possibility that the interval may start at 59 
*--               seconds, and thus that the ending number of seconds will be 
*--               smaller.>
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: TIME2SEC()           Function in TIME.PRG
*-- Called by...: Any
*-- Uses........: Exactime.bin         Binary file
*-- Usage.......: Delay(<nSeconds>)
*-- Example.....: lX= Delay(10)
*-- Returns.....: Logical
*-- Parameters..: nSeconds = number of seconds to delay
*-------------------------------------------------------------------------------

	parameters nSeconds         && up to 86400, one day
	private nTimeout
        nTimeout = mod( Time2Sec( call( "Exactime", space( 11 ) ) ) + nSeconds, 86400 )
        do while Time2Sec( call( "Exactime", space( 11 ) ) ) < nTimeout
	   *-- Nothing to do ...
	enddo

RETURN .T.
*-- EoF: ExactDelay()

FUNCTION Time2Sec
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/92
*-- Notes.......: Convert HH:MM:SS or HH:MM:SS.SS string to seconds.
*-- Written for.: dBASE IV
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Time2Sec("<cTime>")
*-- Example.....: ?Time2Sec("01:24:15")
*-- Returns.....: Numeric
*-- Parameters..: cTime = Time string in format HH:MM:SS or HH:MM:SS.SS
*-------------------------------------------------------------------------------
	
	parameters cTime
	private cTemp, nSecs
	cTemp = cTime
	nSecs = 3600 * val( cTemp )
	cTemp = substr( cTemp, at( ":", cTemp ) + 1 )
	nSecs = nSecs + 60 * val( cTemp )
	
RETURN nSecs + val( substr( cTemp, at( ":", cTemp ) + 1 ) )
*-- EoF: Time2Sec()

*---------------------------------------------------------------------------

*--                 Opening, closing and screen painting

*---------------------------------------------------------------------------
*--                     Startup and exit boilerplate
*----------------------------------------------------------------------------

PROCEDURE Setenv
  SET CONSOLE OFF
  CLEAR WINDOWS
  CLOSE ALL
  CLOSE PROCEDURE
  SET CONSOLE ON
  ON ERROR ??
  SET DISPLAY TO MONO
  SET DISPLAY TO COLOR
  SET DISPLAY TO EGA25
  ON ERROR

  *-- Blank the screen
  SET COLOR TO
  CLEAR
  SET SCOREBOARD OFF
  SET STATUS OFF

  SET ALTERNATE OFF
  SET BELL ON
  SET CARRY OFF
  SET CENTURY OFF
  SET CLOCK OFF
  SET CONFIRM OFF
  SET CURSOR ON
  SET DELETED OFF
  SET DELIMITERS TO ""
  SET DELIMITERS OFF
  SET DEVICE TO SCREEN
  SET ESCAPE ON
  SET EXACT ON
  SET EXCLUSIVE ON
  SET LOCK ON
  SET MESSAGE TO ""
  SET PRINT OFF
  SET REPROCESS TO 4
  SET SAFETY OFF
  SET TALK OFF
  SET TITLE OFF
RETURN
*-- EoP: Setenv

PROCEDURE Resetenv
  *-- Reset environment
  DEACTIVATE WINDOW FullScr
  DO ReColor WITH gc_color
  SET BELL  &gc_bell.
  SET CARRY &gc_carry.
  SET CLOCK &gc_clock.
  SET CENTURY &gc_century.
  SET CONFIRM &gc_confirm.
  SET CONSOLE &gc_console.
  SET CURSOR  &gc_cursor.
  SET DELETED &gc_delete.
  SET DELIMITERS &gc_deli.
  SET DISPLAY TO &gc_display.
  SET ESCAPE &gc_escape.
  SET EXACT &gc_exact.
  SET STATUS &gc_status.
  SET SAFETY &gc_safety.
  SET SCOREBOARD &gc_score.
  SET TALK  &gc_talk.
  SET TITLE &gc_title
  ON KEY LABEL F1
  CLEAR WINDOWS
  CLEAR ALL
  CLOSE ALL
  CLOSE PROCEDURE
  SET MESSAGE TO ""
  CLEAR
RETURN
*-- EoP: Resetenv

*------------------------------------------------------------------------
*--         Miscellaneous opening, screen and closing code
*------------------------------------------------------------------------

PROCEDURE Paint
* MAIN setup routine.  Sets the color scheme.
* It will then paint the menu on the screen.
*---------------------------------------------------------------------

  SET BORDER TO
  SET CLOCK TO 1,68
  SET CLOCK ON
  @ 0,0 TO 2,79 DOUBLE COLOR RG+/GB
  @ 1,1 CLEAR TO 1,78
  @ 1,1 FILL TO 1,78 COLOR W+/N
  @ 1,1 SAY  "Files"  COLOR W+/N
  @ 1,16 SAY "Updates" COLOR W+/N
  @ 1,31 SAY "Reports" COLOR W+/N
  @ 1,46 SAY "Options" COLOR W+/N
  lc_popup = POPUP()                    && Find out the active pulldown

  *-- Determine which menu pad to draw as highlight
  IF lc_popup = "FFILES"
    @ 1,1  SAY "Files" COLOR RG+/GB
  ENDIF
  IF lc_popup = "UPDATES"
    @ 1,16 SAY "Updates" COLOR RG+/GB
  ENDIF
  IF lc_popup = "REPORTS"
    @ 1,31 SAY "Reports" COLOR RG+/GB
  ENDIF
  IF lc_popup = "OPTIONS"
    @ 1,46 SAY "Options" COLOR RG+/GB
  ENDIF

RETURN
*-- EOP: Paint - Menu MAIN

PROCEDURE Showstatus
*-- display what's what on main menu
PARAMETERS cNop
SET CURSOR OFF
IF .NOT. cNop
  @ 12, 30 SAY "Operation completed" COLOR RG+/B
  IF Timerbin
    Trash = ExactDelay ( Delaysecs )
   @ 12, 30 CLEAR TO 12, 50
  ENDIF
ENDIF
nL = 15
@ nL,0 CLEAR TO nL + 3,79
@ nL    , 5 SAY "Catalog file:     " + Catfile
@ nL + 1, 5 SAY "Source file:      " + Source
@ nL + 2, 5 SAY "Target file:      " + Target + iif( Ltype, " (L)", " (P)" )
@ nL + 3, 5 SAY "Description file: " + Descfile

@ nL    , 42 SAY "Printer is:        " + gc_dev
@ nL + 1, 42 SAY "Alternate file is: " + Reptfile
@ nL + 2, 42 SAY "Alternate is set:  " + gc_alt
@ nL + 3, 42 SAY "Overwrite mode is: " + Crash
IF .NOT. ( cNop .OR. Timerbin )
  @ 12, 30 CLEAR TO 12, 50
ENDIF
SET CURSOR ON
RETURN

PROCEDURE Quickout
*-- Cleanup if exit from first-time screen
IF select( "Strufile" ) > 0
  SELECT ( Strufile )
  USE
ENDIF
ERASE ( Strufile + ".DBF" )
DO Resetenv
ON ESCAPE
CANCEL
RETURN

PROCEDURE Shutdown
*-- Cleanup of devices and files
IF gc_alt = "ON"
  SET ALTERNATE OFF
ENDIF
CLOSE ALTERNATE
IF gc_dev='PRN'
  SET PRINT OFF
ENDIF
gc_dev='CON'
IF select( "Strufile" ) > 0
  SELECT ( Strufile )
  USE
ENDIF
ERASE ( Strufile + ".DBF" )

IF nSrce > 0
  Trash = fclose( nSrce )
ENDIF
IF nTarg > 0
  Trash = fclose( nTarg )
ENDIF
IF select( "Tindex" ) > 0
  SELECT Tindex
  USE
ENDIF
IF select( "Sindex" ) > 0
  SELECT Sindex
  USE
ENDIF
IF SELECT( "Callees" ) > 0
  SELECT Callees
  USE
ENDIF

SELECT Made
Recs = reccount()
IF Recs > 0
  IF select( "Catfile" ) > 0
    SELECT Catfile
  ELSE
    SELECT select()
    USE ( Catfile ) ALIAS Catfile
  ENDIF
  SET ORDER TO Procfile
ENDIF
DO WHILE Recs > 0
  Nowrecs = min( recs, 14 )
  @ 3,5 SAY "These indexing files have been created.  Please decide among"
  @ 4,5 SAY "keeping them, renaming them or deleting them:"
  nCount = 1
  DO WHILE nCount <= Nowrecs
    SELECT Made
    GO nCount
    Oldname = ProcName
    @ nCount +5,15 SAY trim( Oldname )+ ".DBF, which indexes "+ FromFile
    Doit = "Keep it  "
    @ nCount +5,55 GET Doit PICTURE "@M Keep it, Rename it, Trash it";
    MESSAGE "Press SPACE to view choices and "+CHR(17)+CHR(196)+CHR(217)+" to Select"
    READ
    DO CASE
      CASE Doit = "Rename it"
        DO WHILE .T.
          Fname = Getfname( Oldname, "the new file" )
          IF .NOT. file( Fname + ".DBF" )
            SELECT select()
            USE ( Oldname ) ALIAS Temp
            COPY TO ( Fname ) WITH PRODUCTION
            USE
            SELECT Made
            ERASE ( Oldname + ".DBF" )
            ERASE ( Oldname + ".MDX" )
            SELECT Catfile
            SEEK Made->FromFile
            REPLACE IndxFile WITH Fname
            EXIT
          ELSE
            IF .NOT. NodShake( "  That name is already in use.  Try another?  ",13,10,2,60,.F.)
              IF NodShake( "  Keep it?  ",13,30,2,20,.F.)
                EXIT
              ELSE
                ERASE ( Oldname + ".DBF" )
                ERASE ( Oldname + ".MDX" )
                SELECT Catfile
                SEEK Made->FromFile
                REPLACE Indxfile WITH " ", Date WITH {}, Used WITH " "
                EXIT
              ENDIF
            ENDIF
          ENDIF
        ENDDO
      CASE Doit = "Trash it"
        ERASE ( Oldname + ".DBF" )
        ERASE ( Oldname + ".MDX" )
        SELECT Catfile
        SEEK Made->FromFile
        REPLACE Indxfile WITH " ", Date WITH {}, Used WITH " "
    ENDCASE
    nCount = nCount + 1
  ENDDO
  Recs = Recs - Nowrecs
ENDDO
USE
IF select( "Catfile" ) > 0
  SELECT Catfile
  USE
ENDIF
IF Ismouse .AND. .NOT. Mouse
  Trash = call( "Setmouse", .T.)
  @ 12,40 SAY " "
ENDIF
RELEASE MODULES
RETURN
*-- EoP: Shutdown

*-----------------------------------------------------------------------

*--             Sign-on banner and help screens

*-----------------------------------------------------------------------
PROCEDURE Showsign
*-- Sign-on banner
  SET BORDER TO
  @ 1,1 FILL TO 23,78 COLOR W+/B
  ln = 4
  @ 0,0 TO 23,79 DOUBLE COLOR RG+/GB
  @ ln,1   SAY "                               "
  @ ln+1,1 SAY "                                    "
  @ ln+2,1 SAY "                            "
  @ ln+3,1 SAY "                                      "
  @ ln+4,1 SAY "                                           the Librarian"
  ln = 13

  @ ln,1   SAY "           Library Maintenance Program in dBASE IV, Version 1.5, for" COLOR RG+/B
  @ ln+1,1 SAY [               KEN MAYER'S PROCEDURE AND FUNCTION LIBRARY, LIB15]           COLOR RG+/B
  @ ln - 1, 8 to ln + 2, 71 double color RG+/N
  @ 19,1 SAY [ "dBASE IV" is a registered trademark                   by Jay Parsons]
  @ 20,1 SAY "    of Borland International, Inc.                Somerset Data Systems, Inc."
  @ 21,1 SAY "                                                      Copyright (c) 1992" COLOR W/B
  @ 22,1 SAY "      Version 1.01: May 9, 1992                       All Rights Reserved" COLOR W/B
RETURN
*-- EoP: Showsign

PROCEDURE Help
ACTIVATE WINDOW Help
DO Showtext WITH upper( popup() ), bar()
DEACTIVATE WINDOW Help
RETURN
*-- Eop: Help

PROCEDURE Showtext
PARAMETERS Pop, Bar
DO CASE
  CASE Bar = -1
  CLEAR
TEXT

                      Welcome to "Marian"

        This program is intended to help you maintain your library files.
As furnished, it is designed to be used either to create your library file,
with routines from Ken Mayer's LIB15.ZIP on the BORBBS, or to add new or
improved routines to your existing library file.  In addition to this program,
you should have in your working directory the .PRG files from LIB15.ZIP, the
README.TXT file from it, and the LIB15MV.MEM, LIB15CAT.DBF and LIB15CAT.MDX
files included with this file as distributed.
        The program allows you to add to any library or to add any .PRG
files to your library.  However, this may not work well unless your .PRG
files follow a precise format.  See the notes in this program file, MARIAN.PRG,
for details.
        This program is copyrighted.  See README for license info.
        Because this program uses low-level file I/O, it can run only under
dBASE IV, Version 1.5 or higher.  It can also destroy your files in its
working directory, so be wise and back them up before going further.  Hit {Esc}
to quit now and back your files up.
        Happy Computing!    Please report bugs to:  Jay Parsons
                                                    CIS 70160,340
                                                    BORBBS Jparsons

ENDTEXT

    CASE Bar = -2
      @  5,15 SAY "Don't ignore that mouse!  You can use it whenever"
      @  6,15 SAY "a key is requested, including to leave this screen,"
      @  7,15 SAY "except, apparently, to pick from @M GET choices."
      @  9,36 TO 11,38 219 COLOR N
      @  9,42 TO 11,44 219 COLOR N
      @ 10,38 TO 14,42 219 COLOR N
      @ 11,39 SAY chr( 22 ) + " " + chr( 22 ) COLOR N/W
      @ 12,39 SAY " " COLOR N/W
      @ 12,41 SAY " " COLOR N/W
      @ 13,39 SAY " " + chr( 196 ) +" " COLOR N/W
      @ 16,10 SAY "Just remember:"
      @ 17,15 SAY "1) Use only the left button."
      @ 18,15 SAY "2) You must double click on lists, and click on the"
      @ 19,20 SAY " top or bottom border to scroll the list."
      @ 20,15 SAY [3) See Appendix A of "Getting Started" for details.]
      @ 23, 0

    CASE Pop = "FFILES" .AND. Bar = 1
      nL = 0
      @ nL, 3 SAY "The catalog file needed by this program is a .dbf of"
      @ nL+1,3 SAY "four fields:"
      @ nL+2,3 SAY "   Procfile, C, 12, 0, Y"
      @ nL+3,3 SAY "   Indxfile, C,  8, 0, N"
      @ nL+4,3 SAY "   Date,     D,  8, 0, N"
      @ nL+5,3 SAY "   Used,     C,  1, 0, N"
      @ nL+7,3 SAY "It is used to keep track of the indexing .dbfs so they"
      @ nL+8,3 SAY "do not have to be rebuilt if existing.  If you designate"
      @ nL+9,3 SAY "a file that does not exist, the program will build it."

    CASE Pop = "FFILES" .AND. Bar = 2
      nL = 1
      @ nL,   3 SAY "The source file is the one from which routines will be"
      @ nL+1, 3 SAY "copied into the target file.  It must be a text, usually"
      @ nL+2, 3 SAY ".PRG, file in the required format for the program to"
      @ nL+3, 3 SAY "locate the routines properly, and its name must be in"
      @ nL+4, 3 SAY "the catalog file."

    CASE Pop = "FFILES" .AND. Bar = 3
      nL = 0
      @ nL,   3 SAY "A library target file is an indexed file of routines into"
      @ nL+1, 3 SAY "which routines will be copied from the source file(s)."
      @ nL+2, 3 SAY "It must be a text, usually .PRG, file containing only"
      @ nL+3, 3 SAY "routines with headers in the required format for the"
      @ nL+4, 3 SAY "program to locate its routines properly, and its name"
      @ nL+5, 3 SAY "must be in the catalog file.  The program will build it"
      @ nL+6, 3 SAY "if it does not exist."
      @ nL+8, 3 SAY "If you want to add routines to the end of a program that"
      @ nL+9, 3 SAY "lacks this structure, choose the next selection."

    CASE Pop = "FFILES" .AND. Bar = 4
      nL = 1
      @ nL,   3 SAY "A program target file is any text file to which you wish"
      @ nL+1, 3 SAY "to add routines.  It needs no special structure and need"
      @ nL+2, 3 SAY "not be in the catalog.  However, since it is not indexed"
      @ nL+3, 3 SAY "no checks can be made that the routines added do not"
      @ nL+4, 3 SAY "already exist in the file, and this program cannot delete"
      @ nL+5, 3 SAY "routines from it or split the file."

    CASE Pop = "UPDATES" .AND. Bar = 1
      nL = 0
      @ nL,   3 SAY "This selection will copy all routines from the source"
      @ nL+1, 3 SAY "file to the target file, consistent with the overwrite"
      @ nL+2, 3 SAY "mode in effect.  If you want most but not all routines"
      @ nL+3, 3 SAY "from the source file, follow this procedure:"
      @ nL+4, 6 SAY [1) Set overwrite to "Never" and choose this selection.]
      @ nL+5, 6 SAY "2) Exit from the program and manually delete the"
      @ nL+6, 9 SAY "records of the routines you don't want from the"
      @ nL+7, 9 SAY "indexing .DBF.  DO NOT PACK it."
      @ nL+8, 6 SAY "3) Start program, reset overwrite mode, and select this."
      @ nL+9, 6 SAY "4) Exit and RECALL ALL in the indexing .DBF."

    CASE Pop = "UPDATES" .AND. Bar = 2
      nL = 1
      @ nL,   3 SAY "This selection adds a single routine to the target file."
      @ nL+2, 3 SAY "To add most but not all routines from a source file,"
      @ nL+3, 3 SAY "either add them all using the preceding selection then"
      @ nL+4, 3 SAY "delete the ones you don't want using the fourth selection,"
      @ nL+5, 3 SAY "or edit the indexing .DBF as described in the Help screen"
      @ nL+6, 3 SAY "for the preceding selection."

    CASE Pop = "UPDATES" .and. Bar = 3
      nL = 1
      @ nL  ,3 SAY [This selection "freshens" the target file by replacing]
      @ nL+1,3 SAY "each routine in the target file with a routine of the"
      @ nL+2,3 SAY "the same name from any of the source files, if the routine"
      @ nL+3,3 SAY "from the source file is later in date."
      @ nL+5,3 SAY "It does not add any additional routines from the"
      @ nL+6,3 SAY "source files."

    CASE Pop = "UPDATES" .AND. Bar = 6
      nL = 0
      @ nL  ,3 SAY "This selection permits you to divide the target file"
      @ nL+1,3 SAY "into two files at a point specified by routine, lines"
      @ nL+2,3 SAY "or bytes.  Either part may be given the name of the"
      @ nL+3,3 SAY "original or another name.  This selection is provided"
      @ nL+4,3 SAY "primarily so you can break up files too long for your"
      @ nL+5,3 SAY "text editor.  You may append the second part to another"
      @ nL+6,3 SAY "existing file.  Supplying the name NUL or a DOS device"
      @ nL+7,3 SAY "for either part will mess up your catalog file."

    CASE Pop = "OPTIONS" .AND. Bar = 4
      nL = 1
      @ nL  ,3 SAY "This selection controls when a routine in the target file"
      @ nL+1,3 SAY "will be overwritten if there exists a routine of the same"
      @ nL+2,3 SAY "type ( PROCEDURE or FUNCTION ) and name in the source file."
      @ nL+3,3 SAY "The options are:"
      @ nL+4,6 SAY "Always   - The target routine will always be overwritten."
      @ nL+5,6 SAY "Never    - The target routine will not be overwritten."
      @ nL+6,6 SAY "Ask_User - The user will be asked at run time."
      @ nL+7,6 SAY "If_Older - The target routine will be overwritten if older."
      @nL+8, 6 SAY "If_Older overrides Always when doing a Freshen operation."

    OTHERWISE
      @ 3,20 SAY "No help available on this topic."
  ENDCASE
  IF Bar >= 0
    @ 12,0
  ENDIF
  WAIT
RETURN
*-- EoP: Showtext
*-- End of File: MARIAN.PRG

