;       Where.asm - obtain address of a memory variable
;       in dBASE IV.

;       Expects a character variable argument giving
;       the name of the variable, padded with spaces to
;       at least 9 characters.  Returns the address in
;       argument as a string of the form "XXXX:XXXX",
;       or as "NOT FOUND".  Returns "E" for "error" if
;       the argument is shorter than 9 bytes.
;       If argument contains more characters than
;       needed for the return, fills balance with
;       spaces.  If no argument is given, just returns.

;       Each dBASE variable is stored in two places:

;       1) The RTSYM table.  Each block of this table
;       contains a number of 17-byte entries.  The
;       first ten bytes are the name of a memory
;       variable, in upper case and null-filled.  The
;       eleventh is a null.  The twelfth though
;       fifteenth are a doubleword giving the ID number
;       of the variable.  Sixteen and seventeen are of
;       unknown use and always appear to be nulls.

;       2) As a 56-byte memory variable entry:

;    Bytes (dec)     Offset (hex)      Meaning

;       10               0      Upper-case name+nulls.
;        1              0A      Always null.
;        1              0B      Variable type, "ACDFLN"
;        4              0C      Data pointer (C and A
;                                   type only, else nulls.)
;        1              10      Length for char. type.
;        1              11      Usually null.
;        2              12      Visibility, one word
;                                (0=public, 1=next level).
;       14              14      Unknown.  Always nulls?
;        2              20      Unknown.  Usually 01 00
;        2              22      Numeric subtype
;                                 (1=int., 2=BCD, 3=float).
;       12              24      Data, see below

;        4              30      Unknown.
;        4              34      ID doubleword.

;       For numeric, date or logical types, the data
;       are included in the entry starting at offset
;       24h, as follows, using up to 12 bytes, with
;       bytes not needed for the particular data type
;       being nulls.

;        4            Integers (-32768 to 32767)
;        8            Dates
;       12            N-type numbers
;        8            Floating-point type numbers
;        1            Logical
;        4            Array dimensions
        
;       The structures used for the data are:

;       Integers - a signed binary word.  Used for N-
;               type numbers whenever they fit.
;       Dates - a Julian number in IEEE 8-byte format,
;               like Floats.  The Julian period used
;               is such that 1/1/1900 = 2,415,021.
;       N-type - the original dBASE binary-coded
;               decimal numeric type.  One word that is
;               a sign bit, 5 bits of precision (top
;               value is 20) and 10 bits of exponent
;               base 10 biased 13Fh, followed by 10
;               bytes (highest first) of BCD digits
;               of the mantissa, with digits past the
;               precision limit being nulls.
;       Float - IEEE 8-byte real numbers.  Zero is all
;               nulls.  Others are from high bit to low
;               (and bytes are stored low byte first)
;               1 bit of sign, 11 of exponent base 2
;               biased 3FFh, and 52 bytes of mantissa
;               with the leading 1 (should be left of
;               binary point) omitted as understood.
;       Logical - 01=.T., 00=.F.
;       Array - Two binary words, # rows and # columns.

;       Since this program is concerned primarily with
;       locating character data for direct manipulation
;       in memory or in using array memory as a buffer,
;       the address returned for character or array
;       variables is the far pointer found at offset
;       0Ch of the memory variable entry.  For
;       character data, this points to the characters
;       themselves in string memory. For an array, it
;       points to the base address of a contiguous set
;       of 56-byte memory variable entries reserved for
;       the array elements.  An entry is provided for
;       each element possible under the array
;       declaration. Until assigned a value, each
;       element is given logical type and value false.

;       For date, logical or numeric types of memory
;       variables, this program returns the base
;       address of the 56-byte entry, the start of the
;       variable name.  To obtain the same result for
;       character or array variables, set the
;       "getpointer" assembly option to false.

;       The value "lowmem" is intended to limit the
;       search to the portion of memory above the dBASE
;       code itself; on a system using many TSR's the value
;       can be increased.

;       Expects arguments in dBASE IV format and
;       locations, i.e.:

;               ES:DI points to table of far pointers
;                       to arguments.
;               CX holds number of arguments.

;       This program must be assembled and linked, then
;       converted to a binary image file by EXE2BIN.
;       It cannot readily be converted to a .COM file.

lowmem          equ 5000h       ; bottom of data memory
true            equ -1
false           equ 0

;       Assembly option - make false before assembly to
;       have the base address of the memory variable
;       entry returned for all types

getpointer      equ true

;------------------------------------------------------

		.model tiny
		code segment
		assume cs:code,ds:code

Where           proc far
		jmp short begin
memtop		dw ?
varname         db 12 dup (?)
membot          dd ?
rtsym           dd ?
ID              dd ?
varlen		dw ?
storage         equ ($-offset memtop)/2
errmess         db 'E'
nofindmess      db 'NOT FOUND'
                db "Copyright (C) 1990 Jay Parsons"
begin:          jcxz alldone    ; must be an argument
                push di         ; push pointer to it
		push es
                mov bp,sp       ; address of pointer
                lds si,es:[di]  ; point to first arg
                mov dx,si       ; and save address
                add dx,9        ; we need 9 bytes
                mov bx,cs
		mov es,bx
                mov di,offset memtop
                mov cx,storage  ; null out storage
		xor ax,ax
                rep stosw       ; and make cx 0
                mov di,offset varname   ; arg here
nameloop:	lodsb
		cmp al,0
		jz namedone
		cmp al,' '
		jz nameloop
		cmp al,'a'
		jb namesave
		and al,5Fh
namesave:	stosb
		inc cx
		jmp short nameloop

;	Exit routines (here to keep jumps short)	

erret:          mov si,offset errmess
		mov cx,1
                jmp short copy
notfound:       mov si,offset nofindmess
copy9:          mov cx,9
copy:           mov sp,bp
                pop es          ; restore arg pointer
		pop di	
                les di,es:[di]  ; make it destination
                rep movsb       ; copy result
                mov al,' '      ; a space
pad:            cmp byte ptr es:[di],0  ; full?
                jz alldone      ; then done
                stosb           ; else pad with spaces
		jmp short pad
alldone:	ret

namedone:       mov ds,bx       ; our segment
                cmp si,dx       ; argument long enough?
                jbe erret       ; jump if not
                mov varlen,cx   ; save length

                int 12h         ; find memory size
		shr ax,1
		shr ax,1
                xchg ah,al      ; convert to segment
                mov memtop,ax   ; save it
                mov word ptr membot+2,lowmem

search:         call look       ; look for rtsym entry
                jnz notfound
                call look       ; then 56-byte entry
                jz searchdone
                xor cx,cx
                mov word ptr ID,cx      ; if not found
                mov word ptr ID+2,cx    ; look again
                mov ax,word ptr rtsym   ; from here
                mov dx,word ptr rtsym+2
                add ax,1                ; add 1 byte to
                adc dx,cx               ; break circle
                mov word ptr membot,ax
                mov word ptr membot+2,dx
                jmp short search

searchdone      equ $

if getpointer
                cmp al,'A'              ; an array?
                jz pointer
                cmp al,'C'              ; characters?
                jnz nopointer           ; if not, jump
pointer:        les bx,es:[bx+12]       ; get data ptr
endif

nopointer:      push bx                 ; save offset
                mov ax,es
                mov cx,4
                mov bx,cs
                mov es,bx
                mov si,offset varname   ; store result
                call bintohex           ; seg to hex
		mov byte ptr [si]-1,':' ; add colon
                pop ax
		mov cx,4 
		call bintohex		; and offset
		mov si,offset varname
                jmp copy9

Where           endp

;       Look - look through memory for variable name.
;       If no ID found yet, test for whether occurrence
;       found is in RTSYM table.  If it could be, save
;       ID doubleword, return address in es:bx and with
;       zero set. If ID already found, look for memory
;       entry. If found, return with address in es:bx,
;       zero set and type byte in al.  If not found
;       below top of memory in either case, return with
;       nonzero.

Look		proc near
                mov di,word ptr membot
                mov es,word ptr membot+2

segloop:        mov al,byte ptr varname
                scasb                   ; first byte
                jz nextchars            ; jump on match

nextlook:       mov cx,di
                neg cx                  ; bytes left
                mov si,offset varname
		lodsb
                repne scasb             ; find match
                jcxz nextseg            ; if none, jmp
nextchars:      mov bx,di               ; save address
                mov cx,varlen           ; length thru null
                repe cmpsb              ; rest match?
                mov di,bx               ; restore
                jnz nextlook            ; look on if n.g.
                call found              ; test more
                jnz nextlook            ; look on if n.g.
                jmp short lookret

nextseg:        xor di,di
                mov ax,es
	        add ax,1000h
                mov es,ax
                cmp ax,word ptr memtop
		jb segloop
                or ax,0FFFFh            ; nz=not found
lookret:        ret

Look		endp

;       Found - test for RTSYM or memory entry.  If ID
;       not found and this might be rtsym entry, place
;       address in rtsym, ID in id, return with zero
;       set. If ID already found, test for memory
;       entry. Return with zero set, address in es:bx.
;       Return with nonzero if no eligible entry found.

Found		proc near
                dec bx          ; point to first char
                mov ax,word ptr ID      ; have ID?
                mov dx,word ptr ID+2
                or ax,dx
                jnz memfind             ; if so, jmp
                mov ax,word ptr es:[bx]+11 ; ID?
		mov dx,word ptr es:[bx]+13
                cmp ax,1Bh      ; lowest is 1Bh
                jae rtfind      ; (sys vars are lower)
		cmp dx,0
                jne rtfind      ; if not ID, jmp
                or dx,0FFFFh    ; nonzero for return
		jmp short findret

rtfind:         mov word ptr ID,ax
                mov word ptr ID+2,dx
                mov word ptr rtsym,bx
                mov word ptr rtsym+2,es
                xor ax,ax       ; set zero flag

findret:        ret

memfind:        cmp bx,word ptr rtsym
                jnz memposs
                mov cx,es
                cmp cx,word ptr rtsym+2
                jnz memposs
nofind:         or cx,0FFFFh    ; same addr is NG
                jmp short findret

memposs:        cmp ax,word ptr es:[bx]+52 ; compare ID
                jnz nofind      ; no match, not found
		cmp dx,word ptr es:[bx]+54
                jnz nofind
                mov al,byte ptr es:[bx]+11
		cmp al,'A'
                jz findret
		cmp al,'C'
                jz findret
                cmp al,'D'      ; test type
		jz findret
                cmp al,'F'
                jz findret
		cmp al,'L'
		jz findret
		cmp al,'N'
                jmp short findret ; nonzero=bad type

Found           endp

;       Bintohex - Translate number in al or ax into
;       ASCII hexadecimal of the number of bytes in cx,
;       store at es/ds:si, adjust si past it and the
;       comma if any.

Bintohex	proc near
                mov dx,ax       ; copy number
                mov bx,cx       ; count
                mov di,si       ; and location
                mov al,'0'      ; fill with zeroes
                rep stosb
                mov cx,bx
                dec di          ; point to last one
		std
binloop:	cmp dx,0
		jz bin3
                mov al,dl       ; get least part
		and al,0Fh		
                add al,30h      ; convert to ASCII
		cmp al,'9'
		jbe bin2
                add al,7        ; correct for letter
bin2:           stosb           ; save it
		shr dx,1
		shr dx,1
		shr dx,1
                shr dx,1        ; shift in next digit
		loop binloop
bin3:		add si,bx
                inc si          ; bump past any comma
		cld
		ret

Bintohex	endp

code	 	ends	
		end

