*******************************************************
*-- Program....: IRR.PRG
*-- Purpose....: Calculate internal rate of return
*-- Version....: 1.0 Written for dBASE IV version 1.0
*--                    by Tony Lima
*--              1.1 Includes modifications
*--                    by Jay Parsons
*-- Authors....: Tony Lima & Jay Parsons
*-- Phone......: (415) 593-6431.
*-- Notice.....: Placed in the public domain 01/31/90.
*                 Authors will assume responsibility
*                 for disseminating updates.  Please
*                 report any errors to TonyLima or
*                 Jparsons on the Ashton-Tate BBS, to
*                 MCI Mail ALIMA or to CIS [70160,340].
*                 Updates will be posted on the
*                 Ashton-Tate BBS and in the CompuServe
*                 Ashton-Tate dBASE Forum as they are
*                 developed.
*
*-- Parameters.: af_cash[gn_period]:  Array with cash
*                                       flows
*                gn_period:   Number of periods
*                gf_hirate:   Maximum discount rate
*                gf_lorate:   Minimum discount rate
*                gn_maxiter:  Maximum iterations
*                gf_abserr:   Absolute error
*                                for convergence
*                gf_relerr:   Relative error
*                                for convergence
*
*                Second letter of variable name is the
*                 data type.
*
*-- Disclaimer.: While the authors have used their best
*                  efforts to ensure that this program
*                  accomplishes the task of calculating
*                  internal rates of return, there is
*                  no guarantee, express or implied,
*                  that the program will perform.  The
*                  authors will not be liable for any
*                  damages which may result from use of
*                  this software.  Those with little or
*                  no knowledge of the theory of
*                  internal rate of return calculations
*                  are strongly encouraged to find
*                  expert help before attempting to use
*                  this program.
*
*-- How it works:
*
* Basically, finding the internal rate of return is the
* same problem as finding the zero of a function.  In
* this particular case, the problem is reduced to
* finding the value of r that makes the following
* equation equal to zero:
*     SUM[CASHFLOW(i)/r^(i-1)]
* where CASHFLOW(i) is the net cash flow in period i,
* SUM is the summation operator, i runs from 1 to n
* (the total number of periods), and the expression r^i
* means r to the i power.
*
* While there are a number of efficient techniques for
* finding the zero of a function, this program uses the
* simplest, namely a bisection and secant search.
* Those interested in more efficient algorithms should
* consult any numerical analysis book.  Probably the
* best way to make this procedure run faster is to
* translate it into C and compile it.  Anyone who knows
* C should have no trouble performing the translation.
*
*-- How to use it:
*
* The main program basically sets up the values for
* the call to the Zeroin() function which performs
* the actual calculation.  It first sets the number of
* periods of the project (gn_period, here set at 11).
* It then declares the array af_cash[] of that many
* periods and initializes it with the cash flows for
* each period.  Note that af_cash[1] is the initial
* cash flow or investment, while af_cash[2] is the
* cash flow at the end of the first period, and so
* forth. In practice, it is unlikely the cash flows
* will be regular as in the sample data used here for
* testing, so it will be necessary to substitute code
* to input the values from the keyboard, or perhaps
* from a file using the dBASE IV COPY TO ARRAY command
* or the equivalent.  The cash flow in any period may
* be negative or zero; it is essential that an array
* element be included for each period of time whether
* or not there is any activity in the period.
*
* The periods chosen may be years, months or such other
* periods as may be required by the data.  The internal
* rate of return will match the periodicity.  That
* means if your periods are months, the program will
* return a monthly IRR.  Multiply the rate returned by
* 12 to get the equivalent annual IRR.
*
* The program then sets a number of constraints to be
* used in the search for a solution.  The variables
* gf_hirate and gf_lorate, set here at 2.00 (200%) and
* zero, set limits for the rates of returns to be
* examined.  In practice, limits of 1.00 and zero would
* suffice for most situations except perhaps some real
* estate deals.  The Zeroin() function requires that
* the npvs of the limiting rates, Npvcrnch(gf_lorate)
* and Npvcrnch(gf_hirate), have opposite signs.  If
* they do not, the variable gn_iflag is set to 4.  In
* that case, try new limits (although you probably have
* a very badly-behaved set of cash flows and are likely
* to get either very rich or very poor from this
* project).  Since (see accompanying article) there is
* no guarantee there is not a root between two limiting
* rates that yield Npvcrnch() values of the same sign,
* it is possible that judicious or serendipitous choice
* of smaller limits will yield a solution.  In most
* cases, enlarging the limits will work better.

* The maximum iterations gn_maxiter is specified to
* gracefully exit from the procedure if the numerical
* process does not converge in a reasonable amount of
* time, while gf_abserr and gf_relerr set the limits
* within which a solution will be accepted as close
* enough.  In some unusual cases it may be necessary to
* increase the value of any of these, allowing either
* more iterations or a larger error and less accurate
* result, to obtain a solution.

* THERE IS NO GUARANTEE THAT THE INTERNAL RATE OF
* RETURN CALCULATED BY THIS PROGRAM WILL BE THE CORRECT
* ONE, OR EVEN THAT IT WILL BE UNIQUE.  If you do not
* understand the problem of non-unique IRR's, you
* should not attempt to use this program.
* For more information, see the comments at the
* beginning of the Zeroin() user-defined function.

*
*-- Beginning of main program

DO Pstart

PUBLIC af_cash,gn_period,gf_hirate,gf_lorate,;
  gn_maxiter,gf_abserr,gf_relerr,gn_iflag
                                         
DO Wndowdef

ON ERROR DO Pause WITH ;
     ERROR(),MESSAGE(),PROGRAM(),LINENO()

* Initialize array and other variables

* Of the nine statements below, only the DECLARE
* statement is required.  The rest should be changed
* to suit the number of periods of your data (including
* the initial investment as [1]) and to store the cash
* flow for each subsequent period to the proper af_cash
* element for that period.

gn_period = 11
DECLARE af_cash[gn_period]
af_cash[1] = FLOAT(-10000.00)
ln_cntr = 2
DO WHILE ln_cntr <= gn_period
  af_cash[ln_cntr] = FLOAT(100.00) ;
     + 500.0*FLOAT(ln_cntr-2)
  ln_cntr = ln_cntr + 1
ENDDO && WHILE ln_cntr <= gn_period
RELEASE ln_cntr

* See above for meaning of the next five variables,
* which may be changed as required.

gf_hirate = FLOAT(2.00)
gf_lorate = FLOAT(0.00)
gn_maxiter = 200
gf_abserr = FLOAT(0.00001)
gf_relerr = FLOAT(0.0001)

* Reset error flag and perform calculation

gn_iflag = 0
lf_irr = Zeroin(gf_lorate,gf_hirate,gf_abserr,;
  gf_relerr,gn_maxiter,gn_iflag)

* Display results

ACTIVATE WINDOW WPause
DO CASE
  CASE gn_iflag <= 2
    * We have a solution
    @ 02,05 SAY "Internal rate of return (%): "
    @ 02,35 SAY STR(100.00*lf_irr,5,2)
  CASE gn_iflag = 3
    @ 02,05 SAY "Zeroin function is diverging.  "+;
      "Badly behaved"
    @ 03,05 SAY "problem.  Re-examine cash flow data."
  CASE gn_iflag = 4
    @ 01,05 SAY ;
       "NPV for both discount rates have the same"
    @ 02,05 SAY ;
       "sign.  Increase initial values or examine"
    @ 03,05 SAY "cash flow data."
  CASE gn_iflag = 5
    @ 03,05 SAY "Maximum interations exceeded."
ENDCASE
WAIT "Hit the space bar to continue..."
DEACTIVATE WINDOW WPause

DO Pstop

RETURN

* EOProg:  IRR.PRG, main program

*-- Function Npvcrnch -- crunches the NPV

* Since this function and its do while loop do the
* main work of the calculation, they should be
* optimized for speed and accuracy.  Therefore:
*   Stepwise multiplication is used to find the
*     discount rate for each period, rather than
*     exponentiation (^).
*   Multiplication is used instead of division
*     where possible.
*   The npv for the first period and the factor
*     to be applied to the discount rate for
*     each period are calculated outside the
*     loop to minimize work done within.

FUNCTION Npvcrnch
* Calculates npv given assumed rate and # periods.
PARAMETERS lf_rate,ln_periods
* All variables assigned values in functions should
* be declared private, so the values assigned do not
* replace those of public variables of the same name.
PRIVATE lf_disc,lf_factor,ln_period,lf_npv
ln_period=1
lf_npv=af_cash[1]
lf_disc=float(1) 
lf_factor=1/(lf_disc+lf_rate)
DO WHILE ln_period<ln_periods
  ln_period=ln_period+1
  lf_disc=lf_disc*lf_factor
  lf_npv=lf_npv+af_cash[ln_period]*lf_disc
ENDDO && ln_period<ln_periods
RETURN lf_npv

* EOFunc: Npvcrnch

*-- Function Zeroin: Calculates zero of nonlinear
*                      function
FUNCTION Zeroin
PARAMETERS lf_nearate,lf_farrate,lf_abserr,lf_relerr,;
   ln_maxiter,gn_iflag
PRIVATE lf_round,lf_minerr,ln_splits,lf_bracket
PRIVATE lf_npvfar,lf_npvnear,lf_maxabs,lf_npvold
PRIVATE ln_iters,ll_bisect,lf_npvdiff,lf_oldrate
PRIVATE lf_temp,lf_halfdif,lf_abshalf,lf_tolrnce
PRIVATE lf_secant

* Zeroin computes a root of the nonlinear function
* Npvcrnch.  It finds an r such that Npvcrnch(r)=0.
* Npvcrnch is continuous and real in the variable r.
* The method used is a combination of the bisection
* and the secant rules.

* The method employs two trial interest rates, here
* called lf_nearate and lf_farrate.  It makes an
* intelligent choice of an intermediate rate, performs
* the calculation of the Npvcrnch() of the new rate,
* substitutes it for one of the previous trial rates,
* and continues.

* The stopping criterion is
* ABS(lf_nearate - lf_farrate) <= 2.0* ;
*     (lf_relerr*ABS(lf_nearate)+lf_abserr)
*
* lf_abserr and lf_relerr are absolute and relative
* errors respectively.  If 0 is a possible value for r,
* do not set lf_abserr = 0.
*
* The output value of lf_nearate is the better
* approximation to a root since lf_nearate and
* lf_farrate are always redefined so that lf_nearate
* is closer to a root, that is:
*   ABS(Npvcrnch(lf_nearate)) <= ;
*                    ABS(Npvcrnch(lf_farrate))
*
* A flag, gn_iflag, is provided as an output quantity.
* Possible values are:
*
*   gn_iflag = 1   if Npvcrnch(lf_farrate)* ;
*                       Npvcrnch(lf_nearate) < 0
*                    and the stopping criterion is met.
*            = 2   if a value of lf_nearate is found
*                    such that the computed value
*                    Npvcrnch(lf_nearate) = 0 exactly.
*            = 3   if ABS(Npvcrnch(lf_nearate)) exceeds
*                    the input values.  In that case,
*                    lf_nearate is likely to be close
*                    to a pole of Npvcrnch().  Choose
*                    different values for gf_hirate and
*                    gf_lorate and try again.
*            = 4   if the Npvcrnch() values of
*                    gf_hirate and gf_lorate have the
*                    same sign.  Choose different
*                    limiting values and try again.
*            = 5   if too many function evaluations
*                    were made (more than allowed by
*                    ln_maxiter).
*
* Method adapted from Shampine and Allen, "Numerical
* Computing", W.B. Saunders Publishing, 1973.
*
* Set lf_round to approximately the unit round-off of
* the specific machine on which you are running.
* The value set here should suffice for most purposes.

lf_round = FLOAT(0.00000009)
lf_minerr = MAX(lf_relerr,lf_round)
ln_splits = 0
lf_bracket = ABS(lf_nearate - lf_farrate)
lf_npvfar = Npvcrnch(lf_farrate,gn_period)
lf_npvnear = Npvcrnch(lf_nearate,gn_period)

IF SIGN(lf_npvnear) = SIGN(lf_npvfar)
  gn_iflag = 4
  RETURN float(0)
ENDIF

*  Note that this makes a real change in the function
*  of the code--if we start with two npvs of the same
*  sign, under the original code we would try once for
*  a third with different sign, but now we just quit.

lf_maxabs = MAX(ABS(lf_npvnear),ABS(lf_npvfar))
ln_iters = 2
gn_iflag = 0

* Main do while loop

DO WHILE .T.
  IF ABS(lf_npvfar) < ABS(lf_npvnear)
    *
    * Interchange lf_nearate and lf_farrate so that
    * lf_nearate is closer to a solution--
    * ABS(Npvcrnch(lf_nearate)) <= ;
    *        ABS(Npvcrnch(lf_farrate))
    *
    lf_temp = lf_nearate
    lf_nearate = lf_farrate
    lf_farrate = lf_temp
    lf_temp = lf_npvnear
    lf_npvnear = lf_npvfar
    lf_npvfar = lf_temp
  ENDIF
  lf_halfdif = 0.5*(lf_farrate - lf_nearate)
  lf_abshalf = ABS(lf_halfdif)
  lf_tolrnce = lf_minerr*ABS(lf_nearate) + lf_abserr
  *
  * Test whether interval is too small to continue
  *
  IF lf_abshalf <= lf_tolrnce
    IF ABS(lf_npvnear) > lf_maxabs
      *
      * Yes, but we are out of bounds
      *
      gn_iflag = 3
      RETURN float(0)
     ELSE
      *
      * Yes, and we have a solution!
      *
      gn_iflag = 1
      RETURN lf_nearate
    ENDIF
  ENDIF
  *
  * Save the last approximation to the rate
  *
  lf_oldrate = lf_nearate
  lf_oldnpv = lf_npvnear
  *
  * Check if reduction in the size of
  * bracketing interval is satisfactory.
  * If not, bisect until it is.
  *
  ll_bisect = .F.
  ln_splits = ln_splits + 1
  IF ln_splits >= 4
    IF 8.0*lf_abshalf >= lf_bracket
      lf_nearate = lf_nearate + lf_halfdif
      ll_bisect = .T.
    ELSE
      ln_splits = 0
      lf_bracket = lf_abshalf
    ENDIF
  ENDIF
  *
  * If we haven't been forced to bisect by the
  * code above:
  *
  IF .NOT. ll_bisect
    *
    * Begin calculation of secant - lf_temp is a
    * convenient variable, but note the presence of
    * lf_npvdiff in both the statements in which it
    * is used.  The concept is equal to:
    *
    * (lf_nearate-lf_farrate)*lf_npvnear/lf_npvdiff
    * or, "that fraction of the difference in the
    * rates that lf_npvnear is of the difference in
    * net present values"
    *
    lf_temp = (lf_nearate - lf_farrate)*lf_npvnear
    lf_npvdiff = lf_npvfar - lf_npvnear
    *
    * Adjust by at least the tolerance
    *
    IF ABS(lf_temp) <= lf_tolrnce*ABS(lf_npvdiff)
      lf_nearate = lf_nearate + lf_tolrnce* ;
          SIGN(lf_halfdif)
    ELSE
      *
      * Or if tolerance is not a problem,
      * by whichever of the secant or bisection
      * is smaller
      *
      lf_secant = lf_temp/lf_npvdiff
      IF lf_secant < lf_halfdif
        lf_nearate = lf_nearate + lf_secant
      ELSE
        lf_nearate = lf_nearate + lf_halfdif
      ENDIF
    ENDIF
  ENDIF
  *
  * Calculate the npv of the new approximation
  *
  lf_npvnear = Npvcrnch(lf_nearate,gn_period)
  *
  * If it's exactly zero, we win!  Run with it
  *
  IF lf_npvnear = 0.00
    gn_iflag = 2
    RETURN lf_nearate
  ENDIF
  *
  * Else adjust iteration count and quit if too
  * many iterations with no solution
  *
  ln_iters = ln_iters + 1
  IF ln_iters > ln_maxiter
    gn_iflag = 5
    RETURN float(0)
  ENDIF
  *
  * And finally keep as the new lf_farrate that one
  * of the previous approximations, lf_farrate and
  * lf_oldrate, of which the npv has a sign opposite
  * to the npv of the new approximation, lf_nearate.
  *
  IF SIGN(lf_npvnear) = SIGN(lf_npvfar)
    lf_farrate = lf_oldrate
    lf_npvfar = lf_oldnpv
  ENDIF

ENDDO && WHILE .T.

* EOFunc:  Zeroin()

*-- Procedure Pause -- Error trap
PROCEDURE Pause
PARAMETERS ln_err,lc_msg,lc_prg,ln_line
lc_msg = SUBSTR(lc_msg,1,35)
ACTIVATE WINDOW WPause
@ 01,02 SAY "Error: " + STR(ln_err,3)
@ 02,02 SAY "Message:"
@ 03,02 SAY lc_msg
@ 04,02 SAY "Program: " + lc_prg
@ 05,02 SAY "Line number: " + STR(ln_line,5)
WAIT
DEACTIVATE WINDOW WPause
DO Pstop
CANCEL
* EOProc: Pause

*-- Procedure Pstart -- Intialization
PROCEDURE Pstart
CLEAR
CLEAR ALL
PUBLIC gc_talk
gc_talk = SET("TALK")
SET TALK OFF
RETURN
* EOProc:  Pstart

*-- Procedure Pstop -- Exit procedures
PROCEDURE Pstop
SET TALK &gc_talk
CLEAR
CLEAR ALL
RETURN
* EOProc:  Pstop

*-- Procedure Wndowdef -- defines all windows
PROCEDURE Wndowdef
DEFINE WINDOW WPause FROM 10,10 TO 20,70 DOUBLE
DEFINE WINDOW WFullscr FROM 1,0 TO 23,79
RETURN
* EOProc: Wndowdef

* EOFile: IRR.PRG

