/**************************************************************

	Program Name:       FoxCln.C

	Description:        The purpose of this program is to erase
						FoxPro temporary files.  For this program
						Temp files are those that have digits in
						all 8 positions of the file name, and have
						a FoxPro file extension.  The program will
						recursively search all sub-directories.

						If the program is called without the /D
						parameter, then it will perform the search
						without erasing the files and report how
						many bytes will be freed.

	Operating System:   MSDOS 3.3 or higher

	Compiler:           ANSI C compiler w/ DOS extensions
						Watcom C, Borland C or Microsoft C

	Written By:         George Sexton
						MH Software
						1006 W 104th Ave #200
						Northglenn, CO 80234

						Phone:  (303) 438-9585
						CIS: 73237,1665

	Legal Stuff:

		I certify that this work is entirely my own.  Any 
		similarity between this work, and other works, copyrighted
		or not, is entirely coincidental.  Coincidental means
		that similarities between this program and others arise
		from the nature of the problem to be solved, and the
		environment in which the problem exists.  The techniques 
		used in this program, represent those that are established
		through "prior art".  
		
		This work is not copyrighted.  You may freely distribute
		this program.  
		

	Liability:

		My liability will be strictly limited to the cost that
		you paid me for the program.  Zero.  Zero dollars, and
		zero cents.  Zero man-hours contributed by myself to
		rectify any problems resulting.  NIL, nought, NULL, 
		(void *) 0, zilch.  Really zero.  
		

	Disclaimer:

		I didn't want to put this stuff in, but my attorney 
		charges me double for advice I don't follow.

**************************************************************/
/**************************************************************

			Include Files Section

**************************************************************/
#include <assert.h>
#include <ctype.h>
#include <direct.h>
#include <dos.h>
#include <io.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/**************************************************************

			System Defines

***************************************************************/
#define TRUE 1
#define FALSE 0

/***************************************************************

			Function Prototypes

***************************************************************/
unsigned long FoxClean(const char);     // File Erase Function
int CheckName(const char *);            // Name Checking Function
unsigned long GetDiskClusterSize(void); // Get Disk Cluster Size

/***************************************************************

				Global System Variables

***************************************************************/
unsigned long ClusterSize;

int main(int argc, char **argv)
/***************************************************************

	Function Name:      main()

	Parameters:         /D, delete files switch
						/?, show help switch

	Returns:            If /? is passed as the command line parameter
						the function returns 1, otherwise it returns
						0.

	Called By:          N/A

	Calls:              FoxCln(), GetDiskClusterSize()

***************************************************************/
{
	char lErase=FALSE;                  // Logical flag, TRUE if files should
										// be erased

	unsigned long BytesProcessed;       // Total number of bytes that were or
										// would be freed

	ClusterSize=GetDiskClusterSize();   // Get the disk cluster size

	/*
		Display banner, and help if necessary.
	*/
	printf("\nFoxCln V1.05  -  FoxPro Temporary File Deletion Utility"
		"\nMH Software, Denver, Colorado, USA, (303)438-9585\n");
		
	if (argc>2 || strcmp(argv[1],"/?")==0) {
		printf("Usage:	FoxCln [  /d[elete] files ]\n");
		return 1;
		}

	if (argc==2 && stricmp(argv[1],"/D")==0)
		lErase=TRUE;

	BytesProcessed=FoxClean(lErase);    //  Call FoxClean to actually perform
										//  Directory traversal

	/*
		Display the results
	*/
	printf("\n\n\t%ld bytes %sfreed!\n",BytesProcessed,lErase ? "" : "would be ");
	return 0;
}


unsigned long FoxClean(const char EraseFiles)
/**********************************************************

	Function Name:      FoxClean

	Parameters:         EraseFiles  TRUE if files should be erased as
						found, FALSE if they should not be erased.

	Returns:            Number of bytes freed, or bytes that would be
						freed.

	Called by:          main(), FoxClean()

	Calls:              FoxClean(), CheckName()

	Description:        This is the main unit of the the FoxCln program.
						This function erases files that match the
						conditions established by CheckName(), and then
						traverses the directory tree downward.

						This function traverses the directory tree by
						changing to the sub-directory, and then calling
						itself to process the actual directory.

***************************************************************/
{
	typedef struct DIRLIST {
		/*
			DIRLIST structure provides a linked list structure used to
			traverse the directory tree.
		*/
		char dirname[13];               // Directory Name
		struct DIRLIST *NextEntry;      // Pointer to next entry
		}
	DirList;

	DirList Head,                       // Pointer to head of linked list
		*Current,                       // Pointer to current element of list
		*scratch;                       // temp variable used in list traversal

	char *cwd;                          // Current Working Directory

	struct find_t ffblk;                // FindFirst/FindNext structure

	int done;                           // Result variable used for find
										// first/find next function

	unsigned long BytesDone=0;          // Number of bytes processed by routine

	Head.NextEntry=NULL;                // Initialize linked list head pointer
	Current=&Head;                      // Initialize Current Element Pointer
										// to head of linked list

	cwd=getcwd(NULL, _MAX_PATH+3);      // Get the current working directory
										// We need this so that we can come
										// back to this directory during the
										// list traversal

	assert(cwd!=NULL);                  // Verify we got the memory for the CWD

	done = _dos_findfirst("*.*",        // Start the find-first/find-next process
		_A_SUBDIR|_A_NORMAL,
		&ffblk);
	while (!done) {
		/*
			main processing loop for the ff/fnext process.
			stay in this loop until all files have been found and DOS
			returns NO_MORE_FILES
		*/

		if ( (ffblk.attrib & _A_SUBDIR) && (ffblk.name[0]!='.') ) {
			/*
				The entry found is a sub-directory.  Add it to the linked
				list for use later in the routine

				First, allocate the memory using calloc(), and then copy
				the name of the entry into the DirName field of the
				linked list structure

				Using calloc() automatically sets the NextEntry pointer to
				NULL.
			*/
			Current->NextEntry=calloc(sizeof(struct DIRLIST),1);
			assert(Current->NextEntry!=NULL);
			Current=Current->NextEntry;
			strcpy(Current->dirname,ffblk.name);
			}
		else
			/*
				The entry found is a file
			*/
			if (CheckName(ffblk.name)) {
				/*
					The file found meets the criteria established in the
					CheckName function and should be deleted.
				*/

				/*
					Increment the BytesDone value

					Cluster size is used to take into account that a file
					takes at least one cluster on a drive regardless of
					the file size.
				*/
				BytesDone+=ffblk.size+ClusterSize-(ffblk.size % ClusterSize);
				if (EraseFiles) {
					/*
						Erase the file
					*/
					if (unlink(ffblk.name)==0)
						printf("\n\tDeleting  %s\\%s",cwd, ffblk.name);
						else {
						/*
							For some reason, the unlink failed.  The most likely
							cause of this is a file in use error.  Decrement
							BytesDone so that we report the correct value.
						*/
						BytesDone-=(ffblk.size+ClusterSize-(ffblk.size % ClusterSize));
						printf("\n\tError Deleting %s\\%s",cwd,ffblk.name);
						}
					}
				else
					printf("\n\t%s\\%s would be deleted!",cwd,ffblk.name);
				}
		/*
			Call findnext to get the next entry in the directory.
		*/
		done = _dos_findnext(&ffblk);
		}

	/*
		Begin the linked list traversal to process any sub-directories
		that were found.
	*/
	Current=Head.NextEntry;
	while (Current) {
		chdir(Current->dirname);            // Change to the new dir
		BytesDone+=FoxClean(EraseFiles);    // Call FoxClean() to process it
		scratch=Current->NextEntry;         // get the address of the next element
		free(Current);                      // Free the current element
		Current=scratch;
		chdir(cwd);                         // Change to our original directory
		}
	free(cwd);                              // Free memory allocated to hold the
											// current working directory name
	fflush(stdout);
	return BytesDone;                       // Return how many bytes were freed
}


int CheckName(const char *fname)
/***************************************************************

	Function Name:      CheckName()

	Parameters:         pointer to file name

	Returns:            TRUE if file should be processed, FALSE
						otherwise

	Called By:          FoxClean()

	Calls:              N/A

	Description:        The purpose of this function is to check to see
						if the passed file name meets the criteria for
						it to be deleted.

						Current criteria are:

							File Name 8 characters, all digits

							File extension contained in the aExtensions
							array.

***************************************************************/
{
#define MAXEXTENSIONS 12

	char n,                     // Loop Variable
	lDelete=TRUE,               // Result variable
	lExtFound=FALSE;            // TRUE if extension was in list

	const char  *ext=NULL,      // Pointer to the file extension
		aExtensions[MAXEXTENSIONS][4]={
		"BAK","CDX","DBF","FKY",
		"FPT","FRT","FRX","IDX",
		"TBK","TMP","TXT","VUE"
		};

	for (n=0;fname[n] && lDelete;n++) {
		/*
			Loop through the name, verifying that each character in the
			name is a digit.  Once the extension is found, initialize the
			ext pointer.
		*/
		if (fname[n]=='.') {
			ext=&fname[++n];
			break;
			}
		lDelete=lDelete && isdigit(fname[n]);
		}
	lDelete&=(n==9);            // Verify that we did find an eight char name
	if (lDelete && ext) {       // So far so good, and we found the extension
		signed int rc;          // result code variable

		/*
			Loop through the extensions array looking for a match.  Once
			one is found, drop out of the loop.
		*/
		for (n=0, rc=1;(n<MAXEXTENSIONS) && (rc>0);n++)
			lExtFound=((rc=strcmp(ext,aExtensions[n]))==0);
		}
	return (lDelete && lExtFound);      // Return whether we should delete
										// this file
}


unsigned long GetDiskClusterSize(void)
/***************************************************************

	Function Name:  GetDiskClusterSize()

	Parameters:     N/A

	Returns:        The cluster size of the current default drive

	Called By:      main()

	Calls:          N/A

	Description:    The purpose of this function is to return the
					cluster size of the current default drive.

					This value is used in calculating the amount
					of space freed by the program.

					The function works by calling dos interrupt
					0x1b, Get Fat Information Current Drive
					and returning the sector size * the number of
					sectors per cluster.

					This does not really work correctly on Novell
					drives with cluster sizes greater than 4K.

***************************************************************/
{
	union REGS regs;
	struct SREGS sr;
	segread(&sr);
	regs.h.ah=0x1b;
	intdosx(&regs, &regs, &sr );
	return (long) (regs.x.cx * regs.h.al);
}
