*********************************************************************
*
*        DEMONSTRATE VARIOUS INTERNAL RATE OF RETURN FUNCTIONS
*
*    The arrays used are for demonstration purposes. Under normal
*    operations, the arrays would most likely be filled from a database.
*
*    This demo is designed to run under FoxPro 2. To run under dBASEIV,
*    the array names must be passed in quotes and calls to the 'flow[]'
*    array within the functions must be preceeded by &, for example,
*    'CV = &FLOW[i]' instead of 'CV = FLOW[i]. Also, to run under 
*    dBASEIV, the 'IF PARAMETERS() = 3' statements should be changed
*    to 'IF PCOUNT() = 3' 
*
*          (dBASEIV required changes made in this file)
*
**********************************************************************


clear
set talk off

* SET UP ARRAY OF PERIODIC CASH FLOWS FOR IRR AND MIRR DEMO.

public array cash[6]
cash[1] = -120000
cash[2] = 39000
cash[3] = -10000
cash[4] = 21000
cash[5] = 37000
cash[6] = 46000

* udfparam = set("UDFPARMS")    && need to pass array by reference
* set udfparms to reference
?
? mirr(6,"cash",.14,.1)*100  &&  Should return 7.17% for above figures
?? "%  (MIRR)"
?


? irr(6,"cash")*100   && Should return 3.05% for above figures
?? "%  (IRR)"
?


* SET UP ARRAY FOR UNEVEN PERIODS OF CASHFLOW
public array ucash[5,2]

store ctod("01/01/91") to ucash[1,1]
store -10000 to ucash[1,2]
store ctod("03/01/91") to ucash[2,1]
store 2750 to ucash[2,2]
store ctod("10/30/91") to ucash[3,1]
store 4250 to ucash[3,2]
store ctod("02/15/92") to ucash[4,1]
store 3250 to ucash[4,2]
store ctod("04/01/92") to ucash[5,1]
store 2750 to ucash[5,2]

? ((xmirr(5,"ucash",.14/365,.1/365)+1)^365-1)*100
?? "% annualized (XMIRR)"    && Should return 29.21% annualized
?

? ((xirr(5,"ucash")+1)^365-1)*100     && Should return 37.46% annualized
?? "% annualized (XIRR)"
?

* set udfparms to &udfparam

***************************************************************************
*  
*    Function MIRR returns Modified Internal (periodic) Rate of Return
*         (Based on equally-spaced periodic uneven cashflows)
*
*    Input parameters: n = number of transactions, flow = the name of
*    an array containing the transactions, rrate = the periodic return 
*    rate on reinvested positive cashflows, frate = the periodic rate
*    on borrowed funds or the 'safe' periodic rate on reserve funds.
*    
***************************************************************************

FUNCTION MIRR
parameters n, flow, rrate, frate

store 0 to i, negv, posv
do while i < n
	i = i+1
	cv = &flow[i]
	cv = cv*(1+iif(cv<0,frate,rrate))^-(i-1)
	if cv < 0
		negv = negv + cv
	else
		posv = posv + cv
	endif
enddo
if abs(negv) = 0 .or. posv = 0
	wait " There must be at least one negative and one positive value! "
	return 0
endif
return  ((-posv * (1+rrate)^(n-1))/negv)^(1/(n-1))-1

***************************************************************************
*
*      FUNCTION IRR returns Internal (periodic) Rate of Return
*        (Based on equally-spaced periodic uneven cashflow)
*
*      Input parameters: n = number of transactions, flow = the name of
*      the array containing the transactions, guess = an optional intitial
*      guess for the periodic rate of return. If a guess is not given,
*      an initializing rate is calculated. The user should be aware that
*      there can be as many solutions for 'IRR' as there are reversals
*      in total present value (how many times a graphic would cross
*      the horizontal axis). If there are multiple solutions, 'guess' may 
*      be varied to isolate all solutions.
*
***************************************************************************

FUNCTION IRR
parameters n, flow, guess
store 0 to i, posv, negv
do while i < n
	i = i+1
	store &flow[i] to cv
	if cv < 0
		negv = negv + cv
	else
		posv = posv + cv
	endif
enddo
if negv = 0 .or. posv = 0
	wait "Must have at least one positive and one negative value"
endif
if pcount() = 3  
newguess = guess
else
	newguess = ((-posv/negv)-1)/n
endif
store 1 to nudelta
do while abs(nudelta) > .000001
	store 0 to i, scvpv, scvd
	do while i < n
		i = i+1
		cv = &flow[i]
		cvpv = cv*(1+newguess)^-i  && incremental present value
		cvd = -i*cvpv/(1+newguess)  && incremental derivative of pv
		scvpv = scvpv + cvpv
		scvd = scvd + cvd
	enddo
	nudelta = scvpv/scvd
	newguess = newguess - nudelta
enddo
return newguess



FUNCTION XMIRR
parameters n, flow, rrate, frate

store 0 to i, negv, posv
store &flow[1,1] to pdate
maxdate = pdate

do while i < n
	i = i+1
	cv = &flow[i,2]
	cdate = &flow[i,1]
	maxdate = max(cdate,maxdate)
	cn = cdate-pdate
	cv = cv*(1+iif(cv<0,frate,rrate))^-cn
	if cv < 0
		negv = negv + cv
	else
		posv = posv + cv
	endif
enddo
if negv = 0 .or. posv = 0
	wait " There must be at least one negative and one positive value! "
	return 0
endif
n = maxdate - pdate
mirr = ((-posv * (1+rrate)^(n-1))/negv)^(1/(n-1))-1
return mirr

***************************************************************************		
*
*       FUNCTION XIRR Returns (daily) Internal Rate of Return
*        (Based on uneven periods and uneven cashflow)
*
*    Input parameters: n = number of transactions, flow = name of n x 2 
*    array containing dates and transactions. &flow[1,1] must contain 
*    earliest date, i.e., date to which present value applies, guess
*    is optional input to guess daily rate of return. See remarks under
*    IRR above for additional information. 
*
***************************************************************************	


FUNCTION XIRR
parameters n, flow, guess

pdate = &flow[1,1]
maxdate = pdate
store 0 to posv, negv
cv = &flow[1,2]
if cv < 0
	negv = cv
else
	posv = cv
endif
	
store 1 to i
do while i < n
	i = i+1
	store &flow[i,2] to cv
	store &flow[i,1] to cdate
	store max(cdate,maxdate) to maxdate
	if cv < 0
		negv = negv + cv
	else
		posv = posv + cv
	endif
enddo
if negv = 0 .or. posv = 0
	wait "Must have at least one positive and one negative value"
endif
if pcount() = 3  
newguess = guess
else
	newguess = ((-posv/negv)-1)/(maxdate-pdate)
endif
store 1 to nudelta
do while abs(nudelta) > .000001
	store 0 to i, scvpv, scvd
	do while i < n
		i = i+1
		cv = &flow[i,2]
		cdate = &flow[i,1]
		cn = cdate-pdate
		cvpv = cv*(1+newguess)^-cn   && incremental pv
		cvd = -cn*cvpv/(1+newguess)  && incremental derivative
		scvpv = scvpv + cvpv
		scvd = scvd + cvd
	enddo
	nudelta = scvpv/scvd
	newguess = newguess - nudelta
enddo
return newguess
