FUNCTION PickList && {ver 1.0}
*-------------------------------------------------------------------------------
*-- Programmer..: Kenneth W. Holloway (HollowayK on BORBBS)
*-- Date........: 10/29/1992
*-- Notes.......: Pick List. This is a "generic" picklist routine designed
*--               to act in a similar fashion to the POPUP commands used
*--               with dBASE for PROMPT FIELDS/FILES/etc. If you type a letter,
*--               this routine will take you to the first record that matches
*--               and typing a second letter will go to the record that matches
*--               the combination of letters. Additionally, if you add between
*--               fields the CHR(29) code (see examples), pressing TAB will
*--               take you to the next field, and then, pressing a letter will
*--               take you to the next record that meets that requirement.
*-- Written for.: dBASE IV, 1.5
*-- Rev. History: 08/12/1992 0.0 - Original version (KWH)
*                 09/11/1992 0.1 - (KWH) Added color settings (x_ClrP*) that
*                                   were Ass-U-Med to be defined elsewhere.
*                 09/16/1992 0.2 - (KWH) Added "set key to" at end of function.
*                                   (BORLAND: What happened to set("KEY")?!?!)
*--               10/14/1992 0.3 - Added (KenMayer) ability to pass colors
*--                                to program ... removed settings for
*--                                alias, order, key. The reason is a lack
*--                                of stack space to call routine, can only send
*--                                x number of parms. The programmer must
*--                                set the database (select .../Use ...), 
*--                                order, and key (set key...) before calling
*--                                this routine, and then reset to prior setting
*--                                (if needed). 
*                 10/15/1992 0.4 - (KWH) Added code for Tab/Shift Tab. Put the
*                                   setting for key back in, as it is required
*                                   for proper SEEKing with SET KEY in effect.
*                 10/19/1992 0.5 - (KWH) Several changes inspired by JOEY:
*                    Now uses setting of SET BORDER TO when drawing borders.
*                    Bell only sounds when SET BELL is ON.
*                    Added code for {Home} and {End}.
*-- Calls.......: COLORBRK()           Function in PROC.PRG
*-- Called by...: Any
*-- Usage.......: PickList(cTitle,cDisplay,cReturn[,cKey[,nFromRow,nFromCol
*                             [,nToRow,nToCol[,cColor1[,cColor2]]]]])
*-- Example.....: ? PickList("Client Name","NAME","JOB_CODE","",5,10,20,55,;
*--                          cColor1,cColor2)
*--                 or
*--               ? PickLsit("Cliant Name","NAME+chr(29)+JOB_CODE","JOB_CODE",;
*--                          5,10,20,55,cColor1,cColor2)
*-- Returns.....: Specified expression, using macro substitution.
*-- Parameters..: cTitle    = Title to be displayed above PickList
*--               cDisplay  = Expression to display, using macro substitution
*--                 Note: If cDisplay includes any chr(29)'s (), the Tab and
*--                       Shift Tab keys can be used to highlight/unhighlight
*--                       everything up to the next/previous chr(29).
*--               cReturn   = Expression to return, using macro substitution
*--               cKey      = Expression for SET KEY TO
*--               nFromRow  \ Upper left corner
*--               nFromCol  / of PickList window
*--               nToRow    \ Lower right corner
*--               nToCol    / of PickList window
*--               cColor1   = message,title,box 
*--               cColor2   = highlight,selected
*--                           Both cColor1, and cColor2 use specific color
*--                           settings of <Foreground>/<Background>  for each
*--                           part of the parm. For example, cColor1 might
*--                           look like:  rg+/gb,w+/b,rg+/gb
*--                           Definitions:
*--                            message   = unselected items in picklist (w+/rb)
*--                            title     = title at top of window (w+/rb)
*--                            box       = border (rg+/rb)
*--                            highlight = highlighted item (g+/n)
*--                            selected  = selected character(s) (r+/n)
*-------------------------------------------------------------------------------

  parameters  cTitle,cDisplay,cReturn,;
              cKey,;
              nFromRow,nFromCol,nToRow,nToCol,;
              cColor1, cColor2
  private all except *_*
  private all like x_ClrP*


  * Check validity of all parameters
  if pcount()<3
    return "***"+program()+" Error***"
  endif

  * Save setting of TALK and turn it off.
  if set("TALK")="ON"
    set talk off
    cTalk     = "ON"
   else
    cTalk     = "OFF"
  endif

  * Save and change settings of other parameters
  cConsole    = set("CONSOLE")
  cCursor     = set("CURSOR")
  cEscape     = set("ESCAPE")
  set cursor        off
  set escape        off

  * Set default values for unspecified parameters
  if type("cKey")="L"
    cKey      = ""
  endif
  if type("nFromRow")="L"
    nFromRow  = 5
  endif
  if type("nFromCol")="L"
    nFromCol  = 5
  endif

  if type("cColor1")="L"
	  x_ClrPMess  = "W+/RB"
	  x_ClrPTitl  = "W+/RB"
	  x_ClrPBox   = "RG+/RB"
  else
     x_ClrPMess  = colorbrk(cColor1,1)
     x_ClrPTitl  = colorbrk(cColor1,2)
     x_ClrPBox   = colorbrk(cColor1,3) 
  endif
  if type("cColor2")="L"
     x_ClrPHigh  = "G+/N"
     x_ClrPSlct  = "R+/N"
  else
     x_ClrPHigh  = colorbrk(cColor2,1)
     x_ClrPSlct  = colorbrk(cColor2,2)
  endif


  *-- Real code starts here
  * Setup specified database environment
  if .not.isblank(cKey)
    set key to cKey
  endif

  * Calculate value of nToRow
  if type("nToRow")="L"
    goto top
    count to nToRow next 21-nFromRow
    nToRow    = nFromRow + max(nToRow,3) + 3
  endif

  * Calculate value of nToCol
  if type("nToCol")="L"
    nToCol    = nFromCol + max(len(cTitle),len(&cDisplay.)) + 1
    if nToCol>79
      nToCol  = 79
    endif
  endif

  * Define and activate title window, draw border and title
  define window wPickList1 from nFromRow,nFromCol to nToRow,nToCol none ;
    color &x_ClrPMess.
  activate window wPickList1
  nWindRow  = nToRow - nFromRow
  nWindCol  = nToCol - nFromCol
  @ 00,00 to nWindRow,nWindCol  color &x_ClrPBox.
  @ 01,01 say cTitle            color &x_ClrPTitl.
  @ 02,01 to 02,nWindCol-1      color &x_ClrPBox.
  cBorder = set("BORDER")
  do case
    case cBorder="NONE"
    case cBorder="SINGLE"
      @ 02,00       say ""                             color &x_ClrPBox.
      @ 02,nWindCol say ""                             color &x_ClrPBox.
    case cBorder="DOUBLE"
      @ 02,00       say ""                             color &x_ClrPBox.
      @ 02,nWindCol say ""                             color &x_ClrPBox.
    case cBorder="PANEL"
      @ 02,00       say ""                             color &x_ClrPBox.
      @ 02,nWindCol say ""                             color &x_ClrPBox.
    otherwise
      @ 02,00       say chr(val(substr(cBorder,17,3)))  color &x_ClrPBox.
      @ 02,nWindCol say chr(val(substr(cBorder,21,3)))  color &x_ClrPBox.
  endcase

  * Define and activate data window
  define window wPickList2 from nFromRow+3,nFromCol+1 to nToRow-1,nToCol-1 none color &x_ClrPMess.
  activate window wPickList2
  nWindRow  = nToRow - nFromRow-4
  nWindCol  = nToCol - nFromCol-2

  * Initialize position and status variables
  goto top
  lBell     = (set("BELL")="ON")
  nCurRow   = 0
  nInkey    = 0
  nNewRow   = 0
  nRecNo    = recno()
  lRepaint  = .t.
  cSeek     = ""
  lSeek     = .F.
  nNewSCur  = 0
  nSeekCur  = 0
  if eof()
    if lBell
      @ 00,00 say chr(7)
    endif
    @ 00,00 say "*** No records to list ***"
    set console off
    wait
    set console on
    cReturn = ""
    nInkey  = 27
  endif


  *-- Display PickList until Enter .or. Ctrl-Q .or. Ctrl-W or Ctrl-End
  *-- .or. Esc is pressed
  do while nInkey#13 .and. nInkey#17 .and. nInkey#23 .and. nInkey#27
    if lSeek
      seek cKey+cSeek
      nNewSCur    = len(cSeek)
      cStr        = &cDisplay.
      nPos  = at(chr(29),substr(cStr,1,nNewSCur+1))
      do while nPos>0
        cStr      = stuff(cStr,nPos,1," ")
        nNewSCur  = nNewSCur + 1
        nPos      = at(chr(29),substr(cStr,1,nNewSCur+1))
      enddo
      nSeek = recno()                   && Save new record number
      n     = 0                         && Counter
      goto nRecNo                       && Record at top of string
      * Look to see if new record is on screen
      scan while recno()#nSeek .and. n<nMaxRow
        n = n + 1
      endscan
      if recno()=nSeek                  && New record is on screen
        nNewRow = n                     && Put cursor on new record
       else                             && New record is not on screen
        nNewRow   = 0                   && Put cursor at top of window
        nRecNo    = nSeek               && New record at top of window
        lRepaint  = .T.                 && Redisplay window
      endif
      lSeek = .F.
    endif

    if lRepaint .or. nNewRow#nCurRow
      * Hide cursor
      @ nCurRow,00 fill to nCurRow,nWindCol color &x_ClrPMess.
    endif

    if lRepaint         && Need to redisplay entire data window
      goto nRecNo                       && Record that should be at top of window
      nMaxRow = 0                       && Number of rows displayed
      scan while nMaxRow<=nWindRow      && nWindRow = number of rows in window
        * Display data
        @ nMaxRow,00 say &cDisplay. picture replicate('X',nWindCol+1) color &x_ClrPMess.
        nMaxRow = nMaxRow + 1           && Increase rows displayed counter
      endscan
      nMaxRow = nMaxRow - 1             && Make rows displayed counter zero-based

      if eof() .and. nMaxRow<nWindRow   && Didn't fill window?
        * Clear unused portion of window
        @ nMaxRow+1,00 clear to nWindRow,nWindCol
      endif
    endif

    if lRepaint .or. nNewRow#nCurRow .or. nNewSCur#nSeekCur
      nSeekCur  = nNewSCur              && New seek cursor length
      nCurRow   = nNewRow               && New cursor position
      if nCurRow>nMaxRow                && Cursor row invalid? (Caused by PgDn)
        nCurRow = nMaxRow               && Put cursor on last displayed row
      endif

      * Display cursor
      if nSeekCur>0
        @ nCurRow,00;
          fill to nCurRow,min(nWindCol,nSeekCur-1);
          color &x_ClrPSlct.
      endif
      if nSeekCur<=nWindCol
        @ nCurRow,max(0,nSeekCur);
          fill to nCurRow,nWindCol;
          color &x_ClrPHigh.
      endif
    endif

    lRepaint = .F.                      && Reset redisplay flag


    nInkey = inkey(0)                   && Get a key-stroke
    do case
      case nInkey=-400                && Shift-Tab
        if isblank(cSeek)
          if lBell
            @ 00,00 say chr(7)
          endif
         else
          if len(cSeek)=nSeekCur
            cSeek = ""
            lSeek = .T.
           else
            goto nRecNo                   && Record at top of window
            skip nCurRow                  && Cursor row
            * Currently seeked string
            cStr  = substr(&cDisplay.,1,nSeekCur)
            * If the last character is a chr(29)
            if substr(cStr,len(cStr),1)=chr(29)
              * Remove the chr(29)
              cStr  = substr(cStr,1,len(cStr)-1)
            endif
            * If there is a chr(29)
            if chr(29)$cStr
              * Remove everything after the last chr(29)
              cSeek = substr(cSeek,1,len(cSeek)-len(cStr)+RAt(chr(29),cStr))
             else
              * Remove everything
              cSeek = ""
            endif
            lSeek = .T.
          endif
        endif

      case nInkey=-6                  && F7
        l_Exit  = .T.
        cReturn = ""                      && Return value

      case nInkey=3                   && PageDown
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        if nCurRow=nMaxRow                && Is cursor on last line in window?
          goto nRecNo                     && Record at top of window
          skip nWindRow+1                 && Number of records in window
          if eof()
            if lBell
              @ 00,00 say chr(7)          && No more records past bottom of window
            endif
           else
            skip -1                       && Put bottom record at top of window
            nRecNo    = recno()           && New record for top of window
            lRepaint  = .T.               && Redisplay window
          endif
         else                             && Cursor is not on last line in window
          nNewRow = nMaxRow               && Put cursor on last line in window
        endif

      case nInkey=5                   && Up Arrow
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        if nCurRow>0                      && Is cursor below top of window?
          nNewRow = nCurRow - 1           && Move cursor up
         else                             && Cursor is at top of window
          goto nRecNo                     && Record at top of window
          skip -1
          if bof()
            if lBell
              @ 00,00 say chr(7)            && No previous record
            endif
           else
            nRecNo    = recno()           && New record for top of window
            lRepaint  = .t.               && Redisplay window
          endif
        endif

      case nInkey=9                   && Tab
        goto nRecNo                       && Record at top of window
        skip nCurRow                      && Cursor row
        * Characters after currently seeked string
        cStr  = substr(&cDisplay.,nSeekCur+1)
        if (chr(29)$cStr)                 && Tab marker included?
          * Seek everything up to the tab marker
          cStr  = substr(cStr,1,at(chr(29),cStr)-1)
          if .not.seek(cKey+cSeek+cStr)
            cStr  = upper(cStr)
          endif
          if seek(cKey+cSeek+cStr)
            cSeek = cSeek + cStr
            lSeek = .T.
           else
            if lBell
              @ 00,00 say chr(7)
            endif
          endif
         else
          if lBell
            @ 00,00 say chr(7)
          endif
        endif

      case nInkey=13 .or. nInkey=23   && Enter .or. Ctrl-W or Ctrl-End
        goto nRecNo                       && Record at top of window
        skip nCurRow                      && Cursor row
        cReturn = &cReturn.               && Return value

      case nInkey=17 .or. nInkey=27   && Ctrl-Q .or. Escape
        cReturn = ""                      && Return value

      case nInkey=18                  && Page Up
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        if nCurRow=0                      && Is cursor on top line of window?
          goto nRecNo                     && Record at top of window
          skip -nWindRow                  && Number of records in window
          if bof()
            if lBell
              @ 00,00 say chr(7)            && No more records above top of window
            endif
           else
            nRecNo    = recno()           && New record for top of window
            lRepaint  = .T.               && Redisplay window
          endif
         else                             && Cursor is not on top line of window
          nNewRow = 0                     && Put cursor on top line of window
        endif

      case nInkey=24                  && Down Arrow
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        if nCurRow<nMaxRow                && Is cursor above bottom of window?
          nNewRow = nCurRow + 1           && Move cursor down
         else                             && Cursor is at bottom of window
          goto nRecNo                     && Record at top of window
          skip nWindRow+1                 && Skip to first record below window
          if eof()
            if lBell
              @ 00,00 say chr(7)            && No records below window
            endif
           else
            goto nRecNo                   && Record at top of window
            skip +1
            nRecNo    = recno()           && New record for top of window
            lRepaint  = .T.               && Redisplay window
          endif
        endif

      case nInkey=2 .or. nInkey=30    && End .or. Ctrl-Page Down
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        goto bottom                       && Last record in database
        skip -nWindRow                    && Number of records in window
        nNewRow   = nWindRow              && Put cursor on bottom line of window
        nRecNo    = recno()               && New record for top of window
        lRepaint  = .T.                   && Redisplay window

      case nInkey=26 .or. nInkey=31   && Home .or. Ctrl-Page Up
        cSeek     = ""                    && Clear seek string
        nNewSCur  = 0                     && Clear seek cursor
        goto top                          && First record in database
        nNewRow   = 0                     && Put cursor on top line of window
        nRecNo    = recno()               && New record for top of window
        lRepaint  = .T.                   && Redisplay window

      case nInkey>31 .and. nInkey<127 && Displayable character - Seek it
        cInkey  = chr(nInkey)
        if .not.seek(cKey+cSeek+cInkey)
          cInkey  = upper(cInkey)
        endif
        if seek(cKey+cSeek+cInkey)        && Seek with new character
          cSeek     = cSeek + cInkey      && Add new character to seek string
          lSeek     = .T.
         else
          if lBell
            @ 00,00 say chr(7)              && Seek with new character failed
          endif
        endif

      case nInkey=127                 && Back Space
        if len(cSeek)>0                   && Seek string is non-blank
          * Remove last character from seek string
          cSeek = left(cSeek,len(cSeek)-1)
          lSeek = .T.
         else
          if lBell
            @ 00,00 say chr(7)              && Seek string is blank
          endif
        endif

      otherwise                       && Unknown key
        b=.t.                             && Breakpoint - used for debugging
        release b
    endcase
  enddo

  * Deactivate and release windows
  deactivate window wPickList2
  deactivate window wPickList1
  release windows wPickList1,wPickList2

  * Restore database environment
  if .not.isblank(cKey)
    set key to
  endif

  *-- Cleanup
  set console       &cConsole.
  set cursor        &cCursor.
  set escape        &cEscape.
  set talk          &cTalk.

RETURN cReturn
*-- EoF: PickList

FUNCTION ColorBrk
*-------------------------------------------------------------------------------
*-- Programmer..: Ken Mayer (KENMAYER)
*-- Date........: 07/22/1992
*-- Notes.......: This routine is designed to be used with any of my functions
*--               and procedures that accept a memory variable for color,
*--               and use a window. It's purpose is to break that color var
*--               into it's components (depending on which one the user wants)
*--               and return those components, so that they can then be used
*--               in SET COLOR OF ... commands.
*-- Written for.: dBASE IV, 1.1, 1.5 (written because of 1.5, but will work in
*--                1.1)
*-- Rev. History: 07/22/1992 - modified to handle memvars/color strings that
*--               may have only two parts to them (no <border>...), so that if
*--               the <nField> parm is 2, we get a valid value.
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: ColorBrk(<cColorVar>,<nField>)
*-- Example.....: set color of normal to ColorBrk(cColor,1)
*-- Returns.....: Either the field you asked for (1 thru 3) or null string ("").
*-- Parameters..: cColorVar = Color variable to extract data from
*--                   Assumes the form: <main color>,<highlight>,<border>
*--                   Where each part uses: <foreground>/<background> format
*--                    i.e., rg+/gb,w+/b,rg+/gb
*--               nField    = Field you want to extract
*-------------------------------------------------------------------------------

	parameters cColorVar, nField
	private cReturn, cExtracted
	
	do case
		case nField = 1
			cReturn = left(cColorVar,at(",",cColorVar)-1)
		case nField = 2
			cExtract = substr(cColorVar,at(",",cColorVar)+1)  && everything to 
			                                                  && right of comma
			if at(",",cExtract) > 0
				cReturn = left(cExtract,at(",",cExtract)-1)    && left of second ,
			else
				cReturn = cExtract
			endif
		case nField = 3
			cExtract = substr(cColorVar,at(",",cColorVar)+1)
			cReturn = substr(cExtract,at(",",cExtract)+1)
		otherwise
			cReturn = ""
	endcase

RETURN cReturn
*-- EoF: ColorBrk()

