/*-----------------------------------------------------------------*
 |  -  DBO.C                                       Public domain   |
 |                                                                 |
 |  -  DBase to Oracle data utility.  Generates BLD, CTL and CVT   |
 |     files for building new table, using ODL to load it, and     |
 |     converting date fields to Oracle internal date format.      |
 |                                                                 |
 |  -  R.Trevithick, Tue  03-28-1989                               |
 |                                                                 |
 |  -  Turbo C v2.0   tcc -w -p -mt -lt -f- -G- -N- -K -O -r -Z -d |
 *-----------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include <dos.h>
#include <dir.h>
#include <io.h>
#include "dbo.h"


/*---------------------------------------- globals --------------------*/
unsigned    f_num = 0,                  /* field counter index         */
            db_length;                  /* adjusted field width        */

char        dbf_name[MAXPATH],          /* input filename              */
            txt_name[MAXPATH],          /* text SDF data file for ODL  */
            tab_name[MAX_NAME],         /* Oracle table name           */
            ctl_name[MAXPATH],          /* output control filename     */
            bld_name[MAXPATH],          /* output BLD filename         */
            cvt_name[MAXPATH],          /* output CVT filename         */
            temp[MAXPATH],              /* misc array for tests, etc.  */
            cvt_needed = 0;             /* flag for date conversions   */


/*---------------------------------------- defined in DBO header file -*/
struct dbf_header   dh;                 /* dBase file header layout    */
struct dbf_record   dr;                 /* dBase record information    */
struct field        f[MAX_FIELDS];      /* our working data storage    */


/*---------------------------------------- function prototypes --------*/
void    parse(int argc, char *argv[]),  /* parser for command line     */
        read_dbf(void),                 /* open and read the .dbf file */
        write_bld(void),                /* write 'create table' file   */
        write_ctl(void),                /* write ODL control file      */
        write_cvt(void),                /* write date conversion file  */
        end_prog(char  code);           /* generic error routine       */
char    user_abort(void),               /* get yes/no response         */
        fcheck(char *filespec);         /* check for existing files    */




/*****************************************************
 * FUNCTION MAIN()                                   *
 * -call each module in turn.                        *
 *****************************************************/
void cdecl main(int argc, char *argv[])
{
    parse(argc, argv);
    read_dbf();
    write_bld();
    write_ctl();
    if (cvt_needed) write_cvt();
    exit(0);
}



/*****************************************************
 * FUNCTION PARSE()                                  *
 * -examine the command line; usage msg on errors.   *
 * -dump the reserved symbol array on /r request.    *
 * -construct all names, checking for conflicts.     *
 *****************************************************/
void parse(int argc, char *argv[])
{
    char    *name_ptr, *cp;
    int     i;

    if (argc < 2) end_prog(ARGS);

    name_ptr = strcpy(temp, strlwr(argv[1]));
    strcpy(dbf_name, name_ptr);

    for (i = 1; i < argc; i++) {
        if (argv[i][0] == '/') {
            strlwr(argv[i]);
            cp = argv[i]+2;
            switch(argv[i][1]) {
                case 'n' :  if (*cp)
                                name_ptr = strcpy(temp, cp);
                            else
                                end_prog(ARGS);
                            break;
                case 'r' :  printf("\n \t\t %s %s\n\n%s",
                                       ORA_VER, "RESERVED SYMBOL TABLE",
                                       reserved);
                            exit(0);
                default  :  end_prog(ARGS);
            }
        }
    }

    if (dbf_name[0] == '/') end_prog(ARGS);

    for (i = strlen(temp); i > -1; i--)
        if (temp[i] == '\\' || temp[i] == ':') {
            name_ptr = temp+i+1;
            break;
        }

    if (NULL != (cp = strchr(name_ptr, '.'))) strcpy(cp, "\0");

    if (strlen(name_ptr) > MAX_NAME-1) end_prog(NAME);

    strcpy(tab_name, name_ptr);
    if (!strchr(dbf_name, '.')) strcat(dbf_name, ".dbf");
    strcpy(txt_name, name_ptr); strcat(txt_name, ".txt");
    strcpy(bld_name, name_ptr); strcat(bld_name, ".bld");
    strcpy(ctl_name, name_ptr); strcat(ctl_name, ".ctl");
    strcpy(cvt_name, name_ptr); strcat(cvt_name, ".cvt");

    strcpy(temp, " "); strcat(temp, tab_name); strcat(temp," ");
    if (NULL != (strstr(reserved, temp)))
        fprintf(stderr, "\nReserved table name changed to %s\n",
                            strcat(tab_name, SYMB_EXT));

    fprintf(stderr,
            "\n\tOracle table....%s\n"
              "\tDBF data file...%s\n"
              "\tTXT data file...%s\n"
              "\tBLD program.....%s\n"
              "\tCTL file........%s\n"
              "\tCVT program.....%s\n\n",
               tab_name, dbf_name, txt_name, bld_name, ctl_name, cvt_name);
}



/*****************************************************
 * FUNCTION READ_DBF()                               *
 * -open, read and verify the .dbf file header.      *
 * -rename fields conflicting with reserved symbols. *
 * -read and store each field, skipping MEMO fields. *
 * -store 'R' for decimals; makes it cleaner later.  *
 *****************************************************/
void read_dbf(void)
{
    char    ren_flag = 0;
    int     fi;


    if (-1 == (fi = open(dbf_name, O_BINARY | O_RDONLY))) end_prog(READ);

    if (sizeof(dh) != (read(fi, (char *) &dh, sizeof(dh)))) end_prog(READ);

    if ( !(dh.id_byte == NORM_DBF || dh.id_byte == MEMO_DBF)
      || !(dh.year > -1 && dh.year < 100)
      || !(dh.month > 0 && dh.month < 13)
      || !(dh.day > 0 && dh.day < 32)
      || !(dh.header_len && dh.record_len) ) end_prog(HEAD);

    db_length = dh.record_len + 1;

    while (sizeof(dr) == (read(fi, (char *) &dr, sizeof(dr)))) {

        if (dr.field_name[0] == 0x0d) break;

        if (dr.data_type == 'D') cvt_needed = 1;

        if (dr.data_type == 'M') {
            fprintf(stderr, "Skipping memo field: %s\n", dr.field_name);
            db_length -= dr.field_length;
            continue;
        }

        strcpy(temp, " "); strcat(temp, dr.field_name); strcat(temp, " ");
        if (NULL != (strstr(reserved, temp))) {
            ren_flag = 1;
            fprintf(stderr, "Reserved field name ");
        }

        f[f_num].type = dr.decimals ? 'R' : dr.data_type;
        f[f_num].len = dr.field_length;
        f[f_num].dec = dr.decimals;
        strcpy(f[f_num].name, strlwr(dr.field_name));

        if (ren_flag) {
            ren_flag = 0;
            fprintf(stderr, "changed to %s\n",
                                strcat(f[f_num].name, SYMB_EXT));
        }

        f_num++;
    }

    fprintf(stderr, "\nInput file processed.  Create output files (Y/N) ? ");
    if (user_abort()) exit(0);
}



/*****************************************************
 * FUNCTION WRITE_BLD()                              *
 * -generate file to build the new Oracle table.     *
 *****************************************************/
void write_bld(void)
{
    struct field    *fp;
    int             i;


    if (fcheck(bld_name)) return;

    if (!(freopen(bld_name, "wt", stdout))) end_prog(WRIT);

    printf("create table %s (\n", tab_name);
    for (i = f_num, fp = f; i; i--, fp++)
        switch (fp->type) {
            case 'R':   printf("%s%s%*s%d,%d%c%s",
                                TAB, fp->name,
                                23-strlen(fp->name), "number(",
                                fp->len, fp->dec, ')',
                                i == 1 ? ")\n/\n" : ",\n");
                                break;
            case 'N':   printf("%s%s%*s%d%c%s",
                                TAB, fp->name,
                                23-strlen(fp->name), "number(",
                                fp->len, ')',
                                i == 1 ? ")\n/\n" : ",\n");
                                break;
            default :   printf("%s%s%*s%d%c%s",
                                TAB, fp->name,
                                21-strlen(fp->name), "char(",
                                fp->len, ')',
                                i == 1 ? ")\n/\n" : ",\n");
        }
}



/*****************************************************
 * FUNCTION WRITE_CTL()                              *
 * -generate 'control-file' for ODL data load util.  *
 *****************************************************/
void write_ctl(void)
{
    struct field    *fp;
    int             i;


    if (fcheck(ctl_name)) return;

    if (!(freopen(ctl_name, "wt", stdout))) end_prog(WRIT);

    printf("define record %s as", REC_NAME);
    for (i = f_num, fp = f; i; i--, fp++)
        printf("\n%s%s%*s%d%s%c",
                TAB, fp->name,
                22-strlen(fp->name), "(char(", fp->len, "))",
                i == 1 ? ';' : ',');

    printf("\ndefine source file"
           "\n%sfrom"
           "%s    %s\n%slength%s  %u\n%scontaining      %s;",
            TAB, TAB, txt_name, TAB, TAB, db_length, TAB, REC_NAME);

    printf("\nfor each record"
           "\ninsert into %s (\n",
            tab_name);

    for (i = f_num, fp = f; i; i--, fp++)
        printf("%s%s%s%s%c\n",
                TAB, TAB, TAB, fp->name, i == 1 ? ')' : ',');

    printf("%svalues (\n", TAB);
    for (i = f_num, fp = f; i; i--, fp++)
        printf("%s%s%s%s%c\n",
                TAB, TAB, TAB, fp->name, i == 1 ? ')' : ',');

    printf("next record\n");
}



/*****************************************************
 * FUNCTION WRITE_CVT()                              *
 * -generate file to convert dates to Oracle format. *
 *****************************************************/
void write_cvt(void)
{
    struct field    *fp;
    int             i;


    if (fcheck(cvt_name)) return;

    if (!(freopen(cvt_name, "wt", stdout))) end_prog(WRIT);

    printf("create table %s (\n", TMP_NAME);
    for (i = f_num, fp = f; i; i--, fp++)
        switch (fp->type) {
            case 'R':   printf("%s%s%*s%d,%d%c%s",
                                TAB, fp->name,
                                23-strlen(fp->name), "number(",
                                fp->len, fp->dec, ')',
                                i == 1 ? ");\n\n" : ",\n");
                                break;
            case 'N':   printf("%s%s%*s%d%c%s",
                                TAB, fp->name,
                                23-strlen(fp->name), "number(",
                                fp->len, ')',
                                i == 1 ? ");\n\n" : ",\n");
                                break;
            case 'D':   printf("%s%s%*s%s",
                                TAB, fp->name,
                                20-strlen(fp->name), "date",
                                i == 1 ? ");\n\n" : ",\n");
                                break;
            default :   printf("%s%s%*s%d%c%s",
                                TAB, fp->name,
                                21-strlen(fp->name), "char(",
                                fp->len, ')',
                                i == 1 ? ");\n\n" : ",\n");
        }

/* table now created, so insert the values into it */

    printf("insert into %s (\n", TMP_NAME);
    for (i = f_num, fp = f; i; i--, fp++)
        printf("%s%s%s%s%c\n",
                TAB, TAB, TAB, fp->name, i == 1 ? ')' : ',');

    printf("select \n");

    for (i = f_num, fp = f; i; i--, fp++)
        switch(fp->type) {
            case 'D':   printf("%s%s%sTo_Date(%s,'YYYYMMDD')%c\n",
                                 TAB, TAB, TAB,
                                 fp->name, i == 1 ? '' : ',');
                                 break;
            default :   printf("%s%s%s%s%c\n",
                                 TAB, TAB, TAB,
                                 fp->name, i == 1 ? '' : ',');
        }

    printf("from %s;\n\n", tab_name);


/* values now inserted into temporary table, so kill original and rename */

    printf("drop table %s;\n\n", tab_name);
    printf("rename %s to %s;\n\n", TMP_NAME, tab_name);
}



/*****************************************************
 * FUNCTION FCHECK()                                 *
 * -warn if file exists; allow him to skip this one. *
 *****************************************************/
char fcheck(char *filespec)
{
    char    u_abort = 0;


    if (0 == access(filespec, 0)) {
        fprintf(stderr, "File %s exists.  Over-write (Y/N) ? ", filespec);
        u_abort = user_abort();
    }

    fprintf(stderr, "\r%79c\r%s file %s\n", 0x20,
                           u_abort ? "Skipping" : "Creating", filespec);
    return(u_abort);
}



/*****************************************************
 * FUNCTION USER_ABORT()                             *
 * -flush keybuffer, check for abort, get y or n.    *
 * -set errorlevel 255 if Ctrl-Break or Escape hit.  *
 * -provide some trivial line-clearing cosmetics.    *
 * -MS/DOS SPECIFIC CODE - change for other systems. *
 *****************************************************/
char user_abort(void)
{
    char    ch;


    do {

        _AX = 0x0C07; __int__(0x21); ch = tolower(_AL);

        if (ch == 0x03 || ch == 0x1b) {
            fprintf(stderr, "\r%79c\rUser-Abort\n", 0x20);
            exit(0xff);
        } 

    } while (ch != 'y' && ch != 'n');

    fprintf(stderr, "\r%79c\r", 0x20);
    return(ch == 'n');
}



/*****************************************************
 * FUNCTION END_PROG()                               *
 * -exit with error message, set DOS errorlevel.     *
 *****************************************************/
void end_prog(char  code)
{
    char *errmsg[] = {
        "DBase -> Oracle data utility \xfe R.Trevithick " __DATE__ "\n\n"
        "    DBO filename[.dbf] [/nName] [/r]\n\n\t"
                "/n  specify new primary Name\n\t"
                "/r  display Reserved symbols",

        "Error reading input file",
        "Invalid dBase file header",
        "Error writing output file",
        "30 chars max table name"
    };


    fprintf(stderr, "\n%s\n", errmsg[code]);
    exit(code);
}

