//--------------------------------------------------------------------------
// CIPHER library for FoxPro 2.0 API
//
// This is a library of encryption and decryption routines written by
// Tom Rettig and Leonard Zerman, modified for the FoxPro 2.0 API by
// Walt Kennamer.
//
//--------------------------------------------------------------------------
#include <string.h>
#include <stdlib.h>
#include "pro_ext.h"
//--------------------------------------------------------------------------
extern long _tr_pnum();
extern char *_tr_crypt();
//--------------------------------------------------------------------------
#define   ERRORNEGL    -1L
#define   PW_MIN_LEN   3         /* number of chars in password */
#define   PW_MIN_NUM   10000000  /* number of digits in password return number */
#define   MAXSTRING   1024       /*    maximum string length   */
#define   WORKAREA    -1         /*    stands for current workarea  */
//--------------------------------------------------------------------------
/*********
*
*  _TR_PNUM.C
*
*  by Tom Rettig
*  modified by Leonard Zerman
*
* Placed in the public domain by Tom Rettig Associates, 10/22/1990.
*
*   long _tr_pnum( *char )
*
*   called by _tr_crypt for encrypt() and decrypt(), and by password()
*
*********/


long _tr_pnum( s )
char FAR *s;
{
   long ret = 1L;
   int  i;

   if ( s[0] && s[1] && s[2] )        /* 3 char minimum password len */
   {
      /* sum the ascii values of each char */
      for ( i = 0; s[i]; i++ )
         ret += (long) (s[i] + i);

      /* bit shift ret 1 position to the left until ret exceeds PW_MIN_NUM */
      for (; ret < (PW_MIN_NUM); ret <<= 1 )
           ;
      return( ret );
   }
   else
      return( ERRORNEGL );     /* -1 for error */
}

/*********
*
*  _TR_CRPT.C
*
*  by Leonard Zerman and Tom Rettig    
*
* Placed in the public domain by Tom Rettig Associates, 10/22/1990.
*
*  Syntax: _tr_crypt( <*string>, <*password>, <*ret> )
*
*  Return: <expC> of <string> encoded/decode according to <password>.
*          Return string will be same length as <string>.
*          Unchanged <string> if <password> is less than 3 characters.
*
*  Note..: All parameters are <*expC>.
*
*          The following encryption algorithm handles the full 
*          international ASCII character set. Passnum is an 
*          integer between 1 and 254 that receives a value
*          returned from _tr_pnum based on the password.
*          Passnum is a seed for the encryption-key variable
*          that is modified in the for-loop by the position
*          of the current character in reference to the total 
*          string (i - len). This produces a non-repeating 
*          pattern even if all the characters are the same, 
*          thus hiding the length of the password key.
*          The password characters are always being rotated
*          in a circular manner (que). 
*          ret[i] gets the bitwise XOR of the character with 
*          the XOR of passnum and the current character in the 
*          password que. This gives 24 bits of encryption for
*          each character. 
*
*          This encryption/decryption program can be broken 
*          by any computer-literate person with access to this
*          source code or a knowledge of cryptography.
*********/
char FAR *_tr_crypt( char FAR *instr, char FAR *pwstr, long len )
{ 
   int j, pwlen, passnum;
   long i;
   char buff[1];

   pwlen  = strlen( pwstr );                   /* length of password */

   /* get seed value */
   passnum = (int) ((((_tr_pnum(pwstr)/997) - 1) % 254 ) + 1);   

   for ( i = j = 0; i < len; i++ )               /* process whole string */
   {
       passnum = (int) (((passnum + ( i - len )) - 1 ) % 254) + 1; 
       buff[0] = (instr[i] ^ (passnum ^ pwstr[j]) );      /* XOR 3 var's */
       instr[i] = (buff[0] ? buff[0] : instr[i]);   /* if NULL return char */              
       j = ( j >= pwlen ? 0 : j++ );    /* password que control variable */
   }
   
   return(instr);                      /* send back encrypted string */
}

/*********
*
* ENCRYPT.C
*
* by Leonard Zerman, Tom Rettig    
*
* Placed in the public domain by Tom Rettig Associates, 10/22/1990.
*
*  Syntax: ENCRYPT( <string>, <password> )
*  Return: <expC> of <string> encoded according to <password>.
*          Return string will be same length as <string>.
*          Unchanged <string> if <password> is less than 3 characters.
*  Note..: Both parameters are <expC>.
*********/

void k_encrypt(char FAR *pwstr, char FAR *instr, long len )
{
   int pwlen;

   pwlen  = strlen(pwstr);                  /* length of password */
                                            /* allocate memory for return */
   if ( pwlen < PW_MIN_LEN )                /* or pwlen < 3 characters */
      return;

   // return encrypted string
   _tr_crypt( instr, pwstr, len );

   return;
}

//--------------------------------------------------------------------------
void far encrypt(ParamBlk FAR *parm)
//--------------------------------------------------------------------------
//
// Shell around Rettig's encrypt function to adapt it to the FoxPro
// API.  This is a reversable algorithm.  To decrypt a string, just
// run it through this function again.  Thus;
//     encrypt(encrypt(plaintext,password),password) = plaintext
//
// Usage:   encrypt(plaintext,password)
//
//        Plaintext can be a character string or a memo field (memo fields
//        must be passed by reference -- encrypt(@memo,password) ).  
//        If a memo field is passed to this function, it will not 
//        be modified, even though it is nominally passed by reference.
//        A character string passed by reference will be modified.  A
//        variable (other than a character string or memo field) that 
//        is passed by reference will be unchanged (and the function return
//        value will be the same value as the variable).
//
// Returns: encrypted string of same length as plaintext
//
// Author:  Walter J. Kennamer
// History: Initially written August 19, 1991
//--------------------------------------------------------------------------
{
   Value FAR *v, FAR *textval, FAR *memvarvalue, FAR *retmemo;
   Value valbuffer;
   Locator FAR *varlocate;

   char password[MAXSTRING];
   int errcode;       // error code from _Load operation

   // Various types for managing memo fields
   int memchan;       // memo file channel
   long memsize, memread, memseek, memfind;
   MHANDLE memohand = 0;

   // Grab the first parameter as a value structure.
   textval  = &parm->p[0].val;

   // Get the password, which must be a character and must be passed
   // by value.  Make sure it doesn't exceed MAXSTRING, which would 
   // cause memory corruption.
   v = &parm->p[1].val;
   _MemMove(password,_HandToPtr(v->ev_handle),min(v->ev_length,MAXSTRING));
   password[min(v->ev_length,MAXSTRING)] = '\0';

   // See what kind of data we received.
   if (textval->ev_type == 'C') {
      // OK, this is a perfectly ordinary character string
      // or character variable passed by value.  Encrypt it and return
      // the results.

      k_encrypt(password,_HandToPtr(textval->ev_handle),textval->ev_length);
      _RetVal(textval);
      }
   else if (textval->ev_type == 'R') {    
      // This is a variable or memo field passed by reference.
      // Grab the locator structure for it.
      varlocate  = &parm->p[0].loc;

      // See if it is a memo file.  If it is, it will have a memo
      // channel greater than zero.  Our strategy is to read the memo
      // into RAM, encrypt it as a character string and return the 
      // results.  Even though the memo field is nominally passed by
      // reference, do not change the memo directly.

      if ( varlocate->l_where > 0 ) {
         memchan = _MemoChan(varlocate->l_where);

         if ((memfind = _FindMemo(varlocate)) < 0)  // find the memo file offset 
            _Error(memfind);

         memsize = _MemoSize(varlocate);        // find the memo size
         memseek = _FSeek(memchan,memfind,0);    // move the file pointer

         // Allocate enough memory to read in the memo field 
         memohand  = _AllocHand(memsize);
         if (memohand == 0) {
            _Error(182);    // insufficient memory
            }

         // Lock the memo handle during the _FRead() callback
         _HLock(memohand);

         // Read the memo into our handle
         memread = _FRead(memchan,_HandToPtr(memohand),memsize);
         if (memread != memsize)
            _Error(182);   // insufficient memory

         // Encrypt or decrypt the memo contents.  
         k_encrypt(password,_HandToPtr(memohand),memsize);

         // Assemble a value structure to pass back to FoxPro.  We need
         // to use a value structure since the memo can contain embedded
         // CHR(0) characters, so we can't just use _RetChar (which is 
         // looking for a null-terminated string and would think the first
         // null it saw was the end of the string).
         retmemo = &valbuffer;
         retmemo->ev_type = 'C';
         retmemo->ev_length = memsize;
         retmemo->ev_handle = memohand;

         _RetVal(retmemo);  // Also frees memohand
         }
      else {   
         // This is some other kind of memory variable passed by reference.

         // Store it into a value structure so that we can learn its secrets.
         memvarvalue = &valbuffer;
         if ( (errcode = _Load(varlocate,memvarvalue)) < 0)
            _Error(-errcode);

         // Lock the handle, though it isn't technically 
         // required to be locked here.
         _HLock(memvarvalue->ev_handle);

         if (memvarvalue->ev_type == 'C') {
            // This is a character string.  Go ahead and encrypt it.
            k_encrypt(password,_HandToPtr(memvarvalue->ev_handle),
                memvarvalue->ev_length);
            if ((errcode = _Store(varlocate,memvarvalue)) < 0)
               _Error(-errcode);
            _RetVal(memvarvalue);   // also frees memvarvalue->ev_handle
            }
         else {
            _Error(302);   // Data type mismatch
            }
         return;
         }
      }
   else {            
      // Numeric, date, logical passed by value
      _Error(302);   // Data type mismatch
   }

   return;
}

// Note how both functions that are visible to FoxPro (ENCRYPT and DECRYPT)
// call the same function inside the API routine.  This works because 
// the API routine is reversible.
FoxInfo myFoxInfo[] = 
   {
   {"ENCRYPT",encrypt,2,"?,C"},
   {"DECRYPT",encrypt,2,"?,C"}
   };

FoxTable _FoxTable = 
   {
   (FoxTable FAR *)0, sizeof(myFoxInfo)/ sizeof(FoxInfo), myFoxInfo
   };
