// Turbo C++ XMS Interface Library
// by Richard Vuduc
// Copyright 1990, Richard Vuduc

// MODULE: XMSHRW.CPP
// The module contains the source for the individual read/writes.
// You may modify this to accomodate your own data types
// See the manual for additional details on making your own
// Get/Store functions.

#include "xms.h"
extern XMSDriver *xmsptr;

// *********Basic Steps
// In general, your get/store routines should do the following
//
//    1) Confirm the state of the cache.  You can do this with
//       a simple GetCache call.  It returns True if the cache
//       is in.
//    2) If the cache is installed, call cache update.  Cache
//       Update will update the cache if the current Index
//       is not within the cache.  If you are doing a Store
//       function, be sure that you do an additional 'CBounds'
//       call to determine whether or not the ENTIRE data
//       item can be found within the cache.  If not, a full
//       cache update needs to be performed.
//    3) After the CacheUpdate, you can be guaranteed that
//       the data item is in the cache (unless the cache is
//       not large enough to hold all of it).  Using a type
//       cast (see the functions below), you can change
//       the value as necessary.
//    4) If you've determined that there is no data cache,
//       you must perform a DIRECT read/write to extended
//       memory.  If you can guarantee that the cache will
//       allows be ON and AVAILABLE, you can omit this
//       special case handling code.
//    5) Lastly, call IncPtr( sizeof( DataItemType ) );
//       where 'DataItemType' is the type of the data.

// *****************Custom single data item Gets************************

// Get a 'long' var from extended memory at the current Index pointer
void XMSHandle::Get( long& a )
{
	// If the cache is installed...
	if( GetCache( ) )
	{
		CacheUpdate( sizeof( long ) );         // Update the cache
		a = *( ((long *)Data) + CIndex );      // Get data from the cache
	}
	else                                       // No cache...Get directly
		xmsptr->EMBMoveToCon( &a, Handle, GetPtr( ), sizeof( long ) );
	IncPtr( sizeof( long ) );                  // Increment pointer
}

// Get the next integer item
// see also: Get( long )
void XMSHandle::Get( int& a )
{
	if( GetCache( ) )                      // Is the cache in?
	{
		CacheUpdate( sizeof( int ) );      // Yes, update the cache
		a = *( (int *)(Data)+CIndex );     // Get from data cache
	}
	else                                   // Nope, direct read
		xmsptr->EMBMoveToCon( &a, Handle, GetPtr( ), sizeof( int ) );
	IncPtr( sizeof( int ) );               // Increment the pointers
}

// Get the next character item
void XMSHandle::Get( char& a )
{
	char t[2];           // Tvar used if the cache is off...(direct reads)
	if( GetCache( ) )    // Is the cache on?
	{
		CacheUpdate( );         // Update the cache
		a = *((char *)(Data)+CIndex);   // Get from the cache buffer
	}
	else                        // No cache in, do a direct read
	{
		// Because the XMSDriver and HIMEM expect an even length,
		// two bytes must be retrieved.
		xmsptr->EMBMoveToCon( t, Handle, GetPtr( ), 2 );
		a = *t;
	}
	IncPtr( sizeof( char ) );   // Increment the pointer
}

// ******************** Custom Store Functions **************************

// Store a long var into extended memory
void XMSHandle::Store( long& a )
{
	if( GetCache( ) )			// The cache must be installed
	{
		CacheUpdate( sizeof( long ) );          // Update the cache
		*((long *)(Data)+CIndex) = (long) a;	// Retrieve from cache
	}
	else				// No cache installed, do a direct write
		xmsptr->EMBMoveToExt( Handle, GetPtr( ), &a, sizeof( long ) );
	IncPtr( sizeof( long ) );
}

void XMSHandle::Store( int& a )
{
	if( GetCache( ) )               // Cache in?
	{
		// The cache is in so update the cache
		CacheUpdate( sizeof( int ) );
		*((int *)(Data) + CIndex) = (int) a;
	}
	else                            // Write to extended memory
		xmsptr->EMBMoveToExt( Handle, GetPtr( ), &a, sizeof( int ) );
	IncPtr( sizeof( int ) );        // Increment the pointer
}

// Store a character variable
void XMSHandle::Store( char& a )
{
	char t[2];	// Remember that direct rd/write lengths must be even
	*t = a;
	if( GetCache( ) )
	{
		CacheUpdate( );
		*((char *)(Data) + CIndex) = (char) a;
	}
	else
		xmsptr->EMBMoveToExt( Handle, GetPtr( ), &t, 2 );
	IncPtr( sizeof( char ) );
}
// END XMSHRW.CPP
