#include <math.h>
#include <memory.h>
#include <stdarg.h>
#include "windows.h"

/****************************************************************************/
/*Title:    SPRITES.C                                                       */
/*Author:   Thomas W. Olsen                                                 */
/*Version:  1.0                                                             */
/*Compiler: Microsoft C/C++ 7.0                                             */
/*  rc /r sprites.rc                                                        */
/*  cl /c /AM /Gsw /W3 /Oas /Zpe /Zi /DSTRICT /DWINVER=0x0300 sprites.c     */
/*  link /CO /NOD sprites,,, libw mlibcew, sprites.def                      */
/*  rc sprites.res sprites.exe                                              */
/****************************************************************************/
/*                            Constants                                     */
/****************************************************************************/

#define TIMER_ID            1
#define TIMER_RESOLUTION    50

/************************************************************************/
/*                          Structure Definitions                       */
/************************************************************************/

typedef struct tagCEL
{
    struct tagCEL *lpPrev;
    struct tagCEL *lpNext;
    BOOL        visible;
    POINT       dimension;          /* Dimensions of Cel */
    POINT       position;           /* Position of Cel in Window */
    HBITMAP     hMaskBitmap;        /* Cel AND Mask Bitmap */
    HBITMAP     hCelBitmap;         /* Cel Bitmap */
} CEL;

typedef CEL *LPCEL;

typedef struct tagSPRITE
{
    WORD        numCels;
    LPCEL       lpCel;
    DWORD       delayTicks;
    DWORD       elapsedTicks;
} SPRITE;

typedef SPRITE *LPSPRITE;

/************************************************************************/
/*                          Structure Definitions                       */
/************************************************************************/

SPRITE  helicopter;
SPRITE  bombs;
SPRITE  explosion;
HBITMAP hBckgrdBitmap = (HBITMAP) NULL;

/************************************************************************/
/*                             Prototypes                               */
/************************************************************************/

LONG FAR PASCAL WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
BOOL CreateSprite( HANDLE hInst, HWND hWnd, LPSPRITE lpSprite, int x, int y, DWORD delay, ... );
void DestroySprite( LPSPRITE lpSprite );
BOOL ShowSprite( HDC hDC, LPSPRITE lpSprite );
BOOL HideSprite( HDC hDC, LPSPRITE lpSprite );

/************************************************************************/
/*                           Program Begins                             */
/************************************************************************/

int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int numCmdShow )
{
    MSG       msg;
    WNDCLASS  wc;
    HWND      hWnd;
    HDC       hDC, hMemDC;
    RECT      rect;

    wc.style         = (UINT) NULL;
    wc.lpfnWndProc   = WindowProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInst;
    wc.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName  = (LPSTR) NULL;
    wc.lpszClassName = "SpriteWndClass";

    if (!RegisterClass(&wc))
        return(FALSE);

    hWnd   = GetDesktopWindow();
    GetClientRect( hWnd, &rect );

    hWnd = CreateWindow( "SpriteWndClass", "Sprites",
                          WS_BORDER | WS_POPUP,
                          0, 0, rect.right, rect.bottom,
                          NULL, NULL, hInst, NULL );

    if (!hWnd)
        return (FALSE);

    /********************************************************************/
    /*                          Create Sprites                          */
    /********************************************************************/

    if (CreateSprite( hInst, hWnd, &helicopter, rect.right, 0, 0, 
                      "HeloCel1", "HeloMask1",
                      "HeloCel2", "HeloMask2", NULL ) == FALSE ||
        CreateSprite( hInst, hWnd, &bombs, 200, 0, 0,
                      "BombCel", "BombMask", NULL ) == FALSE   ||
        CreateSprite( hInst, hWnd, &explosion, 0, 0, 150,
                      "ExpCel1", "ExpMask1",
                      "ExpCel2", "ExpMask2",
                      "ExpCel3", "ExpMask3",
                      "ExpCel4", "ExpMask4",
                      "ExpCel5", "ExpMask5", NULL ) == FALSE )
    {
        return(FALSE);
    }

    /********************************************************************/
    /*                     Capture & Display Screen                     */
    /********************************************************************/

    hDC        = GetDC( hWnd );
    hMemDC     = CreateCompatibleDC( hDC );
    hBckgrdBitmap = CreateCompatibleBitmap( hDC, rect.right, rect.bottom );
    SelectObject( hMemDC, hBckgrdBitmap );
    BitBlt( hMemDC, 0, 0, rect.right, rect.bottom, hDC, 0, 0, SRCCOPY );
    DeleteDC( hMemDC );
    ReleaseDC( hWnd, hDC );

    ShowWindow( hWnd, numCmdShow );

    /********************************************************************/
    /*                       Setup Timer Routine                        */
    /********************************************************************/

    SetTimer(hWnd, TIMER_ID, TIMER_RESOLUTION, (TIMERPROC) NULL);

    while (GetMessage(&msg, NULL, NULL, NULL))  /* Typical Message Loop */
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
    return (msg.wParam);
}


/************************************************************************/
/*                          Window Procedure                            */
/************************************************************************/

LONG FAR PASCAL WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    PAINTSTRUCT paint;
    HDC         hDC, hMemDC;
    HBITMAP     hOldBitmap;
    static POINT dropPoint, impactPoint;
    static BOOL droppedBombs = FALSE;
    static RECT rect;

    switch (message)
    {
        case WM_DESTROY:
            DestroySprite( &helicopter );
            DestroySprite( &bombs );
            DestroySprite( &explosion );
            DeleteObject( hBckgrdBitmap );
            KillTimer( hWnd, TIMER_ID );
            PostQuitMessage( NULL );
            break;

        case WM_TIMER:
            hDC = GetDC( hWnd );

            if (helicopter.lpCel->lpPrev->position.x > dropPoint.x)
            {
                helicopter.lpCel->position.x = helicopter.lpCel->lpPrev->position.x - 10;
                helicopter.lpCel->position.y = helicopter.lpCel->lpPrev->position.y + 3;
            }
            else
            if (droppedBombs == FALSE)
            {
                droppedBombs = TRUE;
                bombs.lpCel->position.x = helicopter.lpCel->lpPrev->position.x +
                                          (helicopter.lpCel->lpPrev->dimension.x / 3);
                bombs.lpCel->position.y = helicopter.lpCel->lpPrev->position.y +
                                          helicopter.lpCel->lpPrev->dimension.y;
                impactPoint.x = bombs.lpCel->position.x;
                impactPoint.y = (rect.bottom * 2) / 3;
                ShowSprite( hDC, &bombs );
            }
            else
            if (bombs.lpCel->lpPrev->position.y < impactPoint.y )
            {
                bombs.lpCel->position.y = bombs.lpCel->lpPrev->position.y + 3;
                ShowSprite( hDC, &bombs );
            }
            else
            {
                explosion.lpCel->position = bombs.lpCel->lpPrev->position;
                HideSprite( hDC, &bombs );
                ShowSprite( hDC, &explosion);
            }
            ShowSprite( hDC, &helicopter );

            ReleaseDC( hWnd, hDC );
            break;

        case WM_PAINT:
            GetClientRect( hWnd, &rect );
            dropPoint.x = rect.right  / 3;
            dropPoint.y = rect.bottom / 3;
            hDC    = BeginPaint( hWnd, &paint );
            hMemDC = CreateCompatibleDC( hDC );
            hOldBitmap = SelectObject( hMemDC, hBckgrdBitmap );
            BitBlt( hDC, 0, 0, rect.right, rect.bottom, hMemDC, 0, 0, SRCCOPY);
            SelectObject( hMemDC, hOldBitmap);
            DeleteDC( hMemDC );
            EndPaint( hWnd, &paint );
            break;

        case WM_LBUTTONDOWN:
        case WM_RBUTTONDOWN:
        case WM_MBUTTONDOWN:
        case WM_KEYDOWN:
        case WM_SYSKEYDOWN:
            SendMessage( hWnd, WM_CLOSE, (WPARAM) NULL, (LPARAM) NULL);

        default:
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
}


/************************************************************************/
/*                             Create Sprite                            */
/************************************************************************/

BOOL CreateSprite( HANDLE hInst, HWND hWnd, LPSPRITE lpSprite, int x, int y, DWORD delay, ... )
{
    char    *lpszCelName, *lpszMaskName;
    LPCEL   lpPrev;
    LPCEL   lpCel;
    BOOL    retVal = FALSE, firstTime = TRUE;
    BITMAP bmInfo;
    va_list marker;
    HDC     hDC;

    memset( lpSprite, 0, sizeof(SPRITE));   /* Initialize Sprite Structure */

    hDC = GetDC( hWnd );
    va_start( marker, delay );      /* Setup stack frame for variable args */

    while ((lpszCelName = va_arg( marker, char *)) &&   /* Read Mask Names */
           (lpszMaskName = va_arg( marker, char *)))
    {
        if ((lpCel = (LPCEL) LocalAlloc(LPTR, sizeof(CEL)))) /* Allocate CEL Memory */
        {                                                    
            if ((lpCel->hCelBitmap  = LoadBitmap( hInst, lpszCelName)) &&   /* Load Mask Bitmaps */
                (lpCel->hMaskBitmap = LoadBitmap( hInst, lpszMaskName)))
            {                                               /* Get Mask Bitmap Dimensions */
                GetObject( lpCel->hCelBitmap, sizeof( BITMAP ), (LPSTR) &bmInfo );

                if (firstTime)
                {
                    firstTime = FALSE;
                    lpSprite->lpCel = lpPrev = lpCel;
                    lpSprite->delayTicks = delay;       /* Establish Sprite Delay Time */
                }
                else
                    lpPrev->lpNext = lpCel;

                lpCel->lpPrev      = lpPrev;
                lpCel->lpNext      = lpSprite->lpCel;
                lpCel->dimension.x = bmInfo.bmWidth;    /* Cel Width */
                lpCel->dimension.y = bmInfo.bmHeight;   /* Cel Height */
                lpCel->position.x  = x;                 /* Starting Row Position */
                lpCel->position.y  = y;                 /* Starting Column Position */
                lpPrev = lpCel;                         /* Save Ptr to Current Cel */
                lpSprite->numCels++;                    /* Increment Sprite Cel Count */
                retVal = TRUE;
                continue;
            }
            DestroySprite( lpSprite );                  /* Couldn't Load Bitmaps */
        }
        retVal = FALSE;
        break;
    }

    va_end( marker );                                   /* Reset variable args */
    ReleaseDC( hWnd, hDC );
    return( retVal );
}

/************************************************************************/
/*                            Destroy Sprite                            */
/************************************************************************/

void DestroySprite( LPSPRITE lpSprite )
{
    LPCEL lpCel, lpNext;

    if ((lpCel = lpSprite->lpCel))          /* Is a Cel Allocated? */
    {
        do                                  
        {
            if ( lpCel->hCelBitmap )        /* Free Allocated Bitmap */
                DeleteObject(lpCel->hCelBitmap);

            if ( lpCel->hMaskBitmap )       /* Free Allocated Bitmap  */
                DeleteObject(lpCel->hMaskBitmap);

            lpNext = lpCel->lpNext;         /* Save Next Cel */

            LocalFree( (HLOCAL) lpCel );    /* Free Current Cel */
        }                                   /* Loop to Next Cel */
        while ((lpCel = lpNext) && lpCel != lpSprite->lpCel);
    }
    memset( lpSprite, 0, sizeof(SPRITE) );  /* Reinitialize Sprite Structure */
    return;
}

/************************************************************************/
/*                             Show Sprite                              */
/************************************************************************/

BOOL ShowSprite( HDC hDC, LPSPRITE lpSprite )
{
    HDC     hMemDC, hTempDC;
    HBITMAP hBitmap;
    LPCEL   lpCel, lpPrev;
    DWORD   currentTicks;

    currentTicks = GetTickCount();              /* Read StopWatch (mSecs) */

    if ((currentTicks - lpSprite->elapsedTicks) < lpSprite->delayTicks)
        return(FALSE);                          /* Has Sufficient Delay Passed? */

    lpCel  = lpSprite->lpCel;                   
    lpPrev = lpCel->lpPrev;
    lpSprite->lpCel = lpCel->lpNext;            /* Point Sprite at Next Cel */
    lpSprite->elapsedTicks = currentTicks;      /* Reset Elapsed Time Counter */

    hMemDC  = CreateCompatibleDC( hDC );
    hTempDC = CreateCompatibleDC( hDC );
    hBitmap = CreateCompatibleBitmap( hDC, lpCel->dimension.x, lpCel->dimension.y );

    SelectObject( hTempDC, hBckgrdBitmap );     /* hTempDC references the background */
    SelectObject( hMemDC, hBitmap );    /* Begin Drawing New Cel on work hBitmap */

    BitBlt( hMemDC, 0, 0, lpCel->dimension.x, lpCel->dimension.y,   /* Copy background into */
            hTempDC, lpCel->position.x, lpCel->position.y, SRCCOPY );  /* work hBitmap */

    SelectObject( hTempDC, lpCel->hMaskBitmap );

    BitBlt( hMemDC, 0, 0, lpCel->dimension.x, lpCel->dimension.y,
            hTempDC, 0, 0, SRCAND );    /* AND the Cel with work bitmap */

    SelectObject( hTempDC, lpCel->hCelBitmap );

    BitBlt( hMemDC, 0, 0, lpCel->dimension.x, lpCel->dimension.y,
            hTempDC, 0, 0, SRCPAINT );  /* OR the Mask with work bitmap */

    BitBlt( hDC, lpCel->position.x, lpCel->position.y,  /* Copy resulting work bitmap  */
            lpCel->dimension.x, lpCel->dimension.y,     /* Directly to Display */
            hMemDC, 0, 0, SRCCOPY );

    lpCel->visible = TRUE;                              /* Flag Cel as Visible */
    DeleteObject( hMemDC );                             /* Cleanup */
    DeleteObject( hTempDC );                            /*   "     */
    DeleteObject( hBitmap );                            /*   "     */
    return(TRUE);
}

/************************************************************************/
/*                            Hide Sprite                               */
/************************************************************************/

BOOL HideSprite( HDC hDC, LPSPRITE lpSprite )
{
    HDC     hTempDC;
    LPCEL   lpCel, lpPrev;
    BOOL    retVal = FALSE;

    lpCel  = lpSprite->lpCel;
    lpPrev = lpCel->lpPrev;

    if (lpPrev->visible == TRUE)                /* Only Hide if Visible */
    {
        hTempDC = CreateCompatibleDC( hDC );
        SelectObject( hTempDC, hBckgrdBitmap );

        BitBlt( hDC, lpPrev->position.x, lpPrev->position.y,
                lpPrev->dimension.x, lpPrev->dimension.y,
                hTempDC, lpPrev->position.x, lpPrev->position.y, SRCCOPY );
        lpPrev->visible = FALSE;
        DeleteObject( hTempDC );
        retVal = TRUE;
    }
    return(retVal);
}
