/*****************************************************/
/* insdll.c                                          */
/* -- DLL that maintains a separate DGROUP for each  */
/*    client task.                                   */
/*****************************************************/

#include <windows.h>
#include <windowsx.h>
#include <memory.h>
#include "insdll.h"

typedef struct
    {
    HTASK   htsk;   /* Registered client task. */
    UINT    wSel;   /* DGROUP selector for task. */
    } CDS;          /* Client Data Structure. */
typedef CDS FAR *   LPCDS;

BOOL             FEnterDll(void);
LPCDS            LpcdsFindCurTask(void);
int CALLBACK     LibMain(HINSTANCE, WORD, WORD, LPSTR);
int CALLBACK     WEP(int);

LPCDS   lrgcds;     /* Array of registered clients. */
int     ccdsAlloc;  /* Size of client array. */
int     ccdsClient; /* Count of registered clients. */
DWORD   tim;        /* Time at registration. */

int CALLBACK
LibMain(HINSTANCE hinsThis, WORD ds, WORD cbHeap,
  LPSTR lsz)
/*****************************************************/
/* -- Entry point during initialization.             */
/*****************************************************/
    {
    /* Start with space for one entry. */
    if (!(lrgcds = (LPCDS)GlobalAllocPtr(
      GMEM_MOVEABLE | GMEM_SHARE, sizeof *lrgcds)))
        return FALSE;

    ccdsAlloc = 1;
    return TRUE;
    }

int CALLBACK
WEP(int nExitCode)
/*****************************************************/
/* -- Exit proc.                                     */
/*****************************************************/
    {
    if (lrgcds)
        GlobalFreePtr(lrgcds);

    return 1;
    }

BOOL _export
FRegisterTask(void)
/*****************************************************/
/* -- This export is to called by each task that     */
/*    requires services provided by this DLL, before */
/*    the services can be provided.                  */
/* -- Return TRUE for success, FALSE for failures    */
/*    out of memory, or task already registered).    */
/*****************************************************/
    {
    LPVOID  lpv     = NULL; /* Task's DGROUP copy. */
    UINT    wSelThis;       /* DLL's DS. */
    UINT    wSelClient;     /* Client's copy of DS. */
    HGLOBAL hgblThis;       /* DLL's DGROUP. */
    DWORD   cb;             /* Size of DGROUP. */
    LPCDS   lpcds;          /* Client's data struct. */

    if (LpcdsFindCurTask())
        return FALSE;   /* Already registered. */

    /* Clone our DGROUP and add the current client */
    /* task to the array. */
    _asm    mov wSelThis, ds;
    hgblThis = (HGLOBAL)LOWORD(GlobalHandle(wSelThis));
    cb = GlobalSize(hgblThis);
    if (!(lpv = GlobalAllocPtr(GMEM_MOVEABLE, cb)))
        return FALSE;   /* Out of memory. */

    /* First look for an empty spot in the array. */
    /* Note: Can't both be zero, so we at least know */
    /* the array has been allocated. */
    if (ccdsClient < ccdsAlloc)
        {
        LPCDS   lpcdsLim;

        for (lpcds = lrgcds,
            lpcdsLim = lpcds + ccdsAlloc;
          lpcds < lpcdsLim; lpcds++)
            if (!lpcds->htsk)
                break;
        }
    else
        {
        LPCDS   lrgcdsNew;

        /* No empty slots, so we we must grow the */
        /* array. */
        if (!(lrgcdsNew = (LPCDS)GlobalReAllocPtr(
          lrgcds, (ccdsAlloc + 1) * sizeof(CDS),
          GMEM_MOVEABLE)))
            {
            GlobalFreePtr(lpv);
            return FALSE;
            }
        lrgcds = lrgcdsNew;
        lpcds = lrgcds + ccdsAlloc++;
        }

    ccdsClient++;
    _fmemcpy(lpv, MAKELP(wSelThis, 0), (int)cb);
    lpcds->htsk = GetCurrentTask();
    lpcds->wSel = wSelClient = SELECTOROF(lpv);

    /* Instantiate some client data. */
    _asm    mov ds, wSelClient;
    tim = GetTickCount();
    return TRUE;
    }

BOOL _export
FUnregisterTask(void)
/*****************************************************/
/* -- Unregister the current task (free associated)  */
/*    memory.                                        */
/* -- A task is ineligible for this DLL's services   */
/*    after making this call.                        */
/*****************************************************/
    {
    LPCDS   lpcds;

    if (!(lpcds = LpcdsFindCurTask()))
        return FALSE;

    GlobalFree(
      (HGLOBAL)LOWORD(GlobalHandle(lpcds->wSel)));
    lpcds->htsk = NULL;
    lpcds->wSel = NULL;
    ccdsClient--;
    return TRUE;
    }

LPCDS
LpcdsFindCurTask(void)
/*****************************************************/
/* -- Return the client data structure of the        */
/*    current task.  Return null if not found.       */
/*****************************************************/
    {
    LPCDS   lpcds, lpcdsLim;
    HTASK   htsk    = GetCurrentTask();

    if (!ccdsClient)
        return NULL;

    for (lpcds = lrgcds, lpcdsLim = lpcds + ccdsAlloc;
      lpcds < lpcdsLim; lpcds++)
        if (lpcds->htsk == htsk)
            return lpcds;

    return NULL;
    }

BOOL
FEnterDll(void)
/*****************************************************/
/* -- Switch to the DGROUP copy of the current task. */
/* -- Return FALSE if task has not registered        */
/*    itself.                                        */
/*****************************************************/
    {
    LPCDS   lpcds;
    UINT    wSel;

    if (!(lpcds = LpcdsFindCurTask()))
        return FALSE;

    wSel = lpcds->wSel;
    _asm   mov ds, wSel;
    return TRUE;
    }

LPSTR _export
LszGetTaskData(void)
/*****************************************************/
/* -- Return instance data for current task.         */
/*****************************************************/
    {
    static  char    szMsg[80];

    if (!FEnterDll())
        return NULL;

    wsprintf(szMsg, "Task %x has been running %ld",
      GetCurrentTask(), (GetTickCount() - tim) / 1000);
    return szMsg;
    }
