/*
** clock.c
**
** Example program for
**      Clock events
**      Background processing
**      Virtual graphics canvas
**
** Revision history:
**
** Thu  10-15-1992
** Original version
*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "smtypes.h"
#include "smwindow.h"
#include "smevent.h"
#include "smgraph.h"
#include "smcolor.h"
#include "smmenu.h"
#include "smlabel.h"
#include "smtext.h"
#include "smbutton.h"


int Clock(WindowType *w);
int InstallClock(void);
int ClockDraw(WindowType *w);
int ClockBkgd(WindowType *w, EventType *ev);
int About(WindowType *w);
int HowItWorks(WindowType *w);
int CloseAbout(ButtonType *b);
int ResizeText(WindowType *w);

/*
** Borlandc requires extra stack
*/
#ifdef __TURBOC__
extern unsigned _stklen = 14336U;
#endif

BackgroundProcIDType    bgproc;

/****************************************************************************
**
** This is the structure we'll attach to the clock window. It holds the
** precalculated coordinates of the center of the window, the radius of
** the clock, and the current time
*/

typedef struct {
    int     cx, cy;
    int     radius;
    int     hour, min, sec;
} tstruct;



main(int argc, char **argv)
{
    EventType   *ev;
    int         msg;
    MenuType    *m;

    /*
    ** Initialize the window system, graphics, fonts, etc.
    */
    SM_Init(argc, argv);

    /*
    ** Demo menu is very simple
    */
    m = SM_CreateMenu();
    SM_AddMenuItem(m, ACTION_ITEM, "About...", About, NULL);
    SM_AddMenuItem(m, ACTION_ITEM, "Source Code", HowItWorks, NULL);
    SM_AddMenuItem(m, ACTION_ITEM, "Quit", SM_Exit, NULL);

    /*
    ** Open the application
    */
    SM_OpenApplication("TWS Demo: Clock", m);

    /*
    ** Set up the clock
    */
    InstallClock();

    /*
    ** Event loop...
    */
    while (True) {
        ev = SM_GetNextEvent();
        msg = SM_ProcessEvent(ev);
        if (msg == KEYPRESS) {
            SM_RemoveBackgroundProc(bgproc);
            SM_Exit(NULL);
        }
    }
}


double  cosines[60];                    /* Cos of angles to clock 'numbers' */
double  sines[60];                      /* Sin of angles to clock 'numbers' */
double  dtheta = 0.10471976;            /* 2*PI/60, delta angle around clock*/


/****************************************************************************
**
** InstallClock
**
** Create the clock window
*/
int InstallClock(void)
{
    RectType    r;                      /* Clock window rectangle           */
    WindowType  *w;                     /* Clock window                     */
    int         i;
    double      startang;
    tstruct     *ts;

    r.Xmin = r.Ymin = 125;
    r.Xmax = r.Ymax = 300;

    startang = -1.5707963;              /* The 12-o'clock position is -pi/2*/

    for (i = 0; i < 60; i++) {
        cosines[i] = cos(startang + i*dtheta);
        sines[i] = sin(startang + i*dtheta);
    }

    /*
    ** Create the clock window
    */
    w = SM_NewWindow(&r, "Clock", DOCUMENT, NULL, NULL);
    /*
    ** This is the procedure that draws the clock face
    */
    SM_SetDrawProc(w, ClockDraw);

    /*
    ** Force the draw canvas to automatically scale as the window is resized
    ** by setting it as a virtual rectangle
    */
    r.Xmin = r.Ymin = 1;
    r.Xmax = r.Ymax = -999;
    GR_CreateGraphState(w, &r, SMDEFAULT);
    GR_SetBackgroundColor(w, SM_GetSystemColor(SMLIGHTGRAY));

    /*
    ** Open the window, drawing the original clock face
    */
    SM_OpenWindow(w);

    /*
    ** Install the clock background procedure to update the hands every second
    */
    SM_InitBackgroundProcs();
    bgproc = SM_RegisterBackgroundProc(w, ClockBkgd, CLOCKTICK | NOTOBSCURED);

    ts = SM_GetUserData(w);

    return True;
}



/****************************************************************************
**
** ClockDraw
**
** Draw the clock face from scratch, including setting the clock time
*/
int ClockDraw(WindowType *w)
{
    int         i;
    int         radius;                 /* Clock radius                     */
    int         cx, cy;
    int         x, y;
    unsigned long ttime;
    static tstruct  ts;
    int         hour, min, sec, hr;

    GR_ClearCanvas(w);
    /*
    ** Get the window clock center and diameter. Use the minimum dimension so
    ** the clock stays within the window
    */
    radius = GR_GetCanvasWidth(w);
    if (GR_GetCanvasDepth(w) < radius) {
        radius = GR_GetCanvasDepth(w);
    }

    /*
    ** Radius is half the diameter, less a little for the 'dots'
    */
    radius = radius / 2 - 5;

    /*
    ** Store the center of the window canvas
    */
    cx = GR_GetCanvasWidth(w)/2;
    cy = GR_GetCanvasDepth(w)/2;

    /*
    ** Draw the 5-minute points around the clock
    */
    GR_SetDrawMode(w, SMREP);
    GR_SetDrawColor(w, SM_GetSystemColor(SMBLACK));
    for (i = 0; i < 60; i++) {
        x = radius * cosines[i];
        y = radius * sines[i];
        GR_DrawSolidCircle(w, cx+x, cy+y, 1);
    }

    for (i = 0; i < 12; i++) {
        x = radius * cosines[i*5];
        y = radius * sines[i*5];
        GR_DrawSolidCircle(w, cx+x, cy+y, 2);
    }

    
    /*
    ** Now draw the cardinal points with bigger dots in yellow or green,
    ** depending on the color scheme selected
    */
    GR_SetDrawColor(w, SM_GetSystemColor(SMLIGHTYG));
    for (i = 0; i < 12; i += 3) {
        x = radius * cosines[i*5];
        y = radius * sines[i*5];
        GR_DrawSolidCircle(w, cx+x, cy+y, 4);
    }

    /*
    ** Get the current time 
    */
    ttime = SM_GetSystemTime();

    /*
    ** Calculate the current hour, min and second from the event data;
    ** adjust 24-hour clock to 12-hour
    */
    SM_TimeOfDay(ttime, &hour, &min, &sec);
    if (hour >= 12) {
        hour -= 12;
    }

    /*
    ** Draw the clock hands
    */
    hr = (hour) * 5 + min / 12;         /* Fractional hour hand position    */

    GR_SetDrawColor(w, SM_GetSystemColor(SMWHITE));
    GR_SetLineWidth(w, 3);
    GR_SetDrawMode(w, SMXOR);
    GR_MoveTo(w, cx, cy);
    GR_LineTo(w,
              (int)((double)cx+cosines[hr]*(double)(radius/2)),
              (int)((double)cy+sines[hr]*(double)(radius/2)));

    GR_MoveTo(w, cx, cy);
    GR_LineTo(w,
              (int)((double)cx+cosines[min]*(double)(radius-5)),
              (int)((double)cy+sines[min]*(double)(radius-5)));

    GR_SetLineWidth(w, 1);
    GR_MoveTo(w, cx, cy);
    GR_LineTo(w,
              (int)((double)cx+cosines[sec]*(double)(radius-5)),
              (int)((double)cy+sines[sec]*(double)(radius-5)));

    /*
    ** Save clock face info and attach it to the clock window
    */
    ts.cx = cx;
    ts.cy = cy;
    ts.radius = radius;
    ts.hour = hour;
    ts.min = min;
    ts.sec = sec;
    SM_SetUserData(w, (void *)&ts);

    return True;
}



/****************************************************************************
**
** ClockBkgd
**
** This is the clock background function. All it does is redraw the clock
** hands, erasing the hands from the old time and drawing in the hands for
** the new time. Note that the clock window drawing mode is set to XOR in
** the draw function, so it's still XOR here.
*/
int ClockBkgd(WindowType *w, EventType *ev)
{
    int         n;
    tstruct     *ts;
    int         hr;
    int         hour, min, sec;
    char        test[15];

    /*
    ** Get our time struct from the clock window
    */
    ts = (tstruct *)SM_GetUserData(w);

    /*
    ** Convert the time field of the event to an hour, minute, and second
    */
    SM_TimeOfDay(ev->localtime, &hour, &min, &sec);

    /*
    ** Convert to 12-hour clock
    */
    if (hour >= 12) {
        hour -= 12;
    }

    /*
    ** Erase the second hand (remember we're in XOR mode)
    */
    GR_SetLineWidth(w, 1);
    GR_MoveTo(w, ts->cx, ts->cy);
    GR_LineTo(w,
              (int)((double)(ts->cx)+cosines[ts->sec]*(double)(ts->radius - 5)),
              (int)((double)(ts->cy)+sines[ts->sec]*(double)(ts->radius - 5)));

    /*
    ** Check if minutes, hours have incremented; if so, erase their hands too
    */
    n = 0;
    if (ts->min != min) {
        GR_SetLineWidth(w, 3);
        GR_MoveTo(w, ts->cx, ts->cy);
        GR_LineTo(w,
                  (int)((double)(ts->cx)+cosines[ts->min]*(double)(ts->radius - 5)),
                  (int)((double)(ts->cy)+sines[ts->min]*(double)(ts->radius - 5)));

        GR_MoveTo(w, ts->cx, ts->cy);
        GR_LineTo(w,
                  (int)((double)(ts->cx)+cosines[ts->hour*5+ts->min/12]*(double)(ts->radius/2)),
                  (int)((double)(ts->cy)+sines[ts->hour*5+ts->min/12]*(double)(ts->radius/2)));
        n++;
    }

    /*
    ** Draw the new second hand
    */
    GR_SetLineWidth(w, 1);
    GR_MoveTo(w, ts->cx, ts->cy);
    GR_LineTo(w,
              (int)((double)(ts->cx)+cosines[sec]*(double)(ts->radius-5)),
              (int)((double)(ts->cy)+sines[sec]*(double)(ts->radius-5)));

    /*
    ** If minutes changed then draw the minute and hour hand
    */
    if (n) {
        GR_SetLineWidth(w, 3);
        GR_MoveTo(w, ts->cx, ts->cy);
        GR_LineTo(w,
                  (int)((double)(ts->cx)+cosines[min]*(double)(ts->radius-5)),
                  (int)((double)(ts->cy)+sines[min]*(double)(ts->radius-5)));

        GR_MoveTo(w, ts->cx, ts->cy);
        GR_LineTo(w,
                  (int)((double)(ts->cx)+cosines[hour*5+min/12]*(double)(ts->radius/2)),
                  (int)((double)(ts->cy)+sines[hour*5+min/12]*(double)(ts->radius/2)));
    }

    /*
    ** Save the event time for the next call
    */
    ts->sec = sec;
    ts->min = min;
    ts->hour = hour;

    return True;
}



int About(WindowType *w)
{
    WindowType  *about;
    RectType    rt;

    rt.Xmin = 50;
    rt.Ymin = 175;
    rt.Xmax = rt.Xmin + 400;
    rt.Ymax = rt.Ymin + 175;
    about = SM_NewWindow(&rt, "About Clock", DIALOG | NOBACKING, NULL, NULL);
    rt.Xmin = 1;
    rt.Xmax = -999;
    rt.Ymin = -200;
    rt.Ymax = -275;
    SM_CreateLabel(about,&rt,"CLOCK",NULL,ALIGNCENTER,False,False,False,NULL);
    rt.Ymin = rt.Ymax - 10;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "Clock demo for the TWS Window System",
                   NULL, ALIGNCENTER, False, False, False, NULL);
    rt.Ymin = rt.Ymax - 175;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "Release:",
                   NULL, ALIGNCENTER, False, False, False, NULL);
    rt.Ymin = rt.Ymax - 10;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "4.0",
                   NULL, ALIGNCENTER, False, False, False, NULL);

    rt.Xmin = -400;
    rt.Xmax = -600;
    rt.Ymax = SM_GetContentDepth(about) - 10;
    rt.Ymin = rt.Ymax - 30;
    SM_CreateButton(about,
                    &rt,
                    "OK",
                    NULL,
                    CloseAbout);
    SM_OpenWindow(about);
}


int CloseAbout(ButtonType *b)
{
    SM_CloseWindow(SM_GetGadgetWindow(b));
    return True;
}


int HowItWorks(WindowType *w)
{
    WindowType  *works;
    RectType    rt;
    TextboxType *tb;

    rt.Xmin = rt.Ymin = 150;
    rt.Xmax = rt.Xmin + 500;
    rt.Ymax = rt.Ymin + 350;
    works = SM_NewWindow(&rt, "clock.c", DOCUMENT | NOBACKING, NULL, NULL);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(works) - 2;
    rt.Ymax = SM_GetContentDepth(works) - 2;
    tb = SM_CreateTextbox(works,
                          &rt,
                          "clock.c",
                          SM_GetSystemFont(),
                          True,
                          False,
                          NULL);
    SM_SetResizeProc(works, ResizeText);
    SM_SetUserData(works, (void *)tb);
    SM_OpenWindow(works);
}


int ResizeText(WindowType *w)
{
    TextboxType *tb;
    RectType    rt;

    tb = (TextboxType *)SM_GetUserData(w);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(w) - 2;
    rt.Ymax = SM_GetContentDepth(w) - 2;
    SM_SetTextboxRect(tb, &rt);
}


