/*
** winimage.c
**
** TWS demo program for image display.
**
** This program reads and displays a PCX image in a window. The image can be
** resized by changing the window size.
**
** Copyright 1993 by
** Dean Clark
** PO Box 37464
** Albuquerque, NM  87176
*/

typedef unsigned char   Byte;
typedef unsigned short  word;


/*
** Include the appropriate C & TWS header files
*/
#include    <stdio.h>
#include    <stdlib.h>
#include    <dos.h>
#include    <string.h>
#ifdef __TURBOC__
#include    <alloc.h>
#else
#include    <malloc.h>
#endif

#include    <smwindow.h>
#include    <smevent.h>
#include    <smgraph.h>
#include    <smmenu.h>
#include    <smpickls.h>
#include    <smcolor.h>
#include "smmenu.h"
#include "smlabel.h"
#include "smtext.h"
#include "smbutton.h"
#include "smpixmap.h"
#include "smicon.h"
#include "smwutil.h"

/*
** The Image Data struct contains all the information we need to store
** the PCX image. One of these structs will be attached to each image window
*/

typedef struct {
    int         width, depth;           /* Image width/depth                */
    int         ncolors;
    char        *fname;                 /* Name of the image file           */
    FILE        *f;                     /* File pointer                     */
    long        imgstart;               /* Offset to beginning of image     */
    unsigned char palette[256*3];       /* Image palette                    */
} ImageDataType;


void main(int argc, char *argv[]);
char *GetPCXFile(void);
int ScaleImage(WindowType *w);
void PCXPalette(ImageDataType *img);
int LoadImage(char *fname);
int Open_PCX( char *fname , ImageDataType *img);
int About(WindowType *w);
int HowItWorks(WindowType *w);
int CloseAbout(ButtonType *b);
int ResizeText(WindowType *w);
void PrntScrn(WindowType *w, RectType *r, char *fname);
int LoadFile(WindowType *w);


#ifdef __TURBOC__
extern unsigned _stklen = 8192U;  /* set stack size to 8K */
#endif

extern FILE *ferr;
PixmapType  *iconpixmap;

void main(int argc, char *argv[])
{
    EventType   *ev;
    char        *fname;
    int         msg;
    MenuType    *m;

    /*
    ** Initialize the window system, graphics, fonts, etc.
    */
    SM_Init(argc, argv);

    /*
    ** Demo menu is very simple
    */
    m = SM_CreateMenu();
    SM_AddMenuItem(m, ACTION_ITEM, "About...", About, NULL);
    SM_AddMenuItem(m, ACTION_ITEM, "Load", LoadFile, NULL);
    SM_AddMenuItem(m, ACTION_ITEM, "Source Code", HowItWorks, NULL);
    SM_AddMenuItem(m, ACTION_ITEM, "Quit", SM_Exit, NULL);

    SM_OpenApplication("TWS Image Display", m);

    SM_SetPrintscreenProc(NULL, NULL, "winimage.pcx", PrntScrn);

    iconpixmap = SM_ReadPixmap("face.icn");

    /*
    ** Just loop around waiting for things to happen
    */
    while (True) {
        ev = SM_GetNextEvent();
        msg = SM_ProcessEvent(ev);
        if (msg == KEYPRESS) {
            if (ev->ASCII == 'q') {
                SM_Exit(NULL);
            }
            /*
            ** Get the PCX file from the user
            */
            if ((fname = GetPCXFile()) == NULL) {
                SM_Exit(NULL);
            }

            /*
            ** Open a window and load the PCX file
            */
            if (LoadImage(fname)) {
                SM_Exit(NULL);
            }
        }
    }
}


int LoadFile(WindowType *w)
{
    char    *fname;

    fname = GetPCXFile();
    if (fname == NULL) return 1;
    return LoadImage(fname);
}


/**************************************************************************
**
** GetPCXFile
**
** Prompts for the name of a PCX image file. Loads the names of .pcx
** files in the current directory and displays a picklist for selecting
** one.
*/
char *GetPCXFile(void)
{
    char        **list;                 /* List of PCX file names           */
    int         msg;                    /* Selected PCX file index          */
    int         nfiles;                 /* Number of PCX files in currdir   */
    struct find_t fdat;                 /* DOS file structure               */
    unsigned    code;                   /* DOS file search return code      */
    RectType    r;                      /* Picklist rectangle               */
    char        *fsel;                  /* Name of file selected            */

    nfiles = 0;

    /*
    ** First find out how many PCX files in the current directory
    */
    code = _dos_findfirst("*.pcx", _A_NORMAL, &fdat);
    while (code == 0) {
        nfiles++;
        code = _dos_findnext(&fdat);
    }

    /*
    ** Allocate enough string pointers for each file name
    */
    list = (char **)malloc(nfiles * sizeof(char *));
    if (list == NULL) {
        return NULL;
    }

    /*
    ** Now go back through and load the stringlist with file names
    */
    code = _dos_findfirst("*.pcx", _A_NORMAL, &fdat);
    nfiles = 0;
    while (code == 0) {
        list[nfiles] = strdup(fdat.name);
        nfiles++;
        code = _dos_findnext(&fdat);
    }

    /*
    ** Display a picklist for the user to select the image. First build
    ** a rectangle big enough for the strings...
    */
    r.Xmin = r.Ymin = 140;
    r.Xmax = r.Xmin + GR_StringWidth("WWWWWWWW.WWW");
    r.Ymax = r.Ymin + 120;
    msg = SM_Picklist("PCX Files",      /* Picklist window title            */
                      &r,               /* Boundary for the strings         */
                      list,             /* Strings to pick from             */
                      nfiles,           /* Number of strings                */
                      0,                /* Index of first string            */
                      0,                /* Which string pre-selected        */
                      0);               /* Destructor should not free list  */

    /*
    ** If the picklist returns a negative number, then the user chose CANCEL;
    ** otherwise, copy the string from the list position
    */
    if (msg < 0) {
        fsel = NULL;
    }
    else {
        fsel = strdup(list[msg]);
    }

    /*
    ** Now free the picklist string list
    */
    while (nfiles > 0) {
        nfiles--;
        free(list[nfiles]);
    }
    free(list);

    return fsel;

}


/****************************************************************************
**
** LoadImage
**
** Creates a window and loads the image file into it.
*/
int LoadImage(char *fname)
{
    WindowType      *w;
    RectType        r;
    int             i, j;
    ImageDataType   *img;

    img = (ImageDataType *)malloc(sizeof(ImageDataType));
    if (img == NULL) {
        return 1;
    }

    /*
    ** Open the image file and load the window header
    */
    if (Open_PCX(fname, img)) {
        return 1;
    }

    /*
    ** Open the window
    */
    r.Xmin = r.Ymin = 100;
    r.Xmax = r.Xmin + img->width;
    r.Ymax = r.Ymin + img->depth;
    img->fname = strdup(fname);
    w = SM_NewWindow(&r, img->fname, DOCUMENT | BACKING, NULL, NULL);

    /*
    ** Create a graphics state for the window
    */
    GR_CreateGraphState(w, NULL, SCULPTED);

    /*
    ** Set the draw and resize procedure functions. We don't strictly have
    ** to set the resize procedure, since if there's a draw procedure attached
    ** to the window but no resize procedure, the draw procedure gets
    ** called automatically when the window is resized.
    */
    SM_SetDrawProc(w, ScaleImage);
    SM_SetResizeProc(w, ScaleImage);

    /*
    ** Attach the image to the window
    */
    SM_SetUserData(w, (void *)img);

    /*
    ** Load the image palette into the window
    */
    SM_CreateWindowPalette(w, img->ncolors);
    for (i = 0; i < img->ncolors; i++) {
        j = i*3;
        SM_SetWindowColor(w, i, img->palette[j], img->palette[j+1], img->palette[j+2], SMSHARE | SMCLOSESTCOLOR);
    }
    GR_SetBackgroundColor(w, SM_GetWindowColor(w, 0));

    /*
    ** Create an icon for this window.  Use the pixmap we read at init
    */
    SM_CreateIcon(w, NULL, iconpixmap);

    /*
    ** Open the window, which will load and display the image
    */
    SM_OpenWindow(w);

    return 0;
}





#define BADSIG  1
#define BADFILE 2
#define BADBITS 3


/*
** ================ The actual pcx reader routines ==========================
*/


typedef struct  _PCXFileHDR
{
    Byte     dMAN;                      /* signature for Image File         */
    Byte     dVER;                      /* version info                     */
    Byte     encoding;                  /* currently 1, RLL                 */
    Byte     pixbits;                   /* bits per pixel                   */
    word     minX;                      /* Image Width (in pixels)          */
    word     minY;                      /* Image Height ( in pixels )       */
    word     maxX;
    word     maxY;
    word     resX;                      /* resolution of device             */
    word     resY;
    Byte     clrmap[48];                /* palette info                     */
    Byte     mode;                      /* usually ignored                  */
    Byte     pixPlanes;                 /* number of planes involved        */
    word     picBytes;                  /* bytes per line in picture        */
    Byte     filler[60];                /* pad to 128 bytes, data starts    */
} PCXFileHDR;

/*
** Read a file and check that it is a valid PCX file. Also check that the
** type of image in the PCX file is compatible with the current port
*/
int Open_PCX( char *fname , ImageDataType *img)
{
    FILE            *datfile;
    PCXFileHDR      PCXInfo;

   /* Open source file */
    if( ( datfile = fopen( fname,"r+b") ) == NULL ) /* binary mode read */
        return BADFILE;

    /*  Read Header */
    if( ( fread( &PCXInfo,sizeof(PCXInfo),1,datfile) ) == 0 ) {
        fclose( datfile );
        return BADFILE;
    }

    /* Validate signatures */
    if( PCXInfo.dMAN != 10 )
        return BADSIG;
    if( PCXInfo.encoding != 1 )
        return BADSIG;

    /* check bits per pixel */
    if( PCXInfo.pixbits != 8) {
        return BADBITS;
    }

    /*
    ** Build an image header for this guy
    */
    img->f          = datfile;
    img->imgstart   = ftell(datfile);
    img->width      = PCXInfo.maxX - PCXInfo.minX + 1;
    img->depth      = PCXInfo.maxY - PCXInfo.minY + 1;
    PCXPalette(img);
    return 0;
}


#define  OK       False
#define  ERROR    True   
#define  PACKED   0xC0  /* two hi bits of byte */ 
#define  MASK     0x3F  /* lower six bits */

/*****************************************************************************
**
** PCX_Get_Row
**
** Get one scan line from the PCX image
**
*/
int PCX_Get_Row(FILE *datfile, int rows, unsigned char *line, int init)
{
    register Byte   unwork, unwork2;
    register int    x, i;
    word            z;
    static int      lineCnt;
    PCXFileHDR      PCXInfo;

    /*
    ** If initializing then position file pointer to beginning of image data
    */
    if (init) {
        fseek(datfile, 0, SEEK_SET);
        fread( &PCXInfo, sizeof(PCXInfo), 1, datfile );
        lineCnt = rows;
        return 1;
    }

    for( x = 0;  x < rows;   ) {
        unwork= (Byte) getc(datfile);
        if( (unwork & PACKED) == PACKED ) {
            z = unwork & MASK;          /* calculate byte value             */
            unwork2 = (Byte) getc(datfile);
            for( i = 0; i < z; i++ ) {  /* unpack info to buffer            */
                line[x++] = unwork2;
            }
         }
         else {                         /* not packed                       */
            line[x++] = unwork;
         }

    }
    return OK;
}


/***************************************************************************
**
** PCXPalette
**
** Load palette information from PCX file
*/
void PCXPalette(ImageDataType *img)
{
    int         i,j;
    long  int   fpos;
    PCXFileHDR  PCXInfo;

    /*
    ** Set to the beginning of the file and read the header again
    */
    fseek(img->f, 0, SEEK_SET);
    if( fread( &PCXInfo, sizeof(PCXInfo), 1, img->f) == 0 ) {
       return ;
    }

    /*
    ** First assume the image has only the 16 header colors...
    */
    for( i = 0; i < 16; i++ ) {
        j= i*3;
        img->palette[j] = PCXInfo.clrmap[j];
        img->palette[j+1] = PCXInfo.clrmap[j+1];
        img->palette[j+2] = PCXInfo.clrmap[j+2];
    }

    /*
    ** Seek to 769 bytes from end of file
    */
    fseek( img->f, 0L, SEEK_END );
    fpos = ftell( img->f);
    fseek( img->f, fpos - 769L, SEEK_SET );

    /*
    ** If there's a 12 there then this image has extended palette
    */
    i = getc(img->f);
    if( i != 12 ) {
        img->ncolors = 16;
        return;
    }
    img->ncolors = 256;
   
    /*
    ** Read the PCX palette
    */
    for(i=0;i<256;i++ ) {
        j = i*3;
        img->palette[j] = getc(img->f);
        img->palette[j+1] = getc(img->f);
        img->palette[j+2] = getc(img->f);
    }
}



int ScaleImage(WindowType *w)
{
    double  zoomh, zoomv;           /* Horizontal and vertical scale factors*/
    int     dday, ddax;             /* Counters for DDA algorithm       */
    int     i, j;                   /* Loop counters                    */
    int     x, y;                   /* Window canvas pixel coordinates  */
    int     izoom;                  /* DDA increment                    */
    ImageDataType   *imgdata;       /* Image data attached to window    */
    unsigned char   *aspline;       /* Image raster buffer              */
    RectType    r;
    ColorType   *c;
/*    ColorType   **array; */
    int         ci;

    /*
    ** Get the image data from the window
    */
    imgdata = (ImageDataType *)SM_GetUserData(w);

    /*
    ** Set the canvas to the same size as the content region is now
    */
    r.Xmin = r.Ymin = 0;
    r.Xmax = r.Xmin + SM_GetContentWidth(w) - 1;
    r.Ymax = r.Ymin + SM_GetContentDepth(w) - 1;
    GR_SetCanvasRect(w, &r);

    /*
    ** Get the canvas background color
    */
    c = GR_GetBackgroundColor(w);

    /*
    ** Erase the graphics canvas
    */
    GR_ClearCanvas(w);

    /*
    ** Figure the scale factor in the horizontal and vertical directions
    */
    zoomh = (double)(GR_GetCanvasWidth(w)) / (double)(imgdata->width);
    zoomv = (double)(GR_GetCanvasDepth(w)) / (double)(imgdata->depth);

    /*
    ** Use the smaller of the two so the image stays in the window. This is
    ** not required; you could also use the larger scale, or the average.
    */
    if (zoomv < zoomh) zoomh = zoomv;

    /*
    ** izoom is the DDA differential. We use an integer so we avoid
    ** floating point math, which ought to make it run a little faster
    */
    izoom = (int)(1.0/zoomh * 1000);

    /*
    ** Setup a color array so we can use faster DrawArray routine
    */
/*    array = (ColorType **)malloc((int)(zoomv*imgdata->width*sizeof(ColorType *))+1); */

    /*
    ** Position the file pointer to the beginning of image data
    */
    fseek(imgdata->f, imgdata->imgstart, SEEK_SET);

    /*
    ** Allocate a buffer for one raster from the image file
    */
    aspline = (unsigned char *)malloc(imgdata->width+2);
    if (aspline == NULL) {
        SM_Exit(NULL);
    }
    dday = 0;
    x = y = 0;

    GR_SetBlocking(w);

    PCX_Get_Row(imgdata->f,imgdata->width,aspline,1);

    for (i = 1; i < imgdata->depth; i++) {
        /*
        ** Get one raster of the image.
        */
        PCX_Get_Row(imgdata->f,imgdata->width,aspline,0);

        /*
        ** One "unit" in image space == 1000
        */
        dday -= 1000;

        while (dday < 0) {
            x = 0;
            /*
            ** As long as dday is less than 0, we haven't advanced to the
            ** next image pixel vertically -- that is, we repeat the same
            ** scan line from the image as many times as it takes to reach
            ** the DDA threshold of 0.
            **
            ** Reset the horizontal DDA sample counter. This ensures that
            ** all rasters start at the same place.
            */
            ddax = 0;
            for (j = 0; j < imgdata->width; j++) {
                ddax -= 1000;
                while (ddax < 0) {
                    /*
                    ** Same deal as above -- keep repeating the same image
                    ** pixel in adjacent window pixels until the DDA counter
                    ** crosses the threshold
                    */
                    GR_SetDrawColor(w, SM_GetWindowColor(w, aspline[j]));
/*                    array[x] = SM_GetWindowColor(w, aspline[j]); */
                    /*
                    ** This example assumes the image is drawn relative to
                    ** the upper left corner. Adjusting for other
                    ** orientations is fairly trivial. Don't bother to draw
                    ** the pixel if it's the same color as the background
                    */
                    GR_DrawPoint(w, x, y);
                    x++;
                    /*
                    ** If x is off the right edge of the window then no
                    ** need to continue
                    */
                    if (x > GR_GetCanvasWidth(w)) {
                        j = imgdata->width;
                    }
                    ddax += izoom;
                }
            }

/*            GR_DrawArray(w, 0, y, x-1, array); */

            dday += izoom;
            y++;
            /*
            ** Test whether the y ordinate is still within the window
            ** boundary, exit if it's out
            */
            if (y > GR_GetCanvasDepth(w)) {
                free(aspline);
                GR_UnsetBlocking();
                return 0;
            }
        }
    }


    free(aspline);
}




int About(WindowType *w)
{
    WindowType  *about;
    RectType    rt;

    rt.Xmin = 50;
    rt.Ymin = 175;
    rt.Xmax = rt.Xmin + 400;
    rt.Ymax = rt.Ymin + 175;
    about = SM_NewWindow(&rt, "About Winimage", DIALOG | NOBACKING, NULL, NULL);
    rt.Xmin = 1;
    rt.Xmax = -999;
    rt.Ymin = -200;
    rt.Ymax = -275;
    SM_CreateLabel(about,&rt,"WINIMAGE",NULL,ALIGNCENTER,False,False,False,NULL);
    rt.Ymin = rt.Ymax - 10;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "PCX image display demo for the TWS Window System",
                   NULL, ALIGNCENTER, False, False, False,NULL);
    rt.Ymin = rt.Ymax - 175;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "Release:",
                   NULL, ALIGNCENTER, False, False, False,NULL);
    rt.Ymin = rt.Ymax - 10;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "4.0",
                   NULL, ALIGNCENTER, False, False, False,NULL);

    rt.Xmin = -400;
    rt.Xmax = -600;
    rt.Ymax = SM_GetContentDepth(about) - 10;
    rt.Ymin = rt.Ymax - 30;
    SM_CreateButton(about,
                    &rt,
                    "OK",
                    NULL,
                    CloseAbout);
    SM_OpenWindow(about);
}


int CloseAbout(ButtonType *b)
{
    SM_CloseWindow(SM_GetGadgetWindow(b));
    return True;
}


int HowItWorks(WindowType *w)
{
    WindowType  *works;
    RectType    rt;
    TextboxType *tb;

    rt.Xmin = rt.Ymin = 150;
    rt.Xmax = rt.Xmin + 500;
    rt.Ymax = rt.Ymin + 350;
    works = SM_NewWindow(&rt, "winimage.c", DOCUMENT | NOBACKING, NULL, NULL);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(works) - 2;
    rt.Ymax = SM_GetContentDepth(works) - 2;
    tb = SM_CreateTextbox(works,
                          &rt,
                          "winimage.c",
                          SM_GetSystemFont(),
                          True,
                          True,
                          NULL);
    SM_SetResizeProc(works, ResizeText);
    SM_SetUserData(works, (void *)tb);
    SM_OpenWindow(works);
}


int ResizeText(WindowType *w)
{
    TextboxType *tb;
    RectType    rt;

    tb = (TextboxType *)SM_GetUserData(w);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(w) - 2;
    rt.Ymax = SM_GetContentDepth(w) - 2;
    SM_SetTextboxRect(tb, &rt);
}

void PrntScrn(WindowType *w, RectType *r, char *fname)
{
    SM_SaveScreenAsPCX(fname);
}


