/*
** Iconedit.c
**
** Utility for building simple TWS icons
**
** Copyright 1993, 1994 by
** Dean Clark
** PO Box 37464
** Albuquerque, NM  87176
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <mem.h>
#include <float.h>
#include <math.h>
#include <dir.h>
#include <string.h>
#include "smwindow.h"
#include "smtypes.h"
#include "smevent.h"
#include "smgraph.h"
#include "smmenu.h"
#include "smlabel.h"
#include "smbutton.h"
#include "smtext.h"
#include "smbutton.h"
#include "smcolor.h"
#include "filedial.h"
#include "smyesno.h"
#include "smync.h"
#include "smok.h"
#include "smwutil.h"


/*
** Borlandc requires extra stack
*/
#ifdef __TURBOC__
extern unsigned _stklen = 16384U;
#endif

#define     LINETOOL    1
#define     CIRCLETOOL  2
#define     FILLTOOL    3
#define     BOXTOOL     4
#define     PENTOOL     0

/*
** Data for icon editor.  We need to pass around the pixmap for the icon
** so a few of our functions can modify it.  The "isize" field is so the
** functions know how big the pixmap is.  We could also get the size from
** the pixmap itself.
** An IconEditData variable will be attached to the iconedit window as user
** data.
*/

typedef struct {
    PixmapType  *pm;
    int         isize;
    char        *filename;
    int         modified;
    int         tool;
    ColorType   *drawcolor, *backcolor;
} IconEditData;

/*
** Function templates for iconedit app
*/
void main(int argc, char *argv[]);
int IconDraw(WindowType *w, EventType *ev);
int About(WindowType *w);
int CloseAbout(ButtonType *b);
int SetDrawColor(ButtonType *b);
int SetBackColor(ButtonType *b);
int DrawEditWin(WindowType *w);
int ClearPixmap(WindowType *button);
int WritePixmap(WindowType *b);
int SavePixmap(WindowType *b);
int ReadPixmap(WindowType *b);
int DrawColorLabel(LabelType *label);
int SetTool(ButtonType *b);
void DrawFatbitsFromPixmap(WindowType *w, IconEditData *ie);
void DrawLine(WindowType *w, IconEditData *IconData);
void SetFatbit(WindowType *w, PixmapType *pm, int x, int y, ColorType *c);
void DrawBox(WindowType *w, IconEditData *IconData);
void DrawCircle(WindowType *w, IconEditData *IconData);
int UndoPixmap(WindowType *w);
int IconExit(WindowType *w);
int DisplayHelp(WindowType *w);
int ResizeText(WindowType *w);
void PrntScrn(WindowType *w, RectType *r, char *fname);


int fatbitsize;                         /* Width of a fat bit in pixels     */
int cindex, bindex;

LabelType   *forelabel, *backlabel;     /* Labels for foreground/background */
PixmapType  *undopm;
PixmapType  *SelectPixmap;

void main(int argc, char *argv[])
{
    WindowType  *w;                     /* The icon editor window           */
    RectType    r;                      /* Rectangle for the window         */
    EventType   *ev;                    /* Event variable                   */
    int         msg;                    /* Event return value               */
    ButtonType  *button,                /* Icon editor control buttons      */
                *LineButton,
                *BoxButton,
                *CircleButton,
                *PenButton,
                *FillButton;
    int         i;
    IconEditData    IconData;           /* "Icon" data (pixmap, actually)   */
    int         colorbuttonsize;        /* Width of the color select buttons*/
    MenuType    *winmenu;
    MenuType    *filemenu;
    MenuType    *editmenu;
    MenuType    *helpmenu;
    PixmapType  *PenPixmap;
    PixmapType  *LinePixmap;
    PixmapType  *BoxPixmap;
    PixmapType  *CirclePixmap;

    /*
    ** Initialize and open the application desktop
    */
    SM_Init(argc, argv);
    SM_OpenApplication("Icon Edit", NULL);

    /*
    ** Set up the window menus
    */
    winmenu = SM_CreateMenu();
    filemenu = SM_CreateMenu();
    editmenu = SM_CreateMenu();
    helpmenu = SM_CreateMenu();
    SM_AddMenuItem(winmenu, SUBMENU_ITEM, "File", NULL, filemenu);
    SM_AddMenuItem(winmenu, SUBMENU_ITEM, "Edit", NULL, editmenu);
    SM_AddMenuItem(winmenu, ACTION_ITEM, "Help", DisplayHelp, NULL);
    SM_AddMenuItem(filemenu, ACTION_ITEM, "Open", ReadPixmap, NULL);
    SM_AddMenuItem(filemenu, ACTION_ITEM, "Save", SavePixmap, NULL);
    SM_AddMenuItem(filemenu, ACTION_ITEM, "SaveAs", WritePixmap, NULL);
    SM_AddMenuItem(filemenu, MENU_DIVIDER, NULL, NULL, NULL);
    SM_AddMenuItem(filemenu, ACTION_ITEM, "About", About, NULL);
    SM_AddMenuItem(filemenu, MENU_DIVIDER, NULL, NULL, NULL);
    SM_AddMenuItem(filemenu, ACTION_ITEM, "Quit", IconExit, NULL);
    SM_AddMenuItem(editmenu, ACTION_ITEM, "Clear", ClearPixmap, NULL);
    SM_AddMenuItem(editmenu, ACTION_ITEM, "Undo", UndoPixmap, NULL);
/*
    SM_AddMenuItem(editmenu, ACTION_ITEM, "Reverse", NULL, NULL);
    SM_AddMenuItem(editmenu, ACTION_ITEM, "Flip Horizontal", NULL, NULL);
    SM_AddMenuItem(editmenu, ACTION_ITEM, "Flip Vertical", NULL, NULL);
*/


    /*
    ** Make each square in the drawing area 5 pixels wide
    */
    fatbitsize = 5;

    /*
    ** Create the drawing window:
    ** The width of the content is the size of the drawing area, plus
    ** room for the color buttons and control buttons to the right of the
    ** canvas
    */
    if (SM_GetDisplayWidth() < 1000) {
        r.Xmax = 40*fatbitsize + 2*SM_GetDefaultBevelDepth() + 4;
        r.Ymax = 40*fatbitsize;
        IconData.isize = 40;
    }
    else {
        r.Xmax = 80*fatbitsize + 2*SM_GetDefaultBevelDepth() + 4;
        r.Ymax = 80*fatbitsize;
        IconData.isize = 80;
    }

    /*
    ** Make the color buttons just fit along the bottom of the fatbit pixmap
    */
    colorbuttonsize = (IconData.isize * fatbitsize - 7)/8;
    r.Xmax += 2*colorbuttonsize + IconData.isize + 4;
    r.Ymax += 2*(colorbuttonsize+3) + 2*SM_GetDefaultBevelDepth();

    /*
    ** Center the editor window on the screen
    */
    r.Xmin = SM_GetDisplayWidth()/2 - r.Xmax/2;
    r.Ymin = SM_GetDisplayDepth()/2 - r.Ymax/2;
    r.Xmax = SM_GetDisplayWidth()/2 + r.Xmax/2;
    r.Ymax = SM_GetDisplayDepth()/2 + r.Ymax/2;

    /*
    ** Create the window and set the window and draw event procedures
    */
    w = SM_NewWindow(&r, "Iconedit:<unnamed>", DOCUMENT | NOBACKING, winmenu, (void *)(&IconData));
    SM_SetWindowProc(w, IconDraw);
    SM_SetDrawProc(w, DrawEditWin);

    /*
    ** Make a pixmap in the corner for the actual-size icon
    */
    r.Xmin = SM_GetContentWidth(w) - IconData.isize - 3;
    r.Ymin = IconData.isize*fatbitsize/2 - IconData.isize/2;
    r.Ymax = r.Ymin + IconData.isize - 1;
    r.Xmax = r.Xmin + IconData.isize - 1;
    IconData.pm = SM_CreatePixmap(w, &r, True);
    IconData.modified = False;
    IconData.filename = NULL;
    IconData.tool = PENTOOL;
    bindex = 0;
    cindex = SMWHITE;
    IconData.backcolor = SM_GetSystemColor(bindex);
    IconData.drawcolor = SM_GetSystemColor(cindex);

    /*
    ** Create the undo pixmap
    */
    undopm = SM_CreatePixmap(NULL, &r, False);

    /*
    ** Create a graphics state so we can draw in the window.  Make the canvas
    ** just big enough for the fatbit pixmap.
    */
    r.Xmin = IconData.isize+3;
    r.Ymin = 1;
    r.Ymax = r.Ymin + IconData.isize*fatbitsize + 2*SM_GetDefaultBevelDepth();
    r.Xmax = r.Xmin + IconData.isize*fatbitsize + 2*SM_GetDefaultBevelDepth();
    GR_CreateGraphState(w, &r, GRRECESSBORDER);

    SM_OpenWindow(w);

    /*
    ** Create the draw tool buttons
    */
    r.Xmin = 3;
/*    r.Xmax = r.Xmin + IconData.isize - 1; */
    r.Xmax = r.Xmin + 32;
    r.Ymin = 1;
/*    r.Ymax = r.Ymin + IconData.isize - 1; */
    r.Ymax = r.Ymin + 32;
    PenButton = SM_CreateButton(w,
                                &r,
                                NULL,
                                (void *)PENTOOL,
                                SetTool);

    PenPixmap = SM_ReadPixmap("..\\icons\\pen.icn");
    SM_SetPixmapBlocking(PenPixmap, True);
    SM_SetButtonPixmap(PenButton, PenPixmap);

    r.Ymin = r.Ymax + 1;
/*    r.Ymax = r.Ymin + IconData.isize - 1; */
    r.Ymax = r.Ymin + 32;
    LineButton = SM_CreateButton(w,
                                 &r,
                                 NULL,
                                 (void *)LINETOOL,
                                 SetTool);
    LinePixmap = SM_ReadPixmap("..\\icons\\line.icn");
    SM_SetPixmapBlocking(LinePixmap, True);
    SM_SetButtonPixmap(LineButton, LinePixmap);
    r.Ymin = r.Ymax + 1;
/*    r.Ymax = r.Ymin + IconData.isize - 1; */
    r.Ymax = r.Ymin + 32;
    BoxButton = SM_CreateButton(w,
                                &r,
                                NULL,
                                (void *)BOXTOOL,
                                SetTool);
    BoxPixmap = SM_ReadPixmap("..\\icons\\box.icn");
    SM_SetPixmapBlocking(BoxPixmap, True);
    SM_SetButtonPixmap(BoxButton, BoxPixmap);
    r.Ymin = r.Ymax + 1;
/*    r.Ymax = r.Ymin + IconData.isize - 1; */
    r.Ymax = r.Ymin + 32;
    CircleButton = SM_CreateButton(w,
                                   &r,
                                   NULL,
                                   (void *)CIRCLETOOL,
                                   SetTool);
    CirclePixmap = SM_ReadPixmap("..\\icons\\circle.icn");
    SM_SetPixmapBlocking(CirclePixmap, True);
    SM_SetButtonPixmap(CircleButton, CirclePixmap);
    r.Ymin = r.Ymax + 1;
/*    r.Ymax = r.Ymin + IconData.isize - 1; */
    r.Ymax = r.Ymin + 32;
    FillButton = SM_CreateButton(w,
                                 &r,
                                 "Fill",
                                 (void *)FILLTOOL,
                                 SetTool);
    SM_SetButtonActive(FillButton, False);
    /*
    ** Create a pixmap to show the currently selected tool
    */
    r.Ymin = r.Ymax + 10;
    r.Ymax = r.Ymin + 31;
    r.Xmin++;
    r.Xmax = r.Xmin + 31;
    SelectPixmap = SM_CreatePixmap(w, &r, False);
    SM_CopyPixmap(SelectPixmap, PenPixmap);

    /*
    ** Build the color buttons...
    */
    r.Xmin = IconData.isize+2;
    r.Ymin = IconData.isize*fatbitsize+2*SM_GetDefaultBevelDepth()+3;
    for (i = 0; i < 16; i++) {
        r.Xmax = r.Xmin + colorbuttonsize;
        r.Ymax = r.Ymin + colorbuttonsize;
        /*
        ** The following makes the buttons alternate into two layers
        */
        if (i % 2) {
            button = SM_CreateButton(w,
                                     &r,
                                     NULL,
                                     (void *)(i/2),
                                     SetBackColor);
            r.Ymin = r.Ymax - colorbuttonsize*2 - 1;
            r.Xmin = r.Xmax + 1;
        }
        else {
            button = SM_CreateButton(w,
                                     &r,
                                     NULL,
                                     (void *)(i/2),
                                     SetDrawColor);
            r.Ymin = r.Ymax + 1;
        }
        if (i > 13) {
            SM_SetButtonColor(button, SM_GetSystemColor(SMWHITE));
            SM_SetButtonData(button, (void *)SMWHITE);
        }
        else {
            SM_SetButtonColor(button, SM_GetSystemColor(i/2));
        }

    }

    /*
    ** Set up labels to display the selected foreground and background colors
    */
    r.Xmin = r.Xmax + 3;
    r.Xmax = r.Xmin + colorbuttonsize*2+1;
    r.Ymin = IconData.isize*fatbitsize+2*SM_GetDefaultBevelDepth()+3;
    r.Ymax = r.Ymin + colorbuttonsize*2+1;
    /*
    ** The two labels occupy the same space, but the background label is
    ** bigger.  We're relying on the fact the TWS draws gadgets in the same
    ** order that they're created to ensure that the foreground drawing
    ** color label is always on top of the background color label.
    */
    backlabel = SM_CreateLabel(w,
                               &r,
                               NULL,
                               DrawColorLabel,
                               0,
                               False,
                               False,
                               True,
                               &bindex);
    SM_SetLabelSculptType(backlabel, SCULPTED);
    GR_InsetRect(&r, IconData.isize/3, IconData.isize/3);
    forelabel = SM_CreateLabel(w,
                               &r,
                               NULL,
                               DrawColorLabel,
                               0,
                               False,
                               False,
                               True,
                               &cindex);
    SM_SetLabelSculptType(forelabel, BEVELED);

    /*
    ** Set the graphics background color
    */
    GR_SetBackgroundColor(w, SM_GetSystemColor(bindex));
    GR_SetDrawColor(w, SM_GetSystemColor(cindex));

    SM_SetPrintscreenProc(NULL, NULL, "panel.pcx", PrntScrn);

    /*
    ** The ubiquitous event loop...
    */
    while (True) {
        ev = SM_GetNextEvent();
        msg = SM_ProcessEvent(ev);
        if (msg == KEYPRESS) {
            SM_Exit(NULL);
        }
    }
}


/***************************************************************************
**
** IconExit
*/
int IconExit(WindowType *w)
{
    IconEditData    *IconEdit;
    char            msg[100];
    int             answer;

    IconEdit = (IconEditData *)SM_GetUserData(w);
    if (IconEdit->modified) {
        sprintf(msg, "Pixmap has been modified. Save changes before quitting?");
        answer = SM_YesNoCancelDialog(msg, "Save First", "Continue", "Cancel");
        if (answer < 0) {
            return 0;
        }
        else if (answer == 0) {
            if (SavePixmap(w)) {
                return 0;
            }
        }
        SM_DestroyPixmap(IconEdit->pm);
    }
    SM_Exit(NULL);
}





int DrawEditWin(WindowType *w)
{
    int             x,y;
    int             i;
    ColorType       c;
    IconEditData    *ie;

    /*
    ** Get the grid size from the window
    */
    ie = (IconEditData *)SM_GetUserData(w);

    /*
    ** Store the current drawing color before drawing dark gray lines
    */
    memcpy(&c, GR_GetDrawColor(w), sizeof(ColorType));
    GR_SetDrawColor(w, SM_GetSystemColor(SMDARKGRAY));

    /*
    ** Set graphics blocking for  faster drawing, but only if this is
    ** the focus window.
    */
    if (SM_IsFocus(w)) {
        GR_SetBlocking(w);
    }

    /*
    ** Draw the grid
    */
    x = y = fatbitsize-1;
    for (i = 0; i < ie->isize-1; i++) {
        GR_MoveTo(w,x,0);
        GR_LineTo(w,x,GR_GetCanvasDepth(w));
        GR_MoveTo(w,0,y);
        GR_LineTo(w,GR_GetCanvasWidth(w),y);
        x += fatbitsize;
        y += fatbitsize;
    }

    DrawFatbitsFromPixmap(w, ie);

    if (SM_IsFocus(w)) {
        GR_UnsetBlocking();
    }

    /*
    ** Restore the original drawing color
    */
    GR_SetDrawColor(w, &c);
    return 0;
}


/**************************************************************************
**
** UndoPixmap
**
** Restore the screen pixmap from the undo pixmap
*/
int UndoPixmap(WindowType *w)
{
    IconEditData    *IconData;

    IconData = (IconEditData *)SM_GetUserData(w);
    SM_CopyPixmap(IconData->pm, undopm);
    DrawFatbitsFromPixmap(w, IconData);
    return 0;
}



/****************************************************************************
**
** DrawFatbitsFromPixmap
*/
void DrawFatbitsFromPixmap(WindowType *w, IconEditData *ie)
{
    int         x, y, blocked = False;
    ColorType   pmcolor;

    /*
    ** Restore the pixmap in the large grid
    */
    if (SM_IsFocus(w)) {
        GR_SetBlocking(w);
        blocked = True;
    }
    for (x = 0; x < ie->isize; x++) {
        for (y = 0; y < ie->isize; y++) {
            SM_GetPixmapPixel(ie->pm,x,y,&pmcolor);
            SetFatbit(w, ie->pm, x, y, &pmcolor);
        }
    }
    if (blocked) {
        GR_UnsetBlocking();
    }
}



/****************************************************************************
**
** IconDraw
**
** This is the editor's window procedure. If the user presses a mouse button
** inside the window.
*/
int IconDraw(WindowType *w, EventType *ev)
{
    int         state, x, y;
    RectType    r;
    int         oldx = 0, oldy = 0;
    IconEditData    *IconData;

    /*
    ** We know if we get here that the window is the focus window, but we
    ** don't know if the event occured in this window.  We can check it...
    */
    if ((ev->event_type & LEFTBUTTONACTIVE) && (ev->region == CANVAS_REGION)) {
        IconData = (IconEditData *)SM_GetUserData(w);
        /*
        ** Copy the current pixmap so we can undo whatever the user does
        ** this time
        */
        SM_CopyPixmap(undopm, IconData->pm);
        IconData->modified = True;
        switch (IconData->tool) {
            case LINETOOL : DrawLine(w, IconData); break;
            case BOXTOOL  : DrawBox(w, IconData); break;
            case CIRCLETOOL  : DrawCircle(w, IconData); break;
            case PENTOOL :
                /*
                ** Trap the mouse here. As long as the mouse button is pressed and
                ** the mouse is inside the cursor keep drawing inside the canvas.
                */
                GR_SetDrawColor(w, IconData->drawcolor);
                GR_GetMouse(w, &state, &x, &y);
                while (state & LEFTBUTTONACTIVE) {
/*
                    r.Xmin = x - (x % fatbitsize) + 1;
                    r.Ymin = y - (y % fatbitsize) + 1;
*/
                    r.Xmin = x - (x % fatbitsize);
                    r.Ymin = y - (y % fatbitsize);
                    if ((r.Xmin != oldx) || (r.Ymin != oldy)) {
                        r.Xmax = r.Xmin + fatbitsize - 1;
                        r.Ymax = r.Ymin + fatbitsize - 1;
                        GR_DrawSolidRect(w, &r);
                        oldx = r.Xmin;
                        oldy = r.Ymin;
                        /*
                        ** Set the icon pixmap also
                        */
                        SM_SetPixmapPixel(IconData->pm,
                                          x/fatbitsize,
                                          y/fatbitsize,
                                          IconData->drawcolor);
                    }
                    GR_GetMouse(w, &state, &x, &y);
                }
                break;
        }
        /*
        ** This value (True) will be returned to the event handler, and
        ** eventually returned to the event loop through SM_ProcessEvent in
        ** the main function
        */
        return True;
    }
    /*
    ** No other actions are relevant if the mouse button is pressed in the
    ** window content region. Return the type of event. This will eventually
    ** get passed back through the SM_ProcessEvent function in Main.
    */
    return ev->event_type;
}



void DrawLine(WindowType *w, IconEditData *IconData)
{
    int         state, x, y;
    int         oldx, oldy;
    int         orgx = 0, orgy = 0;
    double      nexty, slope;
    ColorType   oldcolor;
    int         dy;

    /*
    ** Trap the mouse here. As long as the mouse button is pressed and
    ** the mouse is inside the cursor keep drawing inside the canvas.
    */
    GR_GetMouse(w, &state, &orgx, &orgy);
    GR_SetDrawMode(w, SMXOR);
    memcpy(&oldcolor, GR_GetDrawColor(w), sizeof(ColorType));
    GR_SetDrawColor(w, SM_GetSystemColor(SMWHITE));
    oldx = orgx;
    oldy = orgy;
    GR_DrawLine(w, orgx, orgy, oldx, oldy);
    while (state & LEFTBUTTONACTIVE) {
        GR_GetMouse(w, &state, &x, &y);
        if ((x != oldx) || (y != oldy)) {
            GR_DrawLine(w, orgx, orgy, oldx, oldy);
            GR_DrawLine(w, orgx, orgy, x, y);
            oldx = x;
            oldy = y;
        }
    }
    GR_DrawLine(w, orgx, orgy, x, y);
    GR_SetDrawMode(w, SMREP);

    /*
    ** Find all the fatbit pixels on the path and light them up.
    ** First sort endpoints in X-ascending order
    */
    if (x < orgx) {
        oldx = orgx;
        orgx = x;
        oldy = orgy;
        orgy = y;
    }

    /*
    ** Find the slope of the line
    */
    if ((orgy - oldy) == 0) {
        slope = 9999;
    }
    else {
        slope = ((double)oldx - (double)orgx) / ((double)oldy - (double)orgy);
        dy = (slope < 0) ? -1 : 1;
        slope = fabs(slope);
    }

    /*
    ** Transform the graphics canvas points to pixmap points
    */
    orgx /= fatbitsize;
    orgy /= fatbitsize;
    oldx /= fatbitsize;
    oldy /= fatbitsize;
    y = orgy;

    /*
    ** Draw the line in the pixmap
    */
    nexty = -1;
    GR_SetDrawColor(w,&oldcolor);
    if (orgx == oldx) {
        if (orgy < oldy) {
            for (y = orgy; y <= oldy; y++) {
                SM_SetPixmapPixel(IconData->pm, orgx, y, IconData->drawcolor);
                SetFatbit(w, IconData->pm, orgx, y, IconData->drawcolor);
            }
        }
        else {
            for (y = oldy; y < orgy; y++) {
                SM_SetPixmapPixel(IconData->pm, orgx, y, IconData->drawcolor);
                SetFatbit(w, IconData->pm, orgx, y, IconData->drawcolor);
            }
        }
    }
    else if (orgy == oldy) {
        for (x = orgx; x <= oldx; x++) {
            SM_SetPixmapPixel(IconData->pm, x, orgy, IconData->drawcolor);
            SetFatbit(w, IconData->pm, x, orgy, IconData->drawcolor);
        }
    }
    else {
        for (x = orgx; x <= oldx; x++) {
            SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
            SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
            nexty += 1.0/slope;
            if (nexty >= 0.0) {
                y += dy;
                nexty -=1.0;
                while (nexty > 0.0) {
                    SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
                    SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
                    nexty -=1.0;
                    y += dy;
                    if (y == oldy) {
                        break;
                    }
                }
            }
        }
    }
/*
    DrawFatbitsFromPixmap(w, IconData);
*/
    GR_SetDrawColor(w,&oldcolor);
}

void DrawBox(WindowType *w, IconEditData *IconData)
{
    int         state, x, y;
    RectType    r, oldr;
    int         oldx, oldy;
    int         orgx, orgy;
    ColorType   oldcolor;

    /*
    ** Trap the mouse here. As long as the mouse button is pressed and
    ** the mouse is inside the cursor keep drawing inside the canvas.
    */
    GR_GetMouse(w, &state, &oldr.Xmin, &oldr.Ymin);
    oldr.Xmax = orgx = oldx = oldr.Xmin;
    oldr.Ymax = orgy = oldy = oldr.Ymin;
    GR_SetDrawMode(w, SMXOR);
    memcpy(&oldcolor, GR_GetDrawColor(w), sizeof(ColorType));
    memcpy(&r, &oldr, sizeof(RectType));
    GR_SetDrawColor(w, SM_GetSystemColor(SMWHITE));
    GR_DrawRect(w, &oldr);
    while (state & LEFTBUTTONACTIVE) {
        GR_GetMouse(w, &state, &x, &y);
        if ((x != oldx) || (y != oldy)) {
            if (x < orgx) {
                r.Xmin = x;
                r.Xmax = orgx;
            }
            else {
                r.Xmax = x;
                r.Xmin = orgx;
            }
            if (y < orgy) {
                r.Ymax = orgy;
                r.Ymin = y;
            }
            else {
                r.Ymax = y;
                r.Ymin = orgy;
            }
            GR_DrawRect(w, &oldr);
            GR_DrawRect(w, &r);
            oldx = x;
            oldy = y;
            memcpy(&oldr, &r, sizeof(RectType));
        }
    }
    GR_DrawRect(w, &oldr);
    GR_SetDrawMode(w, SMREP);

    oldr.Xmin /= fatbitsize;
    oldr.Xmax /= fatbitsize;
    oldr.Ymax /= fatbitsize;
    oldr.Ymin /= fatbitsize;

    for (x = oldr.Xmin, y = oldr.Ymin; x < oldr.Xmax; x++) {
        SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
        SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
    }
    for (x = oldr.Xmax, y = oldr.Ymin; y < oldr.Ymax; y++) {
        SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
        SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
    }
    for (x = oldr.Xmax, y = oldr.Ymax; x > oldr.Xmin; x--) {
        SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
        SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
    }
    for (x = oldr.Xmin, y = oldr.Ymax; y > oldr.Ymin; y--) {
        SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
        SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
    }
}



/*****************************************************************************
**
** DrawCircle
*/
void DrawCircle(WindowType *w, IconEditData *IconData)
{
    int         state, x, y, cx, cy, radx, rady;
    RectType    r, oldr;
    int         oldx, oldy;
    int         orgx = 0, orgy = 0;
    double      nexty, slope;
    ColorType   oldcolor;

    /*
    ** Trap the mouse here. As long as the mouse button is pressed and
    ** the mouse is inside the cursor keep drawing inside the canvas.
    */
    GR_GetMouse(w, &state, &oldr.Xmin, &oldr.Ymin);
    /*
    ** Set up the initial (probably degenerate) rectangle
    */
    oldr.Xmax = orgx = oldx = oldr.Xmin;
    oldr.Ymax = orgy = oldy = oldr.Ymin;
    memcpy(&r, &oldr, sizeof(RectType));

    GR_SetDrawMode(w, SMXOR);

    /*
    ** Save the current drawing color; we'll restore it later
    */
    memcpy(&oldcolor, GR_GetDrawColor(w), sizeof(ColorType));
    GR_SetDrawColor(w, SM_GetSystemColor(SMWHITE));

    /*
    ** Center the circle inside the rectangle dragged by the user.  The
    ** circle radius is half the rectangle side; use the shortest side as
    ** the radius so it's always inside the rect.
    */
    radx = (oldr.Xmax - oldr.Xmin)/2;
    rady = (oldr.Ymax - oldr.Ymin)/2;
    cx = oldr.Xmin + radx;
    cy = oldr.Ymin + rady;
    GR_DrawRect(w, &oldr);
    if (rady < radx) {
        radx = rady;
    }
    GR_DrawCircle(w, cx, cy, radx);
    while (state & LEFTBUTTONACTIVE) {
        GR_GetMouse(w, &state, &x, &y);
        if ((x != oldx) || (y != oldy)) {
            if (x < orgx) {
                r.Xmin = x;
                r.Xmax = orgx;
            }
            else {
                r.Xmax = x;
                r.Xmin = orgx;
            }
            if (y < orgy) {
                r.Ymax = orgy;
                r.Ymin = y;
            }
            else {
                r.Ymax = y;
                r.Ymin = orgy;
            }
            radx = (oldr.Xmax - oldr.Xmin)/2;
            rady = (oldr.Ymax - oldr.Ymin)/2;
            cx = oldr.Xmin + radx;
            cy = oldr.Ymin + rady;
            if (rady < radx) {
                radx = rady;
            }
            GR_DrawRect(w, &oldr);
            GR_DrawCircle(w, cx, cy, radx);
            radx = (r.Xmax - r.Xmin) / 2;
            rady = (r.Ymax - r.Ymin) / 2;
            cx = r.Xmin + radx;
            cy = r.Ymin + rady;
            if (rady < radx) {
                radx = rady;
            }
            GR_DrawRect(w, &r);
            GR_DrawCircle(w, cx, cy, radx);
            oldx = x;
            oldy = y;
            memcpy(&oldr, &r, sizeof(RectType));
        }
    }
    radx = (oldr.Xmax - oldr.Xmin)/2;
    rady = (oldr.Ymax - oldr.Ymin)/2;
    cx = oldr.Xmin + radx;
    cy = oldr.Ymin + rady;
    GR_DrawRect(w, &oldr);
    if (rady < radx) {
        radx = rady;
    }
    GR_DrawCircle(w, cx, cy, radx);
    GR_SetDrawMode(w, SMREP);
    cx /= fatbitsize;
    cy /= fatbitsize;
    radx /= fatbitsize;
    slope = 6.283;
    if (radx) {
        for (nexty = 0.0; nexty < slope; nexty += slope/(slope*radx*2)) {
            x = (int)((double)radx * cos(nexty) + cx + 0.5);
            y = (int)((double)radx * sin(nexty) + cy + 0.5);
            if ((x < 0) || (x > IconData->isize) || (y < 0) || (y > IconData->isize)) {
                continue;
            }
            SM_SetPixmapPixel(IconData->pm, x, y, IconData->drawcolor);
            SetFatbit(w, IconData->pm, x, y, IconData->drawcolor);
        }
    }
}




/*****************************************************************************
**
** SetFatbit
**
*/
void SetFatbit(WindowType *w, PixmapType *pm, int x, int y, ColorType *c)
{
    RectType    r;

/*
    r.Xmin = x*fatbitsize + 1;
    r.Ymin = y*fatbitsize + 1;
    r.Xmax = r.Xmin + fatbitsize - 1;
    r.Ymax = r.Ymin + fatbitsize - 1;
*/
    r.Xmin = x*fatbitsize;
    r.Ymin = y*fatbitsize;
    r.Xmax = r.Xmin + fatbitsize - 1;
    r.Ymax = r.Ymin + fatbitsize - 1;
    GR_SetDrawColor(w, c);
    GR_DrawSolidRect(w, &r);
}



int About(WindowType *w)
{
    WindowType  *about;
    RectType    rt;

    rt.Xmin = 50;
    rt.Ymin = 175;
    rt.Xmax = rt.Xmin + 400;
    rt.Ymax = rt.Ymin + 175;
    about = SM_NewWindow(&rt, "About IconEdit", DIALOG | NOBACKING, NULL, NULL);
    rt.Xmin = 1;
    rt.Xmax = -999;
    rt.Ymin = -200;
    rt.Ymax = -275;
    SM_CreateLabel(about,&rt,"ICONEDIT",NULL,ALIGNCENTER,False,False,False,NULL);
    rt.Ymin = rt.Ymax - 10;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "Icon Editor for the TWS Window System",
                   NULL, ALIGNCENTER, False, False, False,NULL);
    rt.Ymin = rt.Ymax - 175;
    rt.Ymax = rt.Ymin - 75;
    SM_CreateLabel(about, &rt,
                   "Release 4.0",
                   NULL, ALIGNCENTER, False, False, False, NULL);

    rt.Xmin = -400;
    rt.Xmax = -600;
    rt.Ymax = SM_GetContentDepth(about) - 10;
    rt.Ymin = rt.Ymax - 30;
    SM_CreateButton(about,
                    &rt,
                    "OK",
                    NULL,
                    CloseAbout);
    SM_OpenWindow(about);
    return 0;
}


int CloseAbout(ButtonType *b)
{
    SM_CloseWindow(SM_GetGadgetWindow(b));
    return True;
}


int SetDrawColor(ButtonType *b)
{
    WindowType  *w;
    IconEditData    *IconData;

    cindex = (int)SM_GetButtonData(b);
    w = SM_GetGadgetWindow(b);
    IconData = (IconEditData *)SM_GetUserData(w);
    IconData->drawcolor = SM_GetSystemColor(cindex);
    SM_SetGadgetRedrawFlag(forelabel, True);
    SM_RefreshGadgets(w);
    return 0;
}


int SetBackColor(ButtonType *b)
{
    WindowType      *w;
    ColorType       curr_bg, curr_pix, new_bg;
    IconEditData    *IconData;
    int             x, y;
    RectType        r;

    /*
    ** The button data value is the index for the new background color
    */
    bindex = (int)SM_GetButtonData(b);

    /*
    ** Get the button's parent window, which contains the icon data struct
    */
    w = SM_GetGadgetWindow(b);
    IconData = (IconEditData *)SM_GetUserData(w);

    /*
    ** Store the current background color and get the new one from the system
    */
    memcpy(&curr_bg, IconData->backcolor, sizeof(ColorType));
    memcpy(&new_bg, SM_GetSystemColor(bindex), sizeof(ColorType));
    IconData->backcolor = SM_GetSystemColor(bindex);

    GR_SetBlocking(w);
    /*
    ** Now set all current background pixels in pixmap to this color
    */
    for (y = 0; y < IconData->isize; y++) {
        for (x = 0; x < IconData->isize; x++) {
            SM_GetPixmapPixel(IconData->pm, x, y, &curr_pix);
            if (SM_IsColorEqual(&curr_pix, &curr_bg)) {
                /*
                ** Change colors in the pixmap that are the old background
                ** color to the new background color
                */
                SM_SetPixmapPixel(IconData->pm, x, y, &new_bg);
/*
                r.Xmin = x*fatbitsize + 1;
                r.Ymin = y*fatbitsize + 1;
*/
                r.Xmin = x*fatbitsize;
                r.Ymin = y*fatbitsize;
                r.Xmax = r.Xmin + fatbitsize - 1;
                r.Ymax = r.Ymin + fatbitsize - 1;
                GR_SetDrawColor(w, &new_bg);
                GR_DrawSolidRect(w, &r);
            }
            else if (SM_IsColorEqual(&curr_pix, &new_bg)) {
                /*
                ** So we don't lose pieces of the drawing that were drawn
                ** in the color that's going to become the background, set
                ** any pixels that are already the new background color to
                ** the old background color
                */
                SM_SetPixmapPixel(IconData->pm, x, y, &curr_bg);
/*
                r.Xmin = x*fatbitsize + 1;
                r.Ymin = y*fatbitsize + 1;
*/
                r.Xmin = x*fatbitsize;
                r.Ymin = y*fatbitsize;
                r.Xmax = r.Xmin + fatbitsize - 1;
                r.Ymax = r.Ymin + fatbitsize - 1;
                GR_SetDrawColor(w, &curr_bg);
                GR_DrawSolidRect(w, &r);
            }
        }
    }
    GR_UnsetBlocking();
    SM_SetGadgetRedrawFlag(backlabel, True);
    SM_SetGadgetRedrawFlag(forelabel, True);
    SM_RefreshGadgets(w);
    return 0;
}




int DrawColorLabel(LabelType *label)
{
    WindowType  *w;
    int         ndx;

    w = SM_GadgetToWindow(label);
    ndx = *(int *)SM_GetLabelData(label);
    SM_SetContentColor(w, SM_GetSystemColor(ndx));
    SM_EraseContent(w);
    SM_FreeWindow(w);
    return 0;
}





/****************************************************************************
**
** ClearPixmap
**
** Set the pixmap back to the graphics background color
*/
int ClearPixmap(WindowType *w)
{
    IconEditData    *IconData;
    int             x, y;
    RectType        r;

    /*
    ** Get the IconData struct from the user data field of the window
    */
    IconData = (IconEditData *)SM_GetUserData(w);

    /*
    ** Set blocking for speedup
    */
    GR_SetBlocking(w);

    /*
    ** Now set the whole pixmap to this color
    */
    for (y = 0; y < IconData->isize; y++) {
        for (x = 0; x < IconData->isize; x++) {
            SM_SetPixmapPixel(IconData->pm, x, y, IconData->backcolor);
/*
            r.Xmin = x*fatbitsize + 1;
            r.Ymin = y*fatbitsize + 1;
*/
            r.Xmin = x*fatbitsize;
            r.Ymin = y*fatbitsize;
            r.Xmax = r.Xmin + fatbitsize - 1;
            r.Ymax = r.Ymin + fatbitsize - 1;
            GR_SetDrawColor(w, IconData->backcolor);
            GR_DrawSolidRect(w, &r);
        }
    }
    GR_UnsetBlocking();
    return 0;
}


/***************************************************************************
**
** WritePixmap
**
** Writes a pixmap to a disk file
*/
int WritePixmap(WindowType *w)
{
    char            *fname;
    char            msg[100];
    FILE            *f;
    int             rsp;
    IconEditData    *icondata;

    icondata = (IconEditData *)SM_GetUserData(w);
    /*
    ** Get a file from the user
    */
    fname = SM_GetFilename("*.icn");
    if (fname == NULL) {
        return 1;
    }
    /*
    ** If the file already exists then warn the user before overwriting
    */
    if ((f = fopen(fname, "r")) != NULL) {
        fclose(f);
        sprintf(msg, "OK to overwrite %s?",fname);
        rsp = SM_YesNoDialog(msg, "OK", "Cancel");
        if (rsp) {
            free(fname);
            return 1;
        }

    }
    /*
    ** Write the pixmap to the file
    */
    if (SM_WritePixmap(icondata->pm, fname)) {
        sprintf("Error writing pixmap file %s",fname);
        SM_OKDialog(msg);
        free(fname);
        return 1;
    }
    icondata->modified = False;
    sprintf(msg, "Iconedit: %s",fname);
    SM_SetWindowTitle(w, msg);
    free(fname);
    return 0;
}


/***************************************************************************
**
** ReadPixmap
**
** Writes a pixmap to a disk file
*/
int ReadPixmap(WindowType *w)
{
    char            *fname;
    char            msg[100];
    int             rsp;
    IconEditData    *icondata;
    PixmapType      *pm;
    ColorType       c;
    int             x, y;
    char            file[MAXFILE], ext[MAXEXT];

    icondata = (IconEditData *)SM_GetUserData(w);
    if (icondata->modified) {
        sprintf(msg,"Discard changes to %s",icondata->filename);
        if (SM_YesNoDialog(msg, "Yes", "Cancel")) {
            return 1;
        }
    }
    rsp = 0;
    while (!rsp) {
        /*
        ** Get a file from the user
        */
        fname = SM_GetFilename("*.icn");
        if (fname == NULL) {
            return 1;
        }
        if (icondata->filename) {
            free(icondata->filename);
        }
        fnsplit(fname,NULL, NULL, file, ext);
        icondata->filename = (char *)malloc(strlen(file) + strlen(ext) + 1);
        strcpy(icondata->filename,file);
        strcat(icondata->filename,ext);
        /*
        ** Load the requested pixmap
        */
        pm = SM_ReadPixmap(fname);
        if (pm == NULL) {
            sprintf(msg,"Error reading pixmap file %s",fname);
            rsp = SM_YesNoDialog(msg, "Retry", "Cancel");
            if (rsp) {
                return 1;
            }
        }
        else {
            rsp = 1;
        }
    }

    /*
    ** Make sure this pixmap is the right size
    */
    if ((SM_GetPixmapWidth(pm) != SM_GetPixmapWidth(icondata->pm))
    || (SM_GetPixmapDepth(pm)) != SM_GetPixmapDepth(icondata->pm)) {
        sprintf(msg, "Pixmap is not correct size for this icon.  Need %dx%d, got %dx%d",
                SM_GetPixmapWidth(icondata->pm), SM_GetPixmapDepth(icondata->pm),
                SM_GetPixmapWidth(pm), SM_GetPixmapDepth(pm));
        SM_OKDialog(msg);
        SM_DestroyPixmap(pm);
        return 1;
    }
    /*
    ** Looks OK, replace the current pixmap with the new one.  All we
    ** have to do is replace the pixmap image.  To do this, traverse
    ** the new pixmap image and set the corrensponding pixels on the
    ** icon pixmap.  Block the existing pixmap so we don't do an image
    ** copy for each pixel.
    */
    SM_SetPixmapBlocking(icondata->pm, True);
    for (y = 0; y < SM_GetPixmapDepth(icondata->pm); y++) {
        for (x = 0; x < SM_GetPixmapWidth(icondata->pm); x++) {
            SM_GetPixmapPixel(pm, x, y, &c);
            SM_SetPixmapPixel(icondata->pm, x, y, &c);
        }
    }
    SM_SetPixmapBlocking(icondata->pm, False);
    SM_DestroyPixmap(pm);
    /*
    ** Redraw the icon editor graphics display
    */
    DrawEditWin(w);
    sprintf(msg,"Iconedit: %s",icondata->filename);
    SM_SetWindowTitle(w,msg);
    return 0;
}


/***************************************************************************
**
** SavePixmap
**
** Writes the pixmap to a file
*/
int SavePixmap(WindowType *w)
{
    char            msg[100];
    FILE            *f;
    int             rsp;
    IconEditData    *icondata;

    icondata = (IconEditData *)SM_GetUserData(w);
    if (icondata->filename == NULL) {
        return WritePixmap(w);
    }
    if ((f = fopen(icondata->filename, "r")) == NULL) {
        sprintf(msg, "WARNING : File %s no longer exists! OK to continue writing?",icondata->filename);
        rsp = SM_YesNoDialog(msg, "Continue", "Cancel");
        if (rsp) {
            return 1;
        }
    }
    fclose(f);
    /*
    ** Write the pixmap to the file
    */
    if (SM_WritePixmap(icondata->pm, icondata->filename)) {
        sprintf("Error writing pixmap file %s",icondata->filename);
        SM_OKDialog(msg);
        return 1;
    }
    icondata->modified = False;
    return 0;
}



/***************************************************************************
**
** SetTool
*/
int SetTool(ButtonType *b)
{
    WindowType      *w;
    IconEditData    *icondata;
    int             tool;

    w = SM_GetGadgetWindow(b);
    icondata = (IconEditData *)SM_GetUserData(w);
    tool = (int)SM_GetButtonData(b);
    icondata->tool = tool;
    if (b->pixmap) {
        SM_CopyPixmap(SelectPixmap, b->pixmap);
    }
    return 0;
}



int DisplayHelp(WindowType *w)
{
    WindowType  *works;
    RectType    rt;
    TextboxType *tb;

    rt.Xmin = rt.Ymin = 150;
    rt.Xmax = rt.Xmin + 500;
    rt.Ymax = rt.Ymin + 350;
    works = SM_NewWindow(&rt, "iconedit.hlp", DOCUMENT | NOBACKING, NULL, NULL);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(works) - 2;
    rt.Ymax = SM_GetContentDepth(works) - 2;
    tb = SM_CreateTextbox(works,
                          &rt,
                          "iconedit.hlp",
                          SM_GetSystemFont(),
                          True,
                          True,
                          NULL);
    SM_SetResizeProc(works, ResizeText);
    SM_SetUserData(works, (void *)tb);
    SM_OpenWindow(works);
    return 0;
}


int ResizeText(WindowType *w)
{
    TextboxType *tb;
    RectType    rt;

    tb = (TextboxType *)SM_GetUserData(w);
    rt.Xmin = rt.Ymin = 1;
    rt.Xmax = SM_GetContentWidth(w) - 2;
    rt.Ymax = SM_GetContentDepth(w) - 2;
    SM_SetTextboxRect(tb, &rt);
    return 0;
}

void PrntScrn(WindowType *w, RectType *r, char *fname)
{
    SM_SaveScreenAsPCX(fname);
}


