//
// Streamable Palette Example
//
// Copyright (c) 1992 CC Software, Inc.
//
//
// REQUIRED FILES:
//
//      EXAMPLE.CPP
//      TVPAL.CPP
//      TVPAL.H
//
// This example program demonstrates the way the class TStreamPalette
// can be used to store the application palette on a stream.  The
// stream file, PALETTE.STM, is checked for existence.  If it exists,
// the application palette is read from the stream using the class
// TStreamPalette.  Normal operation follows.  Upon exit of the app
// the palette is stored back on the stream.
//
// NOTES:
//
// Although the application palette is being stored here, ANY object
// that uses a palette can make use of the TStreamPalette class as
// a way of write/reading the object palette with fpstreams.
//

#define Uses_TKeys
#define Uses_TApplication
#define Uses_TDeskTop
#define Uses_TMenuBar
#define Uses_TSubMenu
#define Uses_TMenuItem
#define Uses_TColorDialog
#define Uses_TColorGroup
#define Uses_TColorItem
#define Uses_TStreamableClass
#define Uses_TResourceCollection
#define Uses_TResourceFile
#define Uses_fpstream
#include <tv.h>
#include "TVPal.h"

__link(RStreamPalette);         // Link in the stream manager record
                                // from TVPAL.CPP.

__link(RResourceCollection);    // Link in the stream manager record
                                // for class TResourceCollection


// Application Commands //

const int cmColorCmd    = 109;


// Palette File Name //

#define RESOURCE   "example.rez"


// Class TApp //

// Description --------------------------------------------------------
//
//      TApp provides the application class for the program.
//
// Member functions
//
//      initDeskTop
//
//      Creates a TDeskTop instance after reading the application
//      palette from the palette file (if it exists).
//
//      initMenuBar
//
//      Initalizes the menu bar for the application operations.
//
//      handleEvent
//
//      Handles the application commands.
//
//      colors
//
//      Opens the TColorDialog instance to allow the user to change the
//      application color palette.  Since the palette is saved during
//      it's destructor, the colors will be persistent between program
//      runs.
//
// END Description ----------------------------------------------------

class TApp : public TApplication {
public:
       TApp();
       ~TApp();

protected:
       static TDeskTop    *initDeskTop( TRect );
       static TMenuBar    *initMenuBar( TRect );

       virtual void handleEvent( TEvent& );
       virtual void colors();
};
// END TApp ===========================================================




// Constructor //

TApp::TApp() : TApplication(), TProgInit( TApp::initStatusLine,
                                          TApp::initMenuBar,
                                          TApp::initDeskTop )
// Summary ------------------------------------------------------------
//
//      Creates the application object.
//
// END Summary --------------------------------------------------------
{
};
// --------------------------------------------------------------------



// Destructor //

TApp::~TApp()
// Summary ------------------------------------------------------------
//
//      Destructor for the TApp class.  Here we attempt to write the
//  application palette to the stream file.  Normal stream operation
//  would need some kludgy code to do this with TPalette.  The class
//  TStreamPalette makes for an ellegant solution for both reading and
//  writing TPalette objects.
//
// END Summary --------------------------------------------------------
{
  fpstream *fp = new fpstream( RESOURCE, ios::out | ios::binary );
  TStreamPalette *sp = new TStreamPalette( getPalette() );
  TResourceFile *res;

  if ( fp->good() )
  {
     res = new TResourceFile( fp );
     if ( res )
        res->put( sp, "Palette" );
  }
  if ( sp ) delete sp;
  if ( res ) destroy( res );
}
// --------------------------------------------------------------------




// Member funciton //

TMenuBar *TApp::initMenuBar( TRect r )
// Summary ------------------------------------------------------------
//
//      Creates the applicaiton menu bar.
//
// Parameters
//
//      r
//
//      A TRect object containing the bounds of the display.
//
// Return Value
//
//      Returns a TMenuBar object.
//
// END Summary --------------------------------------------------------
{
  r.b.y = r.a.y + 1;
  return new TMenuBar( r,
              *new TSubMenu( "~F~ile", kbAltF ) +
              *new TMenuItem( "E~x~it", cmQuit, kbAltX, hcNoContext, "Alt-X" ) +
              *new TSubMenu( "~O~ptions", kbAltO ) +
              *new TMenuItem( "~C~olors", cmColorCmd, kbNoKey, hcNoContext )
             );
};
// --------------------------------------------------------------------




// Member function //

TDeskTop *TApp::initDeskTop( TRect bounds )
{
  fpstream *fp = new fpstream( RESOURCE, ios::nocreate | ios::in | ios::binary );
  TResourceFile *res;
  TStreamPalette *sp;

  if ( fp->good() )
  {
     res = new TResourceFile( fp );
     if ( res )
     {
        sp = (TStreamPalette *)res->get( "Palette" );
        if ( sp )
        {
           application->getPalette() = *sp;
           delete sp;
        };
        delete res;
     }
     else delete fp;
  };
  bounds.grow( 0, -1 );
  return new TDeskTop( bounds );
};
// --------------------------------------------------------------------




// Member function //

void TApp::handleEvent( TEvent& event )
{
  TApplication::handleEvent( event );
  if ( event.what == evCommand )
     switch ( event.message.command )
     {
            case cmColorCmd:
                 colors();
                 clearEvent( event );
                 return;
     };
  clearEvent( event );
};
// --------------------------------------------------------------------




// Member function //

void TApp::colors()
{
    TColorGroup &group1 =
        *new TColorGroup("Desktop") +
            *new TColorItem("Color",             1)+

        *new TColorGroup("Menus") +
            *new TColorItem("Normal",            2)+
            *new TColorItem("Disabled",          3)+
            *new TColorItem("Shortcut",          4)+
            *new TColorItem("Selected",          5)+
            *new TColorItem("Selected disabled", 6)+
            *new TColorItem("Shortcut selected", 7);

    TColorGroup &group2 =
        *new TColorGroup("Dialogs") +
            *new TColorItem("Frame/background",  33)+
            *new TColorItem("Frame icons",       34)+
            *new TColorItem("Scroll bar page",   35)+
            *new TColorItem("Scroll bar icons",  36)+
            *new TColorItem("Static text",       37)+

            *new TColorItem("Label normal",      38)+
            *new TColorItem("Label selected",    39)+
            *new TColorItem("Label shortcut",    40);

    TColorItem &item_coll1 =
        *new TColorItem("Button normal",     41)+
        *new TColorItem("Button default",    42)+
        *new TColorItem("Button selected",   43)+
        *new TColorItem("Button disabled",   44)+
        *new TColorItem("Button shortcut",   45)+
        *new TColorItem("Button shadow",     46)+
        *new TColorItem("Cluster normal",    47)+
        *new TColorItem("Cluster selected",  48)+
        *new TColorItem("Cluster shortcut",  49);

    TColorItem &item_coll2 =
        *new TColorItem("Input normal",      50)+
        *new TColorItem("Input selected",    51)+
        *new TColorItem("Input arrow",       52)+

        *new TColorItem("History button",    53)+
        *new TColorItem("History sides",     54)+
        *new TColorItem("History bar page",  55)+
        *new TColorItem("History bar icons", 56)+

        *new TColorItem("List normal",       57)+
        *new TColorItem("List focused",      58)+
        *new TColorItem("List selected",     59)+
        *new TColorItem("List divider",      60)+

        *new TColorItem("Information pane",  61
        );

     group2 = group2 + item_coll1 + item_coll2;

    TColorGroup &group3 = group1 + group2;

    TColorDialog *c = new TColorDialog(&getPalette(), &group3 );

    c->helpCtx = hcNoContext;
    if( validView( c ) != 0 )
    {
      c->setData(&getPalette());
      if( deskTop->execView( c ) != cmCancel )
      {
        getPalette() = *(c->pal);
        deskTop->redraw();
        redraw();
      }
      destroy( c );
    }
};
// --------------------------------------------------------------------




// Main program //

int main()
{
  TApp  app;
  app.run();
  return 0;
};
