// File    : TNUMVAL.CPP
// Author  : Eric Woodruff,  CIS ID: 72134,1150
// Updated : Thu 07/21/94 21:01:41
// Note    : Hereby declared public domain
// Compiler: Borland C++ 4.02
//
// This is a derived TPXPictureValidator class that will allow TInputLine to
// return its data as a long, short, unsigned short, int, or unsigned int
// by setting the voTransfer bit in the validator's 'options' data member.
// Due to the signed short and signed int types, this class will not handle
// unsigned longs.  It also does not attempt to handle type double values
// to avoid linking in the floating point library when it isn't needed.
// Note that short and int are treated as separate types due to the fact
// that int values are 32 bits long in 32 bit DPMI whereas shorts remain
// 16 bits long.  It is derived from TPXPictureValidator for proper handling
// of +/- when a sign is allowed.
//
// NOTE:  This class automatically sets the voTransfer bit so that the
//        data is automatically returned as the specified type.  To return
//        a character string, you must manually turn the bit off.  However,
//        turning it off makes the class not that much better than a basic
//        TPXPictureValidator.
//

#include <stdio.h>
#include <string.h>

#define Uses_MsgBox
#define Uses_TPXPictureValidator
#include <tv.h>

#define Uses_TNumericValidator
#include <custmval.h>

// Error message string.
const char * _NEAR TNumericValidator::errorMsg =
    "Value not in the range %ld to %ld";

TNumericValidator::TNumericValidator(long aMin, long aMax, numType aType,
  Boolean autoFill) : TPXPictureValidator(NULL, autoFill),
    min(aMin),
    max(aMax),
    type(aType)
{
    // Assume that the user will want a numeric data type returned.
    options |= voTransfer;

    // If min or max is less than zero, allow a sign.
    if(aMin < 0L || aMax < 0L)
        pic = newStr("[+,-]*#");
    else
        pic = newStr("*#");

    // Status *must* be set to vsOk here because the NULL picture
    // passed to the base TPXPictureValidator structure sets it to vsSyntax
    // which indicates an error.  If you leave it like that, validView()
    // interprets it as meaning something is wrong and the dialog
    // will not get executed.
    status = vsOk;
}

// Emulate the TRangeValidator error function.
void TNumericValidator::error(void)
{
    messageBox(mfError | mfOKButton, errorMsg, min, max);
}

ushort TNumericValidator::transfer(char* s, void* buffer, TVTransfer flag)
{
    long   lValue;
    int    iValue;
    uint   uiValue;
    short  sValue;
    ushort usValue, varSize;

    // Set the return size.
    switch(type)
    {
        case nShort:
        case nUShort:
            varSize = sizeof(short);    // Always 16 bits.
            break;

        case nInt:
        case nUInt:
            varSize = sizeof(int);      // 16 bit for real mode & 16 bit DPMI
            break;                      // 32 bit for 32 bit DPMI

        case nLong:
            varSize = sizeof(long);     // Can't do unsigned longs with this
            break;                      // class because of the other signed
                                        // types.
        default:
            varSize = 0;
            break;
    }

    // Force size to 0 if voTransfer is not set.
    if(!(options & voTransfer))
        varSize = 0;

    if(varSize && flag != vtDataSize)
        switch(type)
        {
            case nShort:
                if(flag == vtGetData)
                {
                    sscanf(s, "%d", &sValue);
                    *(short *)buffer = sValue;
                }
                else
                    sprintf(s, "%d", *(short *)buffer);
                break;

            case nUShort:
                if(flag == vtGetData)
                {
                    sscanf(s, "%u", &usValue);
                    *(ushort *)buffer = usValue;
                }
                else
                    sprintf(s, "%u", *(ushort *)buffer);
                break;

            case nInt:
                if(flag == vtGetData)
                {
                    sscanf(s, "%d", &iValue);
                    *(int *)buffer = iValue;
                }
                else
                    sprintf(s, "%d", *(int *)buffer);
                break;

            case nUInt:
                if(flag == vtGetData)
                {
                    sscanf(s, "%u", &uiValue);
                    *(uint *)buffer = uiValue;
                }
                else
                    sprintf(s, "%u", *(int *)buffer);
                break;

            case nLong:
                if(flag == vtGetData)
                {
                    sscanf(s, "%ld", &lValue);
                    *(long*)buffer = lValue;
                }
                else
                    sprintf(s, "%ld", *(long *)buffer);
                break;
        }

    return varSize;
}

Boolean TNumericValidator::isValid(const char* s)
{
    long  value;

    // Any of the other types can be tested as signed long here.
    sscanf(s, "%ld", &value);

    // See if it's within limits.
    if(value < min || value > max)
        return False;

    // If it is, let the base TPXPictureValidator have its say.
    return TPXPictureValidator::isValid(s);
}

#if !defined(NO_STREAMABLE)

TNumericValidator::TNumericValidator( StreamableInit s) : TPXPictureValidator(s)
{
}

void TNumericValidator::write(opstream& os)
{
    TPXPictureValidator::write( os );
    os << min << max << short(type);
}

void *TNumericValidator::read( ipstream& is )
{
    short temp;

    TPXPictureValidator::read( is );
    is >> min >> max >> temp;
    type = numType(temp);

    return this;
}

TStreamable *TNumericValidator::build()
{
    return new TNumericValidator( streamableInit );
}

#endif
