/*
*+-------------------------------------------------------------------------
*| File.........: TIMECLS.CPP
*| Date.........: Wed  12-2-1992
*| Author.......: Anderson P. Bryan
*| Copyright....: None! Use freely.
*| Version......: 1.0    Compile w/ BC++ 3.0
*| Usage........: General purpose time conversion, arithmetic, comparision
*|                and formatting class
*|
*| See TIMECLS.H for acknowledgements.
*+-------------------------------------------------------------------------
*/

#ifndef TIMECLS_CPP

#include "timecls.h"
#include <stdlib.h>
#include <string.h>
#include <mem.h>
#include <strstream.h>
#include <iomanip.h>

int Time :: DisplayFormat = HMS;
unsigned Time :: DisplayOptions = 0;

Time :: Time( void ) : hour(0), minute(0), second(0), ticks(0)
{
}

Time :: Time( const long t ) : ticks(t)
{
    ticks_to_hms();
}

Time :: Time( const int h, const int m, const int s ) :
              hour(h), minute(m), second(s)
{
    hms_to_ticks();
}

Time :: Time( const char *szTime )
{
    if( !stricmp( szTime, "NOW" ) )
    {
        struct dostime_t dt;
        _dos_gettime( &dt );
        hour = dt.hour;
        minute = dt.minute;
        second = dt.second;
    }
    else
    {
        char *sz = strdup( szTime );
        hour = atoi( strtok( sz, ":" ) );
        minute = atoi( strtok( NULL, ":" ) );
        second = atoi( strtok( NULL, ":" ) );
    }

    hms_to_ticks();
}

Time :: Time( const dostime_t &dt )
{
    hour = dt.hour;
    minute = dt.minute;
    second = dt.second;

    hms_to_ticks();
}

Time :: Time( const Time& tm )
{
    hour = tm.hour;
    minute = tm.minute;
    second = tm.second;

    hms_to_ticks();
}


///////////////////
// Time Arithmetic
///////////////////

Time& Time :: operator + ( const long i )
{
    Time *pt = new Time( ticks + i );
    return( *pt );
}

Time& Time :: operator + ( const int i )
{
    Time *pt = new Time( ticks + (long) i );
    return( *pt );
}

Time& Time :: operator - ( const long i )
{
    Time *pt = new Time( ticks - i );
    return( *pt );
}

Time& Time :: operator - ( const int i )
{
    Time *pt = new Time( ticks - (long) i );
    return( *pt );
}

Time& Time :: operator += ( const long i )
{
    ticks += i;
    ticks_to_hms();
    return( *this );
}

Time& Time :: operator -= ( const long i )
{
    ticks -= i;
    ticks_to_hms();
    return( *this );
}

Time& Time :: operator ++ ( void )
{
    ticks++;
    ticks_to_hms();

    return( *this );
}

Time& Time :: operator ++ ( int )
{
    ticks++;
    ticks_to_hms();

    return( *this );
}

Time& Time :: operator -- ( void )
{
    ticks--;
    ticks_to_hms();

    return( *this );
}

Time& Time :: operator -- ( int )
{
    ticks--;
    ticks_to_hms();

    return( *this );
}


////////////////////////////
// Time Comparison
////////////////////////////

int operator < ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks < dt2.ticks );
}

int operator <= ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks <= dt2.ticks );
}

int operator > ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks > dt2.ticks );
}

int operator >= ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks >= dt2.ticks );
}

int operator == ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks == dt2.ticks );
}

int operator != ( const Time& dt1, const Time& dt2 )
{
    return( dt1.ticks != dt2.ticks );
}


////////////////////////////
// ostream operations
///////////////////////////

ostream &operator << ( ostream &os, const Time& tm )
{
    return( os << tm.formatTime() );
}

ostream &operator << ( ostream &os, const dostime_t &dt )
{
    return( os << (int) dt.hour << ':' << (int) dt.minute \
               << (int) dt.second << ':' << dt.hsecond );
}

char * Time :: formatTime( const int format ) const
{
    static char buf[40];

    memset( buf, 0, sizeof( buf ) ) ;
    ostrstream os( buf, sizeof( buf ) );
    if( !isValidHMS( hour, minute, second) )
    {
        os << "Invalid Time" << ends;
        return( buf );
    }

    switch( format )
    {
        case HMS:
             os << setw(2) << (int) (hour % 12 == 0 ? 12 : hour % 12 ) << ':'
                << setfill('0') << setw(2) << minute;
             if( !(DisplayOptions & NO_SECS) )
                 os << ':' << setfill('0') << setw(2) << second;
             if( !(DisplayOptions & NO_AMPM) )
                 os << (hour < 12 ? " AM" : " PM");
             break;
        case ENGLISH:
             os << "Not Implemented";
             break;
        case MILITARY:
        default:
            os << setfill('0') << setw(2) << hour << ':'
               << setw(2) << minute << ':';

            if( !(DisplayOptions & NO_SECS) )
               os << setw(2) << second;
            break;
    }

    os << ends;

    return( buf );
}

void Time :: setFormat( const int format )
{
    DisplayFormat = format;
}

void Time :: setOptions( const int options )
{
    DisplayOptions |= options;
}


//////////////////////////
// Read private variables
/////////////////////////

long Time :: tickTime( void ) const
{
    return( ticks );
}

int Time :: NHour( void ) const
{
    return( hour );
}

int Time :: NMinute( void ) const
{
    return( minute );
}

int Time :: NSecond( void ) const
{
    return( second );
}


/////////////////////////
// Information
////////////////////////

int Time :: isValidHMS( int h, int m, int s ) const
{
    if( h < 0 || h > 23 )
        return( 0 );
    if( m < 0 || m > 59 )
        return( 0 );
    if( s < 0 || s > 59 )
        return( 0 );
    return( 1 );
}

int Time :: isMidNight( void ) const
{
    return( (hour == 0) ? 1 : 0 );
}

int Time :: isNoon( void ) const
{
    return( (hour == 12) ? 1 : 0 );
}


//////////////////////////
// Misc
////////////////////////

dostime_t Time :: getTime( void ) const
{
    static dostime_t dt;
    dt.hour = hour;
    dt.minute = minute;
    dt.second = second;
    dt.hsecond = 0;

    return( dt );
}


/////////////////
// Assignment
////////////////

Time& Time :: Set()
{
    dostime_t dt;
    _dos_gettime( &dt );

    hour = dt.hour;
    minute = dt. minute;
    second = dt.second;

    hms_to_ticks();

    return( *this );
}

Time& Time :: Set( long t )
{
    ticks = t;
    ticks_to_hms();

    return( *this );
}

Time& Time :: Set( int h, int m, int s )
{
    if( isValidHMS( h, m, s ) )
    {
        hour = h;
        minute = m;
        second = s;

        hms_to_ticks();
    }
    else
    {
        hour = minute = second = 0;
        ticks = 0L;
    }

    return( *this );
}


Time& Time :: AddHours( int i )
{
    ticks += 3600L * i;

    ticks_to_hms();

    return( *this );
}

inline Time& Time :: AddMinutes( int i )
{
    ticks += 60L * i;

    ticks_to_hms();

    return( *this );
}

#endif