/*--------------------------------------------------------------------------*
 *
 *  Placed in Public Domain By Damir Nadramija.
 *
 *  Module:         TOOLS.C 
 *  Purpose:        Handy tool functions and routines
 *                                                                            
 *  Created by:     Damir Nadramija 
 *  Date:           November 3, 1989                                            
 *  Note:           Keyboard handling functions were taken and modified from
 *                  All Stevens "Extending Turbo C Professional" book (placed
 *                  in public domain).
 *--------------------------------------------------------------------------*/
#include <math.h>
#include <alloc.h>
#include <ctype.h>
#include <string.h>
#include <bios.h>
#include <mem.h>
#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include <dir.h>
#include <tools.h>

/* Globals resident in this module */
word NoCursor = 0, TallCursor = 0, ShortCursor = 0, OldCursor = 0;
int  DirectVideo = TRUE;

/* Local variables, arrays */
static struct text_info t_info;
static char far *VideoPtr     = MK_FP(0xB800, 0);
static int  far *KeyBuffHead  = MK_FP(0x40, 0x1a);
static int  far *KeyBuffTail  = MK_FP(0x40, 0x1c);


/*--------------------------------------------------------------------------*
 *  Function:       WritePrompt
 *  Description:    Prints a prompt on the screen
 *                                                                           
 *--------------------------------------------------------------------------*/
void WritePrompt(int row, int col, string prompt)
{
    gotoxy(col, row);
    cputs(prompt);

} /* WritePrompt */

/*--------------------------------------------------------------------------*
 *  Function:       WriteXY
 *  Description:    Prints a string in video memory at a selected location 
 *                  in a specified color 
 *                                                                           
 *--------------------------------------------------------------------------*/
void WriteXY(int col, int row, int color, int width, va_list arg_list, ...)
{
    va_list arg_ptr;
    string format = arg_list;
    char output[MAX_LINE_LEN];
    int len;

    va_start(arg_ptr, arg_list);
    vsprintf(output, format, arg_ptr);
    output[width] = 0;
    if ((len = strlen(output)) < width) setmem(&output[len], width - len, ' ');
    Cputs(output, color, col, row);

} /* WriteXY */

/*--------------------------------------------------------------------------*
 *  Function:       SetCursor
 *  Description:    Sets the shape of the cursor 
 *                                                                           
 *--------------------------------------------------------------------------*/
void SetCursor(word shape)
{
    union REGS reg;

    reg.h.ah = 1;
    reg.x.cx = shape;
    int86(0X10, &reg, &reg);

} /* SetCursor */

/*--------------------------------------------------------------------------*
 *  Function:       GetCursor
 *  Description:    Gets the shape of the cursor 
 *                                                                           
 *--------------------------------------------------------------------------*/
word GetCursor(void)
{
    union REGS reg;

    reg.h.ah = 3;
    reg.h.bh = 0;
    int86(0X10, &reg, &reg);
    return(reg.x.cx);

} /* GetCursor */

/*--------------------------------------------------------------------------*
 *  Function:       ChangeCursor
 *  Description:    Changes the cursor shape based on the current insert mode 
 *                                                                           
 *--------------------------------------------------------------------------*/
void ChangeCursor(insmode)
{
    if (insmode)
        SetCursor(TallCursor);
    else
        SetCursor(ShortCursor);

} /* ChangeCursor */

/*--------------------------------------------------------------------------*
 *  Function:       InitCursor
 *  Description:    Initializes the different cursor types 
 *                                                                           
 *--------------------------------------------------------------------------*/
void InitCursor(void)
{
    struct text_info ti;

    gettextinfo(&ti);
    OldCursor = GetCursor();
    if (ti.currmode == MONO)
    {
        ShortCursor = MONO_SHORT_CURSOR;
        TallCursor  = MONO_TALL_CURSOR ;
    }
    else
    {
        ShortCursor = CGA_SHORT_CURSOR;
        TallCursor  = CGA_TALL_CUROSR ;
    }
    NoCursor = 0x2000;

} /* InitCursor */

/*--------------------------------------------------------------------------*
 *  Function:       CursorOn
 *  Description:    Sets the shape of the cursor (using Turbo C function)
 *                                                                           
 *--------------------------------------------------------------------------*/
void CursorOn(void)
{
    _setcursortype(_NORMALCURSOR);

} /* CursorOn */

/*--------------------------------------------------------------------------*
 *  Function:       CursorOff
 *  Description:    Sets the shape of the cursor (using Turbo C function)
 *                                                                           
 *--------------------------------------------------------------------------*/
void CursorOff(void)
{
    _setcursortype(_NOCURSOR);

} /* CursorOff */

/*--------------------------------------------------------------------------*
 *  Function:       InvertColor
 *  Description:    Inverts color value
 *                                                                           
 *--------------------------------------------------------------------------*/
int InvertColor(int attr)
{ 
    int blink = 0, fg = 0, bg = 0;
    blink = attr & 128; 
    bg = (attr & 112) >> 4; 
    fg = attr & 15;
    if (fg > 8) fg -= 8;
    return (bg + (fg << 4) + blink);
}

/*--------------------------------------------------------------------------*
 *  Function:       instr                                                    
 *  Description:    Returns position of a character in the string            
 *                                                                           
 *--------------------------------------------------------------------------*/
int instr(char *instring, int c)
{ 
    int ic = 0, len = 0, in = 0, ch;

    len = strlen(instring);
    while ((!in) && (ic < len))
    { 
        ch = instring[ic];
        ic++;
        if (ch == c) in = ic;
    }
    return (in);

} /* End of instr */

/*--------------------------------------------------------------------------*
 *  Function:       strpos                                                   
 *  Description:    Returns position of a character in the string            
 *                                                                           
 *--------------------------------------------------------------------------*/
int strpos(char *instring, int c)
{ 
    int ic = 0, len = 0, in = 0, ch;

    len = strlen(instring);
    while ((!in) && (ic < len))
    { 
        ch = instring[ic];
        if (ch == c) in = ic;
        ic++;
    }
    return (in);

} /* End of strpos */

/*--------------------------------------------------------------------------*
 *  Function:       rightpad
 *  Description:    
 *                                                                           
 *--------------------------------------------------------------------------*/
string rightpad(string instring, int fieldlength)
{ 
    int len = 0, in = 0;

    len = strlen(instring);
    if (len < fieldlength)
    { 
        for (in = len; in < (fieldlength - 1); in++)  
            instring[in] = 32;
        instring[fieldlength - 1] = 0;
    }
    return(instring);
}

/*--------------------------------------------------------------------------*
 *  Function:       leftpad
 *  Description:    
 *                                                                           
 *--------------------------------------------------------------------------*/
string leftpad(string str, int fldlen)
{ 
    int len = 0;

    len = strlen(str);
    if (len < fldlen)
    { 
        memmove(&str[fldlen - len - 1], &str[0], len);
        memset(&str[0], 32, (fldlen - len - 1));
        str[fldlen - 1] = 0;
    }
    return(str);
}

/*--------------------------------------------------------------------------*
 *  Function:       ltrim                                                    
 *  Description:    Left trimming (returns a left trimmed copy of the string)
 *                                                                            
 *--------------------------------------------------------------------------*/
string ltrim(string instring)                                                 
{
    int ic = 0, ch, in = 0, len, newn = 0;
    string temp;

    len = strlen(instring);
    if ((temp = malloc(len + 2)) != NULL)
        BlankString(temp, len);
    else
        return(instring);

    while ((!in) && (ic < len))
    { 
        ch = instring[ic];
        ic++;
        if (ch != ' ')  
        {
            in = ic;
            break;
        }
    }
    for (ic = in - 1; ic < len; ic++, newn++)
       temp[newn] = instring[ic];
    temp[newn] = 0;
    return(temp);

} /* End of ltrim */

/*--------------------------------------------------------------------------*
 *  Function:       rtrim                                                    
 *  Description:    Right trimming (returns a right trimmed copy of the string)
 *
 *--------------------------------------------------------------------------*/
string rtrim(string instring)
{ 
    int ic = 0, ch, in = 0, newn = 0, len;
    string temp;

    len = strlen(instring);
    if ((temp = malloc(len + 2)) != NULL)
        BlankString(temp, len);
    else
        return(instring);

    ic = strlen(instring) - 1;
    while ((!in) && (ic >= 0))
    { 
        ch = instring[ic];
        if (isspace(ch)) ic--;
        else
        {
            in = ic;
            break;
        }
    }
    for (ic = 0; ic < in; ic++, newn++)
    temp[newn] = instring[ic];
    temp[newn + 1] = 0;
    return(temp);

} /* End of rtrim */

/*--------------------------------------------------------------------------*
 *  Function:       LeftTrim                                                     
 *  Description:    Left trimming (returning the same left trimmed string)
 *                                                                            
 *--------------------------------------------------------------------------*/
string LeftTrim(string instring, string filter)                                               
{                                                                             
    int ic = 0, ch, len, newn = 0; 

    len = strlen(instring);
    while (ic < len)
    {
        ch = instring[ic++];
        if (strchr(filter, ch) == NULL)
        {
            newn = ic;
            break;
        }
    }
    if (newn)
    {
        movmem(&instring[newn - 1], instring, len - newn + 1);
        instring[len - newn + 1] = 0;
    }
    return(instring);

} /* End of LeftTrim */

/*--------------------------------------------------------------------------*
 *  Function:       RightTrim
 *  Description:    Right trimming (returning the same rightt trimmed string)
 *
 *--------------------------------------------------------------------------*/
string RightTrim(string instring,  string filterout)
{
    int ic = 0;

    ic = strlen(instring) - 1;
    while (ic >= 0)
    { 
        if (strchr(filterout, instring[ic]) != NULL)
            instring[ic] = 0;
        else
            break;
        ic--;
    }
    return(instring);

} /* End of RightTrim */

/*--------------------------------------------------------------------------*
 *  Function:       GetKey
 *  Description:    Uses the BIOS to read the next keyboard character
 *                                                                           
 *--------------------------------------------------------------------------*/
int GetKey(void)
{
    int key, lo, hi;

    key = bioskey(0);
    lo = key & 0X00FF;
    hi = (key & 0XFF00) >> 8;
    return((lo == 0) ? hi + 256 : lo);

} /* GetKey */

/*--------------------------------------------------------------------------*
 *  Function:       BoxDraw
 *  Description:    
 *                                                                           
 *--------------------------------------------------------------------------*/
void BoxDraw(int y1, int x1, int y2, int x2, int frame, byte shadow)
{ 
    int width = 0, height = 0, counter = 0;
    int hc = 0, vc = 0, ltc = 0, rtc = 0, lbc = 0, rbc = 0;
    char framestring[80], buffstr[160], attr;

    gettextinfo(&t_info);
    attr  = t_info.attribute;
    width = x2 - x1 - 1;
	switch(frame)
	{
		case 0:	
			 hc = vc = ltc = rtc = lbc = rbc = ' ';
			 break;

		case 1:
			 hc  = 196; vc  = 179; 
             ltc = 218; rtc = 191;
			 lbc = 192; rbc = 217;
             break;

		case 2:
			 hc  = 205; vc  = 186;
			 ltc = 201; rtc = 187;
			 lbc = 200; rbc = 188;
			 break;

		case 3:
			 ltc = ''; rtc = '';
			 lbc = ''; rbc = '';
			 hc  = ''; vc = '';
			 break;

		default:
			 return;
	}

    if (hc != 0)
    {
        memset(framestring, hc, width + 1);
        framestring[0] = ltc; 
        framestring[width + 1] = rtc; 
        framestring[width + 2] = '\0';
        Cputs(framestring, attr, x1, y1);
        for (counter = y1 + 1; counter < y2; counter++)
        {
            Putc(vc, attr, x1, counter);
			if (frame == 3)
                Putc('', attr, x2, counter);
            else
                Putc(vc, attr, x2, counter);
        }
        framestring[0] = lbc; 
        framestring[width + 1] = rbc;
        Cputs(framestring, attr, x1, y2);
    }
    if (shadow)
    {
        lbc = x2 + 1;
        if (lbc > SCREEN_COLUMNS) lbc = SCREEN_COLUMNS;
        width = lbc - x1;
        rbc = y2 + 1;
        if (rbc > SCREEN_ROWS) rbc = SCREEN_ROWS;
        height = rbc - y1;
        gettext(x1 + 1, rbc, lbc, rbc, buffstr);
        for (counter = 0, hc = x1 + 1; counter < 2 * width; hc++, counter += 2)
            Putc(buffstr[counter], SHADOW_COLOR, hc, rbc);
        gettext(x2 + 1, y1 + 1, x2 + 1, rbc, buffstr);
        for (counter = 0, vc = y1 + 1; counter < 2 * height; vc++, counter += 2)
            Putc(buffstr[counter], SHADOW_COLOR, x2 + 1, vc);
    }
    textattr(t_info.attribute);

} /* BoxDraw */

/*--------------------------------------------------------------------------*
 *  Function:       DrawTable
 *  Description:    Draws the table within the screen/window using specified
 *                  parameters (start position, size of the cell, number of
 *                  rows and number of columns of cells)
 *
 *--------------------------------------------------------------------------*/
void DrawTable(int x1, int y1, int nhoriz, int nvert, int width, int height, int frame)
{								   
    int gridheight = 0, gridwidth = 0;
    int xcurr = 0, count1 = 0, count2 = 0, hc = 0, vc = 0, ltc = 0, rtc = 0, 
        lbc = 0, rbc = 0, lmc = 0, rmc = 0, tmc = 0, bmc = 0, mmc = 0, attr;
    char frametop[80], framebottom[80], framemidlle[80];

    gettextinfo(&t_info);
    attr  = t_info.attribute;
    gridwidth  = nhoriz * width - 1;
    gridheight = nvert * height - 1;
    switch (frame)
    {
        case 1: hc  = 196; vc  = 179; ltc = 218; rtc = 191;
                lbc = 192; rbc = 217; lmc = 195; rmc = 180;
                bmc = 193; tmc = 194; mmc = 197;
                break;

        case 2: hc  = 205; vc  = 186; ltc = 201; rtc = 187;
                lbc = 200; rbc = 188; lmc = 204; rmc = 185;
                bmc = 202; tmc = 203; mmc = 206;
                break;

        default:
                return;

    }
    if (vc != 0)
    {
        memset(frametop, hc, gridwidth + 1);
        memset(framebottom, hc, gridwidth + 1);
        memset(framemidlle, hc, gridwidth + 1);
        frametop[0] = ltc;
        frametop[gridwidth + 1] = rtc;
        frametop[gridwidth + 2] = '\0';
        framebottom[0] = lbc;
        framebottom[gridwidth + 1] = rbc;
        framebottom[gridwidth + 2] = '\0';
        framemidlle[0] = lmc;
        framemidlle[gridwidth + 1] = rmc;
        framemidlle[gridwidth + 2] = '\0';
        for (count1 = 1; count1 < nhoriz; count1++)
        {
            xcurr = count1 * width;
            frametop[xcurr] = tmc;
            framebottom[xcurr] = bmc;
            framemidlle[xcurr] = mmc;
        }
        Cputs(frametop, attr, x1, y1);

        for (count1 = 0; count1 <= nhoriz; count1++)
        {
            xcurr = x1 + count1 * width;
            for (count2 = 1; count2 <= gridheight; count2++)
                Putc(vc, attr, xcurr, y1 + count2);
        }
        for (count2 = 1; count2 < nvert; count2++)
            Cputs(framemidlle, attr, x1, y1 + count2 * height);

        Cputs(framebottom, attr, x1, y1 + gridheight + 1);
    }
    textattr(t_info.attribute);

} /* End of DrawTable */


/*--------------------------------------------------------------------------*
 *  Function:       PaintBox
 *  Description:    
 *                                                                           
 *--------------------------------------------------------------------------*/
void PaintBox(int x1, int y1, int x2, int y2, int box, byte shade, byte attr, char *title)
{
    int center = 0;

    gettextinfo(&t_info);
    textattr(attr);
    if (title != NULL) center = (x1 + x2 - strlen(title)) / 2; 
    window(1, 1, t_info.screenwidth, t_info.screenheight);
    BoxDraw(y1, x1, y2, x2, box, shade);
    window(x1 + 1, y1 + 1, x2 - 1, y2 - 1);
    clrscr();
    window(1, 1, t_info.screenwidth, t_info.screenheight);
    if (center) Cputs(title, attr, center, y1);
    textattr(t_info.attribute);

}

/*--------------------------------------------------------------------------*
 *  Function:       keyhit
 *  Description:    Test for a keystroke using BIOS
 *                                                                           
 *--------------------------------------------------------------------------*/
int keyhit()
{
      union REGS rg;

    rg.h.ah = 1;
    int86(0x16, &rg, &rg);
    if (rg.x.flags & 0x40) return 0;
    rg.h.ah = 0;
    int86(0x16, &rg, &rg);  /* Eat the keystroke */
    return 1;
}

/*--------------------------------------------------------------------------*
 *  Function:       BlankString                                                              
 *  Description:    Puts specified number of spaces to the passed string
 *                                                                                                   
 *--------------------------------------------------------------------------*/
void BlankString(string String, int Len)
{
    int len = strlen(String);

    len  = (len <= Len ? len : Len);
    memset(String, SPACE, Len);   
    String[Len] = 0;

}/* End of BlankString */

/*--------------------------------------------------------------------------*
 *  Function:       DialogBoxMgr
 *  Description:    
 *
 *--------------------------------------------------------------------------*/
int DialogBoxMgr(int action)
{
    static int saved;

    gettextinfo(&t_info);
    if (!action)
    {
        window(1, 1, 80, 25);
        textattr(DIALOG_COLOR);
        saved = gettext(Message.x1 - 1, Message.y1 - 1, Message.x2 + 1, 
                        Message.y2 + 1, &(Message.ScreenBuffer));
        BoxDraw(Message.y1, Message.x1, Message.y2, Message.x2,
                                                            Message.FrameType, 1);
        gotoxy((Message.x1 + Message.x2 - 12) / 2, Message.y1);
        cputs(" Dialog Window ");
        window(Message.x1 + 1, Message.y1 + 1, Message.x2 - 1, Message.y2 - 1);
        clrscr();
    }
    else
    {
        if (saved)
            puttext(Message.x1 - 1, Message.y1 - 1, Message.x2 + 1, 
                    Message.y2 + 1, &(Message.ScreenBuffer));
        window(t_info.winleft, t_info.wintop, t_info.winright, t_info.winbottom);
        saved = FALSE;
    }
    textattr(t_info.attribute);
    return(saved);

} /* DialogBoxMgr */


/*--------------------------------------------------------------------------*
 *  Function:       GetString
 *  Description:    Allows the user to edit a string with filter string 
 *                  imposed                                                          
 *                                                                           
 *--------------------------------------------------------------------------*/
int GetString(int row, int col, int attr, string s, string filter, int size)
{
    int len = strlen(s), pos = 0, insert = TRUE, exitchar = 0, c = 0;

    CursorOn();
    gettextinfo(&t_info);
    textattr(InvertColor(attr));
    do
    {
        WriteXY(col, row, attr, len + 1, "%s", s);
        gotoxy(col + pos, row);
        c = GetKey();
        if (isalpha(c)) tolower(c);
        switch(c)
        { 
            case HOME_KEY:
                 pos = 0;
                 break;

            case END_KEY:
                 pos = len;
                 break;

            case INS_KEY:
                 insert = !insert;
                 if (insert)
                    _setcursortype(_NORMALCURSOR);
                 else
                    _setcursortype(_SOLIDCURSOR);
                 break;

            case LEFT_KEY:
                 if (pos > 0)
                    pos--;
                 break;

            case RIGHT_KEY:
                 if (pos < len)
                    pos++;
                 break;

            case BS:
                 if (pos > 0)
                 {
                    movmem(&s[pos], &s[pos - 1], len - pos + 1);
                    pos--;
                    len--;
                 }
                 break;

            case DEL_KEY:
                 if (pos < len)
                 {
                    movmem(&s[pos + 1], &s[pos], len - pos);
                    len--;
                 }
                 break;

            case CR:
                 exitchar = CR;
                 break;

            case UP_KEY:
                 c = CR;
                 exitchar = UP_KEY;
                 break;

            case DOWN_KEY:
                 c = CR;
                 exitchar = DOWN_KEY;
                 break;

            case ESC:
                 len = 0;
                 break;

            default:
                if (filter[0] != 0)
                { 
                    if ((strchr(filter, c) != NULL) && (c >= ' ') && 
                                                        (len <= size))
                    { 
                        if (insert)
                        { 
                            memmove(&s[pos + 1], &s[pos], len - pos + 1);
                            len++;
                        }
                        if (pos >= len) len++;
                        len = (len <= size ? len : size);
                        if (pos < size) s[pos++] = c;
                    }
                }
                else
                { 
                    if (pos <= size)
                    { 
                        if (pos >=len) len++;
                        s[pos++] = c;
                    }
                }
                break;

        } /* switch */
        s[len] = 0;

    } while ((c != CR) && (c != ESC));
    textattr(t_info.attribute);
    CursorOff();
    if (c != ESC)
        return(exitchar);
    else
        return(FALSE);

} /* GetString */


/*--------------------------------------------------------------------------*
 *  Function:       ClearBiosBuff
 *  Description:    Clear the BIOS read-ahead buffer
 *
 *--------------------------------------------------------------------------*/
void ClearBiosBuff(void)
{
    *KeyBuffHead = *KeyBuffTail;
}

/*--------------------------------------------------------------------------*
 *  Function:       Putc
 *  Description:    Prints a character and text attribute at given location.
 *                  Writes directly to the video buffer, assuming full screen!
 *--------------------------------------------------------------------------*/
void Putc(char ch, char attr, int x, int y)
{
    int offset, xx, yy;

    if (DirectVideo)
    {
        gettextinfo(&t_info);
        xx = t_info.winleft + x - 1;
        yy = t_info.wintop  + y - 1;
        if ((xx <= t_info.winright) && (yy <= t_info.winbottom))
        {
            offset = (yy - 1) * 160 + ((xx - 1) << 1);
            VideoPtr[offset++] = ch;
            VideoPtr[offset]   = attr;
        }
    }
    else
    {
        textattr(attr);
        gotoxy(x, y);
        putch(ch);
    }
}

/*--------------------------------------------------------------------------*
 *  Function:       Cputs
 *  Description:    Prints a char string and text attribute at given location
 *                  Writes directly to the video buffer, assuming full screen!
 *--------------------------------------------------------------------------*/
void Cputs(char *ch, char attr, int x, int y)
{
    int offset, xx, yy, len;

    if (DirectVideo)
    {
        gettextinfo(&t_info);
        xx  = t_info.winleft + x - 1;
        yy  = t_info.wintop  + y - 1;
        len = t_info.winright - xx;
        if ((xx <= t_info.winright) && (yy <= t_info.winbottom))
        {
            offset = (yy - 1) * 160 + ((xx - 1) << 1);
            while ((*ch != '\0') && len--)
            {
                VideoPtr[offset++] = *ch++;
                VideoPtr[offset++] = attr;
            }
        }
    }
    else
    {
        textattr(attr);
        gotoxy(x, y);
        cputs(ch);
    }
}

/*--------------------------------------------------------------------------*
 *  Function:       GetScreenMode
 *  Description:    Returns the screen mode
 *--------------------------------------------------------------------------*/
int GetScreenMode(void)
{
    union  REGS Regs;

    Regs.h.ah = 0x0F;           
    int86(0x10, &Regs, &Regs);
    return(Regs.h.al);       
}
