/*---------------------------------------------------------------------------*
 *
 *  Placed in Public Domain By Damir Nadramija.
 *
 *  Module:         CLOCKINT.C 
 *  Purpose:        Clock hardware interrupt service routine used as a driver
 *                  for the "Small Multi Threader" project              
 *                                                                            
 *  Written by:     Damir Nadramija 
 *  Date:           September 17, 1991
 *                                           
 *  Note:           Void interrupt functions OldDisk and OldVideo were taken and
 *                  modified from All Stevens "Extending Turbo C Professional"  
 *                  (placed in public domain).
 *--------------------------------------------------------------------------*/
#include <stddef.h>
#include <dos.h>
#include <setjmp.h>
#include <tools.h>
#include <clockint.h>

/* Local macros */
#define IsSafe() (!peekb(DOSSeg, DOSBusy) && !DiskFlag && !VideoFlag && !ComFlag && !Semafore)
#define IsTaskOK(task)      ((task >= 0) && (task < MAX_TASKS))
#define OKToKillTask(task)  ((task > 0)  && (task < MAX_TASKS))

/* Globals Resident Here */
Time Timer;
int  NumberOfTasks;                        /* Number of tasks         */
int  Semafore;                             /* Safe passage or stop    */  
int  CurrTaskIndex;                        /* Current task index      */
int  FastClock;                            /* Fast/regular clock rate */

/* Local and Static Variables */
static int ReEntrancyFlag = FALSE;

static int      Flags;                      /* Staging area for flags  */
static unsigned DOSSeg;                     /* DOS segment address     */
static unsigned DOSBusy;                    /* offset to InDOS flag    */
static int      DiskFlag;                   /* Disk busy flag          */
static int      VideoFlag;                  /* Video busy flag         */   
static int      ComFlag;                    /* Comm ports busy flag    */   
static int      TotalTasks;                 /* Total number of tasks   */
static int      ModifiedSystem;             /* Number of active tasks  */
static int      ActiveTasksNumber;          /* Number of active tasks  */
static int      LastActiveTask;             /* Index of last task      */
static int      ActivateNewTask;            /* Enforce activating task */
static int      TaskToActivate;             /* Task to be activated    */
static int      TaskIndex[MAX_TASKS];       /* Index array for tasks   */
static TaskData TaskInfo[MAX_TASKS];        /* Array of tasks          */
static TaskData *NewTask;                   /* Task being activated    */
static TaskData *CurrentTask;               /* Current task            */

/*  Interrupt vector chains  */
static void interrupt (*OldDisk)();
static void interrupt (*OldVideo)();
static void interrupt (*OldCom1)();
static void interrupt (*OldCom2)();
static void interrupt (*SysClock)();

/*  ISRs for the TSR  */
static void interrupt NewClock();
static void interrupt NewDisk();
static void interrupt NewCrit();
static void interrupt NewVideo();
static void interrupt NewCom1();
static void interrupt NewCom2();

#pragma warn -pro
#pragma warn -par

/* Local prototypes */
static int  NextTask(void);
static int  LastTaskSlot(void);
static int  EmptyTaskSlot(void);
static void NullTask(int tasknumber);

/*--------------------------------------------------------------------------*
 *  Function:       InitializeCounters                                       
 *  Description:    Initializes internal counters                            
 *                                                                           
 *--------------------------------------------------------------------------*/
void InitializeCounters(void)
{
    Timer.Ticks     = 0;
    Timer.DOSTicks  = 0;
    Timer.MiliSecs  = HMI_CLOCK_RATE;
    Timer.TickTime  = FALSE;

} /* End of InitializeCounters */


/*--------------------------------------------------------------------------*
 *  Function:       InitializeTimer                                          
 *  Description:    Reprogramms the chip timer to run at 1 ms rate           
 *                                                                           
 *--------------------------------------------------------------------------*/
void InitializeTimer(void)
{
    if (!FastClock) return;

    outportb(CLOCK_CONTROL_WORD, SET_CLOCK_COUNTER_0_CW);
    goto NullJump1;

NullJump1:;

    outportb(CLOCK_COUNTER_0, SET_HMI_CLOCK_LSB);
    goto NullJump2;

NullJump2:;

    outportb(CLOCK_COUNTER_0, SET_HMI_CLOCK_MSB);

} /* End of InitializeTimer */


/*--------------------------------------------------------------------------*
 *  Function:       RestoreTimer                                             
 *  Description:    Restores the state of the timer chip                     
 *                                                                           
 *--------------------------------------------------------------------------*/
void RestoreTimer(void)
{
    if (!FastClock) return;

    outportb(CLOCK_CONTROL_WORD, SET_CLOCK_COUNTER_0_CW);
    goto NullJump1;

NullJump1:;

    outportb(CLOCK_COUNTER_0, SET_DOS_CLOCK_LSB);
    goto NullJump2;

NullJump2:;

    outportb(CLOCK_COUNTER_0, SET_DOS_CLOCK_MSB);

} /* End of RestoreTimer */


/*--------------------------------------------------------------------------*
 *  Function:       SetNewClock                                              
 *  Description:    Sets a new interrupt service routine to the hardware clock        
 *                                                                           
 *--------------------------------------------------------------------------*/
void SetNewClock(void)
{
    SysClock = getvect(CLOCK);
    setvect(CLOCK, NewClock);
    InitializeCounters();
    disable();
    InitializeTimer();
    enable();

} /* End of SetNewClock */


/*--------------------------------------------------------------------------*
 *  Function:       NewClock                                                 
 *  Description:    Interrupt service routine for the higher rate timer      
 *                                                                           
 *--------------------------------------------------------------------------*/
void interrupt NewClock(bp, di, si, ds, es, dx, cx, bx, ax)
{
    static word ss, sp, taskss, tasksp;

    if (ReEntrancyFlag)
    {
        Timer.Ticks++;
        outportb(INTERRUPT_CONTROLLER, END_OF_INTERRUPT);
        return;
    }
    disable();
    ss = _SS;
    sp = _SP;
    enable();
    ReEntrancyFlag = TRUE;

    Timer.Ticks++;

    if (FastClock)
    {
        if (--Timer.MiliSecs <= 0)
        {
            Timer.MiliSecs = HMI_CLOCK_RATE;
            Timer.DOSTicks++;
            Timer.Seconds = Timer.MiliSecs / 1000;

            Timer.TickTime = TRUE;
            (*SysClock)();
        }
        else outportb(INTERRUPT_CONTROLLER, END_OF_INTERRUPT);
    }
    else
    {
        Timer.DOSTicks++;
        Timer.Seconds = Timer.DOSTicks / 18;
        (*SysClock)();
    }

    if (IsSafe())
    {
        if ((NumberOfTasks > 1) && IsTaskOK(CurrTaskIndex))
        {
            CurrentTask = &TaskInfo[CurrTaskIndex];
            if (ActivateNewTask && IsTaskOK(TaskToActivate))
            {
                disable();
                NewTask = &TaskInfo[TaskToActivate];
                if (NewTask->ExtStack != NULL)
                {
                    taskss = FP_SEG(&NewTask->ExtStack);
                    tasksp = FP_OFF(&NewTask->ExtStack);
                }
                else
                {
                    taskss = FP_SEG(&NewTask->LocStack[STACK_SIZE - 1]);
                    tasksp = FP_OFF(&NewTask->LocStack[STACK_SIZE - 1]);
                }
                CurrentTask->TaskSS = ss;
                CurrentTask->TaskSP = sp;
                _SS = taskss;
                _SP = tasksp;
                NewTask->TaskTimer = NewTask->TaskPriority;
                NewTask->Active    = TRUE;
                CurrTaskIndex      = TaskToActivate;
                ActivateNewTask    = FALSE;
                ReEntrancyFlag     = FALSE;
                enable();
                NewTask->FunPtr();
            }
            else
            {
                if (CurrentTask->Active)
                {
                    if (CurrentTask->TaskTimer)
                    {
                        if (CurrentTask->SuspendTask || CurrentTask->KillTask)
                            CurrentTask->TaskTimer = 1;

                        if (--CurrentTask->TaskTimer <= 0)
                        {
                            TaskToActivate = NextTask();
                            if (TaskToActivate != CurrTaskIndex)
                            {
                                LastActiveTask = CurrTaskIndex;
                                disable();
                                if (CurrentTask->KillTask)
                                {
	                                TaskIndex[CurrTaskIndex] = 0;
                                    CurrentTask->Active      = 0;
                                    CurrentTask->TaskTimer   = 0;
                                    NumberOfTasks--;
                                }
                                else
                                {
                                    CurrentTask->TaskSS = ss;
                                    CurrentTask->TaskSP = sp;
                                }
                                NewTask = &TaskInfo[TaskToActivate];
                                _SS = NewTask->TaskSS;
                                _SP = NewTask->TaskSP;
                                CurrTaskIndex      = NewTask->TaskIndex;
                                NewTask->TaskTimer = NewTask->TaskPriority;
                                ReEntrancyFlag     = FALSE;
                                enable();
                            }
                        }
                    }
                }
            }
        }
    }
    disable();
    ReEntrancyFlag = FALSE;
    enable();

} /* End of NewClock */


/*--------------------------------------------------------------------------*
 *  Function:       ReadTimer                                                
 *  Description:    Gets the current time in miliseconds (NOT ticks)   
 *                                                                           
 *--------------------------------------------------------------------------*/
long ReadTimer(void)
{
    return(Timer.Ticks);

}   /* End of ReadTimer */


/*--------------------------------------------------------------------------*
 *  Function:       SetOldClock                                              
 *  Description:    Resets the old (BIOS) hardware clock interrupt service routine    
 *                                                                           
 *--------------------------------------------------------------------------*/
void SetOldClock(void)
{
    disable();
    RestoreTimer();
    enable();
    if ((getvect(CLOCK) == NewClock) && (SysClock != NULL)) setvect(CLOCK, SysClock);

}  /* End of SetOldClock */


/*--------------------------------------------------------------------------*
 *  Function:       InitGlobals
 *  Description:    Get address of DOS busy flag.
 *                                                                           
 *--------------------------------------------------------------------------*/
void InitGlobals(void)
{
    union REGS rg;

    rg.h.ah = 0x34;
    intdos(&rg, &rg);
    DOSSeg = _ES;
    DOSBusy = rg.x.bx;
}

/*--------------------------------------------------------------------------*
 *  Function:       NewVideo
 *  Description:    BIOS video functions ISR.  
 *                                                                           
 *--------------------------------------------------------------------------*/
static void interrupt NewVideo(bp, di, si, ds, es, dx, cx, bx, ax)
{
    /* INT 0x10 uses most registers (incl BP for EGA). It can
       call itself, so this code allows 5 reentrant calls */

    static int hbx, funcode, params;
    static unsigned vidbp[5];
    static unsigned vidoutbp;

    vidbp[VideoFlag++] = _BP;       /* TC's BP. This uses bx, so    */

    funcode = ax >> 8;
    params  = ax & 0xFF;

    _AX = ax;                       /* ... restore ax               */
    _BX = bx;                       /* ... restore bx               */
    _CX = cx;                       /*         .                    */
    _DX = dx;                       /*         .                    */
    _BP = bp;                       /* caller's bp (C used it)      */
    (*OldVideo)();                  /* interrupt chain              */
    vidoutbp = _BP;                 /* save the BP return           */
    hbx = _BX;                      /* next line uses bx, save it   */
    _BP = vidbp[--VideoFlag];       /* restore TC's BP              */
    ax = _AX;                       /* for register returns         */
    bx = hbx;                       /* saved from INT 0x10          */
    cx = _CX;
    bp = vidoutbp;                  /* saved from INT 0x10          */
    dx = _DX;
    es = _ES;
    di = _DI;
}

/*--------------------------------------------------------------------------*
 *  Function:       NewDisk
 *  Description:    BIOS disk functions ISR 
 *                                                                           
 *--------------------------------------------------------------------------*/
static void interrupt NewDisk(bp, di, si, ds, es, dx, cx, bx, ax, ip, cs, flags)
{
    DiskFlag++;
    (*OldDisk)();
    ax = _AX;                       /* For the ax return to get current   */
    NewCrit();                      /* Flags register                     */
    flags = Flags;                  /* Newdisk will return oldisk's flags */
    --DiskFlag;
}

/*--------------------------------------------------------------------------*
 *  Function:       NewCom1
 *  Description:    COM1 function ISR 
 *                                                                           
 *--------------------------------------------------------------------------*/
static void interrupt NewCom1(bp, di, si, ds, es, dx, cx, bx, ax, ip, cs, flags)
{
    ComFlag++;
    (*OldCom1)();
    ax = _AX;               
    bx = _BX;               
    --ComFlag;
}

/*--------------------------------------------------------------------------*
 *  Function:       NewCom2
 *  Description:    COM2 function ISR 
 *                                                                           
 *--------------------------------------------------------------------------*/
static void interrupt NewCom2(bp, di, si, ds, es, dx, cx, bx, ax, ip, cs, flags)
{
    ComFlag++;
    (*OldCom2)();
    ax = _AX;           
    bx = _BX;               
    --ComFlag;
}

/*--------------------------------------------------------------------------*
 *  Function:       NewCrit
 *  Description:    Critical error ISR
 *                                                                           
 *--------------------------------------------------------------------------*/
static void interrupt NewCrit(bp, di, si, ds, es, dx, cx, bx, ax, ip, cs, flags)
{
    ax    = 0;                           /* ignore critical errors       */
    Flags = flags;                       /* for newdisk                  */
}

/*--------------------------------------------------------------------------*
 *  Function:       ActivateTask
 *  Description:    Initializes the Task structure to activate the task.
 *                  If using external stack buffer (larger than default 
 *                  STACK_SIZE bytes) be sure that the passed pointer is 
 *                  pointing to the end of the allocated memory as the 
 *                  stack is growing downward.
 *--------------------------------------------------------------------------*/
int ActivateTask(TaskFunc FunPtr, char *ExtStack, int Priority)
{
    int index = INVALID_TASK_INDEX;

    /* First check if we can activate a new task */
    if (ActivateNewTask) return(index); 
    if (NumberOfTasks >= MAX_TASKS) return (index);
    if ((index = EmptyTaskSlot()) == INVALID_TASK_INDEX) return(index);

    /* Set the new task for activation */  
    NullTask(index);
    disable();
    TaskInfo[index].TaskIndex    = index;
    TaskInfo[index].TaskID       = ++TotalTasks;
    TaskInfo[index].FunPtr       = FunPtr;
    TaskInfo[index].ExtStack     = ExtStack;
    TaskInfo[index].TaskPriority = Priority;
    TaskInfo[index].TaskClass    = BACKGROUND_TASK;
	TaskIndex[index]             = index + 1;
    NumberOfTasks++;
    ActivateNewTask++;
    enable();
    TaskToActivate = index;
    return(index);
}

/*--------------------------------------------------------------------------*
 *  Function:       KillTask
 *  Description:    Sets the Kill flag for the task.  
 *--------------------------------------------------------------------------*/
void KillTask(int taskindex)
{
    if (OKToKillTask(taskindex)) TaskInfo[taskindex].KillTask++;
}

/*--------------------------------------------------------------------------*
 *  Function:       SuspendTask
 *  Description:    Sets the SuspendTask flag for the task. 
 *--------------------------------------------------------------------------*/
void SuspendTask(int taskindex)
{
    if (IsTaskOK(taskindex)) TaskInfo[taskindex].SuspendTask++;
}

/*--------------------------------------------------------------------------*
 *  Function:       UnSuspendTask
 *  Description:    Sets the SuspendTask flag for the task. 
 *--------------------------------------------------------------------------*/
void UnSuspendTask(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        if (TaskInfo[taskindex].SuspendTask) TaskInfo[taskindex].SuspendTask--;
}

/*--------------------------------------------------------------------------*
 *  Function:       SetTaskSleep
 *  Description:    Sets the sleep time for the specified task.
 *--------------------------------------------------------------------------*/
void SetTaskSleep(int taskindex, dword sleeptime)
{
    if (IsTaskOK(taskindex)) TaskInfo[taskindex].SleepTimer = sleeptime;
}

/*--------------------------------------------------------------------------*
 *  Function:       SetUpSystem
 *  Description:    Setting up a system for the "Task Scheduler".
 *                                                                           
 *--------------------------------------------------------------------------*/
void SetUpSystem(void)
{
    if (ModifiedSystem) return;

    InitGlobals();
    SetInitialTask();

    /*  Get original interrupt vectors  */
    OldVideo = getvect(VIDEO);
    OldDisk  = getvect(DISK);
    OldCom1  = getvect(COM1);
    OldCom2  = getvect(COM2);

    setvect(DISK,  NewDisk);
    setvect(VIDEO, NewVideo);
    setvect(COM1,  NewCom1);
    setvect(COM2,  NewCom2);

    /* Setup new clock hardware interrupt handler */
    SetNewClock();
    ModifiedSystem++;
}

/*--------------------------------------------------------------------------*
 *  Function:       RestoreSystem
 *  Description:    Restores system's settings before returning to DOS.
 *                                                                           
 *--------------------------------------------------------------------------*/
void RestoreSystem(void)
{
    if (!ModifiedSystem) return;

    /* Restore old clock hardware interrupt handler */
    SetOldClock();

    /*  Restore the interrupt vectors  */
    if (getvect(VIDEO) == NewVideo) setvect(VIDEO, OldVideo);
    if (getvect(DISK)  == NewDisk)  setvect(DISK,  OldDisk);
    if (getvect(COM1)  == NewCom1)  setvect(COM1,  OldCom1);
    if (getvect(COM2)  == NewCom2)  setvect(COM2,  OldCom2);

    ModifiedSystem--;
}

/*--------------------------------------------------------------------------*
 *  Function:         EmptyTaskSlot                                            
 *  Description:      Returns first empty task slot index                      
 *                                                                             
 *--------------------------------------------------------------------------*/
static int EmptyTaskSlot(void)
{
	int taskindex = 0;

	while ((TaskIndex[taskindex]) && (taskindex < MAX_TASKS))  taskindex++;
    if (taskindex < MAX_TASKS) return(taskindex);
    else return(INVALID_INDEX);
 
} /* End of EmptyTaskSlot */


/*--------------------------------------------------------------------------*
 *  Function:         LastTaskSlot
 *  Description:      Returns the last occupied task slot index                
 *                                                                             
 *--------------------------------------------------------------------------*/
static int LastTaskSlot(void)
{
	int taskindex = MAX_TASKS;

	while ((!TaskIndex[taskindex]) && (taskindex > 0))  taskindex--;
	return(taskindex);
 
} /* End of LastTaskSlot */

/*--------------------------------------------------------------------------*
 *  Function:         NextTask                                            
 *  Description:      Returns next task slot index. This is an unusual
 *                    implementation of the Round-Robin scheduler. It starts
 *                    at the current task and goes till the end of the task
 *                    queue, and then goes back to the start of the queue.
 *                    Also, before finding a next task, first decrements all
 *                    sleep time settings for all sleeping tasks. 
 *--------------------------------------------------------------------------*/
static int NextTask(void)
{
	int taskindex, startindex = 0;

    if (!CurrTaskIndex) startindex = 1;

	for (taskindex = 0; taskindex < MAX_TASKS; taskindex++)
        if (TaskIndex[taskindex]) 
            if (TaskInfo[taskindex].SleepTimer) 
                TaskInfo[taskindex].SleepTimer--;

	for (taskindex = CurrTaskIndex + 1; taskindex < MAX_TASKS; taskindex++)
        if (TaskIndex[taskindex]) 
            if (!TaskInfo[taskindex].SuspendTask)
                if (!TaskInfo[taskindex].SleepTimer)
                    return(taskindex);

	for (taskindex = startindex; taskindex < CurrTaskIndex; taskindex++)
        if (TaskIndex[taskindex]) 
            if (!TaskInfo[taskindex].SuspendTask)
                if (!TaskInfo[taskindex].SleepTimer)
                    return(taskindex);

    return(CurrTaskIndex);

} /* End of NextTask */


/*--------------------------------------------------------------------------*
 *  Function:         NullTask                                                  
 *  Description:      Resets/nulls a task slot in the task queue                
 *                                                                              
 *--------------------------------------------------------------------------*/
static void NullTask(int tasknumber)
{
	memset(&(TaskInfo[tasknumber]), 0, sizeof(TaskData));
	TaskIndex[tasknumber] = 0;

} /* End of NullTask */


/*--------------------------------------------------------------------------*
 *  Function:         InitializeTaskQueue                                       
 *  Description:      Initializes task queue                                    
 *                                                                              
 *--------------------------------------------------------------------------*/
void InitializeTaskQueue(void)
{
	int counter;

	for (counter = 0; counter < MAX_TASKS; counter++)   NullTask(counter);
    ActiveTasksNumber = 0;
	NumberOfTasks = 0;
    TotalTasks = 0;

} /* End of InitializeTaskQueue */

/*--------------------------------------------------------------------------*
 *  Function:       SetInitialTask
 *  Description:    Initializes the first task (foreground task). This has
 *                  to be done so that we can go back to the initial task 
 *                  (one that was initiating other (background) tasks).
 *--------------------------------------------------------------------------*/
int SetInitialTask(void)
{
    int index = 0;

    NullTask(index);
    disable();
    TaskInfo[index].TaskIndex    = index;
    TaskInfo[index].TaskID       = ++TotalTasks;
    TaskInfo[index].FunPtr       = NULL;
    TaskInfo[index].ExtStack     = NULL;
    TaskInfo[index].TaskPriority = MAX_PRIORITY;
    TaskInfo[index].TaskClass    = INITIAL_TASK;
    TaskInfo[index].Active       = TRUE;
	TaskIndex[index]             = index + 1;
    CurrTaskIndex = index;
    NumberOfTasks++;
    enable();
    return(index);
}


/*--------------------------------------------------------------------------*
 *  Function:       SuspendAllTasks
 *  Description:    Suspends all tasks
 *--------------------------------------------------------------------------*/
void SuspendAllTasks(void)
{
    int taskindex;

	for (taskindex = 1; taskindex < MAX_TASKS; taskindex++)
        if (TaskIndex[taskindex]) TaskInfo[taskindex].SuspendTask++;
}


/*--------------------------------------------------------------------------*
 *  Function:       UnSuspendAllTasks
 *  Description:    Unsuspends all tasks
 *--------------------------------------------------------------------------*/
void UnSuspendAllTasks(void)
{
    int taskindex;

	for (taskindex = 1; taskindex < MAX_TASKS; taskindex++)
        if (TaskIndex[taskindex]) TaskInfo[taskindex].SuspendTask--;
}

/*--------------------------------------------------------------------------*
 *  Function:         LastTaskIndex
 *  Description:      Returns the last occupied task slot index                
 *                                                                             
 *--------------------------------------------------------------------------*/
int LastTaskIndex(void)
{
    return(LastTaskSlot());
}


/*--------------------------------------------------------------------------*
 *  Function:         GetTaskTimer
 *  Description:      Returns task's timer value
 *                                                                             
 *--------------------------------------------------------------------------*/
int GetTaskTimer(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        return(TaskInfo[taskindex].TaskTimer);
    else
        return(-1);
}

/*--------------------------------------------------------------------------*
 *  Function:         GetTaskSleepTimer
 *  Description:      Returns task's sleep timer value
 *                                                                             
 *--------------------------------------------------------------------------*/
long GetTaskSleepTimer(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        return(TaskInfo[taskindex].SleepTimer);
    else
        return(-1);
}

/*--------------------------------------------------------------------------*
 *  Function:         IsTaskSuspended
 *  Description:      Returns task's suspend flag
 *                                                                             
 *--------------------------------------------------------------------------*/
int IsTaskSuspended(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        return(TaskInfo[taskindex].SuspendTask);
    else
        return(-1);
}


/*--------------------------------------------------------------------------*
 *  Function:         IsTaskActive
 *  Description:      Returns task's active flag
 *                                                                             
 *--------------------------------------------------------------------------*/
int IsTaskActive(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        return(TaskInfo[taskindex].Active);
    else
        return(-1);
}

/*--------------------------------------------------------------------------*
 *  Function:         IsTaskToBeKilled
 *  Description:      Returns task's kill flag
 *                                                                             
 *--------------------------------------------------------------------------*/
int IsTaskToBeKilled(int taskindex)
{
    if (IsTaskOK(taskindex)) 
        return(TaskInfo[taskindex].KillTask);
    else
        return(-1);
}


#pragma warn .pro
#pragma warn .par




