//  SHORT BIT SET CLASS
//      sbitset.h   3.10    04-Aug-1990
//      Turbo C++ 1.0
//
//      Defines a set of up to 65,536 bits
//
//      Written by Scott Robert Ladd. Released into the public domain.

#if !defined(SBITSET_H)
#define SBITSET_H

#include "stddef.h"
#include "string.h"

class ShortBitSet
    {
    public:
        // constructors
        ShortBitSet(unsigned short size);

        ShortBitSet(const ShortBitSet & bs);

        // destructor
        ~ShortBitSet();

        // assignment operator
        void operator = (const ShortBitSet & bs);

        // Get number of bits in set
        unsigned short Size() const;

        // operation methods
        void Include(unsigned short bit);
        void Exclude(unsigned short bit);

        // turn all bits on or off
        void AllOn();
        void AllOff();

        // union operators
        friend ShortBitSet operator & (const ShortBitSet & bs1,
                                       const ShortBitSet & bs2);

        void operator &= (const ShortBitSet & bs);

        // synonyms for union operators
        ShortBitSet operator + (const ShortBitSet & bs);

        void operator += (const ShortBitSet & bs);

        // intersection operators
        ShortBitSet operator |  (const ShortBitSet & bs);

        void operator |= (const ShortBitSet & bs);

        // difference operators
        ShortBitSet operator -  (const ShortBitSet & bs);

        void operator -= (const ShortBitSet & bs);

        // complement operator
        ShortBitSet operator ~ ();

        // comparison operator
        int operator == (const ShortBitSet & bs) const;
        int operator != (const ShortBitSet & bs) const;

        // value retrieval method
        int operator [] (unsigned short bit) const;

    protected:
        unsigned short  Length;
        unsigned char * Data;

        // protected constructor
        ShortBitSet();
    };

// Get number of bits in set
inline unsigned short ShortBitSet::Size() const
    {
    return Length;
    }

// operation methods
inline void ShortBitSet::Include(unsigned short bit)
    {
    if (bit < Length)
        Data[bit / 8] |= (unsigned char)(1 << (bit & 7));
    }

inline void ShortBitSet::Exclude(unsigned short bit)
    {
    if (bit < Length)
        Data[bit / 8] &= ~(unsigned char)(1 << (bit & 7));
    }

// turn all bits in set on
inline void ShortBitSet::AllOn()
    {
    memset(Data,'\xFF',(Length + 7) / 8);
    }

// turn all bits in set off
inline void ShortBitSet::AllOff()
    {
    memset(Data,'\x00',(Length + 7) / 8);
    }

// union operators
inline void ShortBitSet::operator &= (const ShortBitSet & bs)
    {
    *this = *this & bs;
    }

// synonyms for union operators
inline void ShortBitSet::operator += (const ShortBitSet & bs)
    {
    *this &= bs;
    }

// intersection operators
inline void ShortBitSet::operator |= (const ShortBitSet & bs)
    {
    *this = *this | bs;
    }

// difference operators
inline void ShortBitSet::operator -= (const ShortBitSet & bs)
    {
    *this = *this - bs;
    }

// value retrieval method
inline int ShortBitSet::operator [] (unsigned short bit) const
    {
    if (bit < Length)
        return (Data[bit / 8] & (1 << (bit & 7)));
    else
        return 0;
    }

// protected constructor
inline ShortBitSet::ShortBitSet()
    {
    Length = 0;
    Data   = NULL;
    }

#endif
