//  SCREEN CLASS
//      screen.h    v4.00   02-Aug-1990
//      Turbo C++ 1.0
//
//      This class requires MS-DOS w/monochrome or color text display. It
//      provides a class for manipulating a text display.
//
//      To drastically improve the speed of this class, NO RANGE
//      CHECKING is done! Invalid row/column values may cause
//      portions of non-video memory to be corrupted!
//
//      Written by Scott Robert Ladd. Released into the public domain.

#if !defined(SCREEN_H)
#define SCREEN_H

#include "str.h"
#include "condata.h"
#include "dos.h"
#include "stddef.h"

enum BoxType {BT_NONE, BT_SINGLE, BT_DOUBLE, BT_SOLID};

enum ScrMode {SCR_MODE_UNKNOWN = -1,
              SCR_MODE_25x40   =  0,  /* all except monochrome */
              SCR_MODE_25x80   =  1,  /* all (default)         */
              SCR_MODE_30x80   =  2,  /* VGA and MCGA          */
              SCR_MODE_43x80   =  3,  /* EGA                   */
              SCR_MODE_50x80   =  4}; /* VGA and MCGA          */

enum ScrType {SCR_TYPE_UNKNOWN = -1,
              SCR_TYPE_MDA     =  0,
              SCR_TYPE_CGA     =  1,
              SCR_TYPE_HGC     =  2,
              SCR_TYPE_EGA     =  3,
              SCR_TYPE_MCGA    =  4,
              SCR_TYPE_VGA     =  5};

enum ScrAccess {SCR_ACC_DIRECT =  0,
                SCR_ACC_BIOS   =  1};

// definitions of color bits

#define SCR_F_BLACK  0x00
#define SCR_F_BLUE   0x01
#define SCR_F_GREEN  0x02
#define SCR_F_RED    0x04
#define SCR_F_CYAN   (SCR_F_BLUE | SCR_F_GREEN)
#define SCR_F_VIOLET (SCR_F_RED  | SCR_F_BLUE)
#define SCR_F_YELLOW (SCR_F_RED  | SCR_F_GREEN)
#define SCR_F_BROWN  (SCR_F_YELLOW)
#define SCR_F_WHITE  (SCR_F_BLUE | SCR_F_GREEN | SCR_F_RED)
#define SCR_F_BRIGHT 0x08

#define SCR_B_BLACK  0x00
#define SCR_B_BLUE   0x10
#define SCR_B_GREEN  0x20
#define SCR_B_RED    0x40
#define SCR_B_CYAN   (SCR_B_BLUE | SCR_B_GREEN)
#define SCR_B_VIOLET (SCR_B_RED  | SCR_B_BLUE)
#define SCR_B_YELLOW (SCR_B_RED  | SCR_B_GREEN)
#define SCR_B_BROWN  (SCR_B_YELLOW)
#define SCR_B_WHITE  (SCR_B_BLUE | SCR_B_GREEN | SCR_B_RED)
#define SCR_B_BRIGHT 0x80

#define SCR_BLINK    0x80
#define SCR_UNDER    0x08

class Screen
    {
    protected:
        static ScrType        VideoType;    // type of monitor
        static ScrMode        VideoMode;    // video mode
        static ScrMode        OriginalMode; // original video mode
        static ScrAccess      AccessMode;   // current access mode

        static unsigned short Width;        // width of the screen in characters
        static unsigned short Length;       // length of the screen in characters

        static unsigned short far * VideoMem; // base address of character memory

        static unsigned short CursorShape;  // stored shape of the cursor
        static short          CursorHidden; // non-zero if cursor has been hidden

        static unsigned short HowMany;      // How many screens instantiated?

        static short *        SavedScreen;  // saved text of original screen
        static unsigned short SavedCursPos; // cursor position on original screen
        static short          SavedCursShp; // cursor shape on original screen

        static ConsoleData *  SavedConData; // stored MS-DOS Console Data

    public:
        // constructor
        Screen();

        // destructor
        ~Screen();

        // obtain class version number
        static short Version();

        // get current type of screen
        static ScrType GetType();

        // set/get screen mode
        static short SetMode(ScrMode mode);
        static ScrMode GetMode(void);

        // set/get screen access mode
        static void SetAccess(ScrAccess access);
        static ScrAccess GetAccess(void);

        // retrieve screen size method
        static void Dimensions(unsigned short & len, unsigned short & wid);

        // methods which change the purpose of bit 8 in the attribute byte
        void SetBlink(void);
        void SetBright(void);

        // cursor methods
        static void CursorHide();
        static void CursorUnhide();

        static void CursorSetPos(unsigned short row, unsigned short col);
        static void CursorGetPos(unsigned short & row, unsigned short & col);

        // data display methods
        static void Put(unsigned short row, unsigned short col,
                        unsigned char attr, char ch);

        static void Put(unsigned short row, unsigned short col,
                        unsigned short word);

        static void Put(unsigned short row, unsigned short col,
                        unsigned char attr, const char * str);

        static short Printf(unsigned short row, unsigned short col,
                           unsigned char attr,
                           const char * format, ...);

        // data retrieval methods
        static void GetChar(unsigned short row, unsigned short col,
                            unsigned char & attr, char & ch);

        static unsigned short GetWord(unsigned short row, unsigned short col);

        // box display method
        static void DrawBox(unsigned short toprow, unsigned short leftCol,
                            unsigned short btmrow, unsigned short rightCol,
                            unsigned char attr, BoxType typeBox);

        // scrolling methods
        static void ScrollUp(unsigned short toprow, unsigned short leftCol,
                             unsigned short btmrow, unsigned short rightCol,
                             unsigned char attr, unsigned short noOfrows);

        static void ScrollDown(unsigned short toprow, unsigned short leftCol,
                               unsigned short btmrow, unsigned short rightCol,
                               unsigned char attr, unsigned short noOfrows);

        // screen clearing methods
        static void Clear();
        static void Clear(unsigned short row, unsigned short col = 0);
    };

// obtain class version number
inline short Screen::Version()
    {
    return 320;
    }

// get current type of screen
inline ScrType Screen::GetType()
    {
    return VideoType;
    }

// get video mode
inline ScrMode Screen::GetMode(void)
    {
    return VideoMode;
    }

// set video access type
inline void Screen::SetAccess(ScrAccess access)
    {
    AccessMode = access;
    }

// get video access type
inline ScrAccess Screen::GetAccess(void)
    {
    return AccessMode;
    }

// access methods
inline void Screen::Dimensions(unsigned short & len, unsigned short & wid)
    {
    len = Length;
    wid = Width;
    }

// cursor positioning
inline void Screen::CursorSetPos(unsigned short row, unsigned short col)
    {
    _BH = 0;
    _DH = (char)row;
    _DL = (char)col;
    _AH = 2;

    geninterrupt(0x10);
    }

inline void Screen::CursorGetPos(unsigned short & row, unsigned short & col)
    {
    _BH = 0;
    _AH = 3;

    geninterrupt(0x10);

    row = _DH;
    col = _DL;
    }

// character display method
inline void Screen::Put(unsigned short row, unsigned short col,
                        unsigned char attr, char ch)
    {
    if (AccessMode == SCR_ACC_DIRECT)
        VideoMem[row * Width + col] = ((unsigned int)attr << 8) | (unsigned char)ch;
    else
        {
        _BH = 0;
        _DH = (char)row;
        _DL = (char)col;
        _AH = 2;

        geninterrupt(0x10);

        _BH = 0;
        _BL = attr;
        _CX = 1;
        _AH = 9;
        _AL = ch;

        geninterrupt(0x10);
        }
    }

inline void Screen::Put(unsigned short row, unsigned short col,
                        unsigned short word)
    {
    if (AccessMode == SCR_ACC_DIRECT)
        VideoMem[row * Width + col] = word;
    else
        {
        _BH = 0;
        _DH = (char)row;
        _DL = (char)col;
        _AH = 2;

        geninterrupt(0x10);

        _BH = 0;
        _BL = word >> 8;
        _CX = 1;
        _AH = 9;
        _AL = word & 0xFF;

        geninterrupt(0x10);
        }
    }

// data retrieval methods
inline void Screen::GetChar(unsigned short row, unsigned short col,
                            unsigned char & attr, char & ch)
    {
    if (AccessMode == SCR_ACC_DIRECT)
        {
        unsigned short videoData = VideoMem[row * Width + col];

        attr = (unsigned char)(videoData >> 8);
        ch   = (char)(videoData & 0xFF);
        }
    else
        {
        _BH = 0;
        _DH = (char)row;
        _DL = (char)col;
        _AH = 2;

        geninterrupt(0x10);

        _BH = 0;
        _AH = 8;

        geninterrupt(0x10);

        attr = _AH;
        ch   = _AL;
        }
    }

inline unsigned short Screen::GetWord(unsigned short row, unsigned short col)
    {
    if (AccessMode == SCR_ACC_DIRECT)
        {
        return VideoMem[row * Width + col];
        }
    else
        {
        _BH = 0;
        _DH = (char)row;
        _DL = (char)col;
        _AH = 2;

        geninterrupt(0x10);

        _BH = 0;
        _AH = 8;

        geninterrupt(0x10);
        }
    }

// scrolling methods
inline void Screen::ScrollUp(unsigned short topRow, unsigned short leftCol,
                             unsigned short btmRow, unsigned short rightCol,
                             unsigned char attr, unsigned short noOfRows)
    {
    _BH = (char)attr;
    _CH = (char)topRow;
    _CL = (char)leftCol;
    _DH = (char)btmRow;
    _DL = (char)rightCol;
    _AH = 6;
    _AL = (char)noOfRows;

    geninterrupt(0x10);
    }

inline void Screen::ScrollDown(unsigned short topRow, unsigned short leftCol,
                               unsigned short btmRow, unsigned short rightCol,
                               unsigned char attr, unsigned short noOfRows)
    {
    _BH = (char)attr;
    _CH = (char)topRow;
    _CL = (char)leftCol;
    _DH = (char)btmRow;
    _DL = (char)rightCol;
    _AH = 7;
    _AL = (char)noOfRows;

    geninterrupt(0x10);
    }

#endif
