//
// str.h   : str class interface
// Author  : Roy S. Woll
//
// Copyright (c) 1993 by Roy S. Woll
// You may distribute this source freely as long as you leave all files
// in their original form, including the copyright notice as is.
//
// Version 3.0     05/07/94  Borland 4 compatibility, reverse searching
//                           ltostr, strtol, itostr, strtoi, hashValue
// Version 2.2     05/16/93  Add member function read, lowercase,
//                           uppercase, and variations of pad and strip.
// Version 2.01    03/01/93  Support Turbo C++ (no nested class)
// Version 2.00    12/01/92
//
#ifndef _STR_H
#define _STR_H

//#define DEBUG_STR

#include <dlldefs.h>

class _CLASSTYPE ostream;
class _CLASSTYPE istream;
class _CLASSTYPE dynstream;
class _CLASSTYPE regX;

#ifdef __TURBOC__
	#ifndef __BORLANDC__
		#define _SUBSTR substr
	#else
		#define _SUBSTR str::substr
		class _CLASSTYPE str {
		public:
	#endif
#else
	#define _SUBSTR str::substr
	class _CLASSTYPE str {
  	public:
#endif

	class _CLASSTYPE substr{
     friend class str;

     int posReplace;
     int numReplace;
     str * mystr;

     substr(const str * data, int AposReplace, int AnumReplace);

     int compare(const char *) const;
     int compare(const substr&) const;
     int length() const;
     
   public:
     str & operator = (const substr& s);
	  str & operator = (const char * s);
     
     str operator+(const char *) const;
     str operator+(const substr&) const;
     int operator==(const char *) const;
     int operator<=(const char *) const;
     int operator>=(const char *) const;
     int operator!=(const char *) const;
     int operator< (const char *) const;
     int operator> (const char *) const;
     int operator==(const substr&) const;
     int operator<=(const substr&) const;
     int operator>=(const substr&) const;
     int operator!=(const substr&) const;
     int operator< (const substr&) const;
     int operator> (const substr&) const;
     
     operator str() const;
	};

#ifdef __TURBOC__
	#ifndef __BORLANDC__
		class _CLASSTYPE str {
	#endif
#endif

   friend class dynstreambuf;

protected:
   enum {STR_DEFAULT_MEMINCR=256};
   enum {ICASE=1};

#ifdef DEBUG_STR
public:
   static int dynstreamCount;
   static int ObjectCount;
   static int AllocationCount;
   static int TotalObjectCount;
   static int TotalAllocationCount;
#endif

   friend class substr;


   struct strdata{
     enum {STR_DEBUG_BUFSIZE=80};
     int cursize;         // cursize should be defined before buf!
     int curlength;       // current length of buffer
     int refcount;

     dynstream * mystream;// pointer to str's stream
     char strChange;      // Internal flag indicating when the length has been
                          // changed by the str class.  
                          // If true, stream is notified.
     char buf[STR_DEBUG_BUFSIZE];
                         // buf[1] is all that is required, but we allow
                         //   it to be buf[80] so that the debugger can
                         //   treat it as a char *.
   };

	static strdata NullData;
   static int defaultFlags;

   int memsize_init;     // Initial amount of memory to allocate for buffer
   int memsize_incr;     // Memory expansion increment
   int flags;            // Default flags for case sensitivity, etc.

   str& _assign (const char * source, int len);
   str& _concat (const char * source, int len);
   int _checkMemAllocation(int requiredLen=0);

   static void init(strdata *&, int, int);

protected:

   strdata * data;
   char * getNewBuffer( int len, int newbufsize);
   char * getNewBuffer(int newbufsize);

   void setNewBuffer(strdata * newdata, int newbufsize, int len);

   void setlength(int len) const;   // update the current length

   int strncmp(const char * s1, const char * s2, int n) const;
   int strcmp(const char * s1, const char * s2) const;

//
// Special constructors used by implementation
//
   str(const char *, const char *);
   str(const char *, const _SUBSTR&);
   str(const substr&, const char *);
   str(const substr&, const substr&);


public:

   enum {END_OF_STRING=-1};
//
// Constructors
//
   str(void);
   str(int p_bufsize, int = STR_DEFAULT_MEMINCR);
   str(const char * s, int bufsize=0, int = STR_DEFAULT_MEMINCR);
   str(const str&, int bufsize=0, int = STR_DEFAULT_MEMINCR);

   virtual ~str();

//
// Access Operators
//
   int size(void) const;        // return current memory allocated for buffer
   int length(void) const;      // return current string length of buffer

   operator const char * () const;    // for implicit casting 
   const char * operator()() const;           // for explicit casting
   const char * operator()(int index) const;  // 

//
// String substitution using substr "() syntax"
//
   substr operator()(int pos, int numreplace); // create substr from str
   const substr operator()(int pos, int numreplace) const;

//
// String search/replace member functions
//
   int search (char, int * startPtr) const;
   int search (char, int start=0) const;
   int search (const char *, int * startPtr) const;
   int search (const char *, int start=0) const;
   int search (const regX&, int * startPtr) const;
   int search (const regX&, int start) const;
   int search (const regX&, str * matchPtr=0, int start=0) const;
   int search (const regX&, str * matchPtr, int * startPtr) const;
   int searchr (char, int * startPtr) const;
   int searchr (char, int start=0) const;
   int searchr (const char *, int * startPtr) const;
   int searchr (const char *, int start=END_OF_STRING) const;
   int searchr (const regX&, int * startPtr) const;
   int searchr (const regX&, int start) const;
   int searchr (const regX&, str * matchPtr=0, int start=END_OF_STRING) const;
   int searchr (const regX&, str * matchPtr, int * startPtr) const;

   int index (char, int start=0) const;
   int index (const char *, int start=0) const;
   int index (const regX&, int start=0) const;
   int index (const regX&, int * matchLen, int start=0) const;
   int indexr (char, int start=END_OF_STRING) const;
   int indexr (const char *, int start=END_OF_STRING) const;
   int indexr (const regX&, int start=END_OF_STRING) const;
   int indexr (const regX&, int * matchLen, int start=END_OF_STRING) const;


   int replace (const regX&, const char * replaceString, 
               int* startPtr, int numReplacements=1);
	int replace (const char * pattern, const char * replaceString,
               int* startPtr, int numReplacements=1);
   int replace (const regX&, const char * replaceString, 
               int start=0, int numReplacements=1);
   int replace (const char * pattern, const char * replaceString, 
               int start=0, int numReplacements=1);
	int replaceAll (const char *, const char * replaceString, int start=0);
	int replaceAll (const regX&, const char * replaceString, int start=0);
	int replacer (const regX&, const char * replaceString,
               int* startPtr, int numReplacements=1);
   int replacer (const regX&, const char * replaceString, 
               int start=END_OF_STRING, int numReplacements=1);
   int replacer (const char * pattern, const char * replaceString, 
               int* startPtr, int numReplacements=1);
   int replacer (const char * pattern, const char * replaceString, 
               int start=END_OF_STRING, int numReplacements=1);

	int count(const char ch, int start=0) const; // return occurences of ch
	int count(const char *, int start=0);
	int count(const regX&, int start=0);

   char& operator[](int position);            // array indexing
   char operator[](int position) const;       // array indexing for const

   ostream& stream(void);    // return stream for this str
   ostream& stream(int);     // return stream and move put pointer

//
// Assignment & Concatenation Operators
//
   str & operator = (const str & s);     // s = str;
   str & operator = (const substr & s);  // s = substr;
   str & operator = (const char * s);    // s = charptr
   str & operator = (const char s);      // s = character
   str & assign (const char * source, int len=32767);

   str & operator += (const str & s);    // s += str
   str & operator += (const substr & s); // s += substr
   str & operator += (const char * s);   // s += charptr
   str & operator += (const char s);     // s += char

   str & operator << (const str& s);     // s << str
   str & operator << (const substr& s);  // s << substr
   str & operator << (const char * s);   // s << charptr
   str & operator << (const int s);      // s << int
   str & operator << (const char s);     // s << char
   str & operator << (const long s);     // s << long

// Manipulators
   enum PadStripT {left, right, both, leading=0, trailing};
   
   str& strip(PadStripT t=trailing, const char * stripchar=" \t");
   str& strip(PadStripT t, char stripchar);
   str& stripTrailing(const char * stripchar=" \t");
   str& stripLeading(const char * stripchar=" \t");
	str& stripBoth(const char * stripchar=" \t");
	
   str& pad(int padsize, PadStripT t=right, char padchar = ' ');
   str& padRight(int padsize, char padchar = ' ');
   str& padLeft(int padsize, char padchar = ' ');
   str& padBoth(int padsize, char padchar = ' ');
	
   str& lowercase(); // convert this string to lowercase
   str& uppercase(); // convert this string to uppercase

	
// Mutators
   int insert(int pos, char ch);       // insert ch, return 0 if fail
   int insert(int pos, const char * insertStr); 
                                       // insert char *, return 0 if fail
   void remove(int pos, int numdel=1); // remove numdel characters starting at
                                       //   position pos
   void setLength(int len);            // update the current length

   str operator+(const str &) const;
   str operator+(const substr &) const;
   str operator+(const char * b) const;
   str operator+(const char b) const;

   int operator==(const char *) const;  // made member functions so that
   int operator<=(const char *) const;  // derived classes inherit functions
   int operator>=(const char *) const;
   int operator!=(const char *) const;
   int operator< (const char *) const;
   int operator> (const char *) const;

   int operator==(const str &) const;
   int operator<=(const str &) const;
   int operator>=(const str &) const;
   int operator!=(const str &) const;
   int operator< (const str &) const;
   int operator> (const str &) const;

// Friend Stream Operators
	istream& read(istream&, int);
   friend istream& _ExportFunction operator >> (istream&, str &);
   friend ostream& _ExportFunction operator << (ostream&, const str &);

   friend int _ExportFunction compare(const str&, const char *);
   friend int _ExportFunction compare(const str&, const str &);
   friend int _ExportFunction compare(const char *, const str &);

   friend str _ExportFunction operator+(const char *, const str &); 
   friend str _ExportFunction operator+(const char *, const _SUBSTR &); 

   friend str _ExportFunction strip(str, PadStripT t=trailing, 
		                      const char * stripchar=" \t");
   friend str _ExportFunction strip(str, PadStripT t, char stripchar);
   friend str _ExportFunction pad(str s, int padsize, PadStripT t=right, 
                          char padchar= ' ');
   friend str _ExportFunction lowercase(const char *); // return lower case of string
   friend str _ExportFunction uppercase(const char *); // return upper case of string
   friend str _ExportFunction itostr(int);             // convert int to str
	friend int _ExportFunction strtoi(const str&);      // convert str to int
	friend str _ExportFunction ltostr(long);             // convert long to str
	friend long _ExportFunction strtol(const str&);      // convert str to long


 //
 // Case sensitivity functions
 //
   static void setdefaultCaseSensitive(int val);
   void setCaseSensitive(int val);
   int caseSensitive(void) const;
	
   unsigned HashValue() const;
	
};

//
// Define inline functions
//
inline const char * str::operator()() const   // for explicit casting
     {return (const char *)*this;}

inline char str::operator[](int position) const{  // array indexing
   return data->buf[position];
}

inline void str::setlength(int len) const{   // update the current length
   data->buf[len]=0;     // null terminated
   data->curlength=len;
   data->strChange=1;
}

//
// Global member function operators
//
   int _ExportFunction operator==(const char *, const str &);
   int _ExportFunction operator<=(const char *, const str &);
   int _ExportFunction operator>=(const char *, const str &);
   int _ExportFunction operator!=(const char *, const str &);
   int _ExportFunction operator< (const char *, const str &);
   int _ExportFunction operator> (const char *, const str &);

   int _ExportFunction operator==(const char *, const _SUBSTR &);
   int _ExportFunction operator<=(const char *, const _SUBSTR &);
   int _ExportFunction operator>=(const char *, const _SUBSTR &);
   int _ExportFunction operator!=(const char *, const _SUBSTR &);
   int _ExportFunction operator< (const char *, const _SUBSTR &);
   int _ExportFunction operator> (const char *, const _SUBSTR &);


#endif
