/*
   Strip Chart Library Multiple Trace Windows Example.
   This program draws a user-specified number of traces.

   Copyright (c) 1993 by Christopher Lim.  All rights reserved.
*/

/*
   Function func() simulates digitizing a sine wave.
   SPS defined below is the simulated sampling rate.
   The program can be made to run faster by using a lower sampling rate.
*/
#define SPS 300.0


#include "sc.h"
#include <stdio.h>
#include <math.h>
#include <conio.h>
#include <stdlib.h>
#include <malloc.h>

#ifdef BORLAND
#include <alloc.h> /* for coreleft() */
#endif

#define MIN_YHEIGHT (0.25+0.125) /* minimum trace height */
#define MAX_AMPLITUDE 1.0 /* max trace amplitude */
#define MARGIN (1.375) /* space for title and time windows */

/*------------------------------------------------------------*/

int FreeMem(void)
{
   /* Prints out free memory.
      This function is compiler dependent.
    */
#ifdef BORLAND
    printf("Free memory: %ld\n",(unsigned long)coreleft());
#endif
#ifdef MICROSOFT
   printf("Free memory: %ld\n",(unsigned long)_memavl());
#endif
    return 0;
}

/*------------------------------------------------------------*/

double func(int n, double t)
{
   /* Test function response for time t for trace n.
      To make the plot look more interesting, we will have
      8 types of traces.
      First four traces: sine waves with constant amplitude.
      Next four traces: sine waves with periodic amplitudes.
      Pattern repeats as above for each set of 8 traces.
    */
   double x;
   double a; /* amplitude */
   double freq; /* sine wave frequency in hertz */
   double pi = 3.1416;

   switch (n%8) /* index into the set of 8 traces */
   {
      case 0:
         a = MAX_AMPLITUDE;
         freq = 2.0; /* 2 hz */
         break;
      case 1:
         a = MAX_AMPLITUDE;
         freq = 4.0; /* 4 hz */
         break;
      case 2:
         a = MAX_AMPLITUDE;
         freq = 8.0; /* 8 hz */
         break;
      case 3:
         a = MAX_AMPLITUDE;
         freq = 16.0; /* 16 hz */
         break;
      case 4:
         a = cos(2 * pi * t * 0.5);
         freq = 2.0; /* 2 hz */
         break;
      case 5:
         a = cos(2 * pi * t * 0.5);
         freq = 4.0; /* 4 hz */
         break;
      case 6:
         a = cos(2 * pi * t * 0.5);
         freq = 8.0; /* 8 hz */
         break;
      case 7:
         a = cos(2 * pi * t * 0.5);
         freq = 16.0; /* 16 hz */
         break;
   }
   x = a * sin(2 * pi * t * freq);
   return(x);
}

/*------------------------------------------------------------*/

void GetPrinterType(unsigned char *printerType, unsigned char *printerRes,
   char **s)
{
   int c;
   unsigned char i,j;

   printf("Select printer type and resolution code: \n");
   printf("[1] 9 PIN LOW RES\n");
   printf("[2] 9 PIN HIGH RES\n");
   printf("[3] 24 PIN LOW RES\n");
   printf("[4] 24 PIN HIGH RES\n");
   printf("Select code [1,2,3 or 4]: ");
   c = getch();
   printf("%c\n",c);
   switch (c)
   {
        case '1': i = SC_9PIN; j = SC_LOW_RES;
             *s = "9 PIN LOW RES";
             break;
        case '2': i = SC_9PIN; j = SC_HIGH_RES;
             *s = "9 PIN HIGH RES";
             break;
        case '3': i = SC_24PIN; j = SC_LOW_RES;
             *s = "24 PIN LOW RES";
             break;
        case '4': i = SC_24PIN; j = SC_HIGH_RES;
             *s = "24 PIN HIGH RES";
             break;
        default:
             printf("Aborted\n");
             exit(1);
   }
   printf("%s\n",*s);
   *printerType = i;
   *printerRes = j;
   return;
}
/*------------------------------------------------------------*/

void GetPlotParms(double *plotWidth, double *plotDuration, int *ntraces)
{
   /* Get plot parameters from user */
   int maxtraces;
   char buf[80];

   while (1)
   {
      printf("Enter plot or paper width (inches): ");
      gets(buf);
      *plotWidth = atof(buf);
      if (*plotWidth < 4.0 || *plotWidth > 13.6)
         printf("\nPlot Width must be greater than 4.0 and less than 13.6 inches!\n");
      else
         break;
   }
   printf("Plot width = %f inches\n", *plotWidth);
   
   while (1)
   {
      printf("Enter plot duration (inches): ");
      gets(buf);
      *plotDuration = atof(buf);
      if (*plotDuration <= 0)
         printf("\nPlot Duration must be a positive number!\n");
      else
         break;
   }
   printf("Plot duration = %f seconds\n",*plotDuration);

   maxtraces = (*plotWidth-MARGIN)/MIN_YHEIGHT;
   while (1)
   {
      printf("Enter number of traces [max=%d]: ",maxtraces);
      gets(buf);
      *ntraces = atoi(buf);
      if (*ntraces <= 0 || *ntraces > maxtraces)
         printf("\nNumber of traces must be greater than zero and less than %d!\n",maxtraces+1);
      else
         break;
   }
   printf("Number of traces = %d\n",*ntraces);
}

/*------------------------------------------------------------*/

char *Label(SC_WINDOW *w, double seconds, unsigned long labelCount)
{
   static char buf[80];
   static int minutes = 0;
   static int hours = 0;

   while (seconds >= 60.0)
   {
      seconds -= 60.0;
      minutes++;
   }
   if (minutes >= 60)
   {
      minutes %= 60;
      hours += minutes/60;
   }
   sprintf(buf,"%02d:%02d:%05.2f",hours,minutes,seconds);
   return buf;
}

/*------------------------------------------------------------*/

void ErrorTrap(char *msg)
{
    printf("%s\n",msg);
    exit(1);
}

/*------------------------------------------------------------*/

void main()
{
   int i,j;
   int error;
   double scan;
   double a;
   SC_WINDOW *text, *time;
   SC_WINDOW **trace;
   double sps = SPS; /* simulated sampling rate */
   double t;
   unsigned char printerType, printerRes;
   char string[128];
   char *s;
   double plotWidth;
   double plotDuration;
   int ntraces;
   double yheight;

   if (!SC_PrinterReady(LPT1))
   {
      printf("Printer at LPT1 is not ready!\n");
      exit(1);
   }

   GetPrinterType(&printerType,&printerRes,&s);
   GetPlotParms(&plotWidth,&plotDuration,&ntraces);

   FreeMem();

   /* Allocate trace window array */
   trace = (SC_WINDOW **)malloc(sizeof(SC_WINDOW *) * ntraces);
   if (trace == NULL)
   {
      printf("Insufficient memory.\n");
      exit(1);
   }

   /* Error trap */
   SC_SetErrorTrap(ErrorTrap);

   /* Initialize: 8.5 inch paper, plot speed = 1 inch per second */
   SC_Init(LPT1, printerType,printerRes, INCHES, plotWidth, 1.0,&error);

   /* Horizontal Grids: 2 divisions */
   SC_SetHorizontalGrids(2,&error);

   /* Time grids every 1 second */
   SC_SetTimeGrids(1.0,&error);

   /* Text window to display title */
   text = SC_AddTextWindow(0.5,14.0/72.0,&error);

   /* Add trace windows */
   yheight = (plotWidth-MARGIN)/ntraces - 0.125;
   for (i=0; i < ntraces; i++)
   {
      trace[i] = SC_AddTraceWindow(SC_yoffset+0.125, /* window offset */
         yheight, /* window height */
         -1.0*MAX_AMPLITUDE, MAX_AMPLITUDE,  /* window limits */
         SCF_BORDER|SCF_GRID, /* enable borders and grids */
         &error);
   }

   /* Add time axis at the bottom of the plot */
   time = SC_AddTimeWindow(SC_yoffset+0.125, /* window offset */
      20.0/72.0, /* window height */
      LABELS_BELOW_AXIS, /* labels below axis */
      2.0, /* label every 2 seconds */
      5, /* minor divisions */
      Label, /* call back function */
      &error);


   /* initialize text windows */
   sprintf(string,"STRIP CHART LIBRARY EXAMPLES: %d TRACES (%s)",ntraces, s);
   SC_Text(text,string,&error);

   /* draw traces */
   a = 1.0; /* initial amplitude */
   for (t=0.0; t < plotDuration; t += 1.0/sps)
   {
      if (kbhit())
      {
         printf("User aborted.\n");
         break;
      }

      for (i=0; i < ntraces; i++)
      {
         scan = a * func(i,t);
         SC_Trace(trace[i],scan,&error);
      }

      SC_Draw();
      SC_Advance(1.0/sps); /* next */
   }

   SC_Close();
   
   free(trace);

   FreeMem();
}


