//*************************************************************
//  File name: getsym.c
//
//  Description: 
//      Code for looking up the nearest symbol in the given SYM file
//	Note that the Symbol File Format is documented in Chapter 12 of
//	Windows 3.1 SDK Programmer's Reference, Volume 4: Resources
//
//  History:    Date       Author     Comment
//              8/10/92    MSS        Created
//
// Written by Microsoft Product Support Services, Windows Developer Support
// Copyright (c) 1992 Microsoft Corporation. All rights reserved.
//*************************************************************

#include <windows.h>
#include "getsym.h"

//*** Global Variables
long pos;      

//*************************************************************
//  ValidSymFile
//
//  Purpose:
//      Confirm the validity of the .SYM  before proceeding further
//      The following checks are done:
//        - Contains only 16-bit symbols (bFlags) and not 32-bit symbols
//        - No more than 255 segments (cSegs)
//        - And hence the entry point segment is less than 256
//        - the last MAPDEF record must have a NULL for its NextMap pointer
//
//  Parameters:
//      HFILE symfile
//      MAPDEF FAR *mapdef
//      
//  Return: (int)
//
//  History:    Date       Author     Comment
//              8/10/92   MSS        Created
//
//*************************************************************

int WINAPI ValidSymFile(HFILE symfile, MAPDEF FAR *mapdef)
{
  WORD pLastMap;
  
  //*** Read the MAPDEF at the beginning of the file
  _llseek(symfile, 0, SEEK_SET);
  _lread(symfile, mapdef, sizeof(MAPDEF));
  
  
  if ((mapdef->bFlags) ||  (mapdef->cSegs > 255) || (mapdef->pSegEntry > 255))
     return 0;
  _llseek(symfile, mapdef->ppNextMap*16L, SEEK_SET);
  _lread(symfile, &pLastMap, sizeof(WORD));
  if (pLastMap)
      return 0;
  
  return 1;

} //*** ValidSymFile
    
//*************************************************************
//  GetSEGDEF
//
//  Purpose:
//      This routine fetches the SEGDEF which contains the correct
//      segment number that was passed, by iterating through all 
//      SEGDEFs.  The file pointer is saved for future use              
//
//  Parameters:
//      HFILE symfile
//      MAPDEF FAR *mapdef
//      SEGDEF FAR *segdef
//      WORD segment
//      
//  Return: (int)
//
//  History:    Date       Author     Comment
//              8/18/92     MSS        Created
//
//*************************************************************

int WINAPI GetSEGDEF(HFILE symfile, MAPDEF FAR *mapdef, SEGDEF FAR *segdef, WORD segment)
{

  WORD ppSegDef = mapdef->ppSegDef;
  
  //*** loop through all segments  
  while (mapdef->cSegs--)
  {
      //*** the offset from the beginning of the file to the first SEGDEF is given
      //*** by multiplying ppSegDef of the MAPDEF struct by 16. Read the SEGDEFs
      //*** at this offset
  
      pos =  ppSegDef*16L;              // save this file pointer too
     _llseek(symfile, pos, SEEK_SET);
     _lread(symfile, segdef, sizeof(SEGDEF));

     //*** check if it's the right SEGDEF
     //*** Remember the wSegNum field (corresponding to the wReserved1 field
     //*** of SEGDEF in the SYM file documentation) gives the segment number 
     if (segdef->SegNum == segment)
        return 1;
     //*** move the next SEGDEF   
     else ppSegDef = segdef->ppNextSeg;
  }
   
  //*** We didn't find the correct SEGDEF if we got here
  return 0;      

} //*** GetSEGDEF


//*************************************************************
//  GetSYMDEF
//
//  Purpose:
//      This routine looks for the nearest symbol whose offset is less than or equal
//      to  the given offset.  Note that the SEGDEF contains the pointer to the 
//      the array which contains pointers to SYMDEFs.  This offset array is used to
//      obtain the nearest symbol.  If a nearest symbol is not found it returns 0.
//      
//
//  Parameters:
//      HFILE symfile
//      SEGDEF FAR *segdef
//      WORD offset
//      
//  Return: (char *)
//
//  History:    Date       Author     Comment
//              8/10/92     MSS        Created
//
//*************************************************************

LPSTR WINAPI GetSYMDEF(HFILE symfile, SEGDEF FAR *segdef, WORD offset)
{
  char achTemp[256] = "Unknown";
  HGLOBAL hTable;
  WORD i, FAR* pTable;
  SYMDEF symdef;
  char buffer[MAXSYMSIZE] = "Unknown";
  
  //*** allocating space for the offset table
  if (!(hTable = GlobalAlloc(GPTR, segdef->cSymbols*sizeof(WORD))))
    return 0;
  if (!(pTable = (WORD FAR *) GlobalLock(hTable)))
    return 0;
       
  //*** read the offset array from the .SYM file
  _llseek(symfile, pos + segdef->pSymDef, SEEK_SET);
  _lread(symfile, pTable, segdef->cSymbols*sizeof(WORD));
  
  //*** now read each SYMDEF using the offsets given in the offset array
  for (i=0; i < segdef->cSymbols; i++)
  {
    _llseek(symfile, pos + pTable[i], SEEK_SET);
    _lread(symfile, &symdef, sizeof(SYMDEF));
    
    if (symdef.wSymVal > offset)
      break;
  }

  //*** if i == 0, we did not find the symbol as the given offset is before any symbols
  //*** if not, we just passed the symbol that we wanted, so back up and get that SYMDEF  
  if (i != 0)
  {
    _llseek(symfile, pos + pTable[i-1], SEEK_SET);
    _lread(symfile, &symdef, sizeof(SYMDEF));
    _lread(symfile, achTemp, symdef.cbSymName);
    achTemp[symdef.cbSymName] = 0;

    //*** if the offset of the nearest symbol differs by a large amount from the
    //*** given offset, return 0
    if ((offset - symdef.wSymVal) > MAXDIFFERENCE )
        i = 0;
    else
        wsprintf(buffer, "%s + %#04x", (LPSTR) achTemp, offset - symdef.wSymVal);
  }
  
  GlobalUnlock(hTable);
  GlobalFree(hTable);

  if (i)
    return (LPSTR)buffer;  
  else 
    return NULL;

} //*** GetSYMDEF

//*************************************************************
//  FindSymbol
//
//  Purpose:
//      Given a handle to an open SYM file and the segment:offset of the
//      logical address, this routine finds the nearest symbol name in
//      SYM file and retruns it in 'buffer'.  If no matching symbol found,
//      it returns null.
//      
//
//  Parameters:
//      HFILE symfile
//      WORD segment
//      WORD offset
//      
//  Return: (char *)
//
//  History:    Date       Author     Comment
//              8/10/92     MSS        Created
//
//*************************************************************

LPSTR WINAPI FindSymbol (HFILE symfile, WORD segment, WORD offset)
{

  MAPDEF mapdef;
  SEGDEF segdef;
 
  //*** Check if it is a valid .SYM file, return if not
  if (!ValidSymFile(symfile, &mapdef))
     return NULL;
     
  //*** Get to the correct SEGDEF which contains the given segment number
  if (!GetSEGDEF(symfile, &mapdef, &segdef, segment))
    return NULL;

  //*** Get to the correct SYMDEF and use the given offset to get the 
  //*** "nearest" symbol name. 
  return GetSYMDEF(symfile, &segdef, offset);

} //*** FindSymbol
  
//*** EOF: getsym.c
