/****************************************************************************

    PROGRAM: StkTrace.c

    PURPOSE: StkTrace Sample Application
    
    FUNCTIONS:

     WinMain() - calls initialization function, processes message loop
     InitApplication() - initializes window data and registers window
     InitInstance() - saves instance handle and creates main window
     MainWndProc() - processes messages
     About() - processes messages for "About" dialog box

    COMMENTS:

        Windows can have several copies of your application running at the
        same time.  The variable hInst keeps track of which instance this
        application is so that processing will be to the correct window.

****************************************************************************/

#include <windows.h>              
#include "stktrace.h"              /* specific to this program              */

HANDLE ghInst;                      /* current instance                 */

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point 
        for the program.  This function calls the application initialization 
        routine, if no other instance of the program is running, and always 
        calls the instance initialization routine.  It then executes a message 
        retrieval and dispatch loop that is the top-level control structure 
        for the remainder of execution.  The loop is terminated when a WM_QUIT 
        message is received, at which time this function exits the application 
        instance by returning the value passed by PostQuitMessage(). 

        If this function must abort before entering the message loop, it 
        returns the conventional value NULL.  

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                  /* current instance      */
HANDLE hPrevInstance;                   /* previous instance          */
LPSTR lpCmdLine;                   /* command line               */
int nCmdShow;                      /* show-window type (open/icon) */
{
    MSG msg;                       /* message                    */

    if (!hPrevInstance)             /* Other instances of app running? */
     if (!InitApplication(hInstance)) /* Initialize shared things */
         return (FALSE);       /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,      /* message structure                 */
         NULL,         /* handle of window receiving the message */
         NULL,         /* lowest message to examine              */
         NULL))             /* highest message to examine        */
     {
     TranslateMessage(&msg);     /* Translates virtual key codes      */
     DispatchMessage(&msg);      /* Dispatches message to window      */
    }
    return (msg.wParam);    /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other 
        instances of the application are running.  This function performs 
        initialization tasks that can be done once for any number of running 
        instances.  

        In this case, we initialize a window class by filling out a data 
        structure of type WNDCLASS and calling the Windows RegisterClass() 
        function.  Since all instances of this application use the same window 
        class, we only need to do this when the first instance is initialized.  


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                    /* current instance         */
{
   WNDCLASS  wc;
   char szMenuName[STR_LEN];
   char szClassName[STR_LEN];

    // Load strings from the string table.
    LoadString(hInstance, IDS_MAINMENUNAME, szMenuName, STR_LEN);
    LoadString(hInstance, IDS_MAINCLASSNAME, szClassName, STR_LEN);

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName =  szMenuName;      /* Name of menu resource in .RC file. */
    wc.lpszClassName = szClassName;     /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of 
        this application.  This function performs initialization tasks that 
        cannot be shared by multiple instances.  

        In this case, we save the instance handle in a static variable and 
        create and display the main program window.  
        
****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
HANDLE          hInstance;          /* Current instance identifier.       */
int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */
    char            szCaption[STR_LEN];
    char            szClassName[STR_LEN];

    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */
    ghInst = hInstance;
    
   // Load strings from the string table.
   LoadString(hInstance, IDS_CAPTION, szCaption, STR_LEN);
   LoadString(hInstance, IDS_MAINCLASSNAME, szClassName, STR_LEN);
   
   /* Create a main window for this application instance.  */
    
   hWnd = CreateWindow(
        szClassName,                /* See RegisterClass() call.          */
        szCaption,                  /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW,            /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
   );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    return (TRUE);               /* Returns the value from PostQuitMessage */

}



//****************************************************************************
// Function: MainWndProc
//
// Purpose: Message handler for main overlapped window.
//
// Parameters:
//    hWnd    == Handle to _this_ window.
//    message == Message to process.
//    wParam  == WORD parameter -- depends on message
//    lParam  == LONG parameter -- depends on message
//
// Returns: Depends on message.
//
// Comments:
//
// History:  Date       Author                  Reason
//          8/23/92   Sridhar Mandyam          Created
//****************************************************************************

long FAR PASCAL MainWndProc (hWnd, message, wParam, lParam)
HWND hWnd;
UINT message;
WPARAM wParam;
LPARAM lParam;
{
   switch (message)
   {
      // Dispatch WM_COMMAND messages to our command handler, DoCommands().
      case WM_COMMAND:
         return DoCommands(hWnd, message, wParam, lParam);


      // On WM_DESTROY, terminate this app by posting a WM_QUIT message.
      case WM_DESTROY:
         PostQuitMessage(0);
         break;

      // We didn't handle, pass to DefWindowProc.
      default:
         return DefWindowProc(hWnd, message, wParam, lParam);
   }

   return (NULL);
}

//****************************************************************************
// Function: DoCommands
//
// Purpose: Called by MainWndProc() to handle all WM_COMMAND messages.
//
// Parameters:
//    hWnd    == Handle to _this_ window.
//    message == Message to process.
//    wParam  == WORD parameter -- depends on message
//    lParam  == LONG parameter -- depends on message
//
// Returns : Depends on message.
//
// Comments:
//
// History:  Date       Author                  Reason
//           8/23/92  Sridhar Mandyam           Created
//****************************************************************************

long DoCommands (hWnd, message, wParam, lParam)
HWND hWnd;
unsigned message;
WORD wParam;
LONG lParam;
{
   char buffer[256];
   char szTitle[STR_LEN];
   char szError[STR_LEN];

   switch (wParam)
   {
       // Call the Stack Trace function 
       case IDM_TRACE:
       {
         LoadString(ghInst, IDS_STKTITLE, szTitle, STR_LEN);
         LoadString(ghInst, IDS_ERRORMSG, szError, STR_LEN);
         
         if (GetStackTrace((LPSTR) buffer, sizeof(buffer), NUMTRACES))
         {
            OutputDebugString((LPSTR) buffer);
            MessageBox(hWnd, (LPCSTR) buffer, (LPCSTR) szTitle, MB_OK);
         }
         else 
            MessageBox(hWnd, (LPCSTR) szError, (LPCSTR) szTitle, MB_OK);
         break;
       }
   
      // Put up the About box.
      case IDM_ABOUT:
      {
         FARPROC lpAboutDlgProc;

         lpAboutDlgProc = MakeProcInstance((FARPROC) AboutDlgProc, ghInst);

         DialogBox(ghInst,             // current instance
                   "AboutBox",              // resource to use 
                   hWnd,               // parent handle   
                   lpAboutDlgProc);    // About() instance address

         FreeProcInstance(lpAboutDlgProc);
         break;
      }

      // User picked File.Exit, terminate this app.
      case IDM_EXIT:
         DestroyWindow(hWnd);
         break;


      // Must be some system command -- pass it on to the default
      //  window procedure.
      default:
         return DefWindowProc(hWnd, message, wParam, lParam);
   }

   return (NULL);
}

/****************************************************************************

    FUNCTION: AboutDlgProc(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:
     WM_INITDIALOG - initialize dialog box
     WM_COMMAND    - Input received

    COMMENTS:
       Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL FAR PASCAL AboutDlgProc(hDlg, message, wParam, lParam)
HWND hDlg;                                /* window handle of the dialog box */
unsigned message;                         /* type of message                 */
WORD wParam;                              /* message-specific information    */
LONG lParam;
{
        
    switch (message) {
     case WM_INITDIALOG:         /* message: initialize dialog box */
         return (TRUE);

     case WM_COMMAND:               /* message: received a command */
         if (wParam == IDOK                /* "OK" box selected?      */
                || wParam == IDCANCEL) {      /* System menu close command? */
            
          EndDialog(hDlg, TRUE);         /* Exits the dialog box      */
          return (TRUE);
         }
         break;
    }
    return (FALSE);                 /* Didn't process a message    */
}


// End stktrace.c
