/* ------------------------------------------------------------------------ */
/*                                 sstwin.c                                 */
/*                         high-level window routines                       */
/*                                                                          */
/*      CopyRight (C) 1991,1992  Steven Lutrov.   All rights reserved.      */
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include <ctype.h>
#include <stdarg.h>
#include <dos.h>
#include <alloc.h>
#include <stdlib.h>
#include <string.h>

#include "sstwin.h"
#include "sstkey.h"

/* ------------------------------------------------------------------------ */
/*                            local prototypes                              */
/* ------------------------------------------------------------------------ */

void redraw       (WINDOW *wnd);
void wframe       (WINDOW *wnd);
void dtitle       (WINDOW *wnd);
int  *waddr       (WINDOW *wnd, int x, int y);
void vswap        (WINDOW *wnd);
void add_list     (WINDOW *wnd);
void beg_list     (WINDOW *wnd);
void remove_list  (WINDOW *wnd);
void insert_list  (WINDOW *w1, WINDOW *w2);
int dget          (WINDOW *wnd, int x, int y);
verify_wnd        (WINDOW **w1);


/* ------------------------------------------------------------------------ */
/*                   array of border character sets                         */
/* ------------------------------------------------------------------------ */
struct {
	int topleft, topright, botright, botleft, vert, hor;
       } wcs[] = {
	{ 32, 32, 32, 32, 32, 32},	/* blank line */
	{218,191,217,192,179,196},	/* single line */
	{201,187,188,200,186,205},	/* double line */
	{214,183,189,211,186,196},	/* single top, double side */
	{213,184,190,212,179,205},	/* double top, single side */
	{194,194,217,192,179,196},	/* single - pop-down menu */
	{203,203,188,200,186,205},	/* double - pop-down menu */
	{210,210,189,211,186,196},	/* single top pop-down menu */
	{209,209,190,212,179,205}	/* double top pop-down menu */
};

/* ------------------------------------------------------------------------ */
/*                 window structure linked list head & tail                 */
/* ------------------------------------------------------------------------ */

WINDOW *listhead = NULL;     /* head address */
WINDOW *listtail = NULL;     /* tail address */

/* ------------------------------------------------------------------------ */
/*                          establish a new window                          */
/* ------------------------------------------------------------------------ */
WINDOW *Westablish(int x, int y, int h, int w)
{
	WINDOW *wnd;

	if ((wnd = (WINDOW *) malloc(sizeof (WINDOW))) == NULL)
		return NULL;
	WTITLE = "";
	TJUST = JUST_C;
	HEIGHT = min(h, SCREENHEIGHT);
	WIDTH = min(w, SCREENWIDTH);
	COL = max(0, min(x, SCREENWIDTH-WIDTH));
	ROW = max(0, min(y, SCREENHEIGHT-HEIGHT));
	WCURS = 0;
	SCROLL = 0;
	SELECT = 1;
	BTYPE = 1;
	VISIBLE = HIDDEN = 0;
	PREV = NEXT = NULL;
	FHEAD = FTAIL = NULL;
	WBORDER=WFACE=PWIN_FACE=WTITLEC=WFIELD = CLR(BLACK, WHITE, BRIGHT);
	WACCENT = CLR(WHITE, BLACK, DIM);
	if ((SAV = malloc(WIDTH * HEIGHT * 2)) == (char *) 0)
		return NULL;
	add_list(wnd);
	Wclear(wnd);
	wframe(wnd);
	return wnd;
}

/* ------------------------------------------------------------------------ */
/*                         set the window's border                          */
/* ------------------------------------------------------------------------ */
void Wsetborder(WINDOW *wnd, int btype)
{
   if (verify_wnd(&wnd)) {
	BTYPE = btype;
	redraw(wnd);
  }
}


/* ------------------------------------------------------------------------ */
/*                               set colours                                */
/* ------------------------------------------------------------------------ */
void Wsetcolour(WINDOW *wnd,int area,int bg,int fg,int inten)
{
	if (ISMONO() )	{
		if (bg != WHITE && bg != BLACK)
			return;
		if (fg != WHITE && fg != BLACK)
			return;
	}
	if (verify_wnd(&wnd))	{
		if (area == WIN_ALL)
			while (area)
		WCOLOUR [--area] = CLR(bg, fg, inten);
		else
	    WCOLOUR [area] = CLR(bg, fg, inten);
		redraw(wnd);
	}
}

/* ------------------------------------------------------------------------ */
/*                      set the intensity of a window                       */
/* ------------------------------------------------------------------------ */
void Wsetintensity(WINDOW *wnd, int inten)
{
	int area = WIN_ALL;

	if (verify_wnd(&wnd))	{
		while (area)	{
            WCOLOUR [--area] &= ~BRIGHT;
            WCOLOUR [area] |= inten;
		}
		redraw(wnd);
	}
}

/* ------------------------------------------------------------------------ */
/*                               set title                                  */
/* ------------------------------------------------------------------------ */
void Wsettitle(WINDOW *wnd, char *title, int just)
{
	if (verify_wnd(&wnd))	{
		WTITLE = title;
		TJUST = just;
		redraw(wnd);
	}
}

/* ------------------------------------------------------------------------ */
/*                redraw a window when an attribute changes                 */
/* ------------------------------------------------------------------------ */
static void redraw(WINDOW *wnd)
{
    int x, y, chat, atr;

    for (y = 1; y < HEIGHT-1; y++)
	for (x = 1; x < WIDTH-1; x++)	{
		chat = dget(wnd, x, y);
		atr = (((chat>>8)&255) == PWIN_FACE ? WFACE : WACCENT);
		  displ(wnd, x, y, chat&255, atr);
		}
    wframe(wnd);
    PWIN_FACE = WFACE;
}

/* ------------------------------------------------------------------------ */
/*                      display an established window                       */
/* ------------------------------------------------------------------------ */
void Wshow(WINDOW *wnd)
{
  if (verify_wnd(&wnd) && !VISIBLE)	{
	VISIBLE = 1;
	vswap(wnd);
  }
}

/* ------------------------------------------------------------------------ */
/*                          close all windows                               */
/* ------------------------------------------------------------------------ */
void Wcloseall()
{
	WINDOW *sav, *wnd = listtail;

	while (wnd)	{
		sav = PREV;
		Wdelete(wnd);
		wnd = sav;
	}
}

/* ------------------------------------------------------------------------ */
/*                           remove a window                                */
/* ------------------------------------------------------------------------ */
void Wdelete(WINDOW *wnd)
{
	if (verify_wnd(&wnd))		{
		Whide(wnd);
		free(SAV);
		remove_list(wnd);	/* remove window from list */
		free(wnd);
	}
}

/* ------------------------------------------------------------------------ */
/*                             hide a window                                */
/* ------------------------------------------------------------------------ */
void Whide(WINDOW *wnd)
{
	if (verify_wnd(&wnd) && VISIBLE)	{
		vswap(wnd);
		HIDDEN = 1;
		VISIBLE = 0;
	}
}

/* ------------------------------------------------------------------------ */
/*                 reposition the window in its 3-axis plane                */
/* ------------------------------------------------------------------------ */
void repos_wnd(WINDOW *wnd, int x, int y, int z)
{
    WINDOW *twnd;
    int x1, y1, chat;

    if (!verify_wnd(&wnd))
		return;
    twnd = Westablish(x+COL, y+ROW, HEIGHT, WIDTH);
    twnd->_tl = WTITLE;
    twnd->_tjust = TJUST;
    twnd->btype = BTYPE;
    twnd->wcolour[WIN_BORDER] = WBORDER;
    twnd->wcolour[WIN_TITLE] = WTITLEC;
    twnd->wcolour[WIN_ACCENT] = WACCENT;
    twnd->wcolour[WIN_FACE] = WFACE;
	twnd->_wsp = SCROLL;
	twnd->_cr = WCURS;
	if (z != 1)	{
		remove_list(twnd);
		if (z == 0)
			insert_list(twnd, wnd);
		else
			beg_list(twnd);
	}
	for (y1 = 0; y1 < twnd->_wh; y1++)
		for (x1 = 0; x1 < twnd->_ww; x1++)	{
			chat = dget(wnd, x1, y1);
			displ(twnd, x1, y1, chat&255, (chat>>8)&255);
		}
	twnd->_wv = 1;
	vswap(twnd);
	Whide(wnd);
	free(SAV);
	remove_list(wnd);
	*wnd = *twnd;
	insert_list(wnd, twnd);
	remove_list(twnd);
	free(twnd);
}

/* ------------------------------------------------------------------------ */
/*                           clear the window area                          */
/* ------------------------------------------------------------------------ */
void Wclear(WINDOW *wnd)
{
   register int x1, y1;

   if (verify_wnd(&wnd))
	for (y1 = 1; y1 < HEIGHT-1; y1++)
		for (x1 = 1; x1 < WIDTH-1; x1++)
			displ(wnd,x1, y1, ' ', WFACE);
}

/* ------------------------------------------------------------------------ */
/*                           window-oriented printf                         */
/* ------------------------------------------------------------------------ */
void Wprintf(WINDOW *wnd, char *ln, ...)
{
   char dlin [100], *dl = dlin;

   if (verify_wnd(&wnd))   {
	 va_list ap;
	 va_start(ap, ln);
	 vsprintf(dlin, ln, ap);
	 va_end(ap);
	 while (*dl)
        Wputch(wnd, *dl++);
      }
}

/* ------------------------------------------------------------------------ */
/*                     write a character to the window                      */
/* ------------------------------------------------------------------------ */
void Wputch(WINDOW *wnd, int c)
{
	if (!verify_wnd(&wnd))
		return;
	switch (c)	{
		case '\n':
			if (SCROLL == HEIGHT-3)
				scroll(wnd, UP);
			else
				SCROLL++;
			WCURS = 0;
			break;
		case '\t':
			do displ(wnd,(WCURS++)+3,SCROLL+1,' ',WFACE);
				while ((WCURS%TABS) && (WCURS+1) < WIDTH-1);
			break;
		default:
			if ((WCURS+1) < WIDTH-1)	{
				displ(wnd, WCURS+1, SCROLL+1, c, WFACE);
				WCURS++;
			}
			break;
	}
}

/* ------------------------------------------------------------------------ */
/*                write a character and attribute to the window             */
/* ------------------------------------------------------------------------ */
void Wputchat(WINDOW *wnd, int c, int bg, int fg, int i)
{
	if (!verify_wnd(&wnd))
		return;
	switch (c)	{
		case '\n':
			if (SCROLL == HEIGHT-3)
				scroll(wnd, UP);
			else
				SCROLL++;
			WCURS = 0;
			break;
		case '\t':
			do displ(wnd,(WCURS++)+3,SCROLL+1,' ',CLR(bg,fg,i));
				while ((WCURS%TABS) && (WCURS+1) < WIDTH-1);
			break;
		default:
			if ((WCURS+1) < WIDTH-1)	{
				displ(wnd, WCURS+1, SCROLL+1, c,CLR(bg,fg,i));
				WCURS++;
			}
			break;
	}
}

/* ------------------------------------------------------------------------ */
/*                             set window cursor                            */
/* ------------------------------------------------------------------------ */
void Wcursor(WINDOW *wnd, int x, int y)
{
	if (verify_wnd(&wnd) && x < WIDTH-1 && y < HEIGHT-1)	{
		WCURS = x;
		SCROLL = y;
        vsetcur(COL+x+1, ROW+y+1);
	}
}

/* ------------------------------------------------------------------------ */
/*               allow the user to make a window selection                  */
/* ------------------------------------------------------------------------ */
int Wgetsel(WINDOW *wnd, int s, char *keys)
{
   int c = 0, ky;
   if (!verify_wnd(&wnd))
		return 0;
   SELECT = s;
   while (c != ESC && c != '\r' && c != LEFT && c != RIGHT)	{
	accent(wnd);
	c = kgetch();
	deaccent(wnd);
	switch (c)   {
	      case UP:	if (SELECT >  1)
				SELECT--;
			    else
				SELECT = SCROLL+1;
			break;
	      case DOWN: if (SELECT < SCROLL+1)
				SELECT++;
			   else
				SELECT = 1;
			   break;
	      case '\r' :
	      case ESC  :
	      case RIGHT:
	      case LEFT :	break;

	      default   :   if (keys)	{
			      ky = 0;
			      while (*(keys + ky))	{
				if (*(keys+ky)==toupper(c) ||
					*(keys+ky)==tolower(c))
						return ky + 1;
						ky++;
				}
			      }
			     break;
		}
	}
	return 	c == '\r' ? SELECT : c == ESC ? 0 : c;
}

WINDOW *ewnd = NULL;

/* ------------------------------------------------------------------------ */
/*                             error messages                               */
/* ------------------------------------------------------------------------ */
void Wemsg(char *s)
{
	ewnd = Westablish(50, 22, 3, max(10, strlen(s)+2));
	Wsetcolour(ewnd, WIN_ALL, RED, WHITE, BRIGHT);
	Wsettitle(ewnd, " Error ",JUST_C);
	Wshow(ewnd);
	Wprintf(ewnd, s);
	mscbeep(ERROR);
}

/* ------------------------------------------------------------------------ */
/*                         clear an error message                           */
/* ------------------------------------------------------------------------ */
void Wemsgclear()
{
	if (ewnd)
		Wdelete(ewnd);
	ewnd = NULL;
}

union REGS rg;

/* ------------------------------------------------------------------------ */
/*                   scroll a window's contents up or down                  */
/* ------------------------------------------------------------------------ */
void scroll(WINDOW *wnd, int dir)
{
	int row = HEIGHT-1, col, chat;

	if (!verify_wnd(&wnd))
		return;
	if (NEXT == NULL && HEIGHT > 3 && VISIBLE)	{
		rg.h.ah = dir == UP ? 6 : 7;
		rg.h.al = 1;
		rg.h.bh = WFACE;
		rg.h.cl = COL + 1;
		rg.h.ch = ROW + 1;
		rg.h.dl = COL + WIDTH - 2;
		rg.h.dh = ROW + HEIGHT - 2;
		int86(16, &rg, &rg);
		return;
	}
	if (dir == UP)	{
		for (row = 2; row < HEIGHT-1; row++)
			for (col = 1; col < WIDTH-1; col++)	{
				chat = dget(wnd, col, row);
				displ(wnd,col,row-1,chat&255,(chat>>8)&255);
			}
		for (col = 1; col < WIDTH-1; col++)
			displ(wnd, col, row-1, ' ', WFACE);
	}
	else	{
		for (row = HEIGHT-2; row > 1; --row)
			for (col = 1; col < WIDTH-1; col++)	{
				chat = dget(wnd, col, row);
				displ(wnd,col,row+1,chat&255,(chat>>8)&255);
			}
		for (col = 1; col < WIDTH-1; col++)
			displ(wnd, col, row+1, ' ', WFACE);
	}
}




/* ------------------------------------------------------------------------ */
/*                         draw the window frame                            */
/* ------------------------------------------------------------------------ */
static void wframe(WINDOW *wnd)
{
	register int x1, y1;

	if (!verify_wnd(&wnd))
		return;
	/* --------- window title -------------- */
	displ(wnd,0, 0, NW, WBORDER);
	dtitle(wnd);
	displ(wnd,WIDTH-1, 0, NE, WBORDER);
	/* ------------ window sides ----------------- */
	for (y1 = 1; y1 < HEIGHT-1; y1++)	{
		displ(wnd,0, y1, SIDE, WBORDER);
		displ(wnd,WIDTH-1, y1, SIDE, WBORDER);
	}
	/* --------------- bottom of frame ---------------- */
	displ(wnd,0, y1, SW, WBORDER);
	for (x1 = 1; x1 < WIDTH-1; x1++)
		displ(wnd,x1, y1, LINE, WBORDER);
	displ(wnd,x1, y1, SE, WBORDER);
}

/* ------------------------------------------------------------------------ */
/*                         display the window title                         */
/* ------------------------------------------------------------------------ */
static void dtitle(WINDOW *wnd)
{
	int x1 = 1, i, ln;
	char *s = WTITLE;

	if (!verify_wnd(&wnd))
		return;

	switch (TJUST) {
	case JUST_L : {  x1 =1;
			 while (*s)
			     displ(wnd, x1++, 0, *s++, WTITLEC);
			 while (x1 < WIDTH-1)
			     displ(wnd, x1++, 0, LINE, WBORDER);

	             }
		     break;
	case JUST_R : {   i = 1;
			 while (i < WIDTH-1)
			     displ(wnd, i++, 0, LINE, WBORDER);
			 x1 = ( (WIDTH-1) - strlen(s));
			 while (*s)
			     displ(wnd, x1++, 0, *s++, WTITLEC);
		     }
		     break;
	default    : if (s)
		      {
		       ln = strlen(s);
		       if (ln > WIDTH-2) /* if max screen length -2 */
		       i = 0;
			else
			 i = ((WIDTH-2-ln) / 2);  /* centre */
			 if (i > 0)
			   while (i--)
			     displ(wnd, x1++, 0, LINE, WBORDER);
			 while (*s && x1 < WIDTH-1)
			     displ(wnd, x1++, 0, *s++, WTITLEC);
		      }
		    while (x1 < WIDTH-1)
		      displ(wnd, x1++, 0, LINE, WBORDER);
       } /* switch */
}

/* ------------------------------------------------------------------------ */
/*              compute address of a window's display character             */
/* ------------------------------------------------------------------------ */
static int *waddr(WINDOW *wnd, int x, int y)
{
   WINDOW *nxt = NEXT;
   int *vp;

   if (!VISIBLE)
	return (int *) (SAV+y*(WIDTH*2)+x*2);
   x += COL;
   y += ROW;
   while (nxt)	{
	if (nxt->_wv)
	  if (x >= nxt->_wx && x <= nxt->_wx + nxt->_ww-1)
	      if (y >= nxt->_wy && y <= nxt->_wy + nxt->_wh-1)	{
			  x -= nxt->_wx;
			  y -= nxt->_wy;
			  vp = (int *)	((nxt->_ws) +y*(nxt->_ww*2)+x*2);
			  return vp;
		      }
	     nxt = nxt->_nx;
	}
 return NULL;
}

/* ------------------------------------------------------------------------ */
/*                    display a character to a window                       */
/* ------------------------------------------------------------------------ */
void displ(WINDOW *wnd, int x, int y, int ch, int at)
{
     int *vp;
     int vch = (ch&255)|(at<<8);

     if ((vp = waddr(wnd, x, y)) != NULL)
	*vp = vch;
     else
	vputch(x+COL,y+ROW,at,ch);
}

/* ------------------------------------------------------------------------ */
/*                    get a displayed character from a window               */
/* ------------------------------------------------------------------------ */
static int dget(WINDOW *wnd, int x, int y)
{
     int *vp;

     if ((vp = waddr(wnd, x, y)) != NULL)
		return *vp;
     return (vget(x+COL,y+ROW));
}

/* ------------------------------------------------------------------------ */
/*                  swap the video image with the save buffer               */
/* ------------------------------------------------------------------------ */
static void vswap(WINDOW *wnd)
{
	int x, y, chat;
	int *bf = (int *) SAV;

	for (y = 0; y < HEIGHT; y++)
		for (x = 0; x < WIDTH; x++)	{
			chat = *bf;
			*bf++ = dget(wnd, x, y);
			displ(wnd, x, y, chat&255, (chat>>8)&255);
		}
}

/* ------------------------------------------------------------------------ */
/*                 (de)accent the line where SELECT points                  */
/* ------------------------------------------------------------------------ */
void acline(WINDOW *wnd, int set)
{
	int x, ch;

	if (!verify_wnd(&wnd))
		return;
	for (x = 1; x < WIDTH - 1; x++)	{
		ch = dget(wnd, x, SELECT) & 255;
		displ(wnd, x, SELECT, ch, set);
	}
}

/* ------------------------------------------------------------------------ */
/*                   add a window to the end of the list                    */
/* ------------------------------------------------------------------------ */
static void add_list(WINDOW *wnd)
{
	if (listtail)	{
		PREV = listtail;
		listtail->_nx = wnd;
	}
	listtail = wnd;
	if (!listhead)
		listhead = wnd;
}

/* ------------------------------------------------------------------------ */
/*                add a window to the beginning of the list                 */
/* ------------------------------------------------------------------------ */
static void beg_list(WINDOW *wnd)
{
	if (listhead)	{
		NEXT = listhead;
		listhead->_pv = wnd;
	}
	listhead = wnd;
	if (!listtail)
		listtail = wnd;
}

/* ------------------------------------------------------------------------ */
/*                     remove a window from the list                        */
/* ------------------------------------------------------------------------ */
static void remove_list(WINDOW *wnd)
{
	if (NEXT)
		NEXT->_pv = PREV;
	if (PREV)
		PREV->_nx = NEXT;
	if (listhead == wnd)
		listhead = NEXT;
	if (listtail == wnd)
		listtail = PREV;
	NEXT = PREV = NULL;
}

/* ------------------------------------------------------------------------ */
/*                            insert w1 after w2                             */
/* ------------------------------------------------------------------------ */
static void insert_list(WINDOW *w1, WINDOW *w2)
{
	w1->_pv = w2;
	w1->_nx = w2->_nx;
	w2->_nx = w1;
	if (w1->_nx == NULL)
		listtail = w1;
	else
		w1->_nx->_pv = w1;
}

/* ------------------------------------------------------------------------ */
/*                 verify the presence of a window in the list              */
/* ------------------------------------------------------------------------ */
static verify_wnd(WINDOW **w1)
{
	WINDOW *wnd;

	if (*w1 == NULL)
		*w1 = listtail;
	else	{
		wnd = listhead;
		while (wnd != NULL)	{
			if (*w1 == wnd)
				break;
			wnd = NEXT;
		}
	}
	return *w1 != NULL;
}



