/* ------------------------------------------------------------------------ */
/*                                sstfield.c                                */
/*                    high-level formatted input routines                   */
/*                                                                          */
/*      CopyRight (C) 1991,1992  Steven Lutrov.   All rights reserved.      */
/* ------------------------------------------------------------------------ */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <alloc.h>
#include <mem.h>
#include <string.h>

#include "sstwin.h"
#include "sstkey.h"



/* ------------------------------------------------------------------------ */
/*                             local prototypes                             */
/* ------------------------------------------------------------------------ */

void   addfield           (WINDOW *wnd, FIELD *fld);
void   disp_field         (WINDOW *wnd, char *bf, char *msk);
void   data_value         (WINDOW *wnd, FIELD *fld, int n);
void   insert_status      (void);
int    read_field         (WINDOW *wnd, FIELD *fld);
void   rjust              (char *s);
void   rjust_zf           (char *s);
int    default_endstroke  (int c);
int    spaces             (char *c);
int    test_ftype         (int ftype, int c);
void   processhelp        (FIELD *fld);
void   post_process_flags (FIELD *fld);
void   pre_process_flags  (FIELD *fld, int *c);


int    insert_mode = FALSE;	             /* insert mode, TRUE/FALSE */
int    (*fend) (int) = default_endstroke;    /* field end keystroke func */
extern int helpkey;                          /* key to activate help */

/* ------------------------------------------------------------------------ */
/*                          initialize a template                           */
/* ------------------------------------------------------------------------ */
void Finittemplate(WINDOW *wnd)
{
	FIELD *fld, *fl;

	fld = FHEAD;
	while (fld)	{
		fl = fld->fnxt;
		free(fld);
		fld = fl;
	}
	FHEAD = NULL;
}

/* ------------------------------------------------------------------------ */
/*                    establish a field in a template                       */
/* ------------------------------------------------------------------------ */
FIELD *Festablish(WINDOW *wnd, int cl, int rw, char *msk,
		      char *bf, int ty, int flags)
{
	FIELD *fld;

	if ( (fld = malloc(sizeof(FIELD))) == NULL)
		return NULL;
	fld->fmask = msk;
	fld->frow = rw;
	fld->fcol = cl;
	fld->fbuff = bf;
	fld->ftype = ty;
	fld->fprot = 0;
	fld->fnxt = fld->fprv = NULL;
	fld->fvalid = NULL;
	fld->fhelp = NULL;
	fld->helptype = HELP_WAIT;
	fld->fhwin = NULL;
	fld->flx = fld->fly = 0;
	fld->fflags = flags;
	addfield(wnd, fld);
	return fld;
}

/* ------------------------------------------------------------------------ */
/*                   add a field to the end of the list                     */
/* ------------------------------------------------------------------------ */
static void addfield(WINDOW *wnd, FIELD *fld)
{
	if (FTAIL)	{
		fld->fprv = FTAIL;
		FTAIL->fnxt = fld;
	}
	FTAIL = fld;
	if (!FHEAD)
		FHEAD = fld;
}

/* ------------------------------------------------------------------------ */
/*                          display a data field                            */
/* ------------------------------------------------------------------------ */
static void disp_field(WINDOW *wnd, char *bf, char *msk)
{
     while (*msk)	{
	Wputch(wnd, *msk != FIELDCHAR ? *msk : *bf++);
	msk++;
     }
}

/* ------------------------------------------------------------------------ */
/*                    display the data value in a field                     */
/* ------------------------------------------------------------------------ */
static void data_value(WINDOW *wnd, FIELD *fld, int n)
{
     int c;

     c = WFACE;
     WFACE = n;
     Wcursor(wnd, fld->fcol, fld->frow);
	disp_field(wnd, fld->fbuff, fld->fmask);
     WFACE = c;
}

/* ------------------------------------------------------------------------ */
/*                   display all the fields in a window                     */
/* ------------------------------------------------------------------------ */
void Ftally(WINDOW *wnd)
{
	FIELD *fld;

	fld = FHEAD;
	while (fld != NULL)	{
		data_value(wnd, fld,WFIELD);
		fld = fld->fnxt;
	}
}

/* ------------------------------------------------------------------------ */
/*                      set a field's help window                           */
/* ------------------------------------------------------------------------ */
void Fsethelpwin(FIELD *fld, char *hwin, int x, int y)
{
	fld->fhwin=hwin;
	fld->flx = x;
	fld->fly = y;
}

/* ------------------------------------------------------------------------ */
/*                      clear a template to all blanks                      */
/* ------------------------------------------------------------------------ */
void Fcleartemplate(WINDOW *wnd)
{
	FIELD *fld;
	char *bf, *msk;

	fld = FHEAD;
	while (fld != NULL)	{
		bf = fld->fbuff;
		msk = fld->fmask;
		while (*msk)	{
			if (*msk == FIELDCHAR)
				*bf++ = ' ';
			msk++;
		}
		fld = fld->fnxt;
	}
	Ftally(wnd);
}

/* ------------------------------------------------------------------------ */
/*                      clear a template to all nulls                       */
/* ------------------------------------------------------------------------ */
void Fnulltemplate(WINDOW *wnd)
{
	FIELD *fld;
	char *bf, *msk;

	fld = FHEAD;
	while (fld != NULL)	{
		bf = fld->fbuff;
		msk = fld->fmask;
		while (*msk)	{
			if (*msk == FIELDCHAR)
				*bf++ = 0x00;
			msk++;
		}
		fld = fld->fnxt;
	}
	Ftally(wnd);
}

/* ------------------------------------------------------------------------ */
/*                     set insert/exchange cursor shape                     */
/* ------------------------------------------------------------------------ */
static void insert_status(void)
{
   vsetcurtype(insert_mode ? 0x0106 : 0x0607);
}

/* ------------------------------------------------------------------------ */
/*     process field according to flags after the field edit is complete    */
/* ------------------------------------------------------------------------ */
void post_process_flags(FIELD *fld)

{
    if (fld->fflags & FLD_ZFILL)
	    rjust_zf(fld->fbuff);
    else
      if (fld->fflags & JUST_R )
	    rjust(fld->fbuff);
}

/* ------------------------------------------------------------------------ */
/*          process user input according to flags as they are keyed in      */
/* ------------------------------------------------------------------------ */
void pre_process_flags(FIELD *fld, int *c)

{
    if (fld->fflags & FLD_TOUPPER)
	*c = toupper(*c);
    if (fld->fflags & FLD_TOLOWER)
	*c = tolower(*c);
}

/* ------------------------------------------------------------------------ */
/*                     read a field from the keyboard                       */
/* ------------------------------------------------------------------------ */
static int read_field(WINDOW *wnd, FIELD *fld)
{
    char *mask = fld->fmask, *buff = fld->fbuff;
    int done = FALSE, c, column;

    column = fld->fcol;
    while (*mask != FIELDCHAR)	{
	column++;
	mask++;
    }
    while (TRUE)	{
	Wcursor(wnd, column, fld->frow);
	c = kgetch();
	pre_process_flags(fld,&c);
	Wemsgclear();
	switch (c)	{
	     case '\b':
	     case LEFT:
			if (buff == fld->fbuff)	{
			  done = c == LEFT;
			  break;
			}  /* left */
			--buff;
			do	{
				--mask;
				--column;
			} while (*mask != FIELDCHAR);
			if (c == LEFT)
			break;
	     case DEL:
			movmem(buff+1, buff, strlen(buff));
			*(buff+strlen(buff)) = ' ';
			Wcursor(wnd, column, fld->frow);
			disp_field(wnd, buff, mask);
			break;
	     case RIGHT:
			do	{
				column++;
				mask++;
			} while (*mask && *mask != FIELDCHAR);
			buff++;
			break;
	     case INS:
			insert_mode ^= TRUE;
			insert_status();
			break;
	     case '.':
			if (fld->ftype == FLD_CURR)	{
				if (*mask++ && *buff == ' ')	{
					*buff++ = '0';
					if (*mask++ && *buff == ' ')
						*buff++ = '0';
					}
			    rjust(fld->fbuff);
			    Wcursor(wnd, fld->fcol, fld->frow);
			    disp_field(wnd, fld->fbuff, fld->fmask);
			    column = fld->fcol+strlen(fld->fmask)-2;
			    mask = fld->fmask+strlen(fld->fmask)-2;
			    buff = fld->fbuff+strlen(fld->fbuff)-2;
			    break;
			}
	     default:
			if (fend(c))	{
				done = TRUE;
				break;
			}
			if (! test_ftype(fld->ftype,c) )
			      break;
			if (insert_mode)	{
				movmem(buff, buff+1, strlen(buff)-1);
				disp_field(wnd, buff, mask);
				Wcursor(wnd, column, fld->frow);
			}
			*buff++ = c;
			Wputch(wnd, c);
				do	{
					column++;
					mask++;
				} while (*mask && *mask != FIELDCHAR);
			if (!*mask)
			c = RIGHT;
			break;
		}   /* switch */
		if (!*mask)
			done = TRUE;
		if (done)
			break;

	}
	if (c != ESC ) 	{
		if (fld->ftype == FLD_CURR) {
			if (*mask++ && *buff == ' ') {
				*buff++ = '0';
				if (*mask++ && *buff == ' ')
					*buff++ = '0';
				}
			}
	 post_process_flags(fld);
	 Wcursor(wnd, fld->fcol, fld->frow);
	 disp_field(wnd, fld->fbuff, fld->fmask);
	}  /* c != ESC */
	return c;
}

/* ------------------------------------------------------------------------ */
/*                       test for a correct field type                      */
/* ------------------------------------------------------------------------ */
static int test_ftype(int ftype, int c)

{
   int f = 1;

   switch(ftype) {

   case FLD_CURR  :
   case FLD_DIGIT :
   case FLD_DATE  :
   case FLD_INT   : if (! (f = isdigit(c) ))
			Wemsg(" Numbers only ");
		    break;
   case FLD_ALNUM   : if (! (f = isalnum(c) ))
			Wemsg(" Alpha numeric only ");
		    break;
   case FLD_ALPHA   : if (! (f = isalpha(c) ))
			Wemsg(" Alpha only ");
		    break;
   case FLD_ASCII   : if (! (f = isascii(c) ))
			Wemsg(" Ascii char only ");
		    break;
   case FLD_PRINT   : if (! (f = isprint(c) ))
			 Wemsg("Printable char only");
		    break;
   case FLD_XDIGIT   : if (! (f = isxdigit(c) ))
			 Wemsg(" Hexadecimal only ");
		    break;
   }
 return(f);
}

/* ------------------------------------------------------------------------ */
/*                     test for an ending keystroke                         */
/* ------------------------------------------------------------------------ */
static int default_endstroke(int c)
{
	switch (c)	{
		case '\r':
		case '\n':
		case '\t':
		case ESC:
		case F1:
		case F2:
		case F3:
		case F4:
		case F5:
		case F6:
		case F7:
		case F8:
		case F9:
		case F10:
		case PGUP:
		case PGDN:
		case HOME:
		case END:
		case UP:
		case DOWN:
			return TRUE;
		default:
			return FALSE;
	}
}

/* ------------------------------------------------------------------------ */
/*                          right justify, space fill                       */
/* ------------------------------------------------------------------------ */
static void rjust(char *s)
{
	int len;

	len = strlen(s);
	while (*s == ' ' || *s == '0' && len)	{
		len--;
		*s++ = ' ';
	}
	if (len)
		while (*(s+(len-1)) == ' ')	{
			movmem(s, s+1, len-1);
			*s = ' ';
		}
}

/* ------------------------------------------------------------------------ */
/*                         right justify, zero fill                         */
/* ------------------------------------------------------------------------ */
static void rjust_zf(char *s)
{
	int len;

	if (spaces(s))
		return;
	len = strlen(s);
	while (*(s + len - 1) == ' ')	{
		movmem(s, s + 1, len-1);
		*s = '0';
	}
}

/* ------------------------------------------------------------------------ */
/*                          test for spaces                                 */
/* ------------------------------------------------------------------------ */
int spaces(char *c)
{
	while (*c == ' ')
		c++;
	return !*c;
}

/* ------------------------------------------------------------------------ */
/*                                                                          */
/* ------------------------------------------------------------------------ */
void processhelp(FIELD *fld)

{
   if ( (fld->helptype & HELP_NOWAIT) && !(fld->helptype & HELP_DONE) )
	   if (fld->fhelp) {
		  (*(fld->fhelp))(fld->fbuff);
			  fld->helptype |= HELP_DONE;
	     }
}

/* ------------------------------------------------------------------------ */
/*                  Process data entry for a screen template.               */
/* ------------------------------------------------------------------------ */
int Fdataentry(WINDOW *wnd, int p)
{
   FIELD *fld;
   int exitcode, isvalid, done=FALSE, oldhelpkey=helpkey;


   Ftally(wnd);
   fld = FHEAD;

   /* ---- collect data from keyboard into screen ---- */

   while (fld != NULL && done == FALSE)	{
	Hset(fld->fhwin, fld->flx, fld->fly);
	helpkey = (fld->fhelp) ? 0 : oldhelpkey;
	Wcursor(wnd, fld->fcol, fld->frow);
	if (fld->fprot == FALSE)	{
		Wrevvideo(wnd);
		data_value(wnd, fld,WACCENT);
		Wcursor(wnd, fld->fcol, fld->frow);
		exitcode = read_field(wnd, fld); /* process */
		isvalid = (exitcode != ESC && fld->fvalid) ?
					(*(fld->fvalid))(fld->fbuff) : OK;
	}
	else	{
		exitcode = RIGHT;
		isvalid = OK;
	}
	if (isvalid == OK)	{
	   Wresetvideo(wnd);
	   data_value(wnd, fld,WFIELD);
	   switch (exitcode)	{		/* passed edit */
	     case F1     : if (fld->fhelp)	{
			     (*(fld->fhelp))(fld->fbuff);
			     data_value(wnd, fld,WFIELD);
			     }
			   break;
	     case DOWN   :
	     case '\r'   :
	     case '\t'   :
	     case RIGHT  : fld = fld->fnxt;
			   if (fld == NULL)
				fld = FHEAD;
			   processhelp(fld);
			   break;
	     case UP     :
	     case LEFT   : fld = fld->fprv;
			   if (fld == NULL)
				fld = FTAIL;
			   processhelp(fld);
			   break;
	     case PGUP   :
	     case HOME   : if (p)
				done = fend(exitcode);
			   else {
			     fld = FHEAD;
			     processhelp(fld);
			   }
			   break;

	     case PGDN   :
	     case END    : if (p)
				done = fend(exitcode);
			   else {
			     fld = FTAIL;
			     processhelp(fld);
			   }
			   break;

	     default     : done = fend(exitcode);
			   break;
		}
	}
  }
  helpkey = oldhelpkey;
  return (exitcode);
}

/* ------------------------------------------------------------------------ */
/*                  View  data entry for a screen template.                 */
/* ------------------------------------------------------------------------ */
int Fdataview(WINDOW *wnd)
{
   FIELD *fld;
   int key;

   Ftally(wnd);
   fld = FHEAD;

   while (fld != NULL ) {
	 data_value(wnd, fld,WFIELD);
	 fld = fld->fnxt;
     }
   vhidecur();
   key = kgetch();
   vshowcur();
   return key;
}
/* ------------------------------------------------------------------------ */
/*                          display a window prompt                         */
/* ------------------------------------------------------------------------ */
void Wprompt(WINDOW *wnd, int x, int y, char *s)
{
	Wcursor(wnd, x, y);
	Wprintf(wnd, s);
}

/* ------------------------------------------------------------------------ */
/*                      set the help function for a field                   */
/* ------------------------------------------------------------------------ */
void Fsethelp(FIELD *f, void (*h)(char *), int w)

{
  f->fhelp=h;
  f->helptype=w;
}

/* ------------------------------------------------------------------------ */
/*                      set the help function for a field                   */
/* ------------------------------------------------------------------------ */
void Fendfunction(int (*f) (int))

{
  if (f)
      fend = f;
   else
      fend = default_endstroke;
}
