//***************************************************************************
//
//  Library:
//      SPLASH.C
//
//
//  Purpose:
//      Shows a "Splash" screen by poking a hole in the screen DC, saving
//      the contents, and displaying the splash image which can be passed
//      in, or loaded from a resource.  The screen DC is then restored to
//      it's original condition by replacing the previous image.
//
//
//  Author:
//      Terrell K. Deppe
//
//
//  Copyright:
//
//      (C) Copyright Terrell K. Deppe.  All rights reserved.
//
//
//***************************************************************************


//*************************************
//
//  Headers
//
//*************************************
#include <windows.h>
#include <windowsx.h>
#include "splash.h"


//*************************************
//
//  Globals.  ...YES, globals...
//
//*************************************
HDC      hDC = NULL;                   // Handle to the screen DC.
HDC      hMemDC = NULL;                // Handle to the memory DC.
HBITMAP  hBitmap = NULL;               // Splash screen bitmap.
HBITMAP  hMemBitmap = NULL;            // Temporary bitmap handle.
HBITMAP  hScrBitmap = NULL;            // Holds the original image.
BITMAP   bm;                           // Used for GetObject().


//***************************************************************************
//***************************************************************************
//
//  Resource based splash screen routines:
//
//***************************************************************************
//***************************************************************************

//***************************************************************************
//
//
//  Function:
//      BOOL SplashResource(HINSTANCE hInst, LPSTR lpszBitmap, DWORD Duration)
//
//
//  Purpose:
//      Shows a splash screen for a specified period of time before restor-
//      ing the screen to it's original image.  This function sits in a
//      dead loop for the duration, so if you want to do processing (ie,
//      startup) you should use ShowSplashResource(...) and
//      HideSplashResource(...).
//
//
//  Usage:
//      SplashResource( HINSTANCE, LPSTR, DWORD );
//
//
//  Where:
//      HINSTANCE is a handle to the instance where the bitmap is located.
//      LPSTR is the name of the bitmap resource.
//      DWORD is the number of clock ticks to display the image.
//
//
//  Return:
//      TRUE if successful, otherwise FALSE.
//
//
//***************************************************************************
BOOL SplashResource(HINSTANCE hInst, LPSTR lpszBitmap, DWORD Duration){
  // Show the splash image.
  if (!ShowSplashResource(hInst, lpszBitmap)) return FALSE;
  // Wait for the specified duration.
  Duration += GetTickCount();
  while (GetTickCount() < Duration);
  // Restore the screen and return.
  return HideSplashResource();
} // SplashResource


//***************************************************************************
//
//
//  Function:
//      BOOL ShowSplashResource(HINSTANCE hInst, LPSTR lpszBitmap)
//
//
//  Purpose:
//      Shows a splash screen.  This routine MUST be followed up by a call
//      to HideSplashResource(...) so that the DC's, bitmaps, and screen are all
//      restored to their original state.
//
//
//  Usage:
//      ShowSplashResource( HINSTANCE, LPSTR );
//
//
//  Where:
//      HINSTANCE is a handle to the instance where the bitmap is located.
//      LPSTR is the name of the bitmap resource.
//
//
//  Return:
//      TRUE if successful, otherwise FALSE.
//
//
//***************************************************************************
BOOL ShowSplashResource(HINSTANCE hInst, LPSTR lpszBitmap){
  // If the handles are valid, HideSplashBitmap() should've been called.
  if ((hDC)||(hMemDC)||(hMemBitmap)||(hScrBitmap)) return FALSE;
  // Load the splash image.
  hBitmap = LoadBitmap(hInst, lpszBitmap);
  if (!hBitmap) return FALSE;
  // Setup the DC's.
  hDC = GetDC(NULL);
  hMemDC = CreateCompatibleDC( hDC );
  // Get the dimensions of the splash bitmap.
  GetObject(hBitmap, sizeof(BITMAP), &bm);
  // Setup the bitmaps.
  hScrBitmap = CreateCompatibleBitmap(hDC, bm.bmWidth, bm.bmHeight);
  // Select in the bitmap that will hold the original screen image.
  hMemBitmap = SelectBitmap(hMemDC, hScrBitmap);
  // Make a copy of the screen where we'll be putting the image.
  BitBlt(hMemDC, 0, 0, bm.bmWidth, bm.bmHeight, hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         SRCCOPY);
  // Hold that image in the screen bitmap, and select the splash.
  hScrBitmap = SelectBitmap(hMemDC, hBitmap);
  // Paint the splash image.
  BitBlt(hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
  // Select the screen image back in, and delete the splash image.
  DeleteObject(SelectBitmap(hMemDC, hScrBitmap));
  // Return success if the handles are valid.
  if ((hDC)||(hMemDC)||(hMemBitmap)||(hScrBitmap)) return TRUE;
  else return FALSE;
} // ShowSplashResource



//***************************************************************************
//
//
//  Function:
//      BOOL HideSplashResource()
//
//
//  Purpose:
//      Removes a previously displayed splash screen, restoring all DC's,
//      bitmaps, and screen to original states.
//
//
//  Usage:
//      HideSplashResource();
//
//
//  Return:
//      TRUE if successful, otherwise FALSE.
//
//
//***************************************************************************
BOOL HideSplashResource(){
  // Make sure this function isn't getting called out of order.
  if ((!hDC)||(!hMemDC)||(!hMemBitmap)||(!hScrBitmap)) return FALSE;
  // Restore the screen where we put the splash screen.
  BitBlt(hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
  // Delete the screen image, and clean up all DC's.
  DeleteObject(SelectBitmap(hMemDC, hMemBitmap));
  DeleteDC(hMemDC);
  ReleaseDC(NULL, hDC);
  hDC = NULL;
  hMemDC = NULL;
  hMemBitmap = NULL;
  hScrBitmap = NULL;
  return TRUE;
} // HideSplashResource


//***************************************************************************
//***************************************************************************
//
//  Handle based splash screen routines:
//
//***************************************************************************
//***************************************************************************

//***************************************************************************
//
//
//  Function:
//      BOOL SplashBitmap(HBITMAP hBMP, DWORD Duration)
//
//
//  Purpose:
//      Shows a splash screen for a specified period of time before restor-
//      ing the screen to it's original image.  This function sits in a
//      dead loop for the duration, so if you want to do processing (ie,
//      startup) you should use ShowSplashBitmap(...) and
//      HideSplashBitmap(...).
//
//
//  Usage:
//      SplashBitmap( HBITMAP, DWORD );
//
//
//  Where:
//      HBITMAP is a handle to the bitmap to be displayed.
//      DWORD is the number of clock ticks to display the image.
//
//
//  Return:
//      TRUE if successful, otherwise FALSE.
//
//
//***************************************************************************
HBITMAP SplashBitmap(HBITMAP hBMP, DWORD Duration){
  // Show the splash image.
  if (!ShowSplashBitmap(hBMP)) return FALSE;
  // Wait for the specified duration.
  Duration += GetTickCount();
  while (GetTickCount() < Duration);
  // Restore the screen and return.
  return HideSplashBitmap();
} // SplashBitmap


//***************************************************************************
//
//
//  Function:
//      BOOL ShowSplashBitmap(HBITMAP hBMP)
//
//
//  Purpose:
//      Shows a splash screen.  This routine MUST be followed up by a call
//      to HideSplashBitmap(...) so that the DC's, bitmaps, and screen are all
//      restored to their original state.
//
//
//  Usage:
//      ShowSplashBitmap( HBITMAP );
//
//
//  Where:
//      HBITMAP is a handle to the bitmap to be displayed.
//
//
//  Return:
//      TRUE if successful, otherwise FALSE.
//
//
//***************************************************************************
BOOL ShowSplashBitmap(HBITMAP hBMP){
  // If the handles are valid, HideSplashBitmap() should've been called.
  if ((hDC)||(hMemDC)||(hMemBitmap)||(hScrBitmap)) return FALSE;
  // Load the splash image.
  hBitmap = hBMP;
  if (!hBitmap) return FALSE;
  // Setup the DC's.
  hDC = GetDC(NULL);
  hMemDC = CreateCompatibleDC( hDC );
  // Get the dimensions of the splash bitmap.
  GetObject(hBitmap, sizeof(BITMAP), &bm);
  // Setup the bitmaps.
  hScrBitmap = CreateCompatibleBitmap(hDC, bm.bmWidth, bm.bmHeight);
  // Select in the bitmap that will hold the original screen image.
  hMemBitmap = SelectBitmap(hMemDC, hScrBitmap);
  // Make a copy of the screen where we'll be putting the image.
  BitBlt(hMemDC, 0, 0, bm.bmWidth, bm.bmHeight, hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         SRCCOPY);
  // Hold that image in the screen bitmap, and select the splash.
  hScrBitmap = SelectBitmap(hMemDC, hBitmap);
  // Paint the splash image.
  BitBlt(hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
  // Select the screen image back in, and the splash back out.
  hBMP = SelectBitmap(hMemDC, hScrBitmap);
  // Return success if the handles are valid.
  if ((hDC)||(hMemDC)||(hMemBitmap)||(hScrBitmap)) return TRUE;
  else return FALSE;
} // ShowSplashBitmap



//***************************************************************************
//
//
//  Function:
//      BOOL HideSplashBitmap()
//
//
//  Purpose:
//      Removes a previously displayed splash screen, restoring all DC's,
//      bitmaps, and screen to original states.
//
//
//  Usage:
//      HideSplashBitmap();
//
//
//  Return:
//      A handle to the splash bitmap.
//
//
//***************************************************************************
HBITMAP HideSplashBitmap(){
  // Make sure this function isn't getting called out of order.
  if ((!hDC)||(!hMemDC)||(!hMemBitmap)||(!hScrBitmap)) return FALSE;
  // Restore the screen where we put the splash screen.
  BitBlt(hDC,
         ((GetSystemMetrics(SM_CXSCREEN)/2)-(bm.bmWidth/2)),
         ((GetSystemMetrics(SM_CYSCREEN)/2)-(bm.bmHeight/2)),
         bm.bmWidth, bm.bmHeight, hMemDC, 0, 0, SRCCOPY);
  // Delete the screen image, and clean up all DC's.
  DeleteObject(SelectBitmap(hMemDC, hMemBitmap));
  DeleteDC(hMemDC);
  ReleaseDC(NULL, hDC);
  hDC = NULL;
  hMemDC = NULL;
  hMemBitmap = NULL;
  hScrBitmap = NULL;
  return hBitmap;
} // HideSplashBitmap
