/* TSR.C   by Ned Konz 2/89
 *
 * demo of TSR under ZTC using tiny model
 * this program opens a log file, and logs all DOS activity to it:
 *		TSR logfile
 * When run again with any argument starting with 'U', unloads:
 *		TSR unload
 *	compile with:
 *		ztc -mti tsr.c
 * Included here are some of my header files (which you probably will
 * want to separate from the main code)
 * 
 * Ned Konz
 * Jenzano, Inc.
 * 820 Oak St. Port Orange, FL 32019
 * (904)761-4474
 * Compuserve: 76046,223   Bix: nkonz
 */

/* std.h */
#ifndef STD_H
#define STD_H

#if defined(DEBUG) && DEBUG
#define STATIC
#else
#define STATIC static
#endif

#define IMPORT extern
#define EXPORT

#define MAXPATH 64

#endif	/* std.h */

/* tsr.h */
#ifndef TSR_H
#define TSR_H
/*************************************************************
 * Misc defines
 *************************************************************/
#define USR_INT 	0x61
#define DOS_INT 	0x21

#define NOT_OPENED (-1)
#define CARRY_FLAG	0x01
#define ZERO_FLAG	0x40

/*************************************************************
 * Global Data
 *************************************************************/
IMPORT int logfile;

/*************************************************************
 * Global Function Prototypes
 *************************************************************/
/* in tsr.c */
IMPORT int logfile;
IMPORT void set_psp(unsigned new_psp);
IMPORT unsigned get_psp(void);
IMPORT void far_strcpy(unsigned dest_seg, unsigned dest_offs, 
	unsigned src_seg, unsigned src_offs);
IMPORT int outstring(int fd, char *string);
IMPORT int outhex(int fd, unsigned num);

#endif

/* dosfns.h */
#ifndef DOSFNS_H
#define DOSFNS_H

#define ERR_INVALID_FN		0x01
#define ERR_FILE_NOT_FOUND	0x02
#define ERR_PATH_NOT_FOUND	0x03
#define ERR_TOO_MANY_FILES	0x04
#define ERR_ACCESS_DENIED	0x05
#define ERR_BAD_HANDLE		0x06

#define DOS_OLDEXIT 	0x00
#define DOS_FCB_OPEN	0x0F
#define DOS_FCB_CLOSE	0x10
#define DOS_FCB_DELETE	0x13
#define DOS_FCB_CREATE	0x16
#define DOS_FCB_RENAME	0x17

#define DOS_TSR		0x31
#define DOS_MKDIR	0x39
#define DOS_RMDIR	0x3A
#define DOS_CHDIR	0x3B
#define DOS_CREAT	0x3C
#define DOS_OPEN	0x3D
#define DOS_CLOSE	0x3E
#define DOS_READ	0x3F

#define DOS_WRITE	0x40
#define DOS_UNLINK	0x41
#define DOS_LSEEK	0x42
#define DOS_CHMOD	0x43
#define DOS_IOCTL	0x44
#define DOS_DUP		0x45
#define DOS_FORCDUP	0x46

#define DOS_FREEMEM 0x49
#define DOS_EXEC	0x4B
#define DOS_EXIT	0x4C
#define DOS_WAIT	0x4D

#define DOS_SET_PSP 0x50
#define DOS_GET_PSP 0x51
#define DOS_FORK	0x55
#define DOS_RENAME	0x56
#define DOS_TOUCH	0x57
#define DOS_CREAT_NEW	0x5B
#define DOS_LOCK	0x5C

#define DOS_PARSE_NAME	0x60
#endif	/* dosfns.h */

/* tsr.c */
#include <stdlib.h>
#include <dos.h>
#include <int.h>
#include <string.h>

extern unsigned _psp, _datapar, _progpar;
unsigned _stack = 1000;
unsigned _okbigbuf = 0;

EXPORT int logfile = NOT_OPENED;

/*************************************************************
 * File Globals
 *************************************************************/
STATIC unsigned old21_seg, old21_offs;
STATIC unsigned new21_seg, new21_offs;

STATIC unsigned old20_seg, old20_offs;
STATIC unsigned old27_seg, old27_offs;

STATIC const char *CRLF = "\r\n";
STATIC char ourname[ 65 ];		/* for argv[0] */
STATIC unsigned user_psp;

/*************************************************************
 * Free our environment segment to save memory
 * prior to TSR'ing
 *************************************************************/
void free_environment(void)
{
	union REGS regs;
	struct SREGS sregs;
	unsigned far *fp = MK_FP(_psp, 0x2C);	/* environ pointer */
	sregs.es = *fp;
	regs.h.ah = DOS_FREEMEM;	/* free allocated memory */
	intdosx(&regs, &regs, &sregs);
}

/*************************************************************
 * Terminate and stay resident
 * possibly after growing our heap
 *************************************************************/
void stay_resident(unsigned para_more)
{
	extern unsigned _progpar, _datapar;
	union REGS regs;
	/* first deallocate our copy of the environment */
	free_environment();
	/* now terminate and stay resident */
	regs.h.ah = DOS_TSR;
	regs.x.dx = _datapar + _progpar + para_more;
	intdos(&regs, &regs);
}

/*************************************************************
 * Set DOS PSP to ours (using an un-documented DOS call)
 * so we use OUR file descriptors, not the application program's
 *************************************************************/
void set_psp(unsigned new_psp)
{
	union REGS regs;
	regs.h.ah = DOS_SET_PSP;
	regs.x.bx = new_psp;
	intdos(&regs, &regs);
}

/*************************************************************
 * Get DOS' idea of the active PSP
 *************************************************************/
unsigned get_psp(void)
{
	union REGS regs;
	regs.h.ah = DOS_GET_PSP;
	intdos(&regs, &regs);
	return regs.x.bx;
}

/*************************************************************
 * intercept an interrupt and keep the old and new vectors
 *************************************************************/
void do_int_intercept (int which, int (*func)(struct INT_DATA *),
	int stacksize, unsigned *oldseg, unsigned *oldoffs,
	unsigned *newseg, unsigned *newoffs)
{
	int_off();
	peek(0, which*4, oldoffs, 2);	/* save old vector */
	peek(0, which*4+2, oldseg, 2);
	int_intercept(which, func, stacksize);
	if (newoffs) {
		peek(0, which*4, newoffs, 2);	/* save new vector */
		peek(0, which*4+2, newseg, 2);
	}
	int_on();
}

/*************************************************************
 * Used to get strings out of the Appl. program data space
 *************************************************************/
void far_strcpy(unsigned dest_seg, unsigned dest_offs,
	 unsigned src_seg, unsigned src_offs)
{
	char far *src = MK_FP(src_seg, src_offs);
	char far *dest = MK_FP(dest_seg, dest_offs);
	while (*src)
		*dest++ = *src++;
	*dest = 0;
}

/* this is expected to be run with DOS interrupt pointing
 * to its original location, and PSP set to ours.
 */
STATIC void cleanup(int retcode)
{
	outstring(logfile, "Cleaning up\r\n");
	if (old20_seg)
		int_restore(0x20);
	if (old27_seg)
		int_restore(0x27);
	close(logfile);
	_exit(retcode);
}

/* swap DOS interrupt */
STATIC void swap_ints(void)
{
	unsigned int_offs, int_seg;
	peek(0, DOS_INT*4, &int_offs, 2);
	peek(0, DOS_INT*4+2, &int_seg, 2);
	if (int_offs == old21_offs && int_seg == old21_seg) {
		poke(0, DOS_INT*4, &new21_offs, 2);
		poke(0, DOS_INT*4+2, &new21_seg, 2);
	}
	else if (int_offs == new21_offs && int_seg == new21_seg) {
		poke(0, DOS_INT*4, &old21_offs, 2);
		poke(0, DOS_INT*4+2, &old21_seg, 2);
	}
	else
		cleanup(1);
}

STATIC void outpsp(int fd, unsigned psp)
{
	outstring(fd, "  (PSP ");
	outhex(fd, psp);
	outstring(fd, ")\r\n");
}

STATIC int int20handler(struct INT_DATA *pd)
{
	unsigned user_psp = get_psp();
	set_psp(_psp);
	outstring(logfile, "Terminated via int 20h");
	outpsp(logfile, user_psp);
	set_psp(user_psp);
	return 0;	/* chain */
}

STATIC int int27handler(struct INT_DATA *pd)
{
	unsigned user_psp = get_psp();
	set_psp(_psp);
	outstring(logfile, "Terminated and stayed resident via int 27h");
	outpsp(logfile, user_psp);
	set_psp(user_psp);
	return 0;	/* chain */
}

STATIC int int21handler(struct INT_DATA *pd)
{
	char filename[ 65 ];
	unsigned dosfunc = pd->regs.h.ah;
	int rval = 0;		/* default is to chain to prior handler */

	swap_ints();		/* back to original DOS handler */
	user_psp = get_psp();

	if (user_psp == _psp) {		/* ignore our own calls */
		swap_ints();
		return(0);		/* chain to prior handler */
	}

	set_psp(_psp);	/************** USE OUR PSP!! *****************/

	switch (dosfunc) {
		case DOS_MKDIR:    /*0x39*/
		case DOS_RMDIR:    /*0x3A*/
		case DOS_CHDIR:    /*0x3B*/
		case DOS_CREAT:    /*0x3C*/
		case DOS_OPEN:    /*0x3D*/
		case DOS_UNLINK:    /*0x41*/
		case DOS_CHMOD:    /*0x43*/
		case DOS_EXEC:	/*0x4b*/
		case DOS_RENAME:    /*0x56*/
		case DOS_CREAT_NEW:    /*0x5B*/
			far_strcpy(getDS(), (unsigned)filename,
				pd->sregs.ds, pd->regs.x.dx);
			break;

		default:
			break;
	}

	pd->regs.x.flags &= ~CARRY_FLAG;		/* reset CARRY */

	switch (dosfunc) {
		case DOS_OLDEXIT:    /*0x00*/
		case DOS_FCB_OPEN:    /*0x0F*/
		case DOS_FCB_CLOSE:    /*0x10*/
		case DOS_FCB_DELETE:    /*0x13*/
		case DOS_FCB_CREATE:    /*0x16*/
		case DOS_FCB_RENAME:    /*0x17*/
			outstring(logfile, "Warning: old-style DOS function #");
			outhex(logfile, dosfunc);
			outstring(logfile, " used!");
			outpsp(logfile, user_psp);
			break;

		case DOS_MKDIR:    /*0x39*/
			outstring(logfile, "Make");
			goto iddir;
		case DOS_RMDIR:    /*0x3A*/
			outstring(logfile, "Remove");
			goto iddir;
		case DOS_CHDIR:    /*0x3B*/
			outstring(logfile, "Change to");
		iddir:
			outstring(logfile, " directory ");
			goto id;

		case DOS_CREAT:    /*0x3C*/
			outstring(logfile, "Create");
			goto idfile;
		case DOS_OPEN:    /*0x3D*/
			outstring(logfile, "Open");
			goto idfile;
		case DOS_UNLINK:    /*0x41*/
			outstring(logfile, "Delete");
			goto idfile;
		case DOS_CHMOD:    /*0x43*/
			outstring(logfile, "Change Attributes of");
			goto idfile;
		case DOS_RENAME: {   /*0x56*/
			char newname[65];
			far_strcpy(getDS(), (unsigned)newname,
				pd->sregs.es, pd->regs.x.di);
			outstring(logfile, "Rename (to ");
			outstring(logfile, newname);
			outstring(logfile, ")");
		}
			goto idfile;
		case DOS_CREAT_NEW:    /*0x5B*/
			outstring(logfile, "Create new");
		idfile:
			outstring(logfile, " file ");
		id:
			outstring(logfile, filename);
			outpsp(logfile, user_psp);
			break;

		case DOS_CLOSE:    /*0x3E*/
			outstring(logfile, "Close file handle #");
			outhex(logfile, pd->regs.x.bx);
			outpsp(logfile, user_psp);
			break;

		case DOS_EXEC:	/*0x4b*/
			/* check to see if we're being run again */
			if (! strcmp(filename, ourname) && pd->regs.h.al == 0) {
				typedef char far *CFP;
				CFP far *fpp = MK_FP(pd->sregs.es, pd->regs.x.bx+2);	
				char far *p = *fpp;
				int acnt = *p++;
				while (acnt) {
					if (isalnum(*p))
						break;
					p++; acnt--;
				}
				if (acnt && (*p == 'u' || *p == 'U'))
					cleanup(1);
				else {
					outstring(2, ourname);
					outstring(2, ": Can't run me twice!\r\n");
					pd->regs.x.ax = 2;	/* file not found */
					pd->regs.x.flags |= CARRY_FLAG;
					rval = 1;
				}
			}
			else {
				outstring(logfile, "Execute program ");
				outstring(logfile, filename);
				outpsp(logfile, user_psp);
			}
			break;

		case DOS_TSR:    /*0x31*/
			outstring(logfile, "Terminate and stay resident");
			outpsp(logfile, user_psp);
			break;

		case DOS_EXIT:    /*0x4C*/
			outstring(logfile, "Terminate program");
			outpsp(logfile, user_psp);
			break;

		default:
			break;
	}	/* switch (dosfunc) */

	set_psp(user_psp);		/************ RESTORE USER PSP *********/
	swap_ints();	/* now we handle it! */
	return rval;	/* chain to prior (rval==0) or do iret (rval==1) */
}

STATIC void usage(void)
{
	outstring(2, "Usage: ");
	outstring(2, ourname);
}

/* given an ASCIIZ string, output to screen */
EXPORT int outstring(int fd, char *string)
{
	union REGS regs;
	if (fd == NOT_OPENED)
		return 0;
	regs.h.ah = DOS_WRITE;
	regs.x.bx = fd;
	regs.x.cx = strlen(string);
	regs.x.dx = (unsigned)string;
	intdos(&regs, &regs);
	return regs.x.ax;
}

EXPORT int outhex(int fd, unsigned num)
{
	char buf[ 6 ];
	extern char *itoa(int, char *, int);
	if (fd == NOT_OPENED)
		return 0;
	outstring(fd, "0x");
	return outstring(fd, itoa(num, buf, 16));
}

EXPORT void main(int argc, char *argv[])
{
	extern unsigned char _osmajor;
	unsigned needed;

	strcpy(ourname, argv[0]);

	if (_osmajor < 3) {
		outstring(2, ourname);
		outstring(2, ": Must have DOS 3.01 or later\r\n");
		_exit(1);
	}

	if (argc != 2) {
		outstring(2, "Usage: ");
		outstring(2, ourname);
		outstring(2, " logname\r\n");
		_exit(2);
	}

	close(0);	/* stdin */
	close(1);	/* stdout */
	close(3);	/* stdaux */
	close(4);	/* stdprn */

	logfile = creat(argv[1]);

	do_int_intercept(DOS_INT, int21handler, 1000,
		&old21_seg, &old21_offs,
		&new21_seg, &new21_offs);

	do_int_intercept(0x20, int20handler, 256, &old20_seg, &old20_offs, 0, 0);

	do_int_intercept(0x27, int27handler, 256, &old27_seg, &old27_offs, 0, 0);

	stay_resident(1);
}

EXPORT void exit(int retcode)
{
	_exit(retcode);
}

