/*
 *                         PLAYLIB.C
 *
 * Written by:   Lynn R. Lively
 * Date Written: 8/21/89
 *
 * This module gives TurboC background music capabilities.
 */

#include <dos.h>
#include <stdlib.h>

#ifndef PRIVATE
#define PRIVATE static
#endif

#define TIMER_TICK_INTR 0x1c

typedef struct
{
        unsigned int      freq;
        unsigned int      duration;
} NOTE;

PRIVATE NOTE *     n_buff;
PRIVATE unsigned   n_buff_sz;
PRIVATE NOTE *     n_head;
PRIVATE NOTE *     n_tail;
PRIVATE unsigned   play_duration;
PRIVATE unsigned   play_freq;

PRIVATE void interrupt (*n_oldtmint) ();

/*
 * Add note to note buff. Return = 1 (note added), 0 (Out of note buff)
 */

int playb_note (unsigned freq, unsigned duration)
{
        if (++n_tail == (n_buff + n_buff_sz))
                n_tail = n_buff;

        if (n_tail == n_head)
        {
                --n_tail;
                return (0);
        }

        n_tail->freq     = freq;
        n_tail->duration = duration;

        return (1);
}

/*
 * ISR for background music.
 */

PRIVATE void interrupt play_intr (void)
{
        disable ();

        n_oldtmint ();                /* Call Old timer interrupt. */

        if (play_duration == 0)
        {
                nosound ();

                if (++n_head == (n_buff + n_buff_sz))
                        n_head = n_buff;

                if (n_head == n_tail)
                {
                        --n_head;
                        enable ();
                        return;
                }

                play_duration = n_head->duration;
                play_freq     = n_head->freq;
                sound (play_freq);
        }
        else    --play_duration;

        enable ();
}

/*
 * Call this function to init background music. buff_sz is number of
 * notes in the note buffer. Returns pointer to buff or NULL if
 * out of heap space.
 */

NOTE * playb_initialize (unsigned buff_sz)
{
        n_buff =
        n_head =
        n_tail =  (NOTE *) calloc (buff_sz, sizeof (NOTE));

        if (n_buff != (NOTE *) NULL)
        {
                n_buff_sz     = buff_sz;

                play_duration =
                play_freq     = 0;

                n_oldtmint    = getvect (TIMER_TICK_INTR);
                setvect (TIMER_TICK_INTR, play_intr);
        }
        return (n_buff);
}

/*
 * Return things to normal and free allocated space.
 */

void playb_close (void)
{
        nosound ();
        setvect (TIMER_TICK_INTR, n_oldtmint);
        free (n_buff);
}
