/*
** see discussion at end
*/

/******************************************************************/
/* qsort  --  ANSI Quicksort function                             */
/*                                                                */
/* Public domain by Ray Gardner, Englewood CO    May 1990         */
/*                                                                */
/* Usage:                                                         */
/*     qsort(base, nbr_elements, width_bytes, compare_function);  */
/*        const void *base;                                       */
/*        size_t nbr_elements, width_bytes;                       */
/*        int (*compare_function)(const void *, const void *);    */
/*                                                                */
/* Sorts an array starting at base, of length nbr_elements, each  */
/* element of size width_bytes, ordered via compare_function,     */
/* which is called as  (*compare_function)(ptr_to_element1,       */
/* ptr_to_element2) and returns < 0 if element1 < element2,       */
/* 0 if element1 = element2, > 0 if element1 > element2.          */
/* Most refinements are due to R. Sedgewick. See "Implementing    */
/* Quicksort Programs", Comm. ACM, Oct. 1978, and Corrigendum,    */
/* Comm. ACM, June 1979.                                          */
/******************************************************************/
#include <stddef.h>                     /* for size_t definition  */

void qsort(const void *, size_t, size_t,
                         int (*)(const void *, const void *));

static void cswap (char *a, char *b, unsigned width)
{
   char tmp;
   do {
      tmp = *a; *a++ = *b; *b++ = tmp;
   } while ( --width );
}

#ifdef  SPECIAL
static void sswap (char *ap, char *bp, unsigned width)
{
   short *a = (short *)ap, *b = (short *)bp;
   short tmp;
   do {
      tmp = *a; *a++ = *b; *b++ = tmp;
   } while ( --width );
}
#define  SWAP(a,b)  (swapf((char *)(a), (char *)(b), width))

#else

#define  SWAP(a,b)  (cswap((char *)(a), (char *)(b), size))

#endif

#define  COMP(a,b)  ((*comp)((void *)(a), (void *)(b)))

#define  T   7            /* subfiles of T or fewer elements will */
                          /* be sorted by a simple insertion sort */
                          /* Note!  T must be at least 3          */

#define  base  ((char *)basep)  /* base pointer as char *         */

void qsort (const void *basep, size_t nmemb, size_t size,
                            int (*comp)(const void *, const void *))
{
   char *stack[40], **sp;       /* stack and stack pointer        */
   char *i, *j, *limit;         /* scan and limit pointers        */
   size_t thresh;               /* size of T elements in bytes    */

#ifdef  SPECIAL
   unsigned width;              /* width of array element         */
   void (*swapf)(char *, char *, unsigned); /* swap func pointer  */

   width = size;                /* save size for swap routine     */
   swapf = cswap;               /* choose swap function           */
   if ( size % sizeof(short) == 0 ) {
      width /= sizeof(short);
      swapf = sswap;
   }
#endif

   thresh = T * size;           /* init threshold                 */
   sp = stack;                  /* init stack pointer             */
   limit = base + nmemb * size; /* pointer past end of array      */
   for ( ;; ) {                 /* repeat until break...          */
      if ( limit - base > thresh ) {  /* if more than T elements  */
                                      /*   swap base with middle  */
         SWAP((((limit-base)/size)/2)*size+base, base);
         i = base + size;             /* i scans left to right    */
         j = limit - size;            /* j scans right to left    */
         if ( COMP(i, j) > 0 )        /* Sedgewick's              */
            SWAP(i, j);               /*    three-element sort    */
         if ( COMP(base, j) > 0 )     /*        sets things up    */
            SWAP(base, j);            /*            so that       */
         if ( COMP(i, base) > 0 )     /*      *i <= *base <= *j   */
            SWAP(i, base);            /* *base is pivot element   */
         for ( ;; ) {                 /* loop until break         */
            do                        /* move i right             */
               i += size;             /*        until *i >= pivot */
            while ( COMP(i, base) < 0 );
            do                        /* move j left              */
               j -= size;             /*        until *j <= pivot */
            while ( COMP(j, base) > 0 );
            if ( i > j )              /* if pointers crossed      */
               break;                 /*     break loop           */
            SWAP(i, j);       /* else swap elements, keep scanning*/
         }
         SWAP(base, j);         /* move pivot into correct place  */
         if ( j - base > limit - i ) {  /* if left subfile larger */
            sp[0] = base;             /* stack left subfile base  */
            sp[1] = j;                /*    and limit             */
            base = i;                 /* sort the right subfile   */
         } else {                     /* else right subfile larger*/
            sp[0] = i;                /* stack right subfile base */
            sp[1] = limit;            /*    and limit             */
            limit = j;                /* sort the left subfile    */
         }
         sp += 2;                     /* increment stack pointer  */
      } else {      /* else subfile is small, use insertion sort  */
         for ( j = base, i = j+size; i < limit; j = i, i += size )
            for ( ; COMP(j, j+size) > 0; j -= size ) {
               SWAP(j, j+size);
               if ( j == base )
                  break;
            }
         if ( sp != stack ) {         /* if any entries on stack  */
            sp -= 2;                  /* pop the base and limit   */
            base = sp[0];
            limit = sp[1];
         } else                       /* else stack empty, done   */
            break;
      }
   }
}

#if 0

The "enhancement" is this: if you compile with SPECIAL defined (use
-DSPECIAL on most compilers), the function will check if the width of
each element is a multiple of sizeof(short).  If it is, then the
function will call a swap routine that swaps shorts instead of chars.
That will make it a bit faster in many environments.  The downside is
that it may fail on some machines that require shorts to be aligned
on even-byte boundaries (if you should happen to sort items that are
even-length but mis-aligned somehow).  Use with caution if your
machine can choke on fetching on certain boundaries.

#endif
