/*
	trf-charmap.c - Unnecessarily complex routines for mapping special
	characters in the non-ASCII range (128..255) onto troff equivalents.

	This file may require modification for local troff versions.

	Tries to handle the four different character sets allowed in the
	RTF specification (ANSI, Macintosh, PC and PCA).  That part in itself
	is easy, but the complexity stems from trying to allow for local
	variations in troff versions (not all troff's understand the same
	set of special characters) and for particular macro packages (-me,
	-me, -ms).  The last one is silly at this point since no other part
	of rtf2troff knows anything about writing macro-package-specific
	output.  Maybe someday.

	SelectFormatterMaps () - selects character maps specific to the given
		formatter, e.g., xroff, pstroff, eroff.
	SelectMacPackMaps () - select override maps for the given macro
		package.
	SelectCharSetMaps () - select maps for particular character set.
	CharMapping () - return troff equivalent for a special character.
*/

# include	<stdio.h>
# ifndef THINK_C
# include	<sys/types.h>
# endif

# include	"rtf.h"
# include	"rtf2troff.h"


/*
	Structure for mapping character values >= 128 to troff strings
	for different character sets.

	The reason char value/string pairs are used instead of a 128-entry
	table is that while the latter would be faster, it's much easier
	to build an incomplete table with the former, and the entries don't
	need to be in any particular order.
*/

typedef struct CharMap	CharMap;

struct CharMap
{
	int	charVal;
	char	*charStr;
};


/*
	Structure for associating default and override maps with
	particular versions of troff.
*/

# define	ansiMap		0
# define	macMap		1
# define	pcMap		2
# define	pcaMap		3
# define	mapTypes	4

typedef struct CharMapTable	CharMapTable;

struct CharMapTable
{
	char	*formatter;
	CharMap	*defCharMap[mapTypes];
	CharMap	*meCharMap[mapTypes];
	CharMap	*mmCharMap[mapTypes];
	CharMap	*msCharMap[mapTypes];
};


/*
	ANSI char map is taken from Windows 3.0 and Word for
	Windows 1.1 manuals.  Note that these conflict, so there
	are some guesses.
	PC char map is taken from Windows 3.0 manual.
*/

static CharMap	ansiCharMap [] =
{
	0x91,	"`",		/* left curly single quote */
	0x92,	"'",		/* right curly single quote */
	0x93,	"``",		/* left curly double quote */
	0x94,	"''",		/* right curly double quote */
	0x95,	"\\(bu",	/* bullet */
	0x96,	"\\(em",	/* emdash */
	0x97,	"\\(hy",	/* endash */
	0xa2,	"\\(ct",	/* cent */
	0xa6,	"|",		/* vert. bar */
	0xa7,	"\\(sc",	/* section */
	0xa9,	"\\(co",	/* copyright */
	0xab,	"<<",		/* alternate quote */
	0xac,	"\\(no",	/* not */
	0xad,	"\\(hy",	/* short dash */
	0xae,	"\\(rg",	/* registered */
	0xaf,	"\\(em",	/* long dash */
	0xb0,	"\\(de",	/* degree */
	0xb1,	"\\(+-",	/* plus or minus */
	0xb4,	"\\(aa",	/* acute accent */
	0xb5,	"\\(*m",	/* micro (use greek mu) */
	0xbb,	">>",		/* alternate end-quote*/
	0xbc,	"\\(14",	/* one-fourth */
	0xbd,	"\\(12",	/* one-half */
	0xbe,	"\\(34",	/* three-fourths */
	0xc6,	"AE",		/* joined A-E */
	0xd7,	"\\(mu",	/* math multiply */
	0xdf,	"\\(*b",	/* German B? (fake with greek beta) */
	0xe6,	"ae",		/* joined a-e */
	0xf7,	"\\(di",	/* divide */
	0,	(char *) NULL
};


static CharMap	macCharMap [] =
{
	0xa0,	"\\(dg",	/* dagger */
	0xa1,	"\\(de",	/* degree */
	0xa2,	"\\(ct",	/* cent */
	0xa4,	"\\(sc",	/* section */
	0xa5,	"\\(bu",	/* bullet */
	0xa7,	"\\(*b",	/* German B? (fake with greek beta) */
	0xa8,	"\\(rg",	/* registered */
	0xa9,	"\\(co",	/* copyright */
	0xaa,	"(TM)",		/* trademark */
	0xab,	"\\(aa",	/* acute accent */
	0xad,	"\\(!=",	/* not equal */
	0xae,	"AE",		/* joined A-E */
	0xb0,	"\\(if",	/* infinity */
	0xb1,	"\\(+-",	/* plus or minus */
	0xb2,	"\\(<=",	/* less than or equal */
	0xb3,	"\\(>=",	/* greater than or equal */
	0xb5,	"\\(*m",	/* micro (use greek mu) */
	0xb6,	"\\(*d",	/* delta */
	0xb7,	"\\(*S",	/* summation (Sigma) */
	0xb8,	"\\(*P",	/* Pi */
	0xb9,	"\\(*p",	/* pi */
	0xba,	"\\(is",	/* integral sign */
	0xbd,	"\\(*W",	/* ohm-sign (greek Omega) */
	0xbe,	"ae",		/* joined a-e */
	0xc2,	"\\(no",	/* not */
	0xc3,	"\\(sr",	/* square root */
	0xc5,	"\\(ap",	/* approximately */
	0xc6,	"\\(*D",	/* Delta */
	0xc7,	"<<",		/* alternate quote */
	0xc8,	">>",		/* alternate end-quote*/
	0xc9,	"...",		/* ellipsis */
	0xca,	"\\ ",		/* unbreakable space */
	0xd0,	"\\(hy",	/* short dash */
	0xd1,	"\\(em",	/* long dash */
	0xd2,	"``",		/* left curly double quote */
	0xd3,	"''",		/* right curly double quote */
	0xd4,	"`",		/* left curly single quote */
	0xd5,	"'",		/* right curly single quote */
	0xd6,	"\\(di",	/* divide */
	0,	(char *) NULL
};


static CharMap	pcCharMap [] =
{
	0x91,	"ae",		/* joined a-e */
	0x9b,	"\\(ct",	/* cent */
	0xa9,	"\\(em",	/* long dash */
	0xaa,	"\\(no",	/* not */
	0xab,	"\\(12",	/* one-half */
	0xac,	"\\(14",	/* one-fourth */
	0xae,	"<<",		/* alternate quote */
	0xaf,	">>",		/* alternate end-quote*/
	0xf1,	"\\(+-",	/* plus or minus */
	0xf8,	"\\(de",	/* degree */
	0,	(char *) NULL
};


static CharMap	pcaCharMap [] =
{
	0,	(char *) NULL
};


/*
	These -me maps are just here for example
*/


static CharMap	meAnsiCharMap [] =
{
	0,	(char *) NULL
};


static CharMap	meMacCharMap [] =
{
	0,	(char *) NULL
};


static CharMap	mePcCharMap [] =
{
	0,	(char *) NULL
};


static CharMap	mePcaCharMap [] =
{
	0,	(char *) NULL
};


static CharMapTable charMapTable [] =
{
    {
	"troff",
	{ ansiCharMap, macCharMap, pcCharMap, pcaCharMap },	/* defaults */
	{ NULL, NULL, NULL, NULL },			/* -me overrides */
	{ NULL, NULL, NULL, NULL },			/* -mm overrides */
	{ NULL, NULL, NULL, NULL }			/* -ms overrides */
    },
    {
	"xroff",
	{ ansiCharMap, macCharMap, pcCharMap, pcaCharMap },	/* defaults */
	{ NULL, NULL, NULL, NULL },			/* -me overrides */
	{ NULL, NULL, NULL, NULL },			/* -mm overrides */
	{ NULL, NULL, NULL, NULL }			/* -ms overrides */
    },
    {
	"pstroff",
	{ ansiCharMap, macCharMap, pcCharMap, pcaCharMap },	/* defaults */
	{ NULL, NULL, NULL, NULL },			/* -me overrides */
	{ NULL, NULL, NULL, NULL },			/* -mm overrides */
	{ NULL, NULL, NULL, NULL }			/* -ms overrides */
    },
    {
	/*
		This should be the last entry.  Do not change it.
		Put new entries before this one.
	*/
	NULL
    }
};


/*
	curCharMaps points to current entry in character map table.
	macroMaps points to current set of override maps (default NULL
	until SelectMacPackMaps() is called).  defaultMap points to the
	default map within curCharMaps (default is the ANSI map).
	overrideMap points to the map within the macro package override
	maps (default is the ANSI map after SelectMacPackMaps() is called).
*/

static CharMapTable	*curCharMaps = &charMapTable[0];
static CharMap		**macroMaps = (CharMap **) NULL;
static CharMap		*defaultMap = ansiCharMap;
static CharMap		*overrideMap = (CharMap *) NULL;


void SelectFormatterMaps (name)
char	*name;
{
CharMapTable	*cmtp;

	for (cmtp = charMapTable; cmtp->formatter != (char *) NULL; cmtp++)
	{
		if (strcmp (name, cmtp->formatter) == 0)
		{
			curCharMaps = cmtp;
			macroMaps = (CharMap **) NULL;
			defaultMap = curCharMaps->defCharMap[ansiMap];
			overrideMap = (CharMap *) NULL;
			return;
		}
	}
	fprintf (stderr, "Unknown formatter: %s\n", name);
	exit (1);
}


void SelectMacPackMaps (name)
char	*name;
{
	if (strcmp (name, "-me") == 0)
		macroMaps = curCharMaps->meCharMap;
	else if (strcmp (name, "-mm") == 0)
		macroMaps = curCharMaps->mmCharMap;
	else if (strcmp (name, "-ms") == 0)
		macroMaps = curCharMaps->msCharMap;
	else
	{
		fprintf (stderr, "Unknown macro package: %s\n", name);
		exit (1);
	}
	if (macroMaps != (CharMap **) NULL)
		overrideMap = macroMaps[ansiMap];
}


/*
	mapSet should be an rtfCharSet minor number
*/

void SelectCharSetMaps (mapSet)
int	mapSet;
{
int	mapIndex;

	switch (mapSet)
	{
	case rtfAnsiCharSet:
		mapIndex = ansiMap;
		break;
	case rtfMacCharSet:
		mapIndex = macMap;
		break;
	case rtfPcCharSet:
		mapIndex = pcMap;
		break;
	case rtfPcaCharSet:
		mapIndex = pcaMap;
		break;
	default:
		fprintf (stderr, "Unknown charset: %d\n", mapSet);
		exit (1);
	}
	defaultMap = curCharMaps->defCharMap[mapIndex];
	if (macroMaps != (CharMap **) NULL)
		overrideMap = macroMaps[mapIndex];
}


/*
	Figure out what string a special (non-ASCII) character should map
	to.  (Determine troff equivalent using "normal" characters.)
*/

static char *_CharMapping (c, map)
int	c;
CharMap	*map;
{
CharMap	*cmp;
char	*p = (char *) NULL;

	for (cmp = map; cmp->charStr != (char *) NULL; cmp++)
	{
		if (c == cmp->charVal)
		{
			p = cmp->charStr;
			break;
		}
	}
	return (p);
}


char *CharMapping (c)
int	c;
{
CharMap	*cmp;
char	*p = (char *) NULL;

	if (overrideMap != (CharMap *) NULL)
		p = _CharMapping (c, overrideMap);
	if (p == (char *) NULL && defaultMap != (CharMap *) NULL)
		p = _CharMapping (c, defaultMap);
	return (p != (char *) NULL ? p : "<<UNKNOWN>>");
}
