/* testcop.c - tests column delimited put functions */
/* recio version 2.04, release October 10, 1994 */
/* Copyright (C) 1994 William Pierpoint */

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <float.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include "recio.h"

/* errors to stderr */
#define errout  stdout

enum {RESET, INCR, REPORT};
/****************************************************************************/
void                         /* return nothing                              */
    warnttl(                 /* warning totals                              */
        int action,          /* action (0=reset, 1=increment, 2=report)     */
        int warnum)          /* warning number                              */
/****************************************************************************/
{
  static unsigned int widthcnt=0;
  static unsigned int noregcnt=0;

  switch (action) {
  case RESET:
    widthcnt=0;
    return;
  case INCR:
    switch(warnum) {
    case R_WWIDTH: 
      widthcnt++; 
      return;
    case R_WNOREG: 
      noregcnt++; 
      return;
    }
    return;
  case REPORT:
    fprintf(errout, "\n");
    if (widthcnt) 
      fprintf(errout, "WARNING -- %s -- %u times\n", 
       rstrwarning(R_WWIDTH), widthcnt);
    if (noregcnt) 
      fprintf(errout, "WARNING -- %s\n", rstrwarning(R_WNOREG));
    return;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rwarnfn(                 /* recio callback warning function             */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  if (risvalid(rp)) warnttl(INCR, rwarning(rp));
}

/****************************************************************************/
void                         /* returns nothing                             */
     errnofn(                /* errno callback error function               */
        void)                /* no parameters                               */
/****************************************************************************/
{
  switch (errno) {

  /* non-fatal errors */
  case EACCES:
  case EMFILE:
    fprintf(errout, "ERROR: %s\n", strerror(errno));
    break;

  /* fatal errors (EINVAL, ENOMEM) */
  default:
    fprintf(errout, "FATAL ERROR: %s\n", strerror(errno));
    abort();
    break;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rerrfn(                  /* recio callback error function               */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  if (risvalid(rp)) {
  
    /* determine cause of error */
    switch (rerror(rp)) {
 
    case R_ENOPUT:   /* could not write data */
      fprintf(errout, "ERROR %s -- %s\n", rnames(rp), rerrstr(rp));
      break;

    /* fatal errors (R_EINVMOD, R_EINVAL, R_ENOMEM) */
    default:
      fprintf(errout, "FATAL ERROR %s -- %s\n", rnames(rp), rerrstr(rp));
      abort();
      break;
    }
  
  /* invalid record pointer */
  } else {
    errnofn();
  }
}

/****************************************************************************/
void putcolnumbers(void)
/****************************************************************************/
{
puts("");
puts("         1         2         3         4         5         6");
puts("123456789012345678901234567890123456789012345678901234567890");
}

/****************************************************************************
main
*****************************************************************************/
int main()
{
  int j;                        /* loop index */
  int i;                        /* integer field */
  unsigned int ui;              /* unsigned integer field */
  long l;                       /* long field */
  unsigned long ul;             /* unsigned long field */
  float f;                      /* float field */
  double d;                     /* double field */
  int ch;                       /* character field */
  char *str = NULL;             /* string field */
  char str1[] = "A";            /* string consisting of one letter */
  
  /* install error and warning functions */
  rinit(rerrfn, rwarnfn);
  
  /* set beginning column number to 1 */
  rsetbegcolno(recout, 1);
  
  /* if output not redirected */
  if (isatty(fileno(stdout))) {
    /* print instructions */
    puts("TESTCOP version 2.04 Copyright (C) 1994 William Pierpoint");
    puts("Tests recio column delimited put functions.\n");
    puts("Field type           Columns");
    puts("----------------    ---------");
    puts("Integer............  1 to  5");
    puts("Unsigned Integer...  6 to 10");
    puts("Long............... 11 to 20");
    puts("Unsigned Long...... 21 to 30");
    puts("Float.............. 31 to 40");
    puts("Double............. 41 to 50");
    puts("Character..........    51   ");
    puts("String............. 52 to 60");
  }

  /* initialize data */
  i = -1;
  ui = 1;
  l = -1L;
  ul = 1L;
  f = 1.111111;
  d = 1.111111111111111111111;
  ch = 'a';
  scpys(str, str1);

  /* loop through data */
  for (j=0; j<11; j++) {

    putcolnumbers();

    rcputi(recout,   1,  5, i);
    rcputui(recout,  6, 10, ui);
    rcputl(recout,  11, 20, l);
    rcputul(recout, 21, 30, ul);
    rcputf(recout,  31, 40, f);
    rcputd(recout,  41, 50, d);
    rcputc(recout,  51, ch);
    rcputs(recout, 52, 60, str);

    i  *= 10;
    ui *= 10;
    l  *= 10L;
    ul *= 10L;
    f  *= 10.;
    d  *= 10.;
    ch += 1;
    *str1 = toupper(ch);
    scats(str, str1);

    rputrec(recout);
  }
  
  /* free dynamic string fields */
  free (str);
  
  /* check stream for warnings */
  if (rwarning(recout)) warnttl(REPORT, NULL);

  /* check stream for error */
  if (rerror(recout)) { 
    fprintf(stdout, "\nERROR %s -- %s\n", 
     rnames(recout), rerrstr(recout));
    exit (EXIT_FAILURE);
  }
  return EXIT_SUCCESS;
}
