/*****************************************************************************
   MODULE: rbget.c
  PURPOSE: recio character delimited integral number input functions
COPYRIGHT: (C) 1994 William Pierpoint
 COMPILER: Borland C Version 3.1
       OS: MSDOS Version 6.2
  VERSION: 2.04
  RELEASE: October 10, 1994
*****************************************************************************/

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "recio.h"

extern int _rstatus(REC *rp, int mode);
extern char *_rfldstr(REC *rp, size_t len);
extern char *_rerrs(REC *rp, int errnum);
extern unsigned long str2ul(const char *nptr, char **endptr, int base);

#define uint      unsigned int
#define ulong     unsigned long
#define UNSIGNED  0
#define SIGNED    1

/****************************************************************************/
static long                  /* return integral number                      */
    _rbgetl(                 /* get integral number from record stream      */
        REC *rp,             /* record pointer                              */
        int base,            /* radix of number                             */
        int sign,            /* signed number? (0=unsigned; !0=signed)      */
        long min,            /* inclusive valid min value (0 if unsigned)   */
        long max)            /* inclusive valid max value                   */
/****************************************************************************/
{ 
    long result=0L;          /* result to return */
    long val;                /* conversion value */
    char *fldptr;            /* pointer to field string */
    char *endptr;            /* pointer to first invalid field char */

    if (!_rstatus(rp, R_READ)) { 
      fldptr = _rfldstr(rp, 0); 
      if (fldptr) { 
        for (;;) { 
          if (*fldptr != '\0') { 
            endptr = fldptr; 
            errno = 0; 
            if (sign) {
              val = strtol(fldptr, &endptr, base); 
            } else {
              val = str2ul(fldptr, &endptr, base); 
            }
            while (isspace(*endptr)) endptr++; 
            if (errno==ERANGE || !*endptr) { 
              if (!errno) { 
                if (sign) {
                  if (val >= min && val <= max) { 
                    result = val; 
                    goto done; 
                  }
                } else {
                  /* note: unsigned min always assumed to be zero */
                  if ((ulong) val <= (ulong) max) {
                    result = val; 
                    goto done;
                  }
                }
              } /* data out of range */ 
              fldptr = _rerrs(rp, R_ERANGE); 
              if (fldptr) { continue; } else { goto done; } 
            } /* invalid data */ 
            fldptr = _rerrs(rp, R_EINVDAT); 
            if (fldptr) { continue; } else { goto done; } 
          } /* missing data */ 
          fldptr = _rerrs(rp, R_EMISDAT); 
          if (fldptr) { continue; } else { goto done; } 
        } 
      }
    }
done:
    return result;
}

/****************************************************************************/
/* character delimited integral number input functions                      */
/****************************************************************************/
int rbgeti(REC *rp, int base)
{
    return (int) _rbgetl(rp, base, SIGNED, INT_MIN, INT_MAX);
}

unsigned int rbgetui(REC *rp, int base)
{
    return (uint) _rbgetl(rp, base, UNSIGNED, 0, UINT_MAX);
}

long rbgetl(REC *rp, int base)
{
    return _rbgetl(rp, base, SIGNED, LONG_MIN, LONG_MAX);
}

unsigned long rbgetul(REC *rp, int base)
{
    return (ulong) _rbgetl(rp, base, UNSIGNED, 0L, ULONG_MAX);
}
