/* testcog.c - tests column delimited get functions */
/* recio version 2.04, release October 10, 1994 */
/* Copyright (C) 1994 William Pierpoint */

#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <float.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "recio.h"

/* errors normally to stderr; but for test purposes, stdout */
#define errout  stdout

/****************************************************************************/
void                         /* return nothing                              */
    rfixnumber(              /* fix number                                  */
        REC *rp,             /* record pointer                              */
        int  errnum,         /* error number                                */
        void (*rfix)(REC *)) /* rfix function pointer                       */
/****************************************************************************/
{
  switch (errnum) {
  case R_EMISDAT:
    fprintf(errout, "...substituting zero\n");
    rsetfldstr(rp, "0");
    break;
  case R_EINVDAT:
  case R_ERANGE:
    fprintf(errout, "...substituting best guess\n");
    rfix(rp);
    break;
  }
}

/****************************************************************************/
void                         /* return nothing                              */
    rfixchar(                /* fix character                               */
        REC *rp,             /* record pointer                              */
        int  errnum,         /* error number                                */
        void (*rfix)(REC *)) /* rfix function pointer                       */
/****************************************************************************/
{
  switch (errnum) {
  case R_EMISDAT:
    fprintf(errout, "...substituting the letter N\n");
    rsetfldstr(rp, "N");
    break;
  case R_EINVDAT:
    fprintf(errout, "...substituting best guess\n");
    rfix(rp);
    break;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rwarnfn(                 /* recio callback warning function             */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  if (risvalid(rp)) {
    switch (rwarning(rp)) {
    case R_WNOREG:   /* atexit() full */
      fprintf (errout, "WARNING %s\n", rwarnstr(rp));
      break;
    case R_WEMPSTR:  /* empty data string */
      fprintf(errout, "WARNING reading %s at record %lu and field %u -- %s\n", 
       rnames(rp), rrecno(rp), rfldno(rp), rwarnstr(rp));
      break;
    }
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
     errnofn(                /* errno callback error function               */
        void)                /* no parameters                               */
/****************************************************************************/
{
  switch (errno) {

  /* non-fatal errors */
  case EACCES:
  case EMFILE:
    fprintf(errout, "ERROR: %s\n", strerror(errno));
    break;

  /* fatal errors (EINVAL, ENOMEM) */
  default:
    fprintf(errout, "FATAL ERROR: %s\n", strerror(errno));
    abort();
    break;
  }
}

/****************************************************************************/
void                         /* returns nothing                             */
    rerrfn(                  /* recio callback error function               */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
  int errnum;       /* error number */

  if (risvalid(rp)) {
  
    /* if reof indicator set */
    if (reof(rp)) { 
      fprintf(errout, "ERROR reading %s: "
       "tried to read past end of file\n", rnames(rp));
    
    /* else rerror indicator set */
    } else {
 
      /* determine cause of error */
      errnum = rerror(rp);
      switch (errnum) {
 
      /* data errors */
      case R_ERANGE:   /* data out of range */
      case R_EINVDAT:  /* invalid data */
      case R_EMISDAT:  /* missing data */
        fprintf(errout, "DATA ERROR reading %s at record %lu and field %u "
         "-- %s\n", rnames(rp), rrecno(rp), rfldno(rp), rerrstr(rp));
            
        /* determine context */
        switch (rcxtno(rp)) {
        case RECIN:
          
          /* determine field */
          switch (rfldno(rp)) {
          case 1:  /* the integer field i */
            rfixnumber(rp, errnum, rfixi);
            break;
          
          case 2:  /* the unsigned integer field ui */
            rfixnumber(rp, errnum, rfixui);
            break;

           case 3:  /* the long field l */
            rfixnumber(rp, errnum, rfixl);
            break;

           case 4:  /* the unsigned long field ul */
            rfixnumber(rp, errnum, rfixul);
            break;

           case 5:  /* the float field f */
            rfixnumber(rp, errnum, rfixf);
            break;

            case 6:  /* the double field d */
            rfixnumber(rp, errnum, rfixd);
            break;
          
          case 7: /* the character field ch */
            rfixchar(rp, errnum, rfixc);
            break;

          default: /* programming error - no case for field */
            fprintf(errout, "FATAL ERROR %s -- code missing for field %u\n",
             rnames(rp), rfldno(rp));
            break;
          }
          break;
        
        default:  /* programming error - missing context number */
          fprintf(errout, "FATAL ERROR %s -- missing context number\n", 
           rnames(rp));
          abort();
          break;
        }
        break;
        
      /* fatal errors (R_EINVMOD, R_EINVAL, R_ENOMEM) */
      default:
        fprintf(errout, "FATAL ERROR %s -- %s\n", rnames(rp), rerrstr(rp));
        abort();
        break;
      }
    }
  
  /* invalid record pointer */
  } else {
    errnofn();
  }
}

/****************************************************************************/
void putcolnumbers(void)
/****************************************************************************/
{
puts("         1         2         3         4         5         6         7");
puts("1234567890123456789012345678901234567890123456789012345678901234567890");
}

/****************************************************************************
main
*****************************************************************************/
#include <io.h>

int main()
{
  int i;                        /* integer field */
  unsigned int ui;              /* unsigned integer field */
  long l;                       /* long field */
  unsigned long ul;             /* unsigned long field */
  float f;                      /* float field */
  double d;                     /* double field */
  int ch;                       /* character field */
  char *str = NULL;             /* string field */
  
  /* install error and warning functions */
  rinit(rerrfn, rwarnfn);
  
  /* set beginning column number to 1 */
  rsetbegcolno(recin, 1);
  
  /* if input not redirected */
  if (isatty(fileno(stdin))) {
    /* print instructions */
    puts("TESTCOG version 2.04 Copyright (C) 1994 William Pierpoint");
    puts("Tests recio column delimited get functions.");
    puts("It reads eight fields from the console.\n");
    puts("Field type           Columns");
    puts("----------------    ---------");
    puts("Integer............  1 to  5");
    puts("Unsigned Integer...  6 to 10");
    puts("Long............... 11 to 20");
    puts("Unsigned Long...... 21 to 30");
    puts("Float.............. 31 to 40");
    puts("Double............. 41 to 50");
    puts("Character..........    51   ");
    puts("String............. 52 to 70");
    puts("\nPress Ctrl-Z followed by the Enter key to exit program.");
    puts("You may begin now.\n");
  }
  
  putcolnumbers();

  /* loop through input */
  while (rgetrec(recin)) {

    /* if input redirected, echo record contents */
    if (!isatty(fileno(stdin))) puts(rrecs(recin));

    /* parse record */
    i   = rcgeti(recin,   1,  5);
    ui  = rcgetui(recin,  6, 10);
    l   = rcgetl(recin,  11, 20);
    ul  = rcgetul(recin, 21, 30);
    f   = rcgetf(recin,  31, 40);
    d   = rcgetd(recin,  41, 50);
    ch  = rcgetc(recin,  51);
    scpys(str, rcgets(recin, 52, 70));

    /* print results */
    printf("\n");
    printf("         Integer field: %d\n", i);
    printf("Unsigned Integer field: %u\n", ui);
    printf("            Long field: %ld\n", l);
    printf("   Unsigned Long field: %lu\n", ul);
    printf("           Float field: %.*e\n", FLT_DIG-1, f);
    printf("          Double field: %.*e\n", DBL_DIG-1, d);
    printf("       Character field: %c\n", ch);
    printf("          String field: %s\n\n", str);
    
    putcolnumbers();
  }
  
  /* free dynamic string fields */
  free (str);
  
  /* check stream for error */
  if (rerror(recin)) { 
    fprintf(errout, "ERROR reading %s: %s\n", 
     rnames(recin), rerrstr(recin));
    exit (EXIT_FAILURE);
  }
  return EXIT_SUCCESS;
}
