// =========================================================
// PROPDLG.CPP
//
// Chicago-style property sheet dialog classes for OWL 2
//
// Author:  Steve Saxon (Compuserve: 100321,2355)
// Written: 24th June 1994
// =========================================================

#define OEMRESOURCE		// needed to load scrollbar bitmaps

#include <owl\owlpch.h>
#pragma hdrstop

#include <ctype.h>
#include <cstring.h>
#include <checks.h>
#include "propdlg.h"
#include "proptab.h"

#define TABMARGIN			8
#define SCROLLMARGIN		8

#define USEANSIVARFONT		TRUE
#define SELECTIONWEIGHT		FW_BOLD

static void EnableParents (TWindow *wnd, BOOL bEnable);
static char GetPressedKey ();
static BOOL MatchesHotKey (LPCSTR lpszText, char cKey);

// =========================================================
// TPropertyDialog
//
// base class for property dialog (holder) class
// =========================================================

DEFINE_RESPONSE_TABLE1(TPropertyDialog, TDialog)
	EV_COMMAND(IDOK, CmOk),
	EV_COMMAND(IDCANCEL, CmCancel),
	EV_WM_DESTROY,
	EV_WM_SIZE,
	EV_WM_SYSCOMMAND,
END_RESPONSE_TABLE;

static TIArrayAsVector<TPropertyTab> arrayTabs (5, 0, 5);
static HINSTANCE	hInstProc = NULL;
static HHOOK		hookKbd = NULL;

LRESULT CALLBACK PropdlgKbdProc (int, WPARAM, LPARAM);

TPropertyDialog::TPropertyDialog (TWindow *pParent, TResId resID, Tab::Style style, TModule *module)
	:	TDialog (pParent, resID, module)
{
	pTabs = new TPropertyTab (this, style);
	IsModal = FALSE;	// not modal by default

	CHECKX(pTabs, "Tab control could not be created");
}

TPropertyDialog::~TPropertyDialog ()
{
	if (pTabs)
	{
		arrayTabs.Detach (pTabs);

		if (!arrayTabs.GetItemsInContainer () && hookKbd)
		{
			UnhookWindowsHookEx (hookKbd);

			hookKbd = NULL;
		}
	}
}

void TPropertyDialog::SetHookInstance (HINSTANCE hInstance)
{
	hInstProc = hInstance;
}

// =========================================================
// SetupWindow, EvInitDialog
//
// initialise this dialog box.
// EvInitDialog is overriden so I can control the focus
// =========================================================
void TPropertyDialog::SetupWindow ()
{
	TDialog::SetupWindow ();

	if (GetEntryCount ())
	{
		if (!arrayTabs.GetItemsInContainer ())
		{
			if (!hInstProc)
			{
				hInstProc = GetModule ()->GetInstance ();
			}

		#ifndef __WIN32__
			hookKbd = SetWindowsHookEx (WH_KEYBOARD, PropdlgKbdProc, hInstProc, GetCurrentTask ());
		#else
			hookKbd = SetWindowsHookEx (WH_KEYBOARD, PropdlgKbdProc, hInstProc, GetCurrentThreadId ());
		#endif
		}

		arrayTabs.Add (pTabs);

		SizeTabWnd (GetClientRect ());

		PostMessage (WM_NEXTDLGCTL, (WPARAM) pTabs->HWindow, TRUE);
	}
	else
	{
		arrayTabs.Detach (pTabs);

		pTabs->Destroy ();
		delete pTabs;
		pTabs = NULL;
	}
}

BOOL TPropertyDialog::EvInitDialog (HWND hWndFocus)
{
	TDialog::EvInitDialog (hWndFocus);

	// don't set focus if a page was displayed
	return (pTabs == NULL);
}

void TPropertyDialog::EvDestroy ()
{
	if (pTabs)
	{
		pTabs->DestroyPages ();
	}

	TDialog::EvDestroy ();
}

// =========================================================
// EvSysCommand
//
// picks up requests for Alt+letter hot-key switching
// =========================================================
void TPropertyDialog::EvSysCommand (UINT cmdType, TPoint& pt)
{
	if (cmdType == SC_KEYMENU && GetEntryCount ())
	{
		if (GetPageRef ().CheckHotKey ())
		{
			// hotkey to a control on the parent
			return;
		}
	}

	TDialog::EvSysCommand (cmdType, pt);
}

// =========================================================
// Add
//
// add a property page to this dialog
// =========================================================
void TPropertyDialog::Add (LPCSTR lpszTab, TPropertyPage *pPage, BOOL bEnabled)
{
	PRECONDITIONX(lpszTab, "Invalid property page name supplied");
	PRECONDITIONX(pPage, "Invalid property page dialog supplied");

	pPage->pdlgParent	= this;
	pPage->nThisTab		= GetEntryCount ();

	pTabs->Add (lpszTab, pPage, bEnabled);
}

// =========================================================
// AdjustMargin
//
// allow a derived dialog a change to override the position
// of the actual tab area (to allow for comment areas etc.)
// =========================================================
void TPropertyDialog::AdjustMargin (TRect& rc)
{
	HWND	hwndOK = GetDlgItem (IDOK);

	if (hwndOK)
	{
		TRect	rcOK;

		::GetWindowRect (hwndOK, &rcOK);

		ScreenToClient (rcOK.TopLeft ());

		rc.bottom = rcOK.top - 6;
	}
}

void TPropertyDialog::EvSize (UINT sizeType, TSize& size)
{
	TDialog::EvSize (sizeType, size);

	if (sizeType != SIZE_MINIMIZED && pTabs && (pTabs->GetTabStyle () & Tab::AutoTabResize))
	{
		// move the tab window
		TRect	client;

		GetClientRect (client);

		SizeTabWnd (client);
	}
}

void TPropertyDialog::SizeTabWnd (TRect& client)
{
	if (pTabs)
	{
		client = client.InflatedBy (-TABMARGIN, -TABMARGIN);

		AdjustMargin (client);

		pTabs->SetWindowPos (NULL, client, SWP_NOZORDER);
		pTabs->Invalidate ();
	}
}

// =========================================================
// SetActiveColour
//
// defines the colour of the active tab
// (only if using Tab::ColorActive style)
// =========================================================
void TPropertyDialog::SetActiveColor (TColor color)
{
	if (pTabs)
	{
		pTabs->SetActiveColor (color);
	}
}

// =========================================================
// SetFixedTabWidth
//
// defines the width of tabs when using "FixedWidth" style
// =========================================================
void TPropertyDialog::SetFixedTabWidth (int nWidth)
{
	if (pTabs)
	{
		pTabs->SetFixedTabWidth (nWidth);
	}
}

// =========================================================
// SetWideMarginWidth
//
// defines the extra space added to the each side of a tab
// defined with the "WideMargin" style
// =========================================================
void TPropertyDialog::SetWideMarginWidth (int nWidth)
{
	pTabs->SetWideMarginWidth (nWidth);
}

// =========================================================
// CmOk
//
// handler for OK button.  Asks all pages to save their data
// then closes the dialog
// =========================================================
void TPropertyDialog::CmOk ()
{
	if (pTabs == NULL || pTabs->SavePageData ())
	{
		CloseAction (IDOK);
	}
}

// =========================================================
// CmCancel
//
// handler for Cancel button
// =========================================================
void TPropertyDialog::CmCancel ()
{
	CloseAction (IDCANCEL);
}

// =========================================================
// CloseAction
//
// requests a dialog be closed (can be overriden where the
// dialog is the client to a TMDIChild dialog)
// =========================================================
void TPropertyDialog::CloseAction (int retValue)
{
	switch (retValue)
	{
		case IDOK :
			TDialog::CmOk ();
			break;

		case IDCANCEL :
			TDialog::CmCancel ();
			break;

		default :
			CloseWindow (retValue);
			break;
	}
}

// =========================================================
// Destroy
//
// called when the window is destroyed
// =========================================================
void TPropertyDialog::Destroy (int retValue)
{
	if (retValue && IsModal)
	{
		EnableParents (Parent, TRUE);

		GetApplication ()->EndModal (retValue);
	}

	TDialog::Destroy (retValue);
}

static void EnableParents (TWindow *wnd, BOOL bEnable)
{
	while (wnd)
	{
		wnd->EnableWindow (bEnable);

		wnd = wnd->Parent;
	}
}

// =========================================================
// DoExecute
//
// overriden to make the modeless dialog appear modal to the
// caller, whilst allowing me to retain control of messages!
// =========================================================
int TPropertyDialog::DoExecute()
{
	Create ();
	Show (SW_SHOW);

	IsModal = TRUE;	// make the dialog appear modal

	EnableParents (Parent, FALSE);

	return GetApplication ()->BeginModal (Parent);
}

// =========================================================
// SelectTab
//
// select a specific page
// =========================================================
void TPropertyDialog::SelectTab (int nTab)
{
	if (pTabs)
	{
		pTabs->SelectTab (nTab);
	}
}

// =========================================================
// FirstPageThat
//
// functions that identify the first page meeting some
// criteria (determined by the specific function)
// =========================================================
TPropertyPage* TPropertyDialog::FirstPageThat (TPageCondFunc test, void* paramList)
{
	int	nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		TPropertyPage	*pPage = pTabs->GetPage (n);

		if (test (pPage, paramList))
		{
			return pPage;
		}
	}

	return NULL;
}

TPropertyPage* TPropertyDialog::FirstPageThat (TPageCondMemFunc test, void* paramList)
{
	int	nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		TPropertyPage	*pPage = pTabs->GetPage (n);

		if ((this->*test) (pPage, paramList))
		{
			return pPage;
		}
	}

	return NULL;
}

// =========================================================
// ForEachPage
//
// functions called for every page
// =========================================================
void TPropertyDialog::ForEachPage (TPageActionFunc action, void* paramList)
{
	int	nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		action (pTabs->GetPage (n), paramList);
	}
}

void TPropertyDialog::ForEachPage (TPageActionMemFunc action, void* paramList)
{
	int	nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		(this->*action) (pTabs->GetPage (n), paramList);
	}
}

// =========================================================
// GetPageRef
//
// get reference to a particular page
// =========================================================
TPropertyPage& TPropertyDialog::GetPageRef (int nTab)
{
	if (nTab == -1)
	{
		nTab = pTabs->GetSelectedTab ();
	}

	PRECONDITIONX (nTab >= 0 && nTab < GetEntryCount (), "GetPage: Invalid index");

	return *(pTabs->GetPage (nTab));
}

// =========================================================
// GetEntryCount
//
// returns the number of pages in the property dialog
// =========================================================
int TPropertyDialog::GetEntryCount ()
{
	if (pTabs == NULL)
	{
		return NULL;
	}

	return pTabs->GetEntryCount ();
}

// =========================================================
// SendMsgToEachPage
//
// sends the given message to each page
// returns the return value of the last page
// =========================================================
LRESULT TPropertyDialog::SendMsgToEachPage (UINT msg, WPARAM wParam, LPARAM lParam)
{
	int		nCount = GetEntryCount ();
	LRESULT	lResult = NULL;

	for (int n = 0; n < nCount; n++)
	{
		lResult = pTabs->GetPage (n)->SendMessage (msg, wParam, lParam);
	}

	return lResult;
}


// =========================================================
// TabColorChange
//
// picks up colors to use from Control Panel
// =========================================================
static void TabColorChangeIter (TPropertyTab& tab, void *)
{
	tab.TabColorChange ();
}

void TPropertyDialog::TabColorChange (TApplication *application)
{
	arrayTabs.ForEach (TabColorChangeIter, NULL);

	if (application && application->Ctl3dEnabled ())
	{
		FARPROC	proc = application->GetCtl3dModule ()->GetProcAddress("Ctl3dColorChange");

		if (proc)
		{
			proc ();
		}
	}
}

// ======================================================= //

// =========================================================
// TPropertyPage
//
// base class for a property page inside a property dialog
// =========================================================

DEFINE_RESPONSE_TABLE1(TPropertyPage, TDialog)
  EV_COMMAND(IDOK, CmOk),
  EV_COMMAND(IDCANCEL, CmCancel),
  EV_WM_SYSCOMMAND,
END_RESPONSE_TABLE;

TPropertyPage::TPropertyPage (TResId resID, TModule *module)
	:	TDialog (0, resID, module)
{
}

TPropertyPage::~TPropertyPage ()
{
}

void TPropertyPage::CmOk ()
{
	// pass it on to the TPropertyDialog
	pdlgParent->ForwardMessage ();
}

void TPropertyPage::CmCancel ()
{
	// pass it on to the TPropertyDialog
	pdlgParent->ForwardMessage ();
}

BOOL TPropertyPage::SaveData ()
{
	// use transfer buffer to save data
	// (if available)
	TransferData (tdGetData);

	// continue by default
	return TRUE;
}

// =========================================================
// EvSysCommand
//
// picks up requests for Alt+letter hot-key switching
// =========================================================
void TPropertyPage::EvSysCommand (UINT cmdType, TPoint& pt)
{
	if (cmdType == SC_KEYMENU)
	{
		if (CheckHotKey (pdlgParent))
		{
			// hotkey to a control on the parent
			return;
		}
	}

	TDialog::EvSysCommand (cmdType, pt);
}

// =========================================================
// Select..., EnableTab
//
// access functions passed on to the tab control
// =========================================================
void TPropertyPage::SelectTab (int nTab)
{
	if (nTab == -1)
	{
		nTab = nThisTab;
	}

	TYPESAFE_DOWNCAST (Parent, TPropertyTab)->SelectTab (nTab);
}

void TPropertyPage::SelectNext ()
{
	TYPESAFE_DOWNCAST (Parent, TPropertyTab)->SelectNext ();
}

void TPropertyPage::SelectPrevious ()
{
	TYPESAFE_DOWNCAST (Parent, TPropertyTab)->SelectNext (-1);
}

void TPropertyPage::EnableTab (int nTab, BOOL bEnable)
{
	if (nTab == -1)
	{
		nTab = nThisTab;
	}

	TYPESAFE_DOWNCAST (Parent, TPropertyTab)->EnableTab (nTab, bEnable);
}

BOOL TPropertyPage::IsTabEnabled (int nTab)
{
	if (nTab == -1)
	{
		nTab = nThisTab;
	}

	return TYPESAFE_DOWNCAST (Parent, TPropertyTab)->IsTabEnabled (nTab);
}

// =========================================================
// MessageBox
//
// focus-safe way to display a messagebox from inside a page
// =========================================================
int TPropertyPage::MessageBox (const char far* text, const char far* caption, UINT type)
{
	return GetParentWindow ()->MessageBox (text, caption, type);
}

// =========================================================
// GetParentWindow
//
// returns a window that can be used as a parent to a
// popup window from this page
// =========================================================
TWindow *TPropertyPage::GetParentWindow ()
{
	for (	TWindow *parent = pdlgParent;
			parent != NULL && !(parent->Attr.Style & WS_CAPTION);
			parent = parent->Parent) ;

	return parent;
}

// =========================================================
// CheckHotKey
//
// check to see if current alt-letter combination is a
// hot-key either for a tab or in the 'other' dialog (ie. if
// we are currently on a page, check the main property
// dialog, otherwise check the page)
// =========================================================
BOOL TPropertyPage::CheckHotKey (TDialog *dlg)
{
	// work out which key was pressed
	char	cKey = GetPressedKey ();

	if (TYPESAFE_DOWNCAST (Parent, TPropertyTab)->SelectTabByKey (cKey))
	{
		// page hotkey, so don't pass to Windows
		return TRUE;
	}

	if (dlg == NULL)
	{
		dlg = this;
	}

	if (!cKey)
	{
		return FALSE;
	}

	// walk through the childlist looking for a match
	HWND	hwndFirst = ::GetWindow (dlg->HWindow, GW_CHILD);

	for (HWND hwndChild = hwndFirst; hwndChild != NULL; hwndChild = ::GetWindow (hwndChild, GW_HWNDNEXT))
	{
		char	szText[100];

		::GetWindowText (hwndChild, szText, sizeof (szText));

		if (::IsWindowEnabled (hwndChild) && MatchesHotKey (szText, cKey))
		{
			// if I can't go here, find the next available tabbable control
			if (!(::GetWindowLong (hwndChild, GWL_STYLE) & WS_TABSTOP))
			{
				hwndChild = dlg->GetNextDlgTabItem (hwndChild, FALSE);
			}

			// spoof a mouse-click
			::PostMessage (hwndChild, WM_LBUTTONDOWN, MK_LBUTTON, NULL);
			::PostMessage (hwndChild, WM_LBUTTONUP, MK_LBUTTON, NULL);

			return TRUE;
        }
	}

	return FALSE;
}

// ======================================================= //

// =========================================================
// TPropertyTab
//
// class which handles drawing and switching of pages
// =========================================================

DEFINE_RESPONSE_TABLE1(TPropertyTab, TControl)
	EV_WM_LBUTTONDOWN,
	EV_WM_SIZE,
	EV_WM_SETFOCUS,
	EV_WM_KILLFOCUS,
	EV_WM_KEYDOWN,
	EV_WM_GETDLGCODE,
END_RESPONSE_TABLE;

TPropertyTab::TPropertyTab (TWindow *pParent, Tab::Style style)
	: 	TControl (pParent, 29000, "", 0, 0, 1, 1),
		pages (10, 0, 5)
{
	// assign tab style
	styleTabs		= style;

	if (styleTabs & Tab::Stacked)
	{
		// for stacked tabs, disable collapsing and enable justification
		styleTabs = (styleTabs & ~Tab::Collapsing) | Tab::Justified;
	}

	// set up the font used for selected pages
#if USEANSIVARFONT
	pfontNormal = new TFont ((HFONT) GetStockObject (ANSI_VAR_FONT));
#else
	pfontNormal = new TFont ("MS Sans Serif", -11);
#endif

	LOGFONT	lf;

	pfontNormal->GetObject (lf);
	lf.lfWeight = SELECTIONWEIGHT;
	pfontSelect = new TFont (&lf);

	CHECKX (pfontSelect, "Unable to create selected property font");

	// Work out font height (thanks to Paul Winwood)
	TRect rcCalc;

	{
		TClientDC	dc (HWindow);
		dc.SelectObject (*pfontSelect);
		dc.DrawText ("(|&y#", 5, rcCalc, DT_CENTER | DT_CALCRECT);
	}

	nTextHeight = rcCalc.Height();

	// initialise important variables
	nSelect 		= 0;
	nTabHeight		= 8 + nTextHeight + ((styleTabs & Tab::DoubleHeight) ? nTextHeight : 2);
	rcSize.SetEmpty ();

	// set up parameters for scrolling
	nTabOrigin		= 0;
	bLeftScroll 	=
	bRightScroll	= 0;
	bmpLeftArrow	=
	bmpRightArrow	= NULL;

	// hard set the window attributes
/*	// (this is to prevent keystrokes being sent here because of WS_TABSTOP!!)	*/
	Attr.Style = WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | WS_TABSTOP;

	// define default active tab colour (for Tab::ColorActive style)
	ppenActive = new TPen (TColor::Black);

	// reset row order for stacked tabs
	for (int n = 0; n < MAXROWS; n++)
	{
		nRowOrder[n] = n;
	}

	// set fixed tab width to default
	SetFixedTabWidth (DEFTABWIDTH);
	SetWideMarginWidth (DEFWIDEMARGIN);

	// assign tab colors
	TabColorChange ();
}

TPropertyTab::~TPropertyTab ()
{
	delete pfontNormal;
	delete pfontSelect;
	delete ppenActive;

	if (bmpLeftArrow)
	{
		delete bmpLeftArrow;
	}

	if (bmpRightArrow)
	{
		delete bmpRightArrow;
	}
}

// =========================================================
// SetupWindow
//
// creates the page dialogs
// =========================================================
void TPropertyTab::SetupWindow ()
{
	TControl::SetupWindow ();

	//
	// create the page dialogs
	//
	int		nCount = GetEntryCount ();
	int		n;

	for (n = 0; n < nCount; n++)
	{
		TPropertyPage	*pPage = pages[n].GetDialog ();

		if (styleTabs & Tab::AllowDupPages && pPage->HWindow)
		{
			// reusing the same dialog has been allowed, so
			// skip the create call (as it would cause a
			// precondition violation)
			continue;
		}

		pPage->SetParent (this);

		if (!(styleTabs & Tab::CreateOnDemand))
		{
			pPage->Create ();
		}
	}

	//
	// select the first selectable page
	//
	for (n = 0; n < nCount; n++)
	{
		if (IsTabVisible (n) && IsTabEnabled (n))
		{
			nSelect = n;

			// inform interested parties that the
			// selection has changed
			TYPESAFE_DOWNCAST (Parent, TPropertyDialog)->AfterSelected (nSelect);
			pages[nSelect].GetDialog ()->AfterSelected ();
			break;
		}
	}

	//
	// set the focus on the selected page
	//
	if (nCount)
	{
		SetPageFocus ();
	}
}

LPSTR TPropertyTab::GetClassName ()
{
	return "tbProperty";
}

void TPropertyTab::GetWindowClass (WNDCLASS& wc)
{
	TControl::GetWindowClass (wc);

	wc.style 		 =	CS_HREDRAW |
						CS_VREDRAW |
						CS_PARENTDC;

	wc.hbrBackground = NULL;	// don't use a background brush
}

void TPropertyTab::DestroyPages ()
{
	int		nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		pages[n].GetDialog ()->Destroy ();	// thanks Bob!
		delete pages[n].GetDialog ();
	}

	pages.Flush ();
}

// =========================================================
// Add
//
// adds a new page to the control
// =========================================================
void TPropertyTab::Add (LPCSTR lpszTab, TPropertyPage *pPage, BOOL bEnabled)
{
	TPropertyEntry	entry (lpszTab, pPage, bEnabled);

	pages.Add (entry);
}

// =========================================================
// SavePageData
//
// calls the SaveData function on each page before the
// dialog is closed (only after the OK button is pressed)
// =========================================================
BOOL TPropertyTab::SavePageData ()
{
	int		nCount = GetEntryCount ();

	for (int n = 0; n < nCount; n++)
	{
		if (!pages[n].GetDialog ()->SaveData ())
		{
			return FALSE;
		}
	}

	return TRUE;
}

// =========================================================
// Paint, DrawTabs, DrawStackedTabs, DrawTab
//
// functions for painting the actual tabs
// =========================================================
void TPropertyTab::Paint (TDC& dc, BOOL, TRect& rcPaint)
{
	TRect	client;
	TRect	rcGrey;

	GetClientRect (client);
	client.bottom--;
	client.right--;
	client.top += nTabHeight;

	dc.SaveDC ();

	// draw and accomodate left/right scroll buttons
	if (bLeftScroll | bRightScroll)
	{
		TRect	rcScroll;

		GetScrollRect (rcScroll);

		if (rcScroll.Touches (rcPaint))
		{
			TMemoryDC 	bmpDC (dc);

			// draw the left-hand scroll button
			GetScrollRect (rcScroll, 0);
			bmpDC.SelectObject (*bmpLeftArrow);
			dc.BitBlt (rcScroll, bmpDC, TPoint(0, 0), SRCCOPY);
			bmpDC.RestoreBitmap();

			// draw the right-hand scroll button
			GetScrollRect (rcScroll, 1);
			bmpDC.SelectObject (*bmpRightArrow);
			dc.BitBlt (rcScroll, bmpDC, TPoint(0, 0), SRCCOPY);
			bmpDC.RestoreBitmap();

			// draw white line under the scroll buttons
			TPen	penWhite (colorWhite);

			GetScrollRect (rcScroll);
			dc.SelectObject (penWhite);
			dc.MoveTo (rcScroll.left - SCROLLMARGIN, nTabHeight);
			dc.LineTo (rcScroll.right, nTabHeight);

			// don't allow tabs to overpaint scroll buttons
			dc.ExcludeClipRect (rcScroll);
			rcScroll.top	= 0;
			rcScroll.left	-= SCROLLMARGIN;

			// grey the surrounding area
			dc.FillRect (rcScroll, TBrush (colorLtGray));
			rcScroll.left	+= 4;

			dc.ExcludeClipRect (rcScroll);
		}
	}

	if (client.Touches (rcPaint))
	{
		//
		// draw the box surround of the enclosed area
		//
		{
			TPen	penWhite (colorWhite);

			dc.SelectObject (penWhite);

			dc.MoveTo (client.left, nTabHeight);
			dc.LineTo (client.left, client.bottom);
		}

		{
			TPen	penBlack (colorBlack);

			dc.SelectObject (penBlack);

			dc.LineTo (client.right, client.bottom);
			dc.LineTo (client.right, nTabHeight);
		}

		{
			TPen	penGrey (colorDkGray);

			dc.SelectObject (penGrey);

			rcGrey = client.InflatedBy (-1, -1);

			dc.MoveTo (rcGrey.left, rcGrey.bottom);
			dc.LineTo (rcGrey.right, rcGrey.bottom);
			dc.LineTo (rcGrey.right, nTabHeight);
		}

		dc.FillRect (rcGrey, TBrush (colorLtGray));
	}

	//
	// now draw the tabs (if necessary)
	//
	SetRect (client);

	if (client.Touches (rcPaint))
	{
		if (styleTabs & Tab::Stacked)
		{
			DrawStackedTabs (dc);
		}
		else
		{
			DrawTabs (dc);
		}
	}

	dc.RestoreObjects ();
	dc.RestoreDC ();
}

void TPropertyTab::DrawTabs (TDC& dc)
{
	TRect	rcTabs;
	int		x = 0;
	int		nCount	= GetEntryCount ();

	SetRect (rcTabs);

	for (int n = nTabOrigin; n < nCount; n++)
	{
		if (IsTabVisible (n))
		{
			int	xOld = x;

			if (x > xMax)	// don't draw out-of-range tabs!!!
			{
				break;
			}

			DrawTab (dc, x, n);

			// draw the white line under this tab
			if (!IsTabSelected (n))
			{
				{
					TPen	penWhite (colorWhite);

					dc.SelectObject (penWhite);

					dc.MoveTo (xOld, nTabHeight);
					dc.LineTo (x, nTabHeight);
				}

				{
					TPen	penLtGrey (colorLtGray);

					dc.SelectObject (penLtGrey);

					dc.MoveTo (xOld, nTabHeight - 1);
					dc.LineTo (x, nTabHeight - 1);
				}
			}
		}
	}

	if (x < rcTabs.right)
	{
		// draw white line along the top of the enclosed area
		TPen	penWhite (colorWhite);

		dc.SelectObject (penWhite);

		dc.MoveTo (x, nTabHeight);
		dc.LineTo (rcTabs.right, nTabHeight);

		// fill above the white line with grey
		rcTabs.left = x;
		rcTabs.bottom++;

		dc.FillRect (rcTabs, TBrush (colorLtGray));
	}
}

void TPropertyTab::DrawStackedTabs (TDC& dc)
{
	TRect	rcTabs;
	int		y = (nRows - 1) * nTabHeight;

	for (int nRow = 0; nRow < nRows; nRow++)
	{
		int		x = 0;
		int		nRealRow = nRowOrder[nRow];

		for (int iter = ptRanges[nRealRow].Left; iter <= ptRanges[nRealRow].Right; iter++)
		{
			int	xOld = x;

			DrawTab (dc, x, iter, y, nRow == 0);

			// draw the white line under this tab
			if (!nRow && !IsTabSelected (iter))
			{
				{
					TPen	penWhite (colorWhite);

					dc.SelectObject (penWhite);

					dc.MoveTo (xOld, y + nTabHeight);
					dc.LineTo (x, y + nTabHeight);
				}

				{
					TPen	penLtGrey (colorLtGray);

					dc.SelectObject (penLtGrey);

					dc.MoveTo (xOld, y + nTabHeight - 1);
					dc.LineTo (x, y + nTabHeight - 1);
				}
			}
		}

		y -= nTabHeight;
	}
}

void TPropertyTab::DrawTab (TDC& dc, int& xParam, int nTab, int yParam, BOOL bIsLast)
{
	WORD	fuFormat;
	int		nTabWidth 	= GetTabWidth (nTab);
	int		yExtra = IsTabSelected (nTab) ? 0 : 2;
	int		x = xParam + !IsTabSelected (nTab);
	int		x2 = xParam + nTabWidth;
	int		y = yParam + yExtra;

	TRect	rcTab;

	rcTab.Set (x + 1, y + 1, x2 - 2, yParam + nTabHeight + bIsLast);

	// check to see if this tab needs a right hand "jagged edge"
	if (bLeftScroll | bRightScroll)
	{
		TRect	rcScroll;

		GetScrollRect (rcScroll);
		rcScroll.left	-= SCROLLMARGIN;

		if (rcTab.right > rcScroll.left)
		{
			TPen	penGrey (colorDkGray);
			int		yDiff = (rcScroll.bottom - y) / 3;

			// paint the jagged edge
			dc.SelectObject (penGrey);
			dc.MoveTo (rcScroll.left + 1, y);
			dc.LineTo (rcScroll.left + 2, y + yDiff);
			dc.LineTo (rcScroll.left + 0, y + yDiff + yDiff);
			dc.LineTo (rcScroll.left + 2, rcScroll.bottom - 2);
			dc.LineTo (rcScroll.left + 1, rcScroll.bottom);

			rcScroll.top	= y;
			dc.ExcludeClipRect (rcScroll);
		}
	}

	// fill the tab
	dc.FillRect (rcTab, TBrush (colorLtGray));

	if (!bIsLast)
	{
		rcTab.bottom++;
	}

	// draw the frame around the tab
	TPen	penGrey (colorLtGray);
	TPen	penDkGrey (colorDkGray);
	TPen	penWhite (colorWhite);
	TPen	penBlack (colorBlack);

	if ((styleTabs & Tab::ColorActive) && IsTabSelected (nTab))
	{
		dc.SelectObject (*ppenActive);
	}
	else
	{
		dc.SelectObject (penWhite);
	}

	dc.MoveTo (x, yParam + nTabHeight - yExtra);
	dc.LineTo (x, y + 2);
	dc.LineTo (x + 2, y);
	dc.LineTo (x2 - 2, y);

	if (! ((styleTabs & Tab::ColorActive) && IsTabSelected (nTab)))
	{
		dc.SelectObject (penDkGrey);
	}

	dc.MoveTo (x2 - 2, y + 1);
	dc.LineTo (x2 - 2, yParam + nTabHeight + 1 - yExtra);

	dc.SelectObject (penBlack);

	rcTab = rcTab.InflatedBy (-1, -1);

	dc.MoveTo (x2 - 1, y + 2);
	dc.LineTo (x2 - 1, yParam + nTabHeight + 1 - yExtra);

	// blank out previous selection at this position
	if (IsTabSelected (nTab))
	{
		dc.SelectObject (penGrey);

		dc.MoveTo (x, y + 1);
		dc.LineTo (x, y);
		dc.LineTo (x + 2, y);

		dc.MoveTo (x2 - 2, y);
		dc.LineTo (x2, y + 2);
		dc.LineTo (x2, y);
		dc.LineTo (x2 - 2, y);
	}
	else
	{
		TPoint	pt[7];

		pt[0]	= TPoint (x, y + 2);
		pt[1]	= TPoint (x + 2, y);
		pt[2]	= TPoint (x2 - 2, y);
		pt[3]	= TPoint (x2, y + 2);
		pt[4]	= TPoint (x2, yParam);
		pt[5]	= TPoint (x, yParam);
		pt[6]	= TPoint (x, y + 2);

		TBrush brush (colorLtGray);
		TRegion rgn (pt, 7, ALTERNATE);

		dc.SelectObject (penGrey);

		dc.MoveTo (xParam, yParam);
		dc.LineTo (xParam, yParam + nTabHeight + 1);

		dc.SelectObject (brush);
		dc.PaintRgn (rgn);
	}

	//
	// draw the tab text
	//
	{
		if (!IsTabSelected (nTab))
		{
		//	dc.SelectStockObject (ANSI_VAR_FONT);
			dc.SelectObject (*pfontNormal);
		}
		else
		{
			dc.SelectObject (*pfontSelect);
		}

		dc.SetBkMode (TRANSPARENT);
		dc.SetTextColor (IsTabEnabled (nTab) ? TColor (GetSysColor (COLOR_WINDOWTEXT)) : TColor::Gray);

		TRect	rcCalc (rcTab);
		string&	s = pages[nTab].GetText ();

		fuFormat = DT_VCENTER | DT_SINGLELINE;

		dc.DrawText (s.c_str (), s.length (), rcCalc, DT_CENTER | DT_CALCRECT | DT_WORDBREAK);

		if ((styleTabs & Tab::DoubleHeight) && rcCalc.Height () > nTextHeight + 4)
		{
			fuFormat = DT_WORDBREAK;
		}

		dc.DrawText (s.c_str (), s.length (), rcTab, (WORD) (DT_CENTER | fuFormat));

		if (IsTabSelected (nTab) && HWindow == GetFocus ())
		{
			dc.DrawFocusRect (rcTab);
		}
	}

	xParam += nTabWidth;
}

// =========================================================
// SetRect
//
// returns the TRect area that the tabs occupy
// =========================================================
void TPropertyTab::SetRect (TRect& rc, int nRow)
{
	GetClientRect (rc);

	if (nRow == -1)
	{
		rc.bottom = (nTabHeight * nRows) - 1;
	}
	else
	{
		rc.top 		= (nRows - nRow - 1) * nTabHeight;
		rc.bottom 	= rc.top + nTabHeight - 1;
	}

	if (bLeftScroll | bRightScroll)
	{
		TRect	rcScroll;

		GetScrollRect (rcScroll);

		rc.right = rcScroll.left - SCROLLMARGIN;
	}
}

// =========================================================
// IsTabVisible, IsTabSelected, IsTabEnabled
//
// return information about a particular tab
// =========================================================
BOOL TPropertyTab::IsTabVisible (int nTab)
{
	if (styleTabs & Tab::Collapsing)
	{
		return IsTabEnabled (nTab);
	}
	else
	{
		return TRUE;
	}
}

BOOL TPropertyTab::IsTabSelected (int nTab)
{
	PRECONDITIONX (nTab >= 0 && nTab < pages.GetItemsInContainer (), "IsTabSelected: Invalid index");

	return nTab == nSelect;
}

BOOL TPropertyTab::IsTabEnabled (int nTab)
{
	PRECONDITIONX (nTab >= 0 && nTab < pages.GetItemsInContainer (), "IsTabEnabled: Invalid index");

	return pages[nTab].IsEnabled ();
}

// =========================================================
// EvLButtonDown, GetMouseTab
//
// process mouse messages.  GetMouseTab indicates which tab
// the mouse is currently over
// =========================================================
void TPropertyTab::EvLButtonDown (UINT, TPoint& point)
{
	TRect	rc;

	if (bLeftScroll)
	{
		GetScrollRect (rc, 0);

		if (rc.Contains (point))
		{
			NextOrigin (-1);

			return;
		}
	}
	if (bRightScroll)
	{
		GetScrollRect (rc, 1);

		if (rc.Contains (point))
		{
			NextOrigin ();

			return;
		}
	}

	if (bLeftScroll | bRightScroll)
	{
		GetScrollRect (rc);
		rc.left -= SCROLLMARGIN;
		rc.top = 0;

		if (rc.Contains (point))
		{
			// no available
			return;
		}
	}

	SelectTab (GetMouseTab (point));
}

int TPropertyTab::GetMouseTab (TPoint& point)
{
	TRect	rc;
	int		nCount = GetEntryCount ();

	for (int n = nTabOrigin; n < nCount; n++)
	{
		if (IsTabVisible (n))
		{
			if (GetTabRect (rc, n))
			{
				if (rc.Contains (point))
				{
					if (pages[n].IsEnabled ())
					{
						return n;
					}

					break;	// disabled, so don't continue
				}
			}
		}
	}

	return -1;
}

// =========================================================
// GetTabRect
//
// returns the TRect area of a particular tab
// =========================================================
BOOL TPropertyTab::GetTabRect (TRect& rc, int nTab, int nOrigin)
{
	int		nFirst;
	int		n;
	int		nCount = GetEntryCount ();

	SetRect (rc);

	if (nOrigin == -1)
	{
		nOrigin = nTabOrigin;
	}

	if (styleTabs & Tab::Stacked)
	{
		n	= GetTabRow (nTab);

		if (n != -1)
		{
			int	nRealRow = nRowOrder[n];

			nFirst = ptRanges[nRealRow].Left;
			nCount = ptRanges[nRealRow].Right + 1;

			rc.top	= (nRows - 1 - n) * nTabHeight;
			rc.bottom = rc.top + nTabHeight;
		}
	}
	else
	{
		nFirst = nOrigin;
		nCount = GetEntryCount ();
	}

	for (n = nFirst; n < nCount; n++)
	{
		int	nTabWidth = GetTabWidth (n);

		if (n == nTab)
		{
			rc.right = rc.left + nTabWidth - 1;

			return IsTabVisible (n);
		}

		if (IsTabVisible (n))
		{
			rc.left += nTabWidth;
		}
	}

	return FALSE;
}

// =========================================================
// GetTabRow
//
// returns the row of a particular tab
// =========================================================
int TPropertyTab::GetTabRow (int nTab)
{
	for (int n = 0; n < nRows; n++)
	{
		int	nRealRow = nRowOrder[n];

		if (nTab >= ptRanges[nRealRow].Left && nTab <= ptRanges[nRealRow].Right)
		{
			return n;
		}
	}

	return -1;
}

// =========================================================
// CalcTabWidth
//
// returns the display width of a particular tab
// =========================================================
int TPropertyTab::CalcTabWidth (TDC& dc, int nTab)
{
	if (styleTabs & Tab::FixedWidth)
	{
		return nFixedTabWidth;
	}

	TSize	sz;
	string&	s = pages[nTab].GetText ();
	LPCSTR	lpszText = s.c_str ();

	dc.SelectObject (*pfontSelect);

	BOOL 	bAnyBreaks = (strchr (lpszText, ' ') != NULL) ||
						 (strchr (lpszText, '\r') != NULL);

	if ((styleTabs & Tab::DoubleHeight) && bAnyBreaks)
	{
		// keep increasing the width until the tab text
		// will fit on two lines
		sz.cx = nFixedTabWidth;

		for (;;)
		{
			TRect	rc (0, 0, sz.cx, nTextHeight);

			dc.DrawText (lpszText, s.length (), rc, DT_LEFT | DT_TOP | DT_WORDBREAK | DT_CALCRECT);

			if (rc.bottom < nTextHeight * 3)
			{
				sz.cx = rc.right;
				break;
			}

			sz.cx += 8;
		}
	}
	else
	{
		sz.cx = nFixedTabWidth;

		TRect	rc (0, 0, sz.cx, nTextHeight);

		dc.DrawText (lpszText, s.length (), rc, DT_LEFT | DT_TOP | DT_CALCRECT);

		sz.cx = rc.right;
	}

	// allow for borders
	sz.cx += 7;

	if (styleTabs & Tab::WideMargins)
	{
		// add wide margin width
		sz.cx += (nWideMarginWidth << 1);
	}

	if (sz.cx > nFixedTabWidth)
	{
		return sz.cx;
	}
	else
	{
		return nFixedTabWidth;
	}
}

// ==============================================================
// CalculateTabWidths, CalculateBasicTabWidths, ReassessTabWidths
//
// calculates widths of all tabs
// ==============================================================
void TPropertyTab::CalculateTabWidths ()
{
	//
	// check if tab control properly sized yet, if not then hold back
	//
	TRect	rcTabs;

	SetRect (rcTabs);

	if (rcTabs.right - rcTabs.left <= 1)
	{
		return;
	}

	//
	// calculate basic tab widths
	//
	CalculateBasicTabWidths ();

	//
	// space out tabs if justification required
	//
	if (styleTabs & Tab::Justified)
	{
		int 	iter;
		int		nCount;
		int		xRow;

		if (!(styleTabs & Tab::Stacked) && (bLeftScroll | bRightScroll))
		{
			// can't justify non-stacked controls with scrolling
			return;
		}

		for (int n = 0; n < nRows; n++)
		{
			xRow 	=
			nCount 	= 0;

			//
			// calculate the amount of slack on this row
			//
			for (iter = ptRanges[n].Left; iter <= ptRanges[n].Right; iter++)
			{
				if (IsTabVisible (iter))
				{
					xRow += pages[iter].GetWidth ();
					nCount++;
				}
			}

			xRow = (rcTabs.right - rcTabs.left) - xRow;

			//
			// allocate the slack between all tabs on this row
			//
			for (iter = ptRanges[n].Left; iter <= ptRanges[n].Right; iter++)
			{
				if (!xRow || !nCount)	// nothing more to allocate
				{
					break;
				}

				if (IsTabVisible (iter))
				{
					// add in a little more space to this control
					pages[iter].SetWidth (pages[iter].GetWidth () + (xRow / nCount));

					xRow -= xRow / nCount--;	// remove this space from the allocatable space
				}
			}
		}
	}
}

void TPropertyTab::CalculateBasicTabWidths ()
{
	int		nCount = GetEntryCount ();
	int		n;

	TClientDC	dc (HWindow);

	for (n = 0; n < nCount; n++)
	{
		pages[n].SetWidth (CalcTabWidth (dc, n));
	}

	//
	// work out tab-to-row allocations
	//
	if (styleTabs & Tab::Stacked)
	{
		TRect	rcTabs;

		SetRect (rcTabs);

		for (n = nRows = 0; n < nCount; n++)
		{
			ptRanges[nRows].Left	=
			ptRanges[nRows].Right	= n;

			int	x 		= rcTabs.left;

			for (int iter = n; iter < nCount; iter++)
			{
				if (IsTabVisible (iter))
				{
					x += pages[iter].GetWidth ();
				}

				if (x < rcTabs.right)
				{
					ptRanges[nRows].Right = iter;
					n = iter;
				}
				else
				{
					nRows++;
					break;
				}
			}
		}

		if (nCount)
		{
			nRows++;
		}
	}
	else
	{
		nRows = 1;

		ptRanges[0].Left	= 0;
		ptRanges[0].Right	= GetEntryCount () - 1;
	}
}

void TPropertyTab::ReassessTabWidths ()
{
	if (styleTabs & Tab::Justified)
	{
		//
		// recalculate tab widths with/without scroll buttons
		//
		CalculateTabWidths ();

		TRect	rcTabs;

		SetRect (rcTabs);
		rcTabs.right++;
		rcTabs.bottom += 2;
		InvalidateRect (rcTabs);
	}
}

// =========================================================
// GetLastTab
//
// param = TRUE - returns last fully-visible enabled tab
// param = FALSE- returns last enabled tab
// =========================================================
int TPropertyTab::GetLastTab (BOOL bVisible)
{
	int		nTabs = GetEntryCount ();
	TRect	rcTab;

	while (--nTabs >= 0)
	{
		if (bVisible)
		{
			GetTabRect (rcTab, nTabs);

			if (rcTab.right > xMax)
			{
				continue;
			}
		}

#if COLLAPSEITEMS
		if (IsTabEnabled (nTabs))
#endif
		{
			break;
		}
	}

	if (nTabs < 0)
	{
		nTabs = 0;
	}

	return nTabs;
}


// =========================================================
// SelectTab, SelectNext, SelectTabByKey
//
// select another tab and redraw to reflect the change
// =========================================================
void TPropertyTab::SelectTab (int nTab)
{
	if (nTab != -1 && nTab != nSelect)
	{
		PRECONDITIONX (nTab >= 0 && nTab < pages.GetItemsInContainer (), "SelectTab: Invalid index");

		if (nSelect < 0)
		{
			// prevent errors for first selection
			nSelect = 0;
		}

		TRect	rcTab;
		BOOL	bOriginChanged = AdjustOrigin (nTab);
		int		nOldSelect = nSelect;

		if (pages[nSelect].GetDialog () != pages[nTab].GetDialog ())
		{
			// get the existing invalid region
			TRegion	rgn;

			if (bOriginChanged)
			{
				GetUpdateRgn (rgn, FALSE);
			}

			// hide only if actual dialog box is changing
			// (prevents flickers when using Tab::AllowDupPages)
			SendMessage (WM_SETREDRAW, FALSE);

			if (pages[nSelect].GetDialog ()->HWindow)
			{
				if (styleTabs & Tab::CreateOnDemand)
				{
					pages[nSelect].GetDialog ()->SaveData ();
					pages[nSelect].GetDialog ()->Destroy ();
				}
				else
				{
					pages[nSelect].GetDialog ()->Show (SW_HIDE);
				}
			}

			SendMessage (WM_SETREDRAW, TRUE);

			if (bOriginChanged)
			{
				InvalidateRgn (rgn, FALSE);
			}
		}

		if (styleTabs & Tab::Stacked)
		{
			int	nTabRow 	= GetTabRow (nTab);
			int nRealRow 	= nRowOrder[nTabRow];

			//
			// do I need to swap rows
			//
			if (nTabRow != 0)
			{
				TRect		rcRow;
				int			nTemp = nRowOrder[0];

				// swap the rows
				nRowOrder[0] 		= nRealRow;
				nRowOrder[nTabRow] 	= nTemp;

				// redraw bottom row
				SetRect (rcRow, 0);
				rcRow.bottom += 2;
				InvalidateRect (rcRow);

				// redraw original row
				SetRect (rcRow, nTabRow);
				rcRow.bottom += 2;
				InvalidateRect (rcRow);
			}
		}

		nSelect = nTab;

		if (!bOriginChanged)
		{
			GetTabRect (rcTab, nOldSelect);
			rcTab.right++;
			rcTab.bottom += 2;

			InvalidateRect (rcTab, FALSE);

			GetTabRect (rcTab, nTab);
			rcTab.right++;
			rcTab.bottom += 2;
			InvalidateRect (rcTab, FALSE);
		}
		else
		{
			TRect	rcScroll;

			GetScrollRect (rcScroll);

			rcScroll -= SCROLLMARGIN;
			rcScroll.top = 0;

			InvalidateRect (rcScroll);

			AssessScroll ();
		}

		// set the focus to the first child
		SetPageFocus ();

		// inform interested parties that the
		// selection has changed
		TYPESAFE_DOWNCAST (Parent, TPropertyDialog)->AfterSelected (nTab);
		pages[nSelect].GetDialog ()->AfterSelected ();
	}
}

void TPropertyTab::SelectNext (int nDir)
{
	// +-------------------------------------------------
	// select next/previous tab
	// +-------------------------------------------------
	int	nNew = nSelect;

	do
	{
		nNew += nDir;

		if (nNew < 0)
		{
			nNew = GetEntryCount () - 1;
		}
		else
		{
			if (nNew >= GetEntryCount ())
			{
				nNew = 0;
			}
		}

		if (IsTabVisible (nNew) && IsTabEnabled (nNew))
		{
			SelectTab (nNew);
			UpdateWindow ();

			return;
		}
	}
	while (nNew != nSelect);
}

BOOL TPropertyTab::SelectTabByKey (char cKey)
{
	// +-------------------------------------------------
	// select a tab according to some hot key combination
	// +-------------------------------------------------
	int		nCount = GetEntryCount ();

	if (cKey == 0)
	{
		return FALSE;
	}

	for (int nTab = 0; nTab < nCount; nTab++)
	{
		if (IsTabEnabled (nTab))
		{
			if (MatchesHotKey (pages[nTab].GetText ().c_str (), cKey))
			{
				SelectTab (nTab);

				return TRUE;
			}
		}
	}

	return FALSE;
}

BOOL TPropertyTab::AdjustOrigin (int nTab)
{
	int		nOldOrigin = nTabOrigin;

	//
	// check to see if the origin needs to change
	//
	if (nTab < nTabOrigin)
	{
		nTabOrigin = nTab;
	}
	else
	{
		while (nTabOrigin < nTab)
		{
			TRect	rcTab;

			if (GetTabRect (rcTab, nTab))
			{
				if (rcTab.right >= xMax)
				{
					nTabOrigin++;

					continue;	// go around again
				}
			}

			break;
		}
	}

	if (nOldOrigin == nTabOrigin)
	{
		return FALSE;
	}

	TRect	rcAllTabs;

	SetRect (rcAllTabs);
	rcAllTabs.right++;
	rcAllTabs.bottom += 2;
	InvalidateRect (rcAllTabs, FALSE);

	return TRUE;
}

// =========================================================
// EnableTab
//
// used to enable/disable specific tabs
// =========================================================
void TPropertyTab::EnableTab (int nTab, BOOL bEnable)
{
	PRECONDITIONX (nTab >= 0 && nTab < pages.GetItemsInContainer (), "EnabledTab: Invalid index");

	if (IsTabEnabled (nTab) == bEnable)
	{
		// nothing to do
		return;
	}

	pages[nTab].EnableTab (bEnable);

	TRect	rcTab;

	GetTabRect (rcTab, nTab);
	rcTab.right++;
	rcTab.bottom += 2;

	if (styleTabs & Tab::Collapsing)
	{
		CalculateBasicTabWidths ();
		AssessScroll ();

		TRect	client;

		SetRect (client);

		rcTab.right = client.right;

		if (!AdjustOrigin (nSelect))
		{
			AssessScroll ();

			if (styleTabs & Tab::Justified)
			{
				ReassessTabWidths ();
			}

			InvalidateRect (rcTab, TRUE);
		}
	}
	else
	{
		InvalidateRect (rcTab);
	}
}

// =========================================================
// EvSize
//
// called when a WM_SIZE message is received.
// resizes all page dialogs to fit correctly in the tab area
// =========================================================
void TPropertyTab::EvSize (UINT sizeType, TSize& size)
{
	TControl::EvSize (sizeType, size);

	//
	// calculate all tab widths
	//
	CalculateTabWidths ();
	AssessScroll ();

	int		nCount = GetEntryCount ();

	rcSize.Set (1, (nTabHeight * nRows) + 1, size.cx - 2, size.cy - 2);

	//
	// calculate all tab widths
	//
	for (int n = 0; n < nCount; n++)
	{
		pages[n].GetDialog ()->SetWindowPos (NULL, rcSize, SWP_NOZORDER);
	}
}

// =========================================================
// SetPageFocus
//
// sets the focus to the first valid control on a page
// =========================================================
void TPropertyTab::SetPageFocus ()
{
	if ((styleTabs & Tab::CreateOnDemand) && !pages[nSelect].GetDialog ()->HWindow)
	{
		pages[nSelect].GetDialog ()->Create ();
		pages[nSelect].GetDialog ()->SetWindowPos (NULL, rcSize, SWP_NOZORDER);
	}

	// get the focus back onto the tabs
	if (IsWindowVisible () && GetFocus () != HWindow)
	{
		SetFocus ();
	}

	pages[nSelect].GetDialog ()->Show (SW_SHOW);

	return;

/*  // this code prevents the focus being set back to the
	// first control of the page-dialog

	HWND	hwndChild 	= ::GetWindow (pages[nSelect].GetDialog ()->HWindow, GW_CHILD);

	if (hwndChild)
	{
		HWND	hwndLast	= ::GetWindow (hwndChild, GW_HWNDLAST);

		if (hwndLast)
		{
			HWND	hwndTab = pages[nSelect].GetDialog ()->GetNextDlgTabItem (hwndLast);

			if (hwndTab)
			{
				pages[nSelect].GetDialog ()->PostMessage (WM_NEXTDLGCTL, (WPARAM) hwndTab, TRUE);
			}
		}
	}	*/
}

// =========================================================
// AssessScroll
//
// works out whether to display scroll arrows or not
// =========================================================
BOOL TPropertyTab::AssessScroll ()
{
	if (styleTabs & Tab::Stacked)
	{
		xMax = GetClientRect ().right;

		return FALSE;
	}

	TRect	rcTab;
	TRect	rcScroll;
	BOOL	bHadLeftScroll	= bLeftScroll;
	BOOL	bHadRightScroll	= bRightScroll;
	BOOL	bEitherScroll;
	TRect	rcTabs 	= GetClientRect ();
	int		nTabs;

	// should I show the left scroll button ?
	bLeftScroll		= nTabOrigin > 0;

	// work out the last selectable tab
	nTabs = GetLastTab (FALSE);

	// should I show the right scroll button ?
	if (bLeftScroll)
	{
		SetRect (rcTabs);
	}
	GetTabRect (rcTab, nTabs);
	bRightScroll	= rcTab.right > rcTabs.right;

	bEitherScroll	= (bLeftScroll | bRightScroll);

	// get tab area of rightmost VISIBLE tab
	nTabs	= GetEntryCount ();

	while (--nTabs >= 0)
	{
		if (GetTabRect (rcTab, nTabs, 0))
		{
			break;
		}
	}

	// get tab area of rightmost VISIBLE tab
	GetScrollRect (rcScroll);

	xMax			= bEitherScroll ? rcScroll.left - SCROLLMARGIN : GetClientRect ().right;

	if (bEitherScroll)
	{
		// load the left-scroll bitmap
		if (bLeftScroll != bHadLeftScroll || bRightScroll)
		{
			if (bmpLeftArrow)
			{
				delete bmpLeftArrow;
			}

			bmpLeftArrow	= new TBitmap (::LoadBitmap (NULL,
								MAKEINTRESOURCE (bLeftScroll ? OBM_LFARROW : OBM_LFARROWI)),
								AutoDelete);
		}

		// load the right-scroll bitmap
		if (bRightScroll != bHadRightScroll || bLeftScroll)
		{
			if (bmpRightArrow)
			{
				delete bmpRightArrow;
			}

			bmpRightArrow 	= new TBitmap (::LoadBitmap (NULL,
								MAKEINTRESOURCE (bRightScroll ? OBM_RGARROW : OBM_RGARROWI)),
								AutoDelete);
		}
	}
	else
	{
		delete bmpLeftArrow;
		bmpLeftArrow	= NULL;

		delete bmpRightArrow;
		bmpRightArrow 	= NULL;
	}

	// get Windows to redraw the scroll buttons
	if (bLeftScroll != bHadLeftScroll || bRightScroll != bHadRightScroll)
	{
		GetScrollRect (rcScroll);
		rcScroll.top = 0;
		rcScroll.left -= SCROLLMARGIN;
		InvalidateRect (rcScroll);

		ReassessTabWidths ();
	}

	return bEitherScroll;
}

void TPropertyTab::NextOrigin (int nDir)
{
	int	nNew = nTabOrigin;

	do
	{
		nNew += nDir;

		if (nNew >= GetEntryCount () - 1)
		{
			nNew = GetEntryCount () - 1;
			break;
		}

		if (nNew <= 0)
		{
			nNew = 0;

			break;
		}
	}
	while (!IsTabEnabled (nNew));

	if (nNew == nTabOrigin)
	{
		return;
	}

	TRect	rcInvalid;

	SetRect (rcInvalid);
	rcInvalid.bottom += 2;
	InvalidateRect (rcInvalid);

	nTabOrigin = nNew;
	AssessScroll ();

	UpdateWindow ();
}

// =========================================================
// GetScrollRect
//
// get the area of the two scroll buttons
// =========================================================
void TPropertyTab::GetScrollRect (TRect& rcTabs, int nButton)
{
	GetClientRect (rcTabs);

	rcTabs.bottom = nTabHeight;

	rcTabs.top 	= rcTabs.bottom - GetSystemMetrics (SM_CYHSCROLL);
	rcTabs.left = rcTabs.right  - GetSystemMetrics (SM_CXHSCROLL);

	// nButton can be:
	//	0 	- left button
	//	1 	- right button
	//	-1 	- both buttons (default)
	if (nButton != 1)
	{
		if (nButton == 0)
		{
			rcTabs.right = rcTabs.left;
		}

		rcTabs.left -= GetSystemMetrics (SM_CXHSCROLL) - 1;
	}
}

// =========================================================
// EvSetFocus/EvKillFocus
//
// gaining/losing focus event handlers
// =========================================================
void TPropertyTab::EvSetFocus (HWND hwndLostFocus)
{
	TControl::EvSetFocus (hwndLostFocus);

	TRect	rcTab;

	GetTabRect (rcTab, nSelect);
	rcTab.right++;
	rcTab.bottom += 2;

	InvalidateRect (rcTab, FALSE);
}

void TPropertyTab::EvKillFocus (HWND hwndGetFocus)
{
	TControl::EvSetFocus (hwndGetFocus);

	TRect	rcTab;

	GetTabRect (rcTab, nSelect);
	rcTab.right++;
	rcTab.bottom += 2;

	InvalidateRect (rcTab, FALSE);
}

// =========================================================
// EvKeyDown
//
// cursor key handler
// =========================================================
void TPropertyTab::EvKeyDown (UINT key, UINT repeatCount, UINT flags)
{
	switch (key)
	{
		case VK_LEFT :
		case VK_UP :
		case VK_PRIOR :
			SelectNext (-1);
			break;

		case VK_RIGHT :
		case VK_DOWN :
		case VK_NEXT :
			SelectNext (1);
			break;

		default:
			TControl::EvChar (key, repeatCount, flags);
			break;
	}
}

// =========================================================
// EvGetDlgCode
//
// tells dialog box we want to receive cursor keys
// =========================================================
UINT TPropertyTab::EvGetDlgCode (MSG far *msg)
{
	UINT	uiDefault = TControl::EvGetDlgCode (msg);

	return uiDefault | DLGC_WANTARROWS;
}

// =========================================================
// SetActiveColour
//
// defines the colour of the active tab
// (only if using Tab::ColorActive style)
// =========================================================
void TPropertyTab::SetActiveColor (TColor color)
{
	delete ppenActive;
	ppenActive = new TPen (color);

	if (HWindow)
	{
		TRect	rcTab;

		if (GetTabRect (rcTab, nSelect))
		{
			InvalidateRect (rcTab);
		}
	}
}

// =========================================================
// SetFixedTabWidth
//
// defines the width of tabs when using "FixedWidth" style
// =========================================================
void TPropertyTab::SetFixedTabWidth (int nWidth)
{
	if (nWidth)
	{
		nFixedTabWidth = nWidth;
	}
	else
	{
		nFixedTabWidth = DEFTABWIDTH;
	}

	if (HWindow)
	{
		CalculateTabWidths ();

		TRect	rcTabs;

		SetRect (rcTabs);
		InvalidateRect (rcTabs);
	}
}

// =========================================================
// SetWideMarginWidth
//
// defines the extra space added to the each side of a tab
// defined with the "WideMargin" style
// =========================================================
void TPropertyTab::SetWideMarginWidth (int nWidth)
{
	nWideMarginWidth = nWidth;

	if (HWindow)
	{
		CalculateTabWidths ();

		TRect	rcTabs;

		SetRect (rcTabs);
		InvalidateRect (rcTabs);
	}
}

// =========================================================
// TabColorChange
//
// picks up colors to use from Control Panel
// =========================================================
void TPropertyTab::TabColorChange ()
{
	// set up system colors
	if (GetApplication ()->Ctl3dEnabled () ||
		GetApplication ()->BWCCEnabled ())
	{
		colorWhite  = GetSysColor (COLOR_BTNHIGHLIGHT);
		colorLtGray = GetSysColor (COLOR_BTNFACE);
		colorDkGray = GetSysColor (COLOR_BTNSHADOW);
		colorBlack  = GetSysColor (COLOR_WINDOWFRAME);
	}
	else
	{
		colorWhite  = GetSysColor (COLOR_WINDOWFRAME);
		colorLtGray = GetSysColor (COLOR_WINDOW);
		colorDkGray = GetSysColor (COLOR_WINDOWFRAME);
		colorBlack  = GetSysColor (COLOR_WINDOWFRAME);
	}

	// if the window is already on-screen
	// then redraw it
	if (HWindow)
	{
		Invalidate ();
	}
}

// ======================================================= //

// =========================================================
// TPropertyEntry
//
// class which maintains details of a particular page
// =========================================================

TPropertyEntry::TPropertyEntry ()
{
	pPage		= NULL;
	bEnabled	= FALSE;
	nWidth		= 0;
}

TPropertyEntry::TPropertyEntry (string sTabTextV, TPropertyPage *pPageV, BOOL bEnabledV)
{
	sTabText	= sTabTextV;
	pPage		= pPageV;
	bEnabled	= bEnabledV;
	nWidth		= 0;
}

TPropertyEntry&	TPropertyEntry::operator= (const TPropertyEntry& copy)
{
	sTabText	= copy.sTabText;
	pPage		= copy.pPage;
	bEnabled	= copy.bEnabled;
	nWidth		= copy.nWidth;

	return *this;
}

BOOL TPropertyEntry::operator== (const TPropertyEntry&)
{
	// only use for "Find", which I don't use...
	return FALSE;
}

// =========================================================
// UseSmallFont
//
// recursively sends WM_SETFONT to all controls in a dialog
// =========================================================
void UseSmallFont (HWND HWindow)
{
	// use small font for all statics EXCEPT SS_NOPREFIX ones (titles)
	for (HWND hwndChild = ::GetWindow (HWindow, GW_CHILD); hwndChild; )
	{
		HWND 	hwndNext = ::GetWindow(hwndChild, GW_HWNDNEXT);

		::SendMessage (hwndChild, WM_SETFONT, (WPARAM) GetStockObject (ANSI_VAR_FONT), NULL);

		UseSmallFont (hwndChild);

		hwndChild = hwndNext;
	}
}

// =========================================================
// GetNextTabKey
//
// locate first/last tab position
// =========================================================
static HWND GetEndTabPosition (TWindow *window, BOOL bFirst)
{
	// identify last control
	HWND	hwndTab = window->GetNextDlgTabItem (::GetWindow (window->HWindow, GW_CHILD), TRUE);

	if (hwndTab && bFirst)
	{
		// identify first control
		hwndTab = window->GetNextDlgTabItem (hwndTab);
	}

	if (hwndTab)
	{
		//
		// just to make sure it really does have the tabstop style
		//
		if (!(GetWindowLong (hwndTab, GWL_STYLE) & WS_TABSTOP))
		{
			return NULL;
		}
	}

	return hwndTab;
}

// =========================================================
// ProcessTabKey
//
// perform tab-key navigation
// =========================================================
static BOOL ProcessTabKey (TWindow *window, HWND hwndFocus, BOOL bForwards)
{
	TPropertyDialog	*dialog;
	TPropertyPage	*page;
	HWND			hwndLimit;

	// see if it is on the page
	page = dynamic_cast<TPropertyPage *> (window);

	if (!page)
	{
		// see if it is on the page
		dialog = dynamic_cast<TPropertyDialog *> (window);
	}

	if (!page && !dialog)
	{
		// can't handle this as it isn't intended for us..
		return FALSE;
	}

	// identify last control
	hwndLimit = GetEndTabPosition (window, !bForwards);

	if (!hwndLimit)
	{
		// no tabbable items :-(
		return FALSE;
	}

	if (hwndFocus == hwndLimit)
	{
		TWindow	*newwindow;

		// select the destination window
		if (page)
		{
			newwindow = &page->GetPropertyDialog ();
		}
		else
		{
			if (!dialog->GetEntryCount ())
			{
				// no pages to check inside of!
				return FALSE;
			}

			newwindow = &dialog->GetPageRef ();
		}

		hwndLimit = GetEndTabPosition (newwindow, bForwards);

		if (hwndLimit)
		{
			window->PostMessage (WM_NEXTDLGCTL, NULL, TRUE);
			newwindow->PostMessage (WM_NEXTDLGCTL, (WPARAM) hwndLimit, TRUE);

			return TRUE;
		}
	}

	return FALSE;
}

// =========================================================
// PropdlgKbdProc
//
// keyboard hook used to perform tab-key navigation
// =========================================================
#include <owl\combobox.h>

LRESULT CALLBACK PropdlgKbdProc (int code, WPARAM wParam, LPARAM lParam)
{
	BOOL result = FALSE;

	if (code >= 0 && wParam == VK_TAB)
	{
		if (GetKeyState (VK_TAB) < 0 && GetKeyState (VK_MENU) == 0)
		{
			BOOL 	bForwards 	= !(GetKeyState (VK_SHIFT) < 0);
			HWND	hwndFocus	= GetFocus ();
			TWindow *window		= GetWindowPtr (GetParent (hwndFocus));

			// if the focus is in the listbox/edit window of a combobox,
			// use the parent of the combobox
			if (window)
			{
				if (dynamic_cast<TComboBox *> (window))
				{
					hwndFocus = window->HWindow;

					window = window->Parent;
				}
			}
			else
			{
				char	szClassName[16];

				GetClassName (GetParent (hwndFocus), szClassName, sizeof (szClassName));

				if (lstrcmpi (szClassName, "combobox") == 0)
				{
					hwndFocus = GetParent (hwndFocus);

					window = GetWindowPtr (GetParent (hwndFocus));
				}

			}

			if (window)
			{
				result = ProcessTabKey (window, hwndFocus, bForwards);
			}
		}
	}

	CallNextHookEx (hookKbd, code, wParam, lParam);

	return result;
}

// =========================================================
// GetPressedKey
//
// get the ASCII code of the currently depressed key
// =========================================================
static char GetPressedKey ()
{
	BYTE	byStates[256];
	char	cKey;

	GetKeyboardState (byStates);

	for (int n = 0; n <= 255; n++)
	{
	#ifdef __WIN32__
		WORD	dwKey;
	#else
		DWORD	dwKey;
	#endif

		if (n != VK_MENU && (byStates [n] & 0x80))	// if this key is down
		{
			if (ToAscii (n, NULL, byStates, &dwKey, TRUE))
			{
				cKey = (char) (dwKey & 0xFF);
				cKey = LOWORD ((DWORD) AnsiUpper ((LPSTR) MAKELPARAM (cKey, NULL)));

				return cKey;
			}
		}
	}

	return NULL;
}

// =========================================================
// GetPressedKey
//
// get the ASCII code of the currently depressed key
// =========================================================
static BOOL MatchesHotKey (LPCSTR lpszPrefix, char cKey)
{
	for (;;)
	{
		// locate the prefix character
		lpszPrefix = strchr (lpszPrefix, '&');

		// if found, and not a '&&' sequence
		if (lpszPrefix == NULL || *(++lpszPrefix) != '&')
		{
			break;
		}

		lpszPrefix++;	// skip the second '&'
	}

	if (lpszPrefix != NULL)
	{
		char	cChar = *lpszPrefix;

		cChar = LOWORD ((DWORD) AnsiUpper ((LPSTR) MAKELPARAM (cChar, NULL)));

		if (cChar == cKey)
		{
			return TRUE;
		}
	}

	return FALSE;
}

