/*****************************************************************************
*   Program to draw 3D object as wireframe from	any ortographic view	     *
* Options:								     *
* 1. -c : Object is closed. If object is closed each edge is shared by two   *
*         polygons and therefore can bedrawn only once.			     *
* 2. -e #Edges : If the edges are order is specific way (like in DrawFn3D)   *
*         and only the k first edges of each polygons are to be displayed    *
*	  use this option as -e k.					     *
* 3. -i : Internal edges. IRIT solid modeller may generate edges, which one  *
*	  may not want to see (default). -i will draw all of them.	     *
* 4. -m : More flag, to print more imformation on input/errors.		     *
*         Note all messages goes to STDOUT (not stderr!) as this program     *
*	  works in graphic mode, and we can redirect stdout to a file.       *
* 5. -n : Draw vertices normals if the data has them, otherwise ignore.      *
* 6. -o Objects : list of objects to be displayed from the input data files. *
*	  This option should not be last and Object list terminates with     *
*	  next option (- as first char). ALL objects are displayed otherwise.*
* 7. -v ViewFile : optional view to start with.				     *
* 8. -z : Print current version, and some helpfull data.		     *
*									     *
* Note some of those options way be permanently set to a different default   *
* using the configuration file "Poly3D.cfg"				     *
*									     *
* Usage:								     *
*   Poly3D [-c] [-m] [-i] [-e #Edges] [-n] [-N] [-o Objects] [-v ViewFile]   *
*								[-z] Files   *
*									     *
* Written by:  Gershon Elber				Ver 2.2, Apr 1990    *
*****************************************************************************/

#ifdef __MSDOS__
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <graphics.h>
#include <alloc.h>
#endif /* __MSDOS__ */

#include <stdio.h>
#include <string.h>
#include <time.h>
#include "program.h"
#include "getarg.h"
#include "genmat.h"
#include "parser.h"
#include "graphgng.h"
#include "viewobjg.h"
#include "config.h"

#ifdef __MSDOS__
#include "mousedrv.h"
#include "matherr.h"
#endif /* __MSDOS__ */

#ifdef __MSDOS__
extern unsigned int _stklen = 32766;	     /* Increase default stack size. */
#endif /* __MSDOS__ */

#ifdef SYSV
static char *VersionStr =
	"Poly3D		version 2.2,	Gershon Elber,\n\
	 (C) Copyright 1989 Gershon Elber, Non commercial use only.";
#else
static char *VersionStr = "Poly3D	version 2.2,	Gershon Elber,	"
		__DATE__ ",   " __TIME__ "\n"
		"(C) Copyright 1989 Gershon Elber, Non commercial use only.";
#endif /* SYSV */

static char *CtrlStr =
	"poly3d c%- m%- i%- e%-#Edges!d n%- N%- o%-Objects!*s v%-ViewFile!s z%- DFiles!*s";
static BinTree *FreeBinTreeList	= NULL;	    /* List of freed bin tree nodes. */

static long SaveTotalTime;

MatrixType GlblViewMat;				  /* Current view of object. */
MatrixType GlblPerspMat =	     /* Perspective optional transformation. */
    {	{ 1.0,   0.0,   0.0,   0.0 },
	{ 0.0,   1.0,   0.0,   0.0 },
	{ 0.0,   0.0,   0.1,  -0.35 },
	{ 0.0,   0.0,   0.35,  1.0 }
    };
MatrixType CrntViewMat;				/* This is the current view! */
int GlblTransformMode = TRANS_SCREEN; /* Screen, Object coords. trans. mode. */
int GlblViewMode = VIEW_ORTHOGRAPHIC;	   /* Perspective, Orthographic etc. */
int GlblDepthCue = TRUE;			   /* Activate depth cueing. */
int GlblNormalLen = NORMAL_DEFAULT_LENGTH;   /* Scaler for normals if drawn. */

/* The following are setable variables (via configuration file poly3d.cfg).  */
#ifdef __MSDOS__
int MouseExists = FALSE, GraphDriver = DETECT;
#else
int MouseExists = TRUE;
#endif /* __MSDOS__ */

int InterFlag = FALSE, MoreFlag = FALSE, TextFlag = FALSE, NumEdges = 0,
    DrawVNormalsFlag = FALSE, DrawPNormalsFlag = FALSE, ViewFlag = FALSE,
    ClosedObject = FALSE;

static ConfigStruct SetUp[] =
{
#ifdef __MSDOS__
  { "Mouse",		(VoidPtr) &MouseExists,		SU_BOOLEAN_TYPE },
  { "GraphDriver",	(VoidPtr) &GraphDriver,		SU_INTEGER_TYPE },
#endif /* __MSDOS__ */
  { "ViewMode",		(VoidPtr) &GlblViewMode,	SU_INTEGER_TYPE },
  { "TransMode",	(VoidPtr) &GlblTransformMode,	SU_INTEGER_TYPE },
  { "ClosedObject",	(VoidPtr) &ClosedObject,	SU_BOOLEAN_TYPE },
  { "DepthCue",		(VoidPtr) &GlblDepthCue,	SU_BOOLEAN_TYPE },
  { "Internal",		(VoidPtr) &InterFlag,		SU_BOOLEAN_TYPE },
  { "More",		(VoidPtr) &MoreFlag,		SU_BOOLEAN_TYPE },
  { "NormalLength",	(VoidPtr) &GlblNormalLen,	SU_INTEGER_TYPE },
  { "NumOfEdges",	(VoidPtr) &NumEdges,		SU_INTEGER_TYPE } };

#define NUM_SET_UP	(sizeof(SetUp) / sizeof(ConfigStruct))

static void MainGetViewFile(char *ViewFileName);
static FileDescription **MainGetDataFiles(char **DataFileNames,
							int NumOfDataFiles);
static void FreeBinTree(BinTree *Pbin);

/*****************************************************************************
* Main routine - Read Parameter	line and do what you need...		     *
*****************************************************************************/
void main(int argc, char **argv)
{
    int EdgesFlag = 0, ObjFlag = 0, NumObjs = 0, VerFlag = 0, NumFiles = 0,
	Error;
    char **Objects = NULL, *ViewName = NULL, **FileNames = NULL, *ErrorMsg;
    FileDescription **FD;

    SaveTotalTime = time(NULL);

#ifdef __MSDOS__
    ctrlbrk((int (*)()) MyExit);		      /* Kill process if ^C. */
#endif /* __MSDOS__ */

#ifdef __MSDOS__
    MathErrorSetUp(ME_KILL, NULL);     /* Kill process if math error occurs! */

    MouseExists = MouseDetect();	   /* Auto test for mouse existance. */
#endif /* __MSDOS__ */

    Config("poly3d", SetUp, NUM_SET_UP);     /* Read config. file if exists. */

    if ((Error = GAGetArgs (argc, argv, CtrlStr,
		&ClosedObject, &MoreFlag, &InterFlag, &EdgesFlag,
		&NumEdges, &DrawVNormalsFlag, &DrawPNormalsFlag, &ObjFlag,
		&NumObjs, &Objects, &ViewFlag, &ViewName,
		&VerFlag, &NumFiles, &FileNames)) != 0) {
	GAPrintErrMsg(Error);
	GAPrintHowTo(CtrlStr);
	MyExit(1);
    }

    if (VerFlag) {
	printf("\n%s\n\n", VersionStr);
	GAPrintHowTo(CtrlStr);
	ConfigPrint(SetUp, NUM_SET_UP);
	MyExit(0);
    }

    if (!NumFiles) {
	fprintf(stderr, "No data file names where given, exit\n");
	GAPrintHowTo(CtrlStr);
	MyExit(1);
    }

    MainGetViewFile(ViewName);			       /* Into GlblViewMat. */

    /* Get the data files: */
    FD = MainGetDataFiles(FileNames, NumFiles);

#ifdef __MSDOS__
    GGInitGraph();			     /* Initiate the graphic driver. */
#else
    GGInitGraph(argc, argv);		     /* Initiate the graphic driver. */
#endif /* __MSDOS__ */

#ifdef __MSDOS__
    if (MouseExists)
	if ((ErrorMsg = MouseInit()) != 0) {
	    /* Must be called before any usage (but after InitGraph routine)!*/
	    printf("\n%s\n\n\tPress any key to continue:", ErrorMsg);
	    MouseExists = FALSE;
	    getch();
	}
#endif /* __MSDOS__ */

    InteractGeomObject(FD, NumObjs, Objects);

    MyExit(0);
}

/*****************************************************************************
* Main routine to read the view	description file:			     *
*****************************************************************************/
static void MainGetViewFile(char *ViewFileName)
{
    FILE *f;

    if (ViewFlag) {
	f = fopen(ViewFileName, "rt");			   /* Open the file. */
	if (!f)
	{
	     printf("Can't open view file %s\n", ViewFileName);
	     MyExit(1);
	}
    }

    GetViewFile(f, ViewFlag);	    /* Get the view data - into GlblViewMat. */

    if (ViewFlag) fclose(f);	 			  /* Close the file. */
}

/*****************************************************************************
* Main routine to read the data	description files:			     *
* Returns pointer to pointers on FileDescription structures (one per file).  *
*****************************************************************************/
static FileDescription **MainGetDataFiles(char **DataFileNames,
							int NumOfDataFiles)
{
    int	i;
    FILE *f;
    FileDescription **FDbase, **FDinc; /* Used to save memory pointers loc. */

    FDbase = FDinc = (FileDescription **)   /* Allocate the pointers block. */
	 MyMalloc((unsigned) sizeof(FileDescription *) * (NumOfDataFiles + 1));

    for	(i=0; i<NumOfDataFiles;	i++) {
	if (MoreFlag) printf("Reading data file %s\n", *DataFileNames);
	f = fopen(*DataFileNames, "rt");		  /* Open the file. */
	if (!f)
	{
	    printf("Can't open data file %s\n", *DataFileNames);
	    MyExit(1);
	}

	*FDinc = GetDataFile(f);		      /* Get the data file. */
	FreeBinTree((*FDinc) ->	VertexPointer);/* We dont need these trees. */
	FreeBinTree((*FDinc) ->	PolygonPointer); /* objects are all loaded. */
	(*FDinc) -> VertexPointer = (*FDinc) ->	PolygonPointer =
							(BinTree *) NULL;
	*FDinc++;

	(void) fclose(f);				 /* Close the file. */

	DataFileNames++;			 /* Skip to next file name. */
    }

    *FDinc = (FileDescription *) NULL;
    return FDbase;
}

/*****************************************************************************
* My Routine to	allocate dynamic memory. All program requests must call this *
* routine (no direct call to malloc). Dies if no memory.		     *
*****************************************************************************/
char *MyMalloc(unsigned size)
{
    static int Count = 0;
    char *p;
    BinTree *Pbin;

    if ((size == sizeof(BinTree)) && (FreeBinTreeList))	{
	Pbin = FreeBinTreeList;
	FreeBinTreeList	= FreeBinTreeList -> right;
	return (char *)	Pbin;
    }

#ifdef __MSDOS__
    if (Count++ == 50)
    {
	Count = 0;
	printf("Core left: %ldk   \r", coreleft() / 1024);
    }
#endif /* __MSDOS__ */

    if ((p = malloc(size)) != NULL) return p;

    printf("Not enough memory, exit\n");
    MyExit(2);

    return NULL;				    /* Make warnings silent. */
}

/*****************************************************************************
* Routine to free a binary tree, which is not needed any more.		     *
* Saves	the freed nodes	in a linear list to be used by malloc.		     *
*****************************************************************************/
static void FreeBinTree(BinTree *Pbin)
{
    if (Pbin) {
	FreeBinTree(Pbin -> right);
	FreeBinTree(Pbin -> left);
	Pbin ->	right =	FreeBinTreeList;
	FreeBinTreeList	= Pbin;
    }
}

/*****************************************************************************
* MyExit routine. Note it might call to CloseGraph without calling	     *
* InitGraph(), or call MouseClose() without MouseInit() etc. and it is the   *
* responsibility of the individual modules to do nothing in these cases.     *
*****************************************************************************/
void MyExit(int ExitCode)
{
    GGCloseGraph();				/* Close the graphic driver. */
#ifdef __MSDOS__
    MouseClose();				/* Disable mouse interrupts. */
#endif /* __MSDOS__ */

#ifdef __MSDOS__
    fprintf(stderr,
	    "\nPoly3D: Total RealTime %ld seconds, Core left %ldk.\n",
	    time(NULL) - SaveTotalTime, coreleft() / 1024);
#else
    fprintf(stderr,
	    "\nPoly3D: Total RealTime %ld seconds.\n",
	    time(NULL) - SaveTotalTime);
#endif /* __MSDOS__ */

    exit(ExitCode);
}
